package Bio::MUST::Core::SeqId;
# ABSTRACT: Modern and legacy MUST-compliant sequence id
# CONTRIBUTOR: Mick VAN VLIERBERGHE <mvanvlierberghe@doct.uliege.be>
$Bio::MUST::Core::SeqId::VERSION = '0.251140';
use Moose;
use namespace::autoclean;

# AUTOGENERATED CODE! DO NOT MODIFY THIS FILE!

use autodie;
use feature qw(say);

# use Smart::Comments;

use Carp;
use Const::Fast;
use Regexp::Common;

use Bio::MUST::Core::Types;
use Bio::MUST::Core::Constants qw(:ncbi :seqids);
use aliased 'Bio::MUST::Core::SeqId::Filter';


has 'full_id' => (
    is       => 'ro',
    isa      => 'Str',
    required => 1,
    writer   => '_set_full_id',
);


has 'is_' . $_ => (
    traits   => ['Bool'],
    is       => 'ro',
    isa      => 'Bool',
    init_arg => undef,
    default  => 0,
    handles  => {
        '_set_' . $_ => 'set',
    },
) for qw(foreign new);


has $_ => (
    is       => 'ro',
    isa      => 'Maybe[Str]',
    init_arg => undef,
    default  => undef,
    writer   => '_set_' . $_,
) for qw(family tag genus species strain accession tail
         taxon_id gca gca_novers gca_vers gca_prefix gca_number
         gi database identifier);


# array of parts from NCBI FASTA-style GNL ids
# Note: all_parts accessor returns an empty list if undef
has 'parts' => (
    traits   => ['Array'],
    is       => 'ro',
    isa      => 'ArrayRef[Str]',
    init_arg => undef,
    default  => sub { [] },
    writer   => '_set_parts',
    handles  => {
        count_parts => 'count',
          all_parts => 'elements',
          get_part  => 'get',
    },
);

# file-scope constants
# TODO: consider forbidding '|' in full_ids

# regexes for full_id (legacy and new) components
const my $FAMILY     => qr{ # underscore allowed
    [^\-\s@\#]+?            # single word without dash, space, '@', '#'
}xms;

const my $TAG        => qr{
    [a-z]{1,6}              # 1-6 letter(s) (lowercase)
}xms;

# TODO: add accessor methods to analyse this
# this is feasible with a switch table with regexes to generate methods
# return 0/1/undef

# list of standard tags:
#     c: contaminant
#     d: divergent
#     u: unknown taxonomy
#     cpcpt [cp]: plastid-encoded-plastid-targeted
#     nucpt: nuclear-encoded-and-plastid-targeted
#     nuppct: nuclear-encoded-periplastid-compartment-targeted
#     nm: nucleomorph-encoded
#     mt: mitochondrion-encoded

const my $GENUS      => qr{ # underscore and dash allowed
    [^\s@]+?                # single word without space, '@'
}xms;

const my $SPECIES    => qr{ # dash allowed
    [^_\s@]+?               # single word without underscore, space, '@'
}xms;

const my $STRAIN     => qr{ # as genus
    $GENUS
}xms;

const my $ACCESSION  => qr{
    $GENUS                  # as genus
}xms;

const my $TAIL       => qr{
    $GENUS                  # as genus
}xms;

# regex for parsing (valid) full_id
const my $FULL_ID    => qr{
    \A                      # begins with...
    (?: ($FAMILY)           # optional family
    - )?                    #   followed by a literal dash (-)
    (?: ($TAG)              # optional tag
    \# )?                   #   followed by a literal number sign (#)
    ($GENUS)                # genus
    \s+                     # whitespace
    ($SPECIES)              # species
    (?: _                   # optional literal underscore (_)
    ($STRAIN) )?            #   followed by strain
    @                       # literal at sign (@)
    ($ACCESSION)            # accession
    (?: \.{3}               # optional literal ellipsis (...)
    ($TAIL) )?              #   followed by a single-word tail
    ($NEW_TAG)?             # optional #NEW# tag
    \z                      # ... at the end
}xms;

# species regex for genus-only ids
const my $GENUSONLY  => qr{
    \A                      # nothing before!
    (?:
    sspp | ssp | spp | sp   # either species (sp), species pluralis (spp),
    )                       # subspecies (ssp) or subspecies pluralis (sspp)
    \.?                     # optionally followed by a dot
    \z                      # and nothing after!
}xms;

# tag regex for doubtful organisms (from a taxonomic POV)
const my $DOUBTFUL   => qr{
    \A
    [cdu]
    \z
}xms;

# regex for parsing taxonomy-enabled abbreviated ids
const my $TAXABBR_ID => qr{
    \A                      # begins with...
    ($NCBIPKEY|$NCBIGCA)    # NCBI primary key (= taxid) or NCBI GCA/GCF
    \|                      # literal pipe sign (|)
    ($ACCESSION)            # accession
    \z                      # ... at the end
}xms;

# regex for parsing NCBI FASTA-style GI ids
# TODO: merge with $GI_ID in Constants?
# TODO: improve handling of partial ids (e.g., gi|404160475| would fail)
const my $NCBIGI_ID => qr{
    \A                      # begins with...
    gi                      # literal 'gi'
    \|                      # literal pipe sign (|)
    ($NCBIPKEY)             # NCBI primary key (= GI number)
    (?: \|                  # optional literal pipe sign (|)
     $NCBIDBABBR            #   followed by NCBI database abbr (not captured)
    \|                      #   then literal pipe sign (|)
    ($NCBIACC) ) ?          #   then NCBI accession
                            # no end anchor to allow for additional stuff
}xms;

# regex for parsing NCBI FASTA-style GNL ids
# TODO: merge with $GNL_ID in Constants?
# Note: this regex actually returns the full_id for subsequent splitting
const my $NCBIGNL_ID => qr{
    \A                      # begins with...
    (gnl                    # literal 'gnl'
    (?: \|                  #   literal pipe sign (|)
    $NCBIPART)              #   NCBI part (very accommodating)
    {2,} )                  # at least twice but allowing for more
}xms;

BEGIN{

no warnings "qw";

# static lists of hyphenated genera and underscored species
# Note: in BEGIN block because the hash must be built before any call to BUILD

const my @GENERA  => qw(
    0-14-0-10-38-17
    0-14-0-20-30-16
    0-14-0-20-34-12
    0-14-0-20-40-13
    0-14-0-20-44-26
    0-14-0-20-45-22
    0-14-0-20-51-18
    0-14-0-80-44-23
    0-14-0-80-59-12
    0-14-0-80-60-11
    0-14-3-00-41-53
    0-14-3-00-62-12
    01-FULL-36-10b
    01-FULL-36-15b
    01-FULL-38-12b
    01-FULL-39-12b
    01-FULL-43-11b
    01-FULL-43-15b
    01-FULL-44-24b
    01-FULL-45-10b
    01-FULL-45-15b
    01-FULL-45-25b
    01-FULL-45-34b
    01-FULL-48-32b
    01-FULL-49-22b
    01-FULL-54-110
    02-12-FULL-59-9
    02-FULL-45-10c
    02-FULL-45-11b
    02-FULL-45-17b
    1,4-dioxane-degrading
    1-14-0-10-31-34
    1-14-0-10-32-24
    1-14-0-10-34-10
    1-14-0-10-34-76
    1-14-0-10-36-11
    1-14-0-10-36-16
    1-14-0-10-36-19
    1-14-0-10-36-32
    1-14-0-10-37-12
    1-14-0-10-37-14
    1-14-0-10-40-38
    1-14-0-10-41-10
    1-14-0-10-41-12
    1-14-0-10-41-28
    1-14-0-10-42-12
    1-14-0-10-42-19
    1-14-0-10-42-22
    1-14-0-10-43-10
    1-14-0-10-43-11
    1-14-0-10-43-13
    1-14-0-10-43-42
    1-14-0-10-44-13
    1-14-0-10-44-23
    1-14-0-10-45-20
    1-14-0-10-45-29
    1-14-0-10-45-34
    1-14-0-10-46-23
    1-14-0-10-46-36
    1-14-0-10-47-10
    1-14-0-10-47-16
    1-14-0-10-48-11
    1-14-0-10-49-15
    1-14-0-10-49-17
    1-14-0-10-51-16
    1-14-0-10-55-24
    1-14-0-10-56-10
    1-14-0-20-35-14
    1-14-0-20-39-34
    1-14-0-20-39-49
    1-14-0-20-42-23
    1-14-0-20-45-16
    1-14-0-20-46-22
    1-14-0-20-47-16
    1-14-0-20-49-13
    1-14-0-20-64-10
    1-14-0-65-60-17
    1-14-2-50-31-20
    1-14-2-50-48-13
    1-19
    1-butene
    1-hexene
    1162T-S-0a-05
    12-FULL-43-12b
    12-FULL-44-12b
    12-FULL-45-19b
    12-FULL-67-14b
    13-1-40CM-64-14
    13-1-40CM-70-15
    13-2-20CM-66-19
    13-2-20CM-68-14
    14-2
    16-39-46
    16SrXXII-A
    16SrXXII-B
    172606-1
    17J80-11
    188UL27-1
    18H4-34
    1XD8-76
    2,3-D
    2,4-D
    2,4-D-degrading
    2-01-FULL-33-17
    2-01-FULL-36-15
    2-01-FULL-37-20
    2-01-FULL-38-13
    2-01-FULL-38-21
    2-01-FULL-38-26
    2-01-FULL-39-10
    2-01-FULL-39-10-A
    2-01-FULL-39-12
    2-01-FULL-39-13
    2-01-FULL-39-19
    2-01-FULL-39-21
    2-01-FULL-39-24
    2-01-FULL-39-28
    2-01-FULL-39-33
    2-01-FULL-40-13
    2-01-FULL-40-14
    2-01-FULL-40-24
    2-01-FULL-40-39
    2-01-FULL-40-42
    2-01-FULL-41-11
    2-01-FULL-41-12
    2-01-FULL-41-14
    2-01-FULL-41-18
    2-01-FULL-41-18-A
    2-01-FULL-41-21
    2-01-FULL-41-22
    2-01-FULL-41-26
    2-01-FULL-42-13
    2-01-FULL-42-15
    2-01-FULL-42-20
    2-01-FULL-43-14
    2-01-FULL-43-22
    2-01-FULL-43-38
    2-01-FULL-44-10
    2-01-FULL-44-11
    2-01-FULL-44-17
    2-01-FULL-44-21
    2-01-FULL-44-22
    2-01-FULL-45-10
    2-01-FULL-45-10-A
    2-01-FULL-45-10-B
    2-01-FULL-45-11
    2-01-FULL-45-18
    2-01-FULL-45-33
    2-01-FULL-46-13
    2-01-FULL-46-14
    2-01-FULL-46-14-A
    2-01-FULL-47-10
    2-01-FULL-47-10-A
    2-01-FULL-48-10
    2-01-FULL-48-14
    2-01-FULL-48-16
    2-01-FULL-49-10
    2-01-FULL-49-14
    2-01-FULL-49-17
    2-01-FULL-49-22
    2-01-FULL-50-11
    2-01-FULL-50-23
    2-01-FULL-50-24
    2-01-FULL-50-25
    2-01-FULL-50-28
    2-01-FULL-51-10
    2-01-FULL-51-21
    2-01-FULL-51-34
    2-01-FULL-52-15
    2-01-FULL-53-15
    2-01-FULL-56-20
    2-01-FULL-58-14
    2-01-FULL-59-12
    2-01-FULL-60-11
    2-01-FULL-63-20
    2-01-FULL-64-13
    2-01-FULL-66-17
    2-02-FULL-34-10
    2-02-FULL-34-12
    2-02-FULL-35-15
    2-02-FULL-36-13
    2-02-FULL-37-13
    2-02-FULL-38-14
    2-02-FULL-38-14-A
    2-02-FULL-38-15
    2-02-FULL-38-15-A
    2-02-FULL-39-11
    2-02-FULL-39-13
    2-02-FULL-39-14
    2-02-FULL-39-32
    2-02-FULL-40-10
    2-02-FULL-40-11
    2-02-FULL-40-12
    2-02-FULL-40-13
    2-02-FULL-40-20
    2-02-FULL-42-16
    2-02-FULL-42-43
    2-02-FULL-43-11
    2-02-FULL-43-11-A
    2-02-FULL-43-13
    2-02-FULL-43-16
    2-02-FULL-44-16
    2-02-FULL-44-35
    2-02-FULL-45-10
    2-02-FULL-45-11
    2-02-FULL-45-18
    2-02-FULL-45-21
    2-02-FULL-45-22
    2-02-FULL-45-22_A
    2-02-FULL-46-11
    2-02-FULL-46-13
    2-02-FULL-46-19
    2-02-FULL-47-12
    2-02-FULL-47-50
    2-02-FULL-48-11
    2-02-FULL-48-11-A
    2-02-FULL-48-12
    2-02-FULL-48-14
    2-02-FULL-48-16
    2-02-FULL-49-11
    2-02-FULL-49-12
    2-02-FULL-49-16
    2-02-FULL-50-16
    2-02-FULL-50-16-A
    2-02-FULL-50-9b
    2-02-FULL-51-10
    2-02-FULL-51-18
    2-02-FULL-52-10
    2-02-FULL-52-16
    2-02-FULL-52-23
    2-02-FULL-53-10
    2-02-FULL-55-14
    2-02-FULL-56-15
    2-02-FULL-56-16
    2-02-FULL-58-16
    2-02-FULL-61-13
    2-02-FULL-61-28
    2-02-FULL-62-14
    2-02-FULL-62-17
    2-02-FULL-63-19
    2-02-FULL-65-29
    2-02-FULL-66-14
    2-02-FULL-66-22
    2-02-FULL-67-20
    2-02-FULL-67-57
    2-12-FULL-35-11
    2-12-FULL-35-15
    2-12-FULL-35-23
    2-12-FULL-36-22
    2-12-FULL-37-11
    2-12-FULL-37-12
    2-12-FULL-37-14
    2-12-FULL-37-34
    2-12-FULL-37-7b
    2-12-FULL-38-22
    2-12-FULL-39-13
    2-12-FULL-40-10
    2-12-FULL-40-12
    2-12-FULL-40-28
    2-12-FULL-41-12
    2-12-FULL-41-13
    2-12-FULL-41-16
    2-12-FULL-41-17
    2-12-FULL-41-20
    2-12-FULL-42-10
    2-12-FULL-42-15
    2-12-FULL-43-11
    2-12-FULL-43-12
    2-12-FULL-43-28
    2-12-FULL-44-25
    2-12-FULL-45-10
    2-12-FULL-45-12
    2-12-FULL-45-22
    2-12-FULL-48-17
    2-12-FULL-49-11
    2-12-FULL-50-11
    2-12-FULL-50-11-A
    2-12-FULL-52-36
    2-12-FULL-53-21
    2-12-FULL-54-10
    2-12-FULL-55-11
    2-12-FULL-57-11
    2-12-FULL-57-22
    2-12-FULL-60-19
    2-12-FULL-60-25
    2-12-FULL-62-27
    2-12-FULL-64-23
    2-12-FULL-65-11
    2-12-FULL-66-21
    2-12-FULL-67-15
    2-12-FULL-68-20
    2-12-FULL-69-37
    20-14-0-80-47-9
    20-14-0-80-50-8
    20-14-all-45-10
    20-14-all-54-14
    20-60-12
    2011-GWA2-47-8b
    2011-GWC2-44-17
    2013-40CM-41-45
    20CM-2-55-15
    20CM-2-65-7
    20CM-2-70-11
    20CM-4-69-16
    21-14-0-10-32-9
    21-14-0-10-35-9
    21-14-0-10-42-9
    21-14-0-10-47-8
    21-14-0-10-47-8-A
    21-14-0-20-44-7
    21-14-0-20-46-8
    21-14-0-20-56-8
    21-14-0-20-63-9
    21-14-all-36-13
    21-14-all-37-9
    21-14-all-39-27
    21-14-all-40-8
    21-14-all-47-15
    21-14-all-47-17
    21-64-14
    229E-related
    28-57-27
    28-YEA-48
    3-1-20CM-2-70-9
    3-1-20CM-3-53-8
    3-1-20CM-4-69-9
    32-111
    33-17
    336-3
    34-128
    34-65-8
    38-11
    39-52-133
    4-nitrophenol
    40-19
    40-80
    40CM-2-53-6
    40CM-2-70-7
    40CM-3-55-5
    40CM-3-62-11
    40CM-4-65-16
    40CM-4-68-19
    40CM-4-69-5
    40CM-68-15
    41-12-T18
    41-269
    41T-STBD-0c-01a
    42-11
    42-54-T18
    43-108
    43-37
    4484-100
    4484-107
    4484-113
    4484-171
    4484-188
    4484-190-2
    4484-211
    4484-213
    4484-219
    4484-230
    4484-232
    4484-275
    4484-276
    4484-49
    4484-7
    4484-87
    4484-93
    4484-95
    4572-104
    4572-112
    4572-123
    4572-128
    4572-13
    4572-130
    4572-55
    4572-78
    46-47
    46-80
    47-G6
    4M-Z18
    50-218
    50-400-T64
    51-20
    54-19
    55-13
    58-81
    62-47
    629B-23
    63-22
    630D-37
    64-32
    65-79
    65-9
    66-26
    66-474
    67-14
    6H3-1
    70-18
    70-9
    73-13
    79-D21
    9FT-COMBO-42-15
    9FT-COMBO-48-23
    9FT-COMBO-53-11
    9FT-COMBO-55-16
    A-2
    A-MIC-10
    A05DMB-2
    AAA044-D11
    AAA071-K20
    AAA164-E04
    AAA240-E13
    AAA261-N23
    AAA300-D14
    AAA536-G10
    AABM5-125-24
    AABM5-25-91
    AaIE-18
    Aalborg-AAW-1
    AB-137-C09
    AB-539-J10
    AB-629-J13
    AB001-6
    AB1-6
    AC-14
    AC-16
    AC-21
    AC-25
    AC-32
    AC-35
    AC-37
    AC-38
    AC-48
    AC-49
    AC-50
    AC-51
    AC-52
    AC-55
    AC-56
    AC-64
    AC-67
    AC-69
    AC-9
    AcAMD-2
    AcAMD-5
    ACE-ARM1
    acetamiprid-degrading
    Activation-tagging
    AD8-1
    Adeno-Associated
    adeno-associated
    Adeno-associated
    AE-B3A
    AE-B3B
    AEGEAN-183
    AER2-1
    AF33-28
    AG-12
    AG-337-I02
    AG-339-G14
    AG-343-D04
    AG-363-A16
    AG-363-B04
    AG-363-J23
    AG-363-K07
    AG-363-N16
    AG-363-N20
    AG-363-O06
    AG-363-O15
    AG-363-P08
    AG-402-N21
    AG-409-A19
    AG-409-B05
    AG-409-B13
    AG-409-D09
    AG-409-G20
    AG-409-I11
    AG-409-J16
    AG-409-J19
    AG-409-L14
    AG-409-L18
    AG-409-M05
    AG-409-N21
    AG-409-O23
    AG-409-P01
    AG-409-P03
    AG-410-N11
    AG-414-E02
    AG-422-B15
    AG-426-I14
    AG-426-M11
    AG-430-B22
    AG-432-D11
    AG-463-P14
    Agave-associated
    Aids-associated
    Aksy-Durug
    AKYH767-A
    AL-N1
    alfalfa-associated
    Alfalfa-associated
    Alg239-R122
    Alg240-R148
    alk-system
    allo-octoploid
    Alpha-05
    Altai-like
    AM-0621
    AM-1111
    AM-2324
    Amalga-like
    AMARA-1
    ammonia-oxidizing
    aniline-degrading
    ANME-1
    ANME-1-THS
    ANME-2
    ANMV-1
    Anopheline-associated
    ANT-38
    ANT-B6
    antibiotic-resistant
    antimony-reducing
    AP-15
    AP-16
    AQU-01
    AQU-03
    AQU-09
    Araraquara-like
    Arctic95D-9
    Arctic96AD-7
    Ardenna-associated
    Argemone-Cactus
    Aristolochia-associated
    ARK-14
    ARK-15
    ARM-1
    Arrabida-like
    arsenate-reducing
    arsenic-metabolizing
    arsenic-oxidizing
    arsenite-oxidising
    arsenite-oxidizing
    Arthrobacter-like
    ARV-138
    ARV-176
    ASFV-like
    ASO4-4
    ASP10-02a
    ASP10-05a
    ATCC-39006
    ATX-6F1
    Aved-18
    Avian-like
    Avon-Heathcote
    AWTP1-13
    AWTP1-18
    AWTP1-30
    AWTP1-31
    AWTP1-35
    AWTP1-9
    Aydin-like
    B-17B0
    B-17BO
    B-1AR
    B-DKE
    B-lymphotropic
    B1-Br10-U2g19
    B1-Br10-U2g21
    B1-G15
    B1-G9
    B10-G16
    B10-G4
    B10-G9
    B109-G9
    B11-G15
    B11-G9
    B111-G9
    B118-G9
    B119-G9
    B12-G15
    B12-G16
    B12-G17
    B12-G4
    B120-G9
    B129-G9
    B13-G15
    B13-G4
    B130-G9
    B137-G9
    B14-G1
    B14-G17
    B14-G2
    B14-G9
    B140-G9
    B143-G9
    B15-G16
    B15-G2
    B15-G4
    B15-G9
    B16-G9
    B17-G15
    B17-G16
    B17-G17
    B17-G2
    B18-G2
    B18-G4
    B1SED10-159
    B1SED10-16
    B1Sed10-166
    B1SED10-173
    B1SED10-174
    B1Sed10-187
    B1SED10-225
    B1Sed10-231
    B1SED10-245
    B1Sed10-29
    B1SED10-74M
    B1SED10-75
    B2-G16
    B20-G17
    B21-G15
    B21-G16
    B21-G9
    B22-G1
    B22-G15
    B23-G16
    B24-2
    B24-G1
    B24-G6
    B24-G9
    B25-G16
    B26-1
    B26-G2
    B26-G9
    B28-G16
    B28-G17
    B29-G15
    B29-G16
    B3-B38
    B3-Chlor
    B3-G11
    B3-G13
    B3-G15
    B3-G16
    B3-LCP
    B3-TA06
    B3-WOES
    B30-G15
    B30-G16
    B30-G6
    B31-G17
    B32-G16
    B33-G15
    B33-G16
    B36-G15
    B39-G2
    B39-G9
    B4-G1
    B40-G2
    B44-G15
    B46-G15
    B46-G9
    B47-G15
    B47-G16
    B47-G6
    B47-G6B
    B48-G16
    B48-G9
    B5-G15
    B5-G16
    B50-G16
    B51-G9
    B54-G15
    B6-G13
    B6-G9
    B60-G16
    B61-G2
    B62-G9
    B64-G9
    B65-G9
    B66-G16
    B67-G16
    B68-G16
    B68-G9
    B69-G16
    B7-G16
    B7-G4
    B70-G16
    B70-G9
    B72-G16
    B73-G16
    B75-G9
    B78-G9
    B79-G16
    B79-G9
    B8-G13
    B8-G4
    B8-G9
    B82-G9
    B83-G16
    B87-G9
    B88-G9
    B9-G15
    B9-G16
    B9-G17
    B91-G9
    BA-13
    BA-19
    Bact-08
    Bact-11
    Bact-19
    Bacteroides-like
    Badici-like
    BAIKAL-G1
    BAIKAL-G1-2R
    Banna-like
    bar-headed
    Barkedji-like
    Barnacle-like
    Bastrovirus-like_virus/VietNam/Bat/17819_21
    Bat-associated
    Bayan-Khairhan-Ula
    BD1-11
    BD1-5
    BD1-7
    BD2-11
    BE-D
    Bean-associated
    Bean-pod
    Bear-associated
    Begomovirus-associated
    Bemisia-associated
    Bengt-jonsellia
    benzene-decomposing
    benzene-degrading
    benzoapyrene-degrading
    Benzoyl-CoA
    Bercke-Baary
    beta-cyfluthrin-degrading
    beta-symbiont
    Bi-allelically
    Bi-directional
    Bi-molecular
    BICA1-8
    Bidnavirus-like
    Bifidobacterium-E.
    biocide-degrading
    biphenthrin-degrading
    biphenyl-degrading
    Birds-foot
    birds-foot
    BK-30
    Black-and-white
    Black-eyed
    Black-headed
    black-headed
    Blackcurrant-associated
    Bo-Circo-like
    Bog-105
    Bog-1112
    BOG-1112
    Bog-113
    BOG-1155
    Bog-1181
    Bog-1183
    BOG-1184
    Bog-1198
    BOG-1338
    Bog-1340
    BOG-1363
    BOG-1369
    BOG-1370
    BOG-1457
    BOG-1460
    Bog-159
    BOG-224
    BOG-234
    Bog-257
    Bog-366
    Bog-375
    Bog-38
    Bog-473
    Bog-515
    Bog-532
    Bog-564
    Bog-752
    Bog-756
    Bog-793
    Bog-877
    BOG-908
    BOG-930
    BOG-931
    BOG-932
    BOG-933
    BOG-935
    BOG-938
    BOG-944
    Bog-950
    BOG-994
    Bombus-associated
    Bos-associated
    Bovine-like
    BP-101
    BP-133
    BP-173
    BP-191
    BP-197
    BRH-c0
    BRH-c25
    BRH-c32
    BRH-c54
    BRH-c57
    BRH-c8a
    BrNp1-15
    Broad-billed
    Broad-leafed
    bromate-reducing
    bromoform-metabolizing
    Bromus-associated
    Bronnoya-like
    BS150m-G50
    BS150m-G51
    BS150m-G53
    BS150m-G59
    BS150m-G60
    BS150m-G62
    BS150m-G63
    BS150m-G64
    BS150m-G65
    BS150m-G66
    BS150m-G67
    BS750-G32
    BS750m-G10
    BS750m-G14
    BS750m-G15
    BS750m-G21
    BS750m-G22
    BS750m-G24
    BS750m-G25
    BS750m-G28
    BS750m-G30
    BS750m-G34
    BS750m-G37
    BS750m-G39
    BS750m-G4
    BS750m-G44
    BS750m-G57
    BS750m-G60
    BS750m-G62
    BS750m-G65
    BS750m-G69
    BS750m-G71
    BS750m-G77
    BS750m-G78
    BS750m-G79
    BS750m-G81
    BSW11-IV
    BTEX-degrading
    BtMd-BetaCoV/HuB2013
    BtMf-AlphaCoV/AH2011
    BtMf-AlphaCoV/FJ2012
    BtMf-AlphaCoV/GD2012
    BtMf-AlphaCoV/GD2012-a
    BtMf-AlphaCoV/GD2012-b
    BtMf-AlphaCoV/HeN2013
    BtMf-AlphaCoV/HeN2013-a
    BtMf-AlphaCoV/HeN2013-b
    BtMf-AlphaCoV/HuB2013
    BtMf-AlphaCoV/HuB2013-a
    BtMf-AlphaCoV/JX2012
    BtMr-AlphaCoV/SAX2011
    BtNv-AlphaCoV/SC2013
    BtPa-BetaCoV/GD2013
    BtRf-AlphaCoV/HuB2013
    BtRf-AlphaCoV/YN2012
    BtRf-BetaCoV/HeB2013
    BtRf-BetaCoV/HeN2013
    BtRf-BetaCoV/HuB2013
    BtRf-BetaCoV/JL2012
    BtRf-BetaCoV/SX2013
    BtRs-BetaCoV/GX2013
    BtRs-BetaCoV/HuB2013
    BtRs-BetaCoV/YN2013
    BtTp-BetaCoV/GX2012
    BtVs-BetaCoV/SC2013
    bubaline-associated
    Bufavirus-1
    Bufavirus-2
    Bufavirus-3
    Bunya-like
    Burke-Gilman
    butyrate-producing
    Buxtonella-like
    BW-2
    BWSM-131
    C-114
    C-19
    C-53
    C-terminal
    C.glutamicum-E.coli
    C7867-001
    C7867-002
    C7867-005
    C7867-006
    Cachavirus-1B
    CACIAM-22H2
    CACIAM-69d
    Cafeteriavirus-dependent
    CAG-1000
    CAG-1024
    CAG-103
    CAG-1031
    CAG-1138
    CAG-115
    CAG-1193
    CAG-1252
    CAG-138
    CAG-170
    CAG-177
    CAG-1782
    CAG-194
    CAG-196
    CAG-217
    CAG-238
    CAG-239
    CAG-245
    CAG-266
    CAG-267
    CAG-269
    CAG-272
    CAG-273
    CAG-274
    CAG-288
    CAG-302
    CAG-303
    CAG-307
    CAG-312
    CAG-313
    CAG-314
    CAG-345
    CAG-349
    CAG-382
    CAG-390
    CAG-411
    CAG-417
    CAG-433
    CAG-448
    CAG-449
    CAG-452
    CAG-465
    CAG-475
    CAG-488
    CAG-492
    CAG-508
    CAG-510
    CAG-533
    CAG-552
    CAG-557
    CAG-568
    CAG-582
    CAG-590
    CAG-605
    CAG-611
    CAG-628
    CAG-631
    CAG-632
    CAG-698
    CAG-710
    CAG-724
    CAG-762
    CAG-791
    CAG-793
    CAG-826
    CAG-841
    CAG-873
    CAG-877
    CAG-882
    CAG-884
    CAG-914
    CAG-917
    CAG-964
    CAG-977
    Calf-giraffe
    Calhevirus-2a
    Calhevirus-2b
    Camellia-associated
    Campylobacter-like
    Caraparu-like
    carbazole-degrading
    Carpediemonas-like
    Carpotroche-associated
    carrageenase-producing
    Carrot-associated
    Cas-Br-E
    Cas-NS-1
    Casp-actino5
    Casp-actino8
    Casp-alpha2
    Casp-Chloro-G3
    Casp-Chloro-G4
    Cassava-Congo
    CASV-2
    CAT-levansucrase
    Catch-me-cave
    CAU-1509
    CBA120-like
    CC-CFT747
    CCA-013414725
    CCAP-1448
    CECT-8803
    Cell-free
    Ceratium-like
    CF-105
    CF-107
    CF-113
    CF-118
    CF-120
    CF-121
    CF-13
    CF-134
    CF-154
    CF-16
    CF-161
    CF-162
    CF-163
    CF-165
    CF-167
    CF-28
    CF-39
    CF-46
    CF-54
    CF-72
    CF-75
    CF-82
    CF-96
    CFB-group
    CFBP-8762
    CFH-32150
    Cfx2-01
    Cfx3-03
    CG-Epi1
    CG07-land
    CG08-08-20-14
    CG1-02-31-27
    CG1-02-33-12
    CG1-02-35-32
    CG1-02-35-72
    CG1-02-37-114
    CG1-02-37-13
    CG1-02-37-22
    CG1-02-39-14
    CG1-02-40-25
    CG1-02-40-82
    CG1-02-41-12
    CG1-02-41-21
    CG1-02-41-26
    CG1-02-43-26
    CG1-02-43-31
    CG1-02-47-18
    CG1-02-47-37
    CG1-02-47-40
    CG1-02-49-23
    CG1-02-57-44
    CG1-02-64-396
    CG10-46-32
    CG10238-14
    CG2-30-32-10
    CG2-30-33-13
    CG2-30-33-31
    CG2-30-33-46
    CG2-30-36-11
    CG2-30-36-16
    CG2-30-36-38
    CG2-30-36-39
    CG2-30-37-29
    CG2-30-39-44
    CG2-30-40-12
    CG2-30-40-21
    CG2-30-43-20
    CG2-30-44-17
    CG2-30-45-37
    CG2-30-46-19
    CG2-30-46-9
    CG2-30-49-12
    CG2-30-51-40
    CG2-30-53-67
    CG2-30-54-10
    CG2-30-54-11
    CG2-30-58-95
    CG2-30-59-28
    CG2-30-60-27
    CG2-30-61-12
    CG2-30-64-16
    CG2-30-66-41
    CG2-30-68-42
    CG2-30-70-394
    CGMCC-115125
    chain-forming
    Chaq-like
    Cherry-associated
    Chersky-265
    Chersky-267
    Chersky-299
    Chersky-534
    Chersky-683
    Chersky-806
    Chersky-822
    Chersky-840
    CHH4-2
    Chiba-040502
    chieh-qua
    CHK158-818
    chlamydia-associated
    Chlorante-Aster
    chloromethane-metabolizing
    chlortetracycline-resistant
    ChocPot-Mid
    Choricystis-like
    Chryso-hypnum
    Chrysovirus-like
    Chth-196
    chu-like
    cilia-associated
    Circo-like
    Circovirus-like
    Citrus-associated
    Clavibacter-like
    Clostridium-like
    Co-expression
    Co-reporter
    Colwellia-like
    COMBO-56-21
    Corn-reddening
    Corynebacterium-like
    COS-degrading
    Cotton-top
    Counter-selectable
    Cowpox-Vaccinia
    Coxiella-like
    CP2-2F
    CP52-degrading
    Cr(VI)-reducing
    CR-1
    CR-4
    Crab-eating
    Crane-associated
    CrAss-like
    Cre-expression
    Cre-lox
    Crimean-Congo
    CRISPR-Cas
    CRISPR-cas9
    CRISPR-mediated
    Crow-dipper
    Crucivirus-102
    Crucivirus-111
    Crucivirus-112
    Crucivirus-113
    Crucivirus-119
    Crucivirus-122
    Crucivirus-124
    Crucivirus-133
    Crucivirus-141
    Crucivirus-146
    Crucivirus-147
    Crucivirus-148
    Crucivirus-152
    Crucivirus-158
    Crucivirus-163
    Crucivirus-168
    Crucivirus-170
    Crucivirus-174
    Crucivirus-175
    Crucivirus-182
    Crucivirus-183
    Crucivirus-184
    Crucivirus-190
    Crucivirus-195
    Crucivirus-204
    Crucivirus-205
    Crucivirus-207
    Crucivirus-211
    Crucivirus-212
    Crucivirus-224
    Crucivirus-234
    Crucivirus-241
    Crucivirus-243
    Crucivirus-245
    Crucivirus-250
    Crucivirus-251
    Crucivirus-255
    Crucivirus-264
    Crucivirus-265
    Crucivirus-266
    Crucivirus-272
    Crucivirus-275
    Crucivirus-276
    Crucivirus-277
    Crucivirus-278
    Crucivirus-281
    Crucivirus-282
    Crucivirus-291
    Crucivirus-293
    Crucivirus-295
    Crucivirus-298
    Crucivirus-304
    Crucivirus-305
    Crucivirus-306
    Crucivirus-309
    Crucivirus-310
    Crucivirus-311
    Crucivirus-312
    Crucivirus-322
    Crucivirus-323
    Crucivirus-324
    Crucivirus-326
    Crucivirus-339
    Crucivirus-340
    Crucivirus-345
    Crucivirus-349
    Crucivirus-350
    Crucivirus-353
    Crucivirus-359
    Crucivirus-362
    Crucivirus-364
    Crucivirus-371
    Crucivirus-372
    Crucivirus-373
    Crucivirus-380
    Crucivirus-384
    Crucivirus-387
    Crucivirus-389
    Crucivirus-392
    Crucivirus-402
    Crucivirus-406
    Crucivirus-407
    Crucivirus-409
    Crucivirus-410
    Crucivirus-411
    Crucivirus-416
    Crucivirus-419
    Crucivirus-427
    Crucivirus-429
    Crucivirus-432
    Crucivirus-438
    Crucivirus-453
    Crucivirus-454
    Crucivirus-455
    Crucivirus-456
    Crucivirus-458
    Crucivirus-468
    Crucivirus-472
    Crucivirus-474
    Crucivirus-475
    Crucivirus-479
    Crucivirus-480
    Crucivirus-482
    Crucivirus-483
    Crucivirus-495
    Crucivirus-497
    Crucivirus-499
    Crucivirus-501
    Crucivirus-505
    Crucivirus-506
    Crucivirus-509
    Crucivirus-513
    Crucivirus-514
    Crucivirus-520
    Crucivirus-523
    Crucivirus-527
    Crucivirus-528
    Crucivirus-530
    Crucivirus-534
    Crucivirus-536
    Crucivirus-538
    Crucivirus-539
    Crucivirus-540
    Crucivirus-81
    Crucivirus-88
    Crucivirus-93
    Crucivirus-94
    Crucivirus-97
    Crucivirus-99
    Crucivirus-like
    Cryphonectria-Endothia
    Cryptaulaxoides-like
    CS1-C013
    CS2-K091
    CS3-C152
    CS4-C045
    CSBR16-104
    CSBR16-107
    CSBr16-110
    CSBR16-113R1
    CSBR16-119
    CSBr16-161
    CSBR16-174
    CSBR16-193
    CSBR16-202
    CSBR16-224
    CSBR16-49
    CSBr16-57R1
    CSBr16-58
    CSBR16-87
    CSBR16-88
    CSK15Z-1
    CSP1-1
    CSP1-2
    CSP1-3
    CSP1-4
    CSP1-5
    CSP1-6
    CSP1-8
    CSSED10-109M
    CSSED10-239
    CSSED10-269
    CSSED10-310
    CSSED10-335
    CSSED10-376
    CSSed10-48
    CSSED10-83
    CSSED11-109
    CSSED11-154
    CSSed11-175R1
    CSSED11-181
    CSSed11-197
    CSSed11-243R1
    CSSED11-302
    CSSED11-315
    Ctenophore-associated
    CTFE4-defluorinating
    CTSoil-043
    CTSoil-045
    CTSoil-072
    CTSoil-075
    CTSoil-080
    CTSoil-081
    CTSOIL-112
    CU05-9
    CUG-92003
    Culex-associated
    Culicine-associated
    Cx-87
    cyanobacteria-associated
    Cyrto-hypnum
    Cysteine-free
    Cytophaga-like
    D16-63
    D1FN1-002
    D2-FULL-39-29
    DA-C8
    Damh-18
    DAT-1
    DC-out156
    DC-out160
    DCM-degrading
    Deep-sea
    deep-sea
    Deinococcus-like
    Deinococcus-Thermus
    Delta-02
    Dendro-hypnum
    Dermatophilus-like
    DG-23
    DG-26
    DG-26_A
    DG-27
    DG-33
    DG-60
    DG-74-1
    DG-74-3
    DG-75
    DG-78
    DHVEG-1
    dibenzofuran-degrading
    dichloromethane-degrading
    Dietzia-Escherichia
    Diporeia-associated
    diuron-degrading
    DKR-5
    DMSP-degrading
    DNA-binding
    DP-1
    DP-19
    DP-20
    DP-23
    DP-3
    DP-36
    DP-6
    DP16D-bin41
    Dragonfly-associated
    DRI-13
    Drosophila-associated
    DSM-100275
    DSM-100316
    DSM-103792
    DSM-111022
    DSM-12255
    DSM-12270
    DSM-1321
    DSM-13327
    DSM-15969
    DSM-16000
    DSM-16054
    DSM-16500
    DSM-1736
    DSM-17781
    DSM-18226
    DSM-19205
    DSM-19610
    DSM-21159
    DSM-21967
    DSM-22653
    DSM-22679
    DSM-23052
    DSM-23947
    DSM-2461
    DSM-24629
    DSM-25080
    DSM-25186
    DSM-25281
    DSM-26407
    DSM-27016
    DSM-27196
    DSM-28697
    DSM-29158
    DSM-2933
    DSM-4216
    DSM-44207
    DSM-44946
    DSM-45169
    DSM-4660
    DSM-5069
    DSM-5130
    DSM-5807
    DSM-6294
    DSM-6578
    DSM-8532
    DSM-8902
    DSM-9736
    DSM-9756
    DSMP-degrading
    DT-108
    DT-110
    DT-111
    DT-14
    DT-32
    DT-33
    DT-34
    DT-35
    DT-38
    DT-50
    DT-59
    DT-68
    DT-91
    Dual-luciferase
    Dual-tagging
    Duck-associated
    Duck-dominant
    Duin-like
    dye-degrading
    E26-bin7
    E29-bin14
    E29-bin15
    E29-bin54
    E29-bin73
    E4-135
    E4-60
    E44-bin10
    E44-bin15
    E44-bin18
    E44-bin26
    E44-bin32
    E44-bin39
    E44-bin46
    E44-bin52
    E44-bin80
    E44-bin88
    E44-bin89
    EA-19
    EA-27
    ECH-B-1
    Echarate-like
    Echinacea-associated
    ECT2AJA-044
    ECT2AJA-110-A
    EDTA-degrading
    EFPC-124
    EFPC-2
    Ehrlichia-like
    EIAV-based
    ELB16-189
    Endornaviridae-like
    Endornavirus-like
    Entamoeba-associated
    Epstein-barr
    Erythrobacter-like
    ES-bin-26
    EsbW-18
    Escherichia-Pseudomonas
    ESFC-1
    ESH-2018B
    Ess09-12
    Ess09-15
    estrogen-degrading
    ETH-SRB1
    ethanol-producing
    EX-H1
    EX4484-135
    EX4484-15
    EX4484-2
    EX4484-204
    EX4484-205
    EX4484-217-1
    EX4484-218
    EX4484-224
    EX4484-31
    EX4484-36
    EX4484-52
    EX4484-58
    EX4484-6
    EX4484-71
    EX4572-165
    EX4572-44
    EX4572-64
    F-11
    F-183
    F1-12-MAGs170
    F1-120-MAGs023
    F1-120-MAGs055
    F1-120-MAGs118
    F1-120-MAGs170
    F1-120-MAGs238
    F1-140-MAGs086
    F1-140-MAGs142
    F1-20-MAGs016
    F1-20-MAGs119
    F1-20-MAGs160
    F1-60-MAGs027
    F1-60-MAGs104
    F1-60-MAGs140
    F1-60-MAGs149
    F1-60-MAGs163
    F1-80-MAGs030
    F1-80-MAGs062
    F1-80-MAGS106
    F23-B02
    F23-D06
    FACHB-1237
    FACHB-1375
    FACHB-1406
    FACHB-1407
    FACHB-261
    FACHB-36
    FACHB-40
    FACHB-46
    FACHB-68
    FACHB-8
    FACHB-800
    FACHB-831
    FACHB-886
    FACHB-T130
    Faecal-associated
    Fancy-tailed
    FCV-like
    Fe(III)-reducing
    Fe-oxidizing
    FEB-10
    FEB-12
    FEB-17
    FEB-22
    FEB-25
    FEB-35
    FEB-7
    Felisa-like
    Fen-1038
    Fen-1039
    Fen-1058
    Fen-1064
    FEN-1073
    Fen-1077
    Fen-1087
    Fen-1088
    FEN-1088
    FEN-1099
    Fen-1118
    Fen-1135
    Fen-1137
    FEN-1139
    FEN-1143
    FEN-1160
    Fen-1166
    FEN-1173
    Fen-1174
    FEN-1185
    FEN-1191
    FEN-1219
    FEN-1226
    Fen-1231
    Fen-1245
    Fen-1247
    FEN-1250
    FEN-1251
    Fen-1254
    Fen-1255
    FEN-1257
    Fen-1258
    Fen-1260
    Fen-1262
    Fen-1273
    FEN-1279
    FEN-1297
    Fen-1298
    Fen-1301
    Fen-1308
    Fen-1317
    FEN-1320
    FEN-1322
    Fen-1326
    FEN-1329
    Fen-1342
    FEN-1343
    FEN-1344
    FEN-1346
    FEN-1349
    FEN-1350
    Fen-1359
    Fen-1362
    Fen-1364
    FEN-1388
    Fen-1467
    Fen-178
    Fen-181
    Fen-183
    FEN-299
    FEN-33
    Fen-330
    FEN-336
    Fen-336
    FEN-413
    Fen-455
    Fen-549
    Fen-668
    Fen-671
    FEN-672
    FEN-680
    Fen-7
    Fen-727
    FEN-727
    FEN-943
    FEN-979
    FEN-987
    Fen-999
    fenpropathrin-degrading
    fenthion-degrading
    Fer-de-lance
    Fermo-like
    Fesa-like
    Fibro-01
    Fibropapilloma-associated
    Finkel-Biskis-Jinkins
    Firm-04
    Firm-07
    Firm-08
    Firm-09
    Firm-10
    Firm-13
    Firm-16
    Firm-18
    Fisa-like
    Fish-associated
    FJAT-45350
    FLAG-tagging
    Flavi-like
    Flaviridae-like
    Flavobacterium-like
    Flexibacter-like
    Flexistipes-like
    Flying-fox
    Fonticula-like
    Food-grade
    Foot-and-mouth
    Four-toed
    Francisella-like
    FRT-Tp-FRT
    FS08-3
    FT1-004
    FT1-020
    furfural-degrading
    FW021-bin31
    FW021-bin43
    FW300-bin19
    FW305-bin1
    FW602-bin16
    FW602-bin19
    FW602-bin22
    G3-4614
    GABA-v2-NaVbeta1-myc
    GABA-v3-NaVbeta1-myc
    GABA-v4-NaVbeta1-myc
    gamma-symbiont
    Ganwon-do
    Gardner-Arnstein
    GBS-1
    GBS-2
    GBS-DC
    GBV-A-like
    GCA-000496135
    GCA-001602625
    GCA-001730085
    GCA-001735895
    GCA-001788215
    GCA-001856825
    GCA-002050365
    GCA-002282575
    GCA-002401295
    GCA-002401385
    GCA-002402085
    GCA-002684695
    GCA-002686595
    GCA-002687715
    GCA-002688135
    GCA-002690875
    GCA-002691585
    GCA-002694895
    GCA-002697625
    GCA-002700125
    GCA-002701365
    GCA-002701455
    GCA-002701945
    GCA-002704045
    GCA-002704185
    GCA-002705045
    GCA-002705445
    GCA-002707245
    GCA-002707645
    GCA-002707905
    GCA-002708145
    GCA-002708315
    GCA-002708715
    GCA-002712045
    GCA-002712965
    GCA-002713455
    GCA-002715585
    GCA-002715885
    GCA-002716945
    GCA-002717915
    GCA-002718135
    GCA-002718395
    GCA-002718515
    GCA-002718655
    GCA-002719095
    GCA-002719695
    GCA-002720895
    GCA-002721295
    GCA-002722245
    GCA-002723415
    GCA-002724395
    GCA-002724975
    GCA-002726655
    GCA-002727615
    GCA-002727735
    GCA-002728675
    GCA-002728855
    GCA-002729875
    GCA-002731945
    GCA-002733105
    GCA-002733185
    GCA-002733465
    GCA-002748355
    GCA-002748415
    GCA-002770795
    GCA-002772825
    GCA-002772895
    GCA-002773005
    GCA-002779355
    GCA-002779425
    GCA-002779465
    GCA-002779875
    GCA-002780085
    GCA-002780795
    GCA-002782225
    GCA-002782285
    GCA-002782345
    GCA-002784865
    GCA-002785045
    GCA-002787175
    GCA-002787255
    GCA-002787375
    GCA-00278855
    GCA-002788555
    GCA-002789015
    GCA-002790365
    GCA-002790575
    GCA-002790615
    GCA-002790695
    GCA-002791395
    GCA-002792115
    GCA-002793235
    GCA-002793355
    GCA-002793575
    GCA-002795985
    GCA-002796325
    GCA-002838935
    GCA-002840015
    GCA-002840305
    GCA-002840935
    GCA-002841655
    GCA-002862325
    GCA-003054495
    GCA-003244245
    GCA-004216475
    GCA-012562765
    GCA-013348725
    GCA-013414725
    GCA-013693735
    GCA-014075295
    GCA-014931565
    GCA-016432245
    GCA-016432345
    GCA-016432425
    GCA-016432445
    GCA-016432465
    GCA-016432525
    GCA-016432625
    GCA-016699085
    GCA-016699105
    GCA-016699145
    GCA-016699305
    GCA-016699365
    GCA-016699445
    GCA-016699515
    GCA-016699535
    GCA-016699595
    GCA-016699755
    GCA-016699835
    GCA-016699855
    GCA-016699895
    GCA-016699975
    GCA-016700035
    GCA-016700095
    GCA-016700155
    GCA-016700315
    GCA-016700335
    GCA-020341775
    GCA-020343435
    GCA-020343575
    GCA-020343615
    GCA-020343675
    GCA-020343715
    GCA-020344235
    GCA-020344395
    GCA-020344555
    GCA-020344715
    GCA-020345705
    GCA-020345945
    GCA-020346225
    GCA-020346805
    GCA-020347585
    GCA-020348065
    GCA-020348265
    GCA-020348365
    GCA-020348445
    GCA-020350385
    GCA-020350865
    GCA-020351305
    GCA-020352575
    GCA-020352645
    GCA-020354555
    GCA-020354575
    GCA-020355125
    GCA-020356005
    GCA-020356105
    GCA-020356305
    GCA-021513135
    GCA-022846635
    GCA-023898365
    GCA-024661895
    GCA-030583725
    GCA-16432305
    GCA-1730015
    GCA-2400775
    GCA-2400975
    GCA-2401155
    GCA-2401345
    GCA-2401425
    GCA-2401445
    GCA-2401635
    GCA-2401725
    GCA-2401735
    GCA-2401785
    GCA-2401865
    GCA-2402025
    GCA-2402105
    GCA-2402195
    GCA-2683135
    GCA-2683315
    GCA-2683405
    GCA-2683775
    GCA-2683825
    GCA-2684655
    GCA-2685815
    GCA-2685855
    GCA-2685935
    GCA-2685965
    GCA-2686065
    GCA-2686265
    GCA-2686295
    GCA-2686365
    GCA-2686445
    GCA-2686465
    GCA-2686595
    GCA-2686765
    GCA-2686855
    GCA-2686945
    GCA-2686955
    GCA-2686995
    GCA-2687015
    GCA-2687025
    GCA-2687185
    GCA-2687275
    GCA-2687515
    GCA-2687715
    GCA-2687795
    GCA-2687815
    GCA-2688035
    GCA-2688095
    GCA-2688105
    GCA-2688265
    GCA-2688355
    GCA-2688435
    GCA-2688775
    GCA-2688925
    GCA-2688965
    GCA-2689605
    GCA-2689925
    GCA-2689995
    GCA-2690215
    GCA-2690565
    GCA-2690645
    GCA-2691085
    GCA-2691145
    GCA-2691245
    GCA-2691785
    GCA-2692065
    GCA-2693325
    GCA-2693335
    GCA-2696645
    GCA-2696965
    GCA-2697345
    GCA-2697505
    GCA-2698665
    GCA-2698995
    GCA-2699025
    GCA-2699105
    GCA-2699335
    GCA-2699365
    GCA-2699445
    GCA-2700365
    GCA-2700405
    GCA-2701545
    GCA-2701705
    GCA-2701885
    GCA-2701925
    GCA-2702065
    GCA-2702235
    GCA-2702655
    GCA-2704625
    GCA-2704665
    GCA-2705055
    GCA-2705205
    GCA-2705445
    GCA-2705995
    GCA-2706205
    GCA-2706465
    GCA-2706885
    GCA-2707145
    GCA-2707185
    GCA-2707205
    GCA-2707495
    GCA-2707695
    GCA-2707915
    GCA-2708265
    GCA-2708415
    GCA-2709075
    GCA-2709385
    GCA-2709565
    GCA-2709665
    GCA-2710765
    GCA-2711125
    GCA-2711415
    GCA-2711515
    GCA-2711565
    GCA-2711625
    GCA-2711925
    GCA-2712005
    GCA-2712105
    GCA-2712585
    GCA-2712665
    GCA-2712715
    GCA-2712985
    GCA-2713145
    GCA-2714055
    GCA-2715305
    GCA-2715465
    GCA-2715505
    GCA-2715565
    GCA-2715645
    GCA-2715965
    GCA-2716065
    GCA-2716345
    GCA-2716485
    GCA-2716585
    GCA-2716765
    GCA-2716945
    GCA-2717185
    GCA-2717285
    GCA-2717385
    GCA-2717465
    GCA-2717485
    GCA-2717565
    GCA-2717625
    GCA-2717925
    GCA-2718035
    GCA-2718595
    GCA-2718715
    GCA-2718815
    GCA-2718845
    GCA-2718895
    GCA-2719315
    GCA-2719675
    GCA-2719775
    GCA-2720125
    GCA-2720195
    GCA-2720855
    GCA-2720865
    GCA-2720935
    GCA-2720975
    GCA-2721105
    GCA-2721365
    GCA-2721545
    GCA-2721815
    GCA-2722105
    GCA-2722175
    GCA-2722315
    GCA-2722455
    GCA-2722775
    GCA-2722955
    GCA-2723085
    GCA-2723275
    GCA-2724185
    GCA-2724215
    GCA-2724505
    GCA-2724675
    GCA-2725445
    GCA-2725625
    GCA-2725915
    GCA-2725955
    GCA-2726205
    GCA-2726245
    GCA-2726865
    GCA-2728135
    GCA-2728255
    GCA-2728355
    GCA-2729495
    GCA-2729515
    GCA-2730555
    GCA-2730855
    GCA-2730975
    GCA-2731375
    GCA-2732015
    GCA-2732755
    GCA-2732805
    GCA-2733415
    GCA-2733545
    GCA-2733575
    GCA-2733885
    GCA-2737125
    GCA-2737595
    GCA-2737665
    GCA-2737725
    GCA-2742885
    GCA-2742935
    GCA-2746115
    GCA-2746155
    GCA-2746185
    GCA-2746235
    GCA-2746255
    GCA-2746335
    GCA-2746365
    GCA-2746425
    GCA-2746535
    GCA-2746565
    GCA-2746605
    GCA-2746795
    GCA-2746885
    GCA-2747035
    GCA-2747105
    GCA-2747255
    GCA-2747325
    GCA-2747355
    GCA-2747515
    GCA-2747695
    GCA-2747955
    GCA-2748055
    GCA-2748265
    GCA-2748355
    GCA-2748425
    GCA-2748505
    GCA-2748575
    GCA-2748835
    GCA-2749205
    GCA-2753255
    GCA-2753275
    GCA-2753565
    GCA-2753745
    GCA-2770975
    GCA-2774365
    GCA-2785325
    GCA-2786165
    GCA-2786225
    GCA-2792135
    GCA-2794515
    GCA-2794595
    GCA-2861585
    GCA-2861595
    GCA-2862085
    GCA-2862225
    GCA-2862545
    GCA-2862585
    GCA-2862715
    GCA-2863045
    GCA-2863065
    GCA-2863085
    GCA-900066905
    GCA-918320885
    GCA-958296115
    GCA-965745
    GCF-000287335
    GCF-000287355
    GCF-000787395
    GCF-001939115
    GCF-002002485
    GCF-002020805
    GCF-002020875
    GCF-002259525
    GCF-002776555
    GCF-013343005
    GCF-1484045
    GCF-900161835
    Gemmata-like
    Gene-editing
    Gene-trap
    Gene-trapping
    Giardia-associated
    Gill-associated
    Ginkgoaceae-associated
    GJ-E10
    Glaucous-winged
    Glaucus-gull
    GLO-12
    Glo-13
    Glo-17
    GLO-3
    Gluconacetobacter-like
    glycerol-degrading
    GMQP-bins7
    GN02-872
    GN02-873
    GO-respiring
    Golden-headed
    Gonad-specific
    Goose-dominant
    Gordonia-like
    Gp1-AA112
    Gp1-AA117
    Gp1-AA122
    Gp1-AA124
    Gp1-AA133
    Gp1-AA142
    Gp1-AA145
    Gp13-AA74
    Gp13-AA77
    Gp18-AA60
    Gp22-AA2
    Gp22-AA3
    Gp6-AA38
    Gp6-AA40
    Gp6-AA45
    Gp7-AA10
    Gp7-AA6
    Gp7-AA8
    GPF-1
    Gram-negative
    Gram-positive
    Grapevine-associated
    Grass-Cactus
    Gray-bellied
    GSA-30
    GSPS2-BC2016
    GW-Firmicutes-8
    GW-Nitrospira-1
    GW2011-AR1
    GW2011-AR10
    GW2011-AR11
    GW2011-AR13
    GW2011-AR15
    GW2011-AR17
    GW2011-AR18
    GW2011-AR20
    GW2011-AR4
    GW2011-AR5
    GW2011-AR9
    GW928-bin9
    GWA1-39-11
    GWA1-40-21
    GWA1-42-19
    GWA1-42-9
    GWA1-44-24
    GWA1-44-25
    GWA1-44-29
    GWA1-48-11
    GWA1-52-35
    GWA1-54-10
    GWA1-54-12
    GWA2-31-9
    GWA2-32-17
    GWA2-33-14
    GWA2-35-9
    GWA2-36-10
    GWA2-37-10
    GWA2-37-8
    GWA2-38-11
    GWA2-38-13
    GWA2-38-13b
    GWA2-38-27
    GWA2-38-27_A
    GWA2-41-24
    GWA2-42-14
    GWA2-42-15
    GWA2-43-13
    GWA2-43-14
    GWA2-44-10
    GWA2-44-7
    GWA2-44-9
    GWA2-45-9
    GWA2-46-10
    GWA2-46-15
    GWA2-46-9
    GWA2-47-10
    GWA2-47-10b
    GWA2-47-26
    GWA2-47-8
    GWA2-49-15
    GWA2-52-8
    GWA2-54-12
    GWA2-56-11
    GWA2-58-10
    GWA2-66-18
    GWA2-73-35
    GWB1-36-13
    GWB1-36-5
    GWB1-37-8
    GWB1-38-13
    GWB1-39-10
    GWB1-39-12
    GWB1-40-14
    GWB1-41-6
    GWB1-42-6
    GWB1-45-5
    GWB1-45-9
    GWB1-50-10
    GWB1-54-5
    GWB1-55-8
    GWB2-37-7
    GWB2-47-37
    GWB2-55-19
    GWC1-27-15
    GWC1-37-10
    GWC1-37-12b
    GWC1-40-35
    GWC1-40-9
    GWC1-42-9
    GWC1-47-15
    GWC1-49-16
    GWC2-33-13
    GWC2-36-17
    GWC2-37-13
    GWC2-37-14
    GWC2-37-73
    GWC2-37-8
    GWC2-39-10
    GWC2-39-14
    GWC2-40-17
    GWC2-42-11
    GWC2-42-12
    GWC2-42-7
    GWC2-42-7-A
    GWC2-45-19
    GWC2-45-44
    GWC2-55-46
    GWC2-56-14
    GWC2-57-13
    GWC2-65-14
    GWC2-70-10
    GWC2-71-9
    GWC2-73-18
    GWE1-39-12
    GWE2-29-8
    GWE2-31-10
    GWE2-37-16
    GWE2-39-37
    GWE2-40-63
    GWE2-41-16
    GWE2-42-24
    GWE2-42-39
    GWE2-51-13
    GWF1-31-35
    GWF1-45-5
    GWF1-51-8
    GWF2-30-66
    GWF2-32-72
    GWF2-33-10
    GWF2-33-38
    GWF2-33-9
    GWF2-35-18
    GWF2-35-48
    GWF2-35-9
    GWF2-37-15
    GWF2-37-49
    GWF2-38-10
    GWF2-38-335
    GWF2-38-69
    GWF2-38-85
    GWF2-39-127
    GWF2-39-15
    GWF2-40-263
    GWF2-43-52
    GWF2-43-63
    GWF2-44-16
    GWF2-45-14
    GWF2-50-10
    GWF2-50-93
    GWF2-57-35
    Gyrovirus-like
    GYS-P2D
    H-1
    H-Pelican
    H-Stinger
    H02-43-120
    H02-43-30
    H02-45-13
    H02-45-28
    H02-48-17
    H1-PLA3
    H2-BAC3
    H2/CO2-metabolizing
    H3-PLA7
    H5-PLA8
    H5-SPI1
    H5-UNCL2
    Hantaan-like
    Hardy-Zuckerman
    Haruka-associated
    HB2-32-21
    HCV/GBV-B
    HDB-SIOI813
    HDS-20
    HDS-28
    Heart-leaved
    Hedyotis-Oldenlandia
    HEL-GB-A
    HEL-GB-B
    Hel1-33-131
    Helper-independent
    Hemibarbus-Squalidus
    Henipa-like
    Hepe-virga-like
    HERV-H/env59
    HERV-H/env60
    HERV-H/env62
    HGW-1
    HGW-15
    HGW-17
    HGW-BRC1-1
    Hibiscus-infecting
    HIGH02-12-38-15
    HIGHO2-12-38-15
    Himar1-delivery
    his-3
    HIV-1
    HIV-2
    HIV-2.D205
    HIV-like
    HK-STAS-PATE-10
    HK-STAS-PATE-2
    HK-STAS-PATE-3
    HK-STAS-PATE-34
    HK-STAS-PATE-36
    HK-STAS-PATE-42
    HK-STAS-PATE-9
    HKI-0501
    HKST-UBA178
    HKST-UBA22
    HKST-UBA223
    HKST-UBA85
    HMWdDNA-degrading
    HO2-37-13b
    HO2-39-17
    HO2-43-120
    HO2-43-30
    HO2-45-11b
    HO2-45-13
    HO2-45-28
    HO2-48-17
    HO2-49-34
    HO2-52-23
    HO2-FULL-38-9
    HO2-FULL-47-45
    Hom-1
    HOT-345
    HPIV-1
    HS-7
    HS-T3
    HSV-tk-containing
    hSyn-NaVbeta1-myc
    HT-58-2
    HT1-32
    HTLV-1
    HTLV-3
    Humaita-Tubiacanga
    Human-associated
    Human-Simian
    Hump-back
    Husa-like
    HVAC-associated
    HXMU1420-13
    HXMU1420-2
    HXMU1428-3
    hydrocarbon-degrading
    hydrogen-producing
    Hyphomicrobium-like
    HyVt-105
    HyVt-16
    HyVt-206
    HyVt-337
    HyVt-429
    HyVt-443
    HyVt-448
    HyVt-477
    HyVt-539
    HyVt-560
    HyVt-613
    Ichthyobodo-related
    ICN-92133
    IDS-52
    IDS-8
    Iflaviridae-like
    IGHO2-01-FULL-4-A
    IGHO2-12-FULL-4
    IncP-1
    IncP-1beta
    IncQ-like
    Insect-associated
    Io17-Chloro-G2
    iodate-reducing
    iodide-accumulating
    iodide-oxidizing
    IRE/CTVM19-associated
    iron-reducing
    iron-reduction
    ISCAR-4553
    Isosphaera-like
    Issyk-Kul
    Janibacter-like
    JCM-10596
    JCM-19050
    JCM-19058
    JCVI-SCAAA005
    JdFR-07
    JDFR-13
    JdFR-19
    JdFR-22
    JdFR-37
    JdFR-41
    JdFR-42
    JdFR-43
    JdFR-45
    JdFR-47
    JdFR-54
    JdFR-56
    JdFR-76
    JdFR-78
    JdFR-80
    JdFR-81
    JdFR-85
    JdFR-86
    JdFR-88
    JdFR-95
    JdFR-97
    jellyfish-degrading
    JGIOTU-2
    Jiangxi12-B14
    JIR-001
    JN-9
    Jodiemicrovirus-1
    Johnson-sea-linkia
    JS7-9
    JSC-12
    Juquitiba-like
    K139-like
    K23-9
    KACC-23026
    Kalu-18
    KAN-5
    Kazan-3B-28
    KBS-83
    KBS-96
    Kineococcus-like
    KJ1-10-99
    KM106-2
    KMM-3653
    Knock-in
    Korle-bu
    KS3-K002
    L227-5C
    L227-S17
    LANV-2
    Large-insert
    LB-PLM-3
    LB3-1
    LC-2
    LC-3
    LC2-1
    LC3-1
    LC5-1
    LC5-5
    LD1-PB3_B
    LDC-1
    Leafhopper-borne
    Leafy-liverwort
    LEGE-06147
    LEGE-07092
    LEGE-11386
    Legionella-like
    Leguminosae-associated
    Lenti-01
    Lenti-02
    Leptotrichia-like
    LFW-46
    LH-M37
    lignocellulose-degrading
    Lily-associated
    LiPW-15
    LiPW-41
    LLD-282
    LM3-B42
    LMEP-10873
    LMEP-6097
    LMWdDNA-degrading
    LNNU-3342
    Loki-b32
    Long-fingered
    long-tailed
    LOWO2-01-FULL-3
    LPP-group
    LS-NOB
    Luna-1
    Luna-2
    lung-eye-trachea
    Lupinus-Cactus
    LW40-45
    LW60-42
    LZORAL124-50-18
    LZORAL124-64-63
    M1-B18
    M10-OD1-1
    M10-OD1-4
    M3-22
    MA-ANB-1
    Maedi-Visna-related
    MAG-120531
    MAG-120802
    MAG-121220-bin8
    MAG-NZ
    magnetite-containing
    magneto-ovoid
    Magpie-robin
    Maize-associated
    male-killer
    male-killing
    manganese-oxidation
    manganese-oxidizing
    Many-flowered
    Marcus-kochia
    MarineAlpha4-Bin1
    MarineAlpha5-Bin12
    MarineAlpha5-Bin3
    MarineAlpha5-Bin7
    MarineAlpha9-Bin2
    Marinobacter-like
    Marseille-P2935
    Marseille-P3160
    Marseille-P3662
    Mason-Pfizer
    Massilia-like
    MAST-4
    Maus-Elberfeld
    Maverick-related
    MB-PLM-1
    MB-PLM-2
    MB1-2
    MB2-83
    MCG-1
    MCG-10
    MCG-11
    MCG-12
    MCG-13
    MCG-14
    MCG-15
    MCG-16
    MCG-17
    MCG-2
    MCG-3
    MCG-4
    MCG-5a
    MCG-5b
    MCG-7
    MCG-8
    MCG-9
    MCMV-bacmid
    mDlx-NaVbeta1-myc
    Meaban-like
    MED-G09
    MED-G10
    MED-G105
    MED-G107
    MED-G116
    MED-G13
    MED-G138
    MED-G14
    MED-G16
    MED-G17
    MED-G20
    MED-G21
    MED-G40
    MED-G52
    MED-G78
    MED-G82
    MED-G85
    MedAcidi-G1
    MedAcidi-G2B
    MedAcidi-G3
    MEDG-81
    MedPE-SWsnd-G2
    Melon-headed
    mercury-resistant
    Merida-like
    metal-contaminated
    methane-oxidizing
    methane-seep
    methylotroph-related
    Metopion-like
    MEX13ELK6-19
    MFBS3-17
    MGIIa-I
    MGIIa-K1
    MGIIa-K2
    MGIIa-L1
    MGIIa-L2
    MGIIa-L3
    MGIIa-L4
    MGIIb-N1
    MGIIb-N2
    MGIIb-O1
    MGIIb-O2
    MGIIb-O3
    MGIIb-O5
    MGIIb-P
    MGIIb-Q1
    MGR-bin208
    MGR-bin340
    Mimivirus-dependent
    mini-chromosome
    mini-Tn5*7-lux
    Mini-Tn7
    Mint-like
    Mirim-like
    MIT-9215
    Mito-like
    MK-D1
    ML615J-28
    ML635J-15
    ML8-F1
    MLS-D
    MLV-like
    MLV-related
    MMGLQ5-1
    MMTV-luciferase
    MnB-11
    Mongoose-associated
    mongoose-associated
    monochloroacetic-acid-degrading
    Mononegavirus-like
    mosquito-borne
    mouse-rat
    MP5-2226
    MP8T-1
    MR-S7
    MS024-2A
    MS499-59
    MSB-5A5
    MSJ-26
    MSJ-34
    Mu-like
    Multi-host
    Multi-promoter
    Murine-associated
    Murray-Darling
    MWH-TA3
    Mx-Lox
    Mycobacteria-E.coli
    MYW30-H2
    N-peptide-mCherry
    N-terminal
    N-tropic
    NAC60-12
    Nanovirus-like
    Nap2-2B
    naphthalene-degrading
    naphthalene-utilizing
    narna-like
    Narnaviridae-like
    Narrow-ridged
    NBRC-103111
    NBRC-107566
    NBRC-107688
    NEAU-GS5
    Nebraska-like
    nege-like
    Nege-like
    Negelike-virus
    Negev-like
    Negeviridae-like
    Neo-uvaria
    NFAT-DsRed
    NIC37A-2
    NICIL-2
    NIES-3974
    NIES-981
    NIOZ-UU106
    NIOZ-UU13
    NIOZ-UU35
    NIT-T3
    nitrite-oxidizing
    nitrogen-fixing
    Nivun-Haamir
    NL63-related
    Nodaviridae-like
    Non-A,
    non-culturable
    Non-geniculate
    Non-human
    non-mammal
    Non-primate
    non-primate
    Norovirus/GII.4/1993-6/UK
    Norovirus/Hu/GII.2/V1/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/1732/07/07-Jun-2007/Slovenia
    Norovirus/Hu/GII.4/1991/07/26-Jun-2007/Slovenia
    Norovirus/Hu/GII.4/220/10/28-Jan-2010/Slovenia
    Norovirus/Hu/GII.4/2791/09/09-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/2830/09/15-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/288/05/04-Feb-2005/Slovenia
    Norovirus/Hu/GII.4/2889/09/23-Oct-2009/Slovenia
    Norovirus/Hu/GII.4/290/05/04-Feb-2005/Slovenia
    Norovirus/Hu/GII.4/580/07/26-Feb-2007/Slovenia
    Norovirus/Hu/GII.4/755/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/756/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/757/10/17-Mar-2010/Slovenia
    Norovirus/Hu/GII.4/V17/08/05-Feb-2009/Slovenia
    Norovirus/Hu/GII.4/V2/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/V3/09/18-Jan-2009/Slovenia
    Norovirus/Hu/GII.4/V47/09/25-Feb-2009/Slovenia
    Norovirus/Hu/GII.4/V7/09/11-Jan-2009/Slovenia
    Norwalk-like
    NP-4
    NP-7
    NPL-UPA2
    NS-11
    NS-12
    NS-4
    NS-7
    NS11-12g
    NSJ-53
    Nuclearia-like
    NZ13-RE01
    O2-01-FULL-39-8
    O2-01-FULL-48-27b
    O2-02-FULL-42-17
    O2-02-FULL-42-9
    O2-02-FULL-48-8
    O2-02-FULL-57-9
    O2-12-FULL-27-8
    O2-12-FULL-42-9
    O2-12-FULL-43-9
    O2-12-FULL-44-13
    O2-12-FULL-45-9
    O2-12-FULL-45-9-A
    O2-12-FULL-49-9
    O2-12-FULL-51-8
    O2-12-FULL-59-9
    O2-FULL-52-10
    OACT-916
    Oak-Vale
    Oceanospirillum-like
    Ochlerotatus-associated
    Odonata-associated
    Of7-15
    oil-degrading
    oil-susceptible
    Okutama-like
    OLZORAL124-38-8
    OM05-12
    Onyong-nyong
    oomycete-like
    Opi-474
    OPU3-GD-OMZ
    oral-nasopharyngeal
    Orange-breasted
    Orange-spotted
    Orthomyxo-like
    OSU-TB39
    Ourmia-like
    oxamyl-degrading
    OXYB2-FULL-32-7
    OXYB2-FULL-41-6
    OXYB2-FULL-48-7
    OXYB2-FULL-48-7-A
    OXYB2-FULL-49-7
    OXYD1-FULL-39-9
    P-decomposing
    P-element
    P1-specific
    P2-11E
    P52-10
    PAH-contaminated
    PALSA-1003
    PALSA-1004
    PALSA-1005
    PALSA-1006
    Palsa-1032
    Palsa-1033
    PALSA-1081
    PALSA-1104
    Palsa-1104
    Palsa-1150
    PALSA-1152
    PALSA-1153
    PALSA-1176
    Palsa-1178
    PALSA-1178
    Palsa-1180
    PALSA-1180
    Palsa-1188
    PALSA-1196
    Palsa-1233
    Palsa-1248
    PALSA-129
    Palsa-1295
    PALSA-1295
    PALSA-1296
    Palsa-1315
    PALSA-1316
    PALSA-1332
    PALSA-1335
    PALSA-1336
    PALSA-1337
    PALSA-1353
    PALSA-1355
    Palsa-1382
    Palsa-1392
    PALSA-1396
    Palsa-1400
    Palsa-1439
    PALSA-1440
    PALSA-1444
    Palsa-1447
    Palsa-147
    Palsa-1515
    Palsa-187
    PALSA-188
    Palsa-189
    PALSA-243
    PALSA-270
    PALSA-278
    Palsa-288
    Palsa-295
    Palsa-343
    PALSA-350
    Palsa-360
    PALSA-457
    Palsa-459
    Palsa-461
    Palsa-465
    Palsa-504
    PALSA-505
    Palsa-506
    Palsa-513
    PALSA-555
    PALSA-600
    Palsa-601
    PALSA-607
    PALSA-610
    PALSA-612
    PALSA-647
    PALSA-660
    Palsa-661
    PALSA-688
    Palsa-688
    PALSA-693
    Palsa-739
    PALSA-743
    Palsa-744
    PALSA-747
    Palsa-851
    Palsa-870
    Palsa-872
    Palsa-875
    Palsa-881
    Palsa-883
    Palsa-89
    Palsa-892
    PALSA-894
    PALSA-911
    PALSA-926
    Palsa-948
    Palsa-948_A
    Palsa-965
    PALSA-967
    PALSA-968
    Paracoccus-like
    Parecho-like
    Paromomycin-resistant
    Partiti-like
    Partitiviridae-like
    Partitivirus-like
    parvo-like
    Parvo-like
    Pasla-948_A
    Passerine-like
    Passionfruit-associated
    PB-SRB1
    PC08-66
    PCB-dechlorinating
    PCC-10605
    PCC-6303
    PCC-6304
    PCC-6307
    PCC-6312
    PCC-6406
    PCC-6802
    PCC-7336
    PCC-7367
    PCC-7407
    PCC-7502
    PCC-9006
    PCC-9228
    PCC-9333
    PD-1
    peach-associated
    Peach-associated
    Pechuel-loeschea
    Peixe-Boi
    Pepper-associated
    Peptococcus-like
    Percalates-clade
    Perchlorate-reducing
    perchlorate-reducing
    Perkinsela-like
    Perkinsiella-like
    peste-des-petits-ruminants
    pesticide-degrading
    Petroica-associated
    Pfiesteria-like
    PH28-bin88
    PHA-producing
    pharmaceutical-degrading
    Pheasant-associated
    Phellodendron-associated
    phenanthrene-degrading
    phenanthrene-metabolizing
    phenol-degrading
    Phenuiviridae-like
    Phlebotomus-associated
    Phoma-like
    PHOS-HE28
    phosphorous-removal
    Picorna-like
    Picornavirales-like
    Pig-tailed
    Pigeon-associated
    Pigeon-dominant
    piggyBac-based
    piggyBac-helper
    Pike-perch
    Pink-eared
    PinkBerry-associated
    Piscirickettsia-like
    Pistacia-associated
    pkBIG-alpha
    PL-Br10-E2g29
    PL-Br10-U2g16
    PLanc-01
    pLEV-TeLS
    PLOWO2-12-38-17
    PLOWO2-12-39-15
    Plut-88865
    Plut-88900
    PMG-095
    PMM-0068
    PN-J185
    pnpB-pnpA
    Po-Circo-like
    Point-Douro
    Polycystine-like
    porcine-murine
    Porphyromonas-like
    Port-miou
    Posa-like
    PreXMRV-1
    PreXMRV-1/PreXMRV-2
    PreXMRV-2
    Promoter-probe
    Promoter-trap
    propionate-degrading
    PRT-SC02
    Pseudo-nitzschia
    Pseudoalteromonas-like
    PTE-S5
    Puumala-like
    PVB_OTU_4-like
    Q3-R57-64
    Qinvirus-like
    QS-1-68-17
    QS-1-68-20
    QS-4-48-99
    QS-4-69-31
    QS-5-54-17
    QS-5-68-33
    QS-5-72-10
    QS-8-64-10
    QS-8-64-29
    QS-9-68-17
    Quinoa-associated
    R-H-3
    R-RK-3
    R3-B46
    R6K-based
    RAAP-1
    RAAP-2
    Raccoon-associated
    RACS-045
    RACS-047
    RAOX-1
    ray-finned
    RBC-AMX1
    RBG-1
    RBG-13-33-26
    RBG-13-36-15
    RBG-13-38-9
    RBG-13-39-12
    RBG-13-40-15
    RBG-13-40-8-A
    RBG-13-41-10
    RBG-13-42-11
    RBG-13-43-14
    RBG-13-43-22
    RBG-13-44-9
    RBG-13-46-10
    RBG-13-46-13
    RBG-13-46-9
    RBG-13-49-15
    RBG-13-50-9
    RBG-13-51-18
    RBG-13-51-36
    RBG-13-52-11
    RBG-13-53-26
    RBG-13-55-18
    RBG-13-61-14
    RBG-13-66-14
    RBG-13-68-16
    RBG-16-34-14
    RBG-16-39-9b
    RBG-16-40-8
    RBG-16-42-10
    RBG-16-42-24
    RBG-16-43-11
    RBG-16-43-9
    RBG-16-45-19
    RBG-16-48-13
    RBG-16-49-10
    RBG-16-49-21
    RBG-16-49-23
    RBG-16-51-14
    RBG-16-51-16
    RBG-16-52-11
    RBG-16-54-11
    RBG-16-54-18
    RBG-16-55-9
    RBG-16-56-11
    RBG-16-57-11
    RBG-16-57-12
    RBG-16-57-9
    RBG-16-58-11
    RBG-16-60-22
    RBG-16-64-13
    RBG-16-64-22
    RBG-16-64-32
    RBG-16-64-43
    RBG-16-65-34
    RBG-16-66-20
    RBG-16-66-30
    RBG-16-68-12
    RBG-16-68-14
    RBG-16-68-21
    RBG-16-71-46
    RBS10-35
    RDHV-like
    Reagent-associated
    Red-bellied
    Red-billed
    Red-capped
    Red-crowned
    Red-eared
    Red-faced
    Red-fronted
    Red-handed
    Red-necked
    Red-recombineering
    red-rumped
    REDSEA-S09-B13
    Regulator-reporter
    Reoviridae-like
    Reptile-associated
    Retro-megavirus
    Retropinna-associated
    RH-AL1
    RH-chloro-G1
    RH2-MAG17b
    Rhabdoviridae-like
    RHBSTW-00994
    Rhinopithecus-associated
    RI-106
    RI-112
    RI-121
    RI-153
    RI-182
    RI-185
    RI-211
    RI-212
    RI-213
    RI-231
    RI-242
    RI-283
    RI-29
    RI-34
    RI-342
    RI-372
    RI-382
    RI-421
    RI-43
    RI-461
    RI-55
    RI-92
    RI-95
    rice-associated
    Rigidomastix-like
    RIOMV-3
    RIOMV-4
    Rm-Bat-CoV
    RML-105355
    rod-shaped
    Rodent-like
    root-associated
    Rosaceae-associated
    Roseobacter-like
    Rous-associated
    rpL23-fusion
    Rs-D84
    Rubrivivax-like
    Rudolf-kamelinia
    RumEn-M2
    RV-African
    RV-Bower
    RV-Brook
    RV-Brown
    RV-Common
    RV-Edible
    RV-European
    RV-Freshwater
    RV-Gharial
    RV-Horse
    RV-Iberian
    RV-Leopard
    RV-Painted
    RV-Palmate
    RV-Pit
    RV-Pufferfish
    RV-Rhinatremid
    RV-Rocket
    RV-Slider
    RV-Stickleback
    RV-Stripe
    RV-Tiger
    RV-Tinamou
    RV-Tuatara
    RVA/Human-wt/MWI/1473/2001/G8P4
    RVA/Human-wt/ZAF/3133WC/2009/G12P4
    RVA/Human-wt/ZAF/3176WC/2009/G12P6
    RVA/Human-wt/ZAF/3203WC/2009/G2P4
    RVW-14
    RYN-111
    RYN-115
    RYN-122
    RYN-124
    RYN-133
    RYN-140
    RYN-150
    RYN-170
    RYN-182
    RYN-227
    RYN-230
    RYN-353
    RYN-404
    RYN-405
    RYN-417
    RYN-425
    RYN-427
    RYN-428
    RYN-431
    RYN-433
    RYN-46
    RYN-50
    RYN-84
    RYN-85
    RYN-87
    S009-29
    S009-33
    S010-17
    S010-25
    S010-51
    S010-53
    S012-109
    S012-113
    S012-128
    S012-135
    S012-138
    S012-15
    S012-22
    S012-25
    S012-40
    S012-47
    S012-72
    S012-77
    S012-89
    S012-90
    S013-73
    S014-12
    S014-18
    S014-19
    S014-41
    S014-50
    S014-63
    S015-18
    S015-6
    S016-13
    S016-19
    S016-27
    S016-38
    S016-54
    S016-85
    S016-97
    S017-3
    S10-like
    S139-112
    S139-113
    S139-126
    S139-133
    S139-149
    S139-169
    S139-18
    S139-21
    S139-23
    S139-57
    S139-71
    S139-92
    S140-20
    S140-43
    S140-46
    S141-128
    S141-161
    S141-23
    S141-3
    S141-38
    S141-43
    S141-50
    S141-55
    S141-66
    S141-70
    S141-80
    S141-84
    S143-107
    S143-136
    S143-140
    S143-145
    S143-32
    S143-33
    S143-34
    S143-42
    S143-5
    S143-63
    S143-66
    S143-82
    S143-98
    S144-17
    S144-22
    S144-23
    S144-34
    S144-35
    S144-45
    S144-5
    S144-51
    S145-11
    S145-15
    S145-22
    S145-25
    S145-28
    S145-42
    S146-22
    S146-30
    S146-37
    S146-42
    S146-62
    S146-64
    S146-70
    S146-73
    S146-80
    S146-88
    S15-B10
    S15B-MN24
    S2-like
    S20-B6
    S3-3
    S36-B12
    S5-A14a
    S5-K13
    S5-like
    S7-like
    SA5d-4
    Saffron-associated
    SAfia-400D
    SAfia-838D_Ambidensovirus
    Saint-Floris
    Sam29DG08-bin30
    Santee-Cooper
    Sapelovirus-like
    Sapporo-HK299
    SAR202-VII-2
    SARS-CoV-2
    SARS-like
    SARS-related
    Saw-shelled
    Saxo-fridericia
    SB-45
    SB3-54
    SCADC1-2-3
    SCBR16-9
    SCGC-AAA003-L08
    SCGC-AAA011-G17
    SCGC-AAA076-P13
    SCGC-AAA160-P02
    SCGC-AAA252-I15
    SCGC-AAA471-B05
    SCGC-AG-212-F23
    SCGC-AG-212-J23
    Sciuro-hypnum
    SCN-52-12
    SCN-57-10
    SCN-67-18
    SCN-69-37
    SCN-70-22
    SCSIO-06110
    SCSIO-12696
    SCSIO-12741
    SCSIO-59966
    SCSIO-75732
    SCUT-3
    SDB-A
    SDU3-3
    Sea-firefly
    Sebastiano-Schaueria
    selenate-reducing
    Selenomonas-like
    Self-inactivating
    Sewage-associated
    SG-bin7
    SG-bin9
    SG8-11
    SG8-13
    SG8-24
    SG8-30
    SG8-32-3
    SG8-38
    SG8-39
    SG8-4
    SG8-40
    SG8-41
    SG8-5
    SG8-52-3
    SH-MAG3
    SHA-98
    Sharp-tailed
    short-finned
    Short-finned
    short-tailed
    Shuttle-expression
    Shuttle/allelic-replacement
    Sikhote-Alin
    simazine-degrading
    Simian-Chimpanzee
    Simian-Human
    Sindbis-like
    Single-copy
    Sint-Jan
    Sisal-associated
    Site-B1
    Site-B15
    Site-specific
    SIV-derived
    SIV-wrc
    SJA-28
    SK-Y3
    SL1-B47
    SLM-1
    SM1-40
    SM1-46
    SM1-50
    SM1-73
    SM1-B181
    SM23-28-2
    SM23-30
    SM23-31
    SM23-32
    SM23-33
    SM23-35
    SM23-39
    SM23-42
    SM23-61
    SM23-62
    SM23-73-2
    SM23-73-3
    SM23-78
    SM23-81
    SM3-B121
    SM3-B41
    SM3-B62
    SMO-BIN1
    SMTZ1-45
    SMTZ1-83
    Snail-Cactus
    SNAP-XDocII
    Snyder-Theilen
    Sodalis-like
    Soft-shell
    Soft-shelled
    Soil-borne
    Solemoviridae-like
    Solidago-associated
    Solms-laubachia
    Soudan-22
    Soudan-30
    Soybean-associated
    Spawner-isolated
    Sphingobacterium-like
    Sphingomonas-like
    Sphinx1.76-related
    Sphinx2.36-related
    Spiro-02
    Spiroplasma-symbiont
    Spirosoma-like
    SPLOWO2-01-44-7
    Spot-less
    SpSt-1050
    SpSt-1133
    SpSt-1165
    SpSt-1174
    SpSt-1190
    SpSt-150
    SpSt-205
    SpSt-223
    SpSt-227
    SpSt-28
    SpSt-313
    SpSt-315
    SpSt-333
    SpSt-398
    SpSt-501
    SpSt-539
    SpSt-583
    SpSt-600
    SpSt-77
    SpSt-899
    SpSt-956
    Spumella-like
    Spur-thighed
    SR-FBR-E99
    SS1-B-03-39
    SSGW-16
    St-Valerien
    star-like
    STB-63
    STB-87
    STC-113
    STC-54
    STC-74
    STD1-117
    STD1-204
    STD1-80
    STD1-84
    STD2-102
    STD2-12
    STD2-154
    STD2-204
    STD2-258
    STD2-77
    STE-11
    STE-63
    STE-71
    STE-72
    Stenotrophomonas-like
    STF1-107
    STF1-138
    STF1-152
    STF1-184
    STF1-284
    STF1-71
    STF1-98
    STF2-111
    STF2-152
    STF2-20
    STF2-203
    STF2-222
    STF2-37
    STF2-47
    Sticholonche-like
    Stolbur-It
    Stolbur-Rus
    stool-associated
    strawberry-associated
    Striga-associated
    String-of-pearls
    Stripa-derived
    Stx1-converting
    Stx1a-converting
    Stx2-converting
    Stx2a-converting
    Stx2c-converting
    Stx2d-converting
    StxXX-converting
    Styphnolobium-associated
    sulfate-reducing
    sulfide-oxidizing
    sulfite-reducing
    sulfo-oxidizing
    sulfur-oxidizing
    Sunn-hemp
    SURF-10
    SURF-12
    SURF-13
    SURF-15
    SURF-16
    SURF-17
    SURF-19
    SURF-21
    SURF-23
    SURF-24
    SURF-25
    SURF-27
    SURF-28
    SURF-3
    SURF-32
    SURF-33
    SURF-36
    SURF-37
    SURF-38
    SURF-40
    SURF-43
    SURF-47
    SURF-5
    SURF-52
    SURF-54
    SURF-55
    SURF-56
    SURF-57
    SURF-58
    SURF-60
    SURF-62
    SURF-65
    SURF-7
    SURF-8
    SURF-9
    SV3-B141
    SV3-B162
    Sva-07
    Sva-25
    SW-10-69-26
    SW-11-45-7
    SW-4-43-9
    SW-4-49-11
    SW-6-46-9
    SW-7-71-33
    SW-9-67-25
    Synechococcus-like
    Synechocystis-like
    Syner-03
    SYSU-D60014
    SYSU-D60015
    SZ-28-30
    SZAS-1
    SZAS-103
    SZAS-13
    SZAS-15
    SZAS-17
    SZAS-18
    SZAS-26
    SZAS-3
    SZAS-5
    SZAS-6
    SZAS-7
    SZAS-72
    SZAS-79
    SZUA-100
    SZUA-104
    SZUA-1055
    SZUA-115
    SZUA-116
    SZUA-117
    SZUA-124
    SZUA-1377
    SZUA-140
    SZUA-1403
    SZUA-1433
    SZUA-1435
    SZUA-1443
    SZUA-1452
    SZUA-146
    SZUA-147
    SZUA-149
    SZUA-1492
    SZUA-1497
    SZUA-150
    SZUA-1508
    SZUA-1514
    SZUA-1515
    SZUA-152
    SZUA-1535
    SZUA-161
    SZUA-178
    SZUA-179
    SZUA-181
    SZUA-182
    SZUA-185
    SZUA-186
    SZUA-191
    SZUA-204
    SZUA-21
    SZUA-217
    SZUA-22
    SZUA-224
    SZUA-226
    SZUA-229
    SZUA-230
    SZUA-232
    SZUA-236
    SZUA-24
    SZUA-242
    SZUA-250
    SZUA-252
    SZUA-254
    SZUA-259
    SZUA-265
    SZUA-272
    SZUA-287
    SZUA-3
    SZUA-309
    SZUA-318
    SZUA-320
    SZUA-33
    SZUA-336
    SZUA-35
    SZUA-359
    SZUA-36
    SZUA-360
    SZUA-365
    SZUA-368
    SZUA-37
    SZUA-370
    SZUA-371
    SZUA-378
    SZUA-38
    SZUA-381
    SZUA-401
    SZUA-41
    SZUA-411
    SZUA-415
    SZUA-421
    SZUA-423
    SZUA-425
    SZUA-426
    SZUA-430
    SZUA-442
    SZUA-444
    SZUA-448
    SZUA-451
    SZUA-46
    SZUA-467
    SZUA-47
    SZUA-472
    SZUA-474
    SZUA-486
    SZUA-493
    SZUA-5
    SZUA-500
    SZUA-51
    SZUA-521
    SZUA-533
    SZUA-534
    SZUA-540
    SZUA-544
    SZUA-545
    SZUA-547
    SZUA-55
    SZUA-561
    SZUA-567
    SZUA-575
    SZUA-580
    SZUA-582
    SZUA-584
    SZUA-592
    SZUA-6
    SZUA-601
    SZUA-605
    SZUA-611
    SZUA-76
    SZUA-79
    SZUA-8
    SZUA-82
    SZUA-87
    T-DNA
    T-vector
    T1Sed10-126
    T1SED10-157
    T1SED10-198M
    T1SED10-24
    T1SED10-26
    T1SED10-28
    T1Sed10-49
    T1Sed10-7
    T1SED10-78
    T1SED10-81
    T1SED10-84
    T1Sed10-97
    T1SED10-99
    T33-2
    T3Sed10-11
    T3Sed10-140
    T3Sed10-183
    T3Sed10-213
    T3Sed10-241
    T3SED10-260
    T3Sed10-336
    T3SED10-354
    T3SED10-47
    T3Sed10-66
    T3SED10-7
    T4-like
    T7-like
    TA-13
    TA-20
    TA-21
    TAE3-ERU5
    Taro-associated
    Taxeobacter-like
    Tb-RUG-53
    Tb-RUG-64
    Tb-RUG-66
    TC-15
    TC-195
    TC-71
    Tc-Br11
    Tc-Br11-B2g6-7
    Tc-Br11-E2g1
    Tc-Br11-E2g8
    TCS-49
    Tea-oil
    TEKIR-12S
    TEKIR-14
    TEKIR-21
    TEKIR-8
    Temperature-sensitive
    Tener-01
    Tent-making
    Teosinte-associated
    Tetracycline-inducible
    TFI-002
    TH-plancto1
    TH1-2
    Theileria-related
    Theilers-like
    thiocyanate-degrading
    thiosulfate-disproportionating
    thiosulfate-reducing
    Thrips-associated
    Ti-curing
    Tick-associated
    tick-borne
    Tick-borne
    TIM5-like
    TIUS-1
    TK10-74A
    TMED-70
    TMP-24
    TMP-7
    TMP-86
    TNE-4
    toluene-degrading
    Tombus-like
    Toti-like
    Totivirus-like
    Tous-C4FEB
    Tri-shuttle
    trichloroacetic-acid-degrading
    triclosan-degrading
    Trifolium-associated
    tris(2-butoxyethyl)
    TSE-associated
    TTA-H9
    TTV-like
    Twisted-stalk
    TYLCAxV-Sic1-IT:Sic2/2:04
    TYLCAxV-Sic2-IT:Sic2/5:04
    TYLCCNV-Y322
    tylosin-resistant
    Type-B
    Type-C
    Type-D
    Type-E
    Type-F
    U1Ap-mCherry
    UAG-readthrough
    UAS-less
    UAS-Red
    UAS-Stinger
    UDEA-SF1
    Ui-te-Rangiora
    UKL13-2
    UKL13-3
    UM-FILTER-33-9
    UM-FILTER-34-10
    UM-FILTER-34-12
    UM-FILTER-36-11
    UM-FILTER-36-13
    UM-FILTER-36-7
    UM-FILTER-37-10
    UM-FILTER-37-12
    UM-FILTER-38-24
    UM-FILTER-39-15
    UM-FILTER-39-64
    UM-FILTER-40-11
    UM-FILTER-40-16
    UM-FILTER-40-17
    UM-FILTER-40-9
    UM-FILTER-41-14
    UM-FILTER-41-21
    UM-FILTER-41-35
    UM-FILTER-42-10
    UM-FILTER-42-14
    UM-FILTER-42-9
    UM-FILTER-43-11
    UM-FILTER-43-11-A
    UM-FILTER-45-8
    UM-FILTER-46-8
    UM-FILTER-47-13
    UM-FILTER-48-10
    UM-FILTER-49-7
    UM-FILTER-50-9
    UM-FILTER-53-11
    UM-FILTER-53-9
    UM-FILTER-63-12
    UphvI-Ar2
    USCg-Taylor
    UWMA-0041
    UWMA-0117
    UWMA-0217
    UWMA-0277
    UWMA-0346
    UWPOB-MICAV1
    Uzun-Agach
    V1-115
    V1-33
    VA-1
    Vaccinium-associated
    Van-royena
    VDV-1/DWV
    Vera-duthiea
    Verruco-01
    VESV-like
    Violaceae-associated
    Virga-like
    Virgaviridae-like
    Viroid-like
    Virus-associated
    Virus-induced
    Virus-like
    Visna-maedi
    Vittaforma-like
    VKM-B-2647
    VKMB-3413
    W-Chloroflexi-9
    W-Firmicutes-14
    W02-35-19
    W0P28-013
    W0P31-042
    W0P33-017
    W1P20-047
    W2P13-069
    W3P20-009
    WA-like
    wall-less
    WCHB1-27
    WCHB1-69
    Wei-like
    WG-1
    WGA-3G
    WGA-4E
    WH-5701
    White-eye
    White-flower
    White-headed
    white-rot
    White-rumped
    White-sloanea
    White-tailed
    white-throated
    White-tufted-ear
    whitefly-associated
    Whitefly-associated
    Whitefly-transmitted
    WM-009
    WM-3-3
    WO2-01-48-18
    WO2-35-19
    WO2-39-10b
    WO2-40-15
    WO2-42-50
    WO2-43-9
    WO2-44-18
    WO2-46-25
    WO2-47-17b
    WO2-48-25
    WO2-49-40
    WO2-54-10
    WO2-FULL-39-10
    WO2-FULL-39-9
    WO2-FULL-40-11
    WOR-1
    WOR-3
    WOR-SM1-SCG
    WPS-2
    WS-11
    WS-2
    WS-3
    WS-7
    WS-8
    WS-9
    WYZ-LMO10
    WYZ-LMO11
    WYZ-LMO2
    WYZ-LMO6
    WYZ-LMO8
    X-symbiont
    Xanthomonas-like
    XMRV-like
    XMRV-related
    Xuzhou-associated
    XY-R5
    XYA12-FULL-58-9
    XYA2-FULL-35-8
    XYA2-FULL-39-19
    XYA2-FULL-43-10
    XYA2-FULL-43-15
    XYA2-FULL-47-19
    XYA2-FULL-47-53
    XYB1-36-10
    XYB1-40-15
    XYB1-FULL-32-11
    XYB1-FULL-40-15
    XYB1-FULL-42-16
    XYB1-FULL-44-23
    XYB1-FULL-47-11
    XYB12-FULL-38-5
    XYB2-FULL-36-35
    XYB2-FULL-38-15
    XYB2-FULL-45-11
    XYB2-FULL-45-9
    XYB2-FULL-48-7
    XYB2-FULL-49-35
    XYB2-FULL-50-12
    XYC1-FULL-37-79
    XYC1-FULL-56-13
    XYC2-34-220
    XYC2-FULL-35-21
    XYC2-FULL-37-10
    XYC2-FULL-38-9
    XYC2-FULL-41-22
    XYC2-FULL-46-14
    XYC2-FULL-47-12
    XYD1-37-29
    XYD1-FULL-39-28
    XYD1-FULL-40-9
    XYD1-FULL-44-32
    XYD1-FULL-46-19
    XYD1-FULL-53-11
    XYD1-FULL-55-31
    XYD12-FULL-36-8
    XYD12-FULL-50-9
    XYD2-FULL-39-9
    XYD2-FULL-41-14
    XYD2-FULL-41-16
    XYD2-FULL-41-8
    XYD2-FULL-50-16
    xylene-degrading
    XZGYJ-43
    Y-symbiont
    YA12-FULL-48-11
    YA12-FULL-51-18
    YA12-FULL-60-10
    YA12-FULL-61-11
    Yaba-7
    Yaba-like
    YAC-BAC
    Yado-kari
    yado-kari
    Yado-nushi
    Yaounde-like
    YB1-3
    YB12-FULL-65-16
    YB3-1
    YC12-FULL-57-11
    YC2-FULL-38-9
    YD12-FULL-39-22
    YD12-FULL-57-12
    YD12-FULL-61-37
    Yellow-bellied
    Yellow-breasted
    Yellow-eyed
    Yellow-fronted
    yellow-necked
    Yellow-wax
    Yes70-04
    Yes70-35
    Yes71-27
    YIM-151500
    YIM-152171
    YIM-78166
    YIM-B00363
    YT-029
    YT-063
    YT1-039
    YT1-057
    YT1-065
    Z-2901
    Z2-YC6860
    ZCTH02-B2
    ZCTH02-B3
    ZCTH02-B6
    ZG21-1
    Zgenome-0002
    Zgenome-0027
    Zostera-associated
    );

const my @SPECIES => qw(
    023Pt_psg01
    04_1_1
    070605-23_N11_16
    0a_pSin-ctrl
    0b_pSin-L7Ae
    100_oclvp78
    101_oclvp44
    102_oclvp93
    103_oclvp94
    104_oclvp99
    105_oclvp561
    106_oclvp92
    107_oclvp18
    108_oclvp22
    109_oclvp514
    10_xplvp428
    110_oclvp17
    111_oclvp16
    112_oclvp26
    113_oclvp23
    114_oclvp28
    115_oclvp27
    1168_2009
    116_oclvp532
    1178_2009
    117_oclvp558
    118_39
    118_oclvp546
    119_MR-2014
    119_xplvp419
    11_xplvp442
    11GM_10_3
    120_oclvp102
    121_oclvp513
    1228_2009
    122_oclvp550
    123_oclvp549
    124_oclvp198
    125_oclvp12
    126_oclvp498
    127_xplvp336
    128_xplvp197
    129_MR-2014
    129_xplvp391
    12_oclvp105
    130_MR-2014
    130_xplvp347
    131_xplvp388
    132_xplvp394
    133_xplvp560
    134_xplvp393
    135_xplvp202
    136_MR-2014
    136_xplvp446
    137_MR-2014
    137_xplvp455
    138_xplvp451
    139_xplvp295
    13_1_20CM_2_51_10
    13_1_20CM_2_51_12
    13_1_20CM_2_54_9
    13_1_20CM_2_63_93
    13_1_20CM_3_53_6
    13_1_20CM_4_52_8
    13_1_20CM_52_20
    13_1_40CM_2_52_13
    13_1_40CM_2_52_4
    13_1_40CM_2_55_8
    13_1_40CM_3_53_8
    13_1_40CM_4_53_4
    13_1_40CM_4_53_8
    13_2_20CM_2_52_21
    13_2_20CM_2_53_6
    13_xplvp186
    140_xplvp374
    141_xplvp294
    142_xplvp291
    143_xplvp334
    144_xplvp290
    145_xplvp335
    146_xplvp378
    147_xplvp204
    1489_2009
    148_xplvp1
    1490_2009
    149_xplvp363
    14_xplvp362
    150907-44_B22_ML8_968
    150907-44_D20_ML1_713
    150907-44_F20_ML2_781
    150907-44_H20_ML3_785
    150907-44_L20_ML5_758
    150907-44_N20_ML6_768
    150907-44_P20_ML7_756
    150_xplvp379
    151_xplvp405
    152_xplvp404
    153_xplvp376
    154_xplvp377
    155_xplvp369
    156_xplvp375
    157_xplvp365
    158_xplvp352
    159_xplvp212
    15_xplvp210
    160_xplvp214
    161_xplvp465
    162_xplvp226
    163_xplvp569
    164_xplvp565
    165_MR-2014
    165_xplvp467
    166_oclvp485b
    167_xplvp346
    168_xplvp345
    169_xplvp344
    16_xplvp576
    16F2019_GII.p16_Guangdong_CHN_2016
    16F2025_GII.p16_Guangdong_CHN_2016
    16F2026_GII.p16_Guangdong_CHN_2016
    16F2029_GII.p16_Guangdong_CHN_2016
    16F21132_GII.p16_Guangdong_CHN_2016
    16F21134_GII.p16_Guangdong_CHN_2016
    16F2149_GII.2_Guangdong_CHN_2016
    16F2152_GII.2_Guangdong_CHN_2016
    16F2161_GII.2_Guangdong_CHN_2016
    16F2284_GII.p16_Guangdong_CHN_2016
    16F2285_GII.p16_Guangdong_CHN_2016
    16SF2025_GII.2_Guangdong_CHN_2016
    16SF2026_GII.2_Guangdong_CHN_2016
    16SF2029_GII.2_Guangdong_CHN_2016
    16SF2030_GII.2_Guangdong_CHN_2016
    16SF21132_GII.2_Guangdong_CHN_2016
    16SF21134_GII.2_Guangdong_CHN_2016
    16SF2151_GII.2_Guangdong_CHN_2016
    16SF2153_GII.2_Guangdong_CHN_2016
    16SF2285_GII.2_Guangdong_CHN_2016
    170_xplvp340
    171_xplvp255
    172_xplvp245
    173_xplvp338
    174_xplvp339
    175_xplvp267
    176_xplvp289
    177_xplvp288
    178_oclvp200
    179_oclvp274
    17_xplvp121
    180_oclvp537
    181_MR-2014
    181_oclvp548
    182_oclvp528
    183_oclvp263
    184_MR-2014
    184_oclvp523
    185_oclvp234
    186_oclvp236
    187_oclvp241
    188_oclvp265
    189_oclvp280
    18_xplvp100
    190_oclvp281
    191_oclvp493
    192_oclvp270
    193_oclvp262
    194_oclvp520
    195_oclvp221
    196_oclvp220
    197_oclvp508
    198_oclvp491
    199_oclvp489
    19_xplvp109
    1_xplvp296
    1a_pSin-SEAP-PP7
    1AZ_DGR
    1b_pSin-SEAP-MS2
    1c_pSin-secNluc-PP7
    1d_pSin-secNluc-MS2
    200_oclvp497
    2010_Msc_17636_VA3/HMO
    2010_Msc_18001_MLB
    201_oclvp283
    202_oclvp284
    203_oclvp551
    204_oclvp480
    205_oclvp481
    206_oclvp485
    207_oclvp487
    20_xplvp148
    2126_10
    21_oclvp54
    21a7_large
    21a7_orange
    21a7_pink
    21a7_small
    22_MR-2014
    22_xplvp150
    22B9_orange
    22C12_green
    22c12_orange
    22E1_Frozen
    2316_26
    2316_32
    23_oclvp517
    24_xplvp459
    25_xplvp140
    26_xplvp434
    27_xplvp395
    2885_18H
    2885_18K
    2885_18L
    28_xplvp13
    29_xplvp147
    2_xplvp435
    2A3B_DGR
    2a_pSin-PCP-cNOT7
    2ANA_DGR
    2b_pSin-PCP-ppvS-cNOT7
    2c_pSin-PCP-tevS-cNOT7
    2d_pSin-PCP-cNOT7-ppvS-degron
    2e_pSin-degron-tevS-PCP-cNOT7
    2f_pSin-PPVp-AND-TEVp
    2g_pSin-PPVp-OR-TEVp
    2h_pSin-PPVp-NAND-TEVp
    2i_pSin-PPVp-NOR-TEVp
    2j_pSin-PPVp-XOR-TEVp
    2k_pSin-PPVp-XNOR-TEVp
    2l_pSin-PPVp-IMPLY-TEVp
    2m_pSin-PPVp-NIMPLY-TEVp
    2n_pSin-TEVp-IMPLY-PPVp
    2o_pSin-TEVp-NIMPLY-PPVp
    2p_pSin-PCP-tevS-cNOT7-MS2
    3-5_YC-ZSS-LKJ-3024
    3-7_YC-ZSS-LKJ-1723
    30_xplvp384
    31_xplvp325
    32_xplvp209
    33_17
    33_xplvp322
    34_xplvp425
    35_xplvp171
    36_xplvp225
    37_xplvp454
    380_MR-2014
    38_xplvp95
    397_MR-2014
    39_xplvp47
    3_xplvp302
    3a_pSin-MCP-cNOT7
    3b_pSin-MCP-smvS-cNOT7
    3c_pSin-MCP-tvmvS-cNOT7
    3d_pSin-MCP-cNOT7-smvS-degron
    3e_pSin-degron-tvmvS-MCP-cNOT7
    3f_pSin-SMVp-AND-TVMVp
    3g_pSin-SMVp-OR-TVMVp
    3h_pSin-SMVp-NAND-TVMVp
    3i_pSin-SMVp-NOR-TVMVp
    3j_pSin-SMVp-XOR-TVMVp
    3k_pSin-MCP-smvS-cNOT7-PP7
    4-1_YC-ZSS-LKJ-010-0902
    40_xplvp46
    41_xplvp469
    42_11
    42_xplvp429
    43_xplvp216
    44_xplvp252
    453a_as
    453b_as
    45_oclvp525
    46_xplvp399
    47_oclvp120
    480_MR-2014
    48_xplvp213
    495_MR-2014
    49_oclvp161
    4_57p
    4_xplvp418
    4a_pSin-PPVp
    4b_pSin-Kt-PPVp
    4c_pSin-TEVp
    4d_pSin-nTEVp-PLY1-2A-ABI-cTEVp
    4e_pSin-nTEVp-smvS-PLY1-2A-ABI-smvS-cTEVp
    4f_pSin-nTEVp-smvS-tvmvS-PLY1-2A-ABI-smvS-tvmvS-cTEVp
    4g_pSin-SMVp
    4h_pSin-SMVp-PP7
    4i_pSin-nSMVp-tevs-tevs-cSMVp-MS2
    4j_pSin-nSMVp-FRB-2A-FKBP-cSMVp
    4k_pSin-nSMVp-tevS-FRB-2A-FKBP-tevS-cSMVp
    4l_pSin-TVMVp
    4m_pSin-nTVMVp-CIBN-2A-CRY2-cTVMVp
    4n_pSin-nTVMVp-tevS-CIBN-2A-CRY2-tevS-cTVMVp
    507_MR-2014
    50_oclvp203
    51_oclvp114
    52_oclvp282
    533_MR-2014
    537_MR-2014
    53_oclvp165
    54_oclvp90
    55_oclvp119
    56_oclvp163
    57_xplvp332
    58_oclvp104
    59_13
    59_oclvp166
    5_25as
    5_xplvp440
    5P_1
    5P_2
    5P_3
    603_MR-2014
    60_oclvp187
    61_oclvp108
    62_oclvp107
    63_oclvp173
    64_12
    64_oclvp174
    65_10
    65_oclvp553
    66_1
    66_12
    66_oclvp134
    67_12
    67_oclvp138
    68_12
    68_oclvp142
    69_oclvp189
    6_xplvp63
    70_oclvp488
    71_18
    71_oclvp188
    72_oclvp145
    73_oclvp578
    73_VP7_H6
    73_VP8_A3
    73_VP8_A4
    73_VP8_B3
    73_VP8_B4
    73_VP8_C3
    73_VP8_C4
    73_VP8_D3
    73_VP8_D4
    73_VP8_E3
    73_VP8_E4
    73_VP8_F3
    73_VP8_G3
    73_VP8_H3
    73_VP9_A1
    74_oclvp132
    755_MR-2014
    75_oclvp484
    763_MR-2014
    76_oclvp68
    774_MR-2014
    77_oclvp66
    78_oclvp237
    79_oclvp69
    7_oclvp249
    80_oclvp279
    81_oclvp45
    82_oclvp510
    82_VP7_A8
    82_VP7_B8
    82_VP7_D8
    82_VP7_E7
    82_VP7_E8
    82_VP7_F8
    82_VP7_G7
    82_VP7_H7
    82_VP9_B1
    82_VP9_D1
    82_VP9_E1
    83_oclvp219
    84_MR-2014
    84_oclvp575
    85_oclvp50
    86_oclvp5
    87_oclvp554
    89_oclvp526
    8_xplvp75
    90_oclvp79
    91_11
    91_27
    91_oclvp479
    92_oclvp33
    93_oclvp84
    94_oclvp83
    95_oclvp82
    96_oclvp81
    97_xplvp333
    98_oclvp43
    99_oclvp542
    9_xplvp439
    A18_CLW
    A19_CLW
    A1_395
    A22_CLW
    A3M_VP7_A10
    A3M_VP7_A12
    A3M_VP7_A9
    A3M_VP7_B10
    A3M_VP7_B11
    A3M_VP7_B9
    A3M_VP7_C10
    A3M_VP7_C12
    A3M_VP7_C9
    A3M_VP7_D10
    A3M_VP7_D11
    A3M_VP7_D12
    A3M_VP7_D9
    A3M_VP7_E10
    A3M_VP7_E9
    A3M_VP7_F10
    A3M_VP7_F11
    A3M_VP7_F9
    A3M_VP7_G10
    A3M_VP7_G9
    A3M_VP7_H10
    A3M_VP7_H8
    A3S_VP5_A6
    A3S_VP5_A8
    A3S_VP5_B7
    A3S_VP5_B8
    A3S_VP5_C7
    A3S_VP5_C8
    A3S_VP5_D7
    A3S_VP5_D8
    A3S_VP5_E7
    A3S_VP5_E8
    A3S_VP5_F7
    A3S_VP5_G6
    A3S_VP5_G8
    A3S_VP8_A5
    A3S_VP8_B5
    A3S_VP8_C5
    A3S_VP8_G4
    A3T_VP5_A5
    A3T_VP5_B4
    A3T_VP5_B5
    A3T_VP5_C4
    A3T_VP5_D4
    A3T_VP5_D5
    A3T_VP5_E4
    A3T_VP5_E5
    A3T_VP5_F4
    A3T_VP5_F5
    A3T_VP5_G4
    A3T_VP5_G5
    A3T_VP7_C6
    A3T_VP7_G12
    A3T_VP7_H12
    A3T_VP8_A1
    A3T_VP8_B1
    A3T_VP8_C1
    A3T_VP8_D1
    A3T_VP8_E1
    A3T_VP8_F1
    A3T_VP8_G1
    A4M_VP5_A10
    A4M_VP5_A11
    A4M_VP5_A9
    A4M_VP5_B10
    A4M_VP5_B9
    A4M_VP5_C10
    A4M_VP5_C9
    A4M_VP5_D10
    A4M_VP5_D9
    A4M_VP5_E10
    A4M_VP5_E9
    A4M_VP5_F10
    A4M_VP5_F9
    A4M_VP5_G10
    A4M_VP5_G9
    A4M_VP5_H10
    A4M_VP8_A6
    A4M_VP8_C6
    A4M_VP8_D5
    A4M_VP8_D6
    A4M_VP8_F5
    A4M_VP8_F6
    A4M_VP8_G5
    A4M_VP8_G6
    A4M_VP8_H5
    A4S_VP6_A4
    A4S_VP6_B3
    A4S_VP6_B6
    A4S_VP6_C3
    A4S_VP6_C4
    A4S_VP6_C5
    A4S_VP6_D3
    A4S_VP6_F3
    A4S_VP6_G2
    A4S_VP6_G4
    A4S_VP6_G5
    A4S_VP6_H2
    A4S_VP6_H3
    A4S_VP8_A11
    A4S_VP8_H10
    A4T_VP5_A12
    A4T_VP5_B12
    A4T_VP5_C12
    A4T_VP5_D11
    A4T_VP5_D12
    A4T_VP5_E11
    A4T_VP5_E12
    A4T_VP5_F11
    A4T_VP5_F12
    A4T_VP5_G11
    A4T_VP5_G12
    A4T_VP5_H11
    A4T_VP5_H12
    A4T_VP6_A1
    A4T_VP6_A2
    A4T_VP6_B1
    A4T_VP6_B2
    A4T_VP6_C1
    A4T_VP6_C2
    A4T_VP6_D1
    A4T_VP6_D2
    A4T_VP6_E1
    A4T_VP6_E2
    A4T_VP6_F2
    A5M_1
    A5M_2
    A5M_VP8_B2
    A5M_VP8_C2
    A5M_VP8_D2
    A5M_VP8_E2
    A5M_VP8_F2
    A5M_VP8_G2
    A5M_VP8_H2
    A5S_VP6_A10
    A5S_VP6_B10
    A5S_VP6_B11
    A5S_VP6_D11
    A5S_VP6_F10
    A5S_VP6_F9
    A5S_VP6_G11
    A5S_VP6_G9
    A5S_VP6_H11
    A5S_VP6_H9
    A5S_VP9_B2
    A5T_VP6_A7
    A5T_VP6_A8
    A5T_VP6_A9
    A5T_VP6_B7
    A5T_VP6_B9
    A5T_VP6_C6
    A5T_VP6_C7
    A5T_VP6_C9
    A5T_VP6_E6
    A5T_VP6_E7
    A5T_VP6_E8
    A5T_VP6_F6
    A5T_VP6_F8
    A5T_VP6_G6
    A5T_VP6_G7
    A5T_VP6_G8
    A5T_VP6_H7
    A5T_VP8_A12
    A5T_VP8_B12
    A5T_VP8_E12
    A5T_VP8_H11
    A6M_VP6_B12
    A6M_VP6_C12
    A6M_VP6_D12
    A6M_VP6_H12
    A6M_VP7_A2
    A6M_VP7_A3
    A6M_VP7_B2
    A6M_VP7_C1
    A6M_VP7_D1
    A6M_VP7_D2
    A6M_VP7_E1
    A6M_VP7_F1
    A6M_VP7_G1
    A6M_VP7_H1
    A6M_VP8_B11
    A6M_VP8_C11
    A6M_VP8_D11
    A6M_VP8_E11
    A6M_VP8_F11
    A6S_VP7_A6
    A6S_VP7_B5
    A6S_VP7_B6
    A6S_VP7_C5
    A6S_VP7_D5
    A6S_VP7_D6
    A6S_VP7_E5
    A6S_VP7_E6
    A6S_VP7_F4
    A6S_VP7_F5
    A6S_VP7_F6
    A6S_VP7_G4
    A6S_VP7_G5
    A6S_VP7_H4
    A6S_VP7_H5
    A6S_VP8_A10
    A6S_VP8_A9
    A6S_VP8_B10
    A6S_VP8_F9
    A6S_VP8_H8
    A6S_VP8_H9
    A6T_VP7_A4
    A6T_VP7_A5
    A6T_VP7_B3
    A6T_VP7_B4
    A6T_VP7_C3
    A6T_VP7_C4
    A6T_VP7_D3
    A6T_VP7_D4
    A6T_VP7_E2
    A6T_VP7_E3
    A6T_VP7_E4
    A6T_VP7_F2
    A6T_VP7_F3
    A6T_VP7_G2
    A6T_VP7_G3
    A6T_VP7_H3
    A6T_VP8_A8
    A6T_VP8_B7
    A6T_VP8_D7
    A6T_VP8_F7
    A6T_VP8_G7
    A6T_VP8_H6
    A_20
    abscessus_A
    abyssi_A
    acanthamoebae_A
    acetatoxydans_A
    aceti_B
    acetivorans_A
    acetylicum_B
    acetylicum_C
    achromogenes_A
    acidaminiphila_A
    acidiceleris_A
    acidiceleris_B
    acidilactici_A
    acidiphilum_A
    acidocaldarius_A
    acidophilus_A
    acidophilus_C
    acidovorans_A
    actinomycetemcomitans_A
    actinomycetemcomitans_B
    activa_A
    activa_B
    actuosus_A
    ADB_BD003
    ADB_BD015
    ADB_BD042
    ADB_BD111
    adecarboxylata_A
    adecarboxylata_B
    adecarboxylata_C
    adecarboxylata_D
    adhaerens_A
    adhaerens_B
    adhaerens_D
    adhaerens_E
    adhaerens_G
    adhaerens_H
    adhaerens_I
    adiacens_A
    Ae_JM910
    aeolica_A
    aeolicus_A
    aeolicus_B
    aeriphila_A
    aerocolonigenes_A
    aerofaciens_A
    aerofaciens_AA
    aerofaciens_AB
    aerofaciens_AC
    aerofaciens_AD
    aerofaciens_AE
    aerofaciens_AF
    aerofaciens_AG
    aerofaciens_AH
    aerofaciens_AI
    aerofaciens_AJ
    aerofaciens_AK
    aerofaciens_AL
    aerofaciens_AM
    aerofaciens_AN
    aerofaciens_AO
    aerofaciens_AP
    aerofaciens_AQ
    aerofaciens_AR
    aerofaciens_AS
    aerofaciens_AT
    aerofaciens_AU
    aerofaciens_AV
    aerofaciens_AW
    aerofaciens_AX
    aerofaciens_AY
    aerofaciens_AZ
    aerofaciens_BA
    aerofaciens_E
    aerofaciens_F
    aerofaciens_G
    aerofaciens_H
    aerofaciens_I
    aerofaciens_J
    aerofaciens_K
    aerofaciens_L
    aerofaciens_M
    aerofaciens_N
    aerofaciens_O
    aerofaciens_P
    aerofaciens_Q
    aerofaciens_R
    aerofaciens_S
    aerofaciens_T
    aerofaciens_V
    aerofaciens_W
    aerofaciens_X
    aerofaciens_Y
    aerofaciens_Z
    aerogenes_A
    aerosaccus_A
    aerosaccus_B
    aerosaccus_C
    aeruginosa_C
    aeruginosa_D
    aeruginosa_E
    aeruginosa_F
    aeruginosa_G
    aeruginosa_H
    aeruginosa_I
    aeruginosavorus_A
    aeruginosavorus_B
    aeruginosavorus_C
    aestuarii_A
    aethiopicum_A
    AF_NAK1-3
    agalactiae_A
    agalactiae_B
    agarilyticus_A
    agarilyticus_B
    agarilyticus_C
    agarilyticus_D
    agarivorans_A
    agassizii_A
    AGG1095_KMOsgRNAs
    AGG1186_pIE1-Koz-FF-SV4
    AGG1187_-_pIE1-Lep-FF-S
    AGG1188_pIE1-BmHi-FF-SV
    AGG1189_-_pIE1-BmLo-FF-
    AGG1191_-_pIE1-Koz-FF-K
    AGG1192_-_pIE1-Lep-FF-K
    AGG1193_-_pIE1-BmHi-FF-
    AGG1194_-_pIE1-BmLo-FF-
    AGG1196_-_pIE1-Koz-FF-P
    AGG1197_-_pIE1-Lep-FF-P
    AGG1198_-_pIE1-BmHi-FF-
    AGG1199_-_pIE1-BmLo-FF-
    AGG1207_AeBGCN-Cas9
    AGG1208_sds3-Cas9
    AGG1245_Pub_AaHyPiggyBac_SV40
    AGG1448_zpg2kb-Cas9
    AGG1547_nos-Cas9
    AGG1551_shu-Cas9
    AGG1552_Ewald-Cas9
    AGG1687_zpg4kb-Cas9
    AGG1747_pGL3_PUb(-565)-FF
    AGG1748_pGL3_PUb(-465)-FF
    AGG1749_pGL3_PUb(-365)-FF
    AGG1750_pGL3_PUb(-265)-FF
    AGG1751_pGL3_PUb(-143)-FF
    AGG1752_pGL3_PUb(-133_77bp-intron)-FF
    AGG2143_PUb(-133_77bp-intron)-AmC-SV40
    AGG2144_PUb(-265)-AmC-SV40
    AGG2145_PUb(-565)-AmC-SV40
    AGG2146_PUb(-2565)-AmC-SV40
    AGG2196_pGL3_PUb(-2565)-FF
    aggregans_A
    aggregans_B
    aggregatum_A
    aggregatum_B
    aggregatum_C
    agilis_B
    agilis_D
    agilis_E
    agri_A
    AH_HY1
    aidingensis_A
    Al31_1
    alba_A
    alba_B
    alba_C
    alba_D
    albidus_A
    albireticuli_B
    albulus_A
    albulus_C
    album_A
    album_C
    album_D
    albus_A
    albus_B
    albus_E
    alcalifaciens_A
    alcalifaciens_B
    alcalifaciens_C
    alcaligenes_A
    alcaligenes_B
    alcaligenes_C
    alcaliphila_B
    alexandrii_A
    algarum_A
    algericum_A
    algicola_B
    alimentarius_A
    aliphaticivorans_A
    aliphaticivorans_B
    alkaliantarctica_A
    alkalilenta_A
    alkaliphila_A
    alkaliphila_B
    alkaliphilum_B
    alkaliphilus_A
    alkaliphilus_B
    alkylphenolica_A
    alkylphenolica_B
    alocis_A
    alpina_A
    alpinus_A
    alpinus_C
    altamirensis_A
    altamirensis_B
    alvei_B
    alvi_B
    alvi_C
    alvi_D
    alvi_E
    alvi_G
    amalonaticus_A
    amalonaticus_C
    amalonaticus_E
    amalonaticus_F
    amalonaticus_G
    amamiensis_A
    amazonense_A
    amazonense_B
    amazonense_C
    ambifaria_A
    americana_A
    aminisulfidivorans_A
    aminophilum_A
    aminophilus_A
    aminovorans_B
    amorphae_A
    ampullae_A
    amycolatum_A
    amygdalina_A
    amylolytica_A
    amylolyticus_A
    amylolyticus_B
    amylolyticus_D
    amylolyticus_E
    anatipestifer_B
    anatipestifer_C
    anatipestifer_D
    anatipestifer_E
    anguilliseptica_A
    angulatum_A
    angusta_A
    angustum_A
    anhuiense_A
    animalis_B
    anomala_A
    anseris_A
    anserisalpingitidis_A
    ansorpii_A
    antarctica_A
    antarctica_B
    antarctica_C
    antarcticus_A
    anthina_B
    antibioticus_A
    antibioticus_B
    antiquum_A
    anulatus_C
    AO2_32
    aphanizomenoides_A
    aphidicola_A
    aphidicola_AA
    aphidicola_AB
    aphidicola_AC
    aphidicola_AD
    aphidicola_AE
    aphidicola_AF
    aphidicola_AG
    aphidicola_AH
    aphidicola_AI
    aphidicola_AJ
    aphidicola_AK
    aphidicola_AL
    aphidicola_AM
    aphidicola_AN
    aphidicola_AO
    aphidicola_AP
    aphidicola_AQ
    aphidicola_AR
    aphidicola_AS
    aphidicola_AT
    aphidicola_AU
    aphidicola_AV
    aphidicola_AW
    aphidicola_AX
    aphidicola_AY
    aphidicola_AZ
    aphidicola_B
    aphidicola_BA
    aphidicola_BB
    aphidicola_BC
    aphidicola_BD
    aphidicola_BE
    aphidicola_BF
    aphidicola_BG
    aphidicola_BH
    aphidicola_BI
    aphidicola_BJ
    aphidicola_BK
    aphidicola_C
    aphidicola_D
    aphidicola_E
    aphidicola_F
    aphidicola_G
    aphidicola_H
    aphidicola_I
    aphidicola_J
    aphidicola_K
    aphidicola_L
    aphidicola_M
    aphidicola_N
    aphidicola_P
    aphidicola_R
    aphidicola_S
    aphidicola_T
    aphidicola_U
    aphidicola_V
    aphidicola_W
    aphidicola_X
    aphidicola_Y
    aphidicola_Z
    apicola_A
    apicola_C
    apicola_E
    apicola_F
    apicola_G
    apicola_H
    apicola_I
    apicola_J
    apicola_K
    apicola_L
    apicola_M
    apicola_N
    apicola_P
    apicola_Q
    apicola_R
    apis_A
    apr34_000142F
    apr34_1784
    apr34_1789
    apr34_1792
    aquaemixtae_A
    aquamarina_A
    aquarii_A
    aquatica_A
    aquaticum_A
    aquaticum_B
    aquaticum_C
    aquidurense_B
    aquimarina_A
    aquimarinus_A
    aquimaris_A
    aquimaris_B
    aquimaris_C
    aquimaris_D
    aquimaris_E
    aquimaris_F
    aquimixticola_A
    arboricola_A
    arboricola_B
    arboricola_C
    arboricola_E
    arboricola_F
    arboriphilus_A
    arcachonense_A
    arctica_A
    arctica_B
    arcticum_A
    arenae_A
    arenilitoris_A
    arenilitoris_B
    arenilitoris_C
    arenilitoris_D
    arenilitoris_E
    arenilitoris_F
    argentinensis_B
    argentoratense_A
    arginini_A
    aridicollis_A
    aromatica_A
    aromaticivorans_A
    aromaticus_B
    arosiense_A
    ARS04068_00002
    ARS04158_00005
    ARS04414_00003
    ARS04571_00004
    ARS06627_00002
    ARS06759_00001
    ARS06980_00002
    ARS07370_00002
    arsenicoxydans_A
    arsenicus_B
    arthritidis_A
    aryabhattai_A
    aryabhattai_B
    asahii_A
    asburiae_B
    asburiae_C
    asburiae_D
    asburiae_E
    asburiae_F
    ascorbata_B
    ascorbata_C
    asiaticum_A
    asiaticum_B
    asiaticum_C
    asiaticum_D
    astaxanthinifaciens_A
    asteroides_A
    asteroides_F
    asteroides_G
    asteroides_H
    asteroides_I
    asteroides_J
    AstV_Rat1
    AstV_Rat4
    asymbioticus_C
    ATA3_5Q_KO9_cl1
    ATA3_5Q_KO9_cl2
    atlantica_A
    atlantica_B
    atlantica_C
    atlantica_D
    atlantica_E
    atlantica_F
    atlantica_G
    atlanticus_A
    atratus_B
    atrocyanea_A
    atrocyanea_B
    atsumiense_A
    aurantia_A
    aurantiaca_A
    aurantiacum_A
    aurantiacum_B
    aurantiacus_A
    aurea_B
    aureofaciens_A
    aurescens_A
    aurescens_B
    aureus_A
    aureus_B
    auricola_A
    aurimucosum_F
    auris_A
    aurum_A
    australicum_A
    australiensis_A
    australiensis_B
    australis_A
    australis_B
    australis_C
    australis_D
    autotrophica_A
    autotrophicus_A
    autotrophicus_B
    autotrophicus_C
    autotrophicus_D
    avenae_A
    aviarius_B
    aviarius_C
    AZ4_17
    azadirachtae_A
    AZD10a_EZLN1
    AZD10a_EZLN2
    AZD11_3
    AZD5_G10
    AZD8_5
    AZD9_B1
    AZD9_B2
    AZD9_W1
    AZD9_W2
    azotifigens_A
    azotoformans_A
    B01_8wb_b10_2005-12-15
    B15_CLW
    B18_CLW
    B24_CLW
    B29_CLW
    B3_2_cr
    B3_3_cr
    B7_CLW
    B_36
    B_A17_127
    B_A17_147
    B_C14_122
    B_C14_139
    B_C14_162
    B_C14_165
    B_C49_12
    B_C49_13
    B_C49_14
    B_C49_51
    B_C49_9
    B_C49_91
    B_D75_40
    B_D75_46
    B_D75_48
    B_D75_5
    B_D75_6
    B_D75_66
    B_D75_7
    B_D75_82
    B_EHO
    B_UC1
    BAC_Igh_FGFR3-EGFP
    bacchi_A
    bacilliformis_A
    backbone_CONEXER_BAC_Universal1
    backbone_CONEXER_BAC_Universal2
    Bact_1
    Bact_10
    Bact_11
    Bact_12
    Bact_13
    Bact_14
    Bact_15
    Bact_16
    Bact_17
    Bact_18
    Bact_19
    Bact_2
    Bact_20
    Bact_21
    Bact_22
    Bact_23
    Bact_24
    Bact_25
    Bact_26
    Bact_27
    Bact_28
    Bact_29
    Bact_3
    Bact_30
    Bact_31
    Bact_32
    Bact_33
    Bact_34
    Bact_35
    Bact_36
    Bact_37
    Bact_38
    Bact_39
    Bact_4
    Bact_40
    Bact_41
    Bact_42
    Bact_5
    Bact_6
    Bact_7
    Bact_8
    Bact_9
    bacteremicum_A
    bacteriovorus_A
    bacteriovorus_B
    bacteriovorus_C
    bacteriovorus_D
    bacteriovorus_E
    bacteriovorus_F
    bacteriovorus_G
    bacteriumRM12EL_02D
    bacteriumRM12EL_05A
    bactoclasticus_A
    baderi_A
    baekdonensis_A
    baekdonensis_B
    baetica_A
    baldri_A
    balearica_A
    balearica_B
    balneolensis_A
    baltica_A
    bambusae_A
    bambusae_B
    bangladeshense_B
    bannensis_A
    baratii_A
    barkeri_A
    barkeri_B
    barkeri_C
    bartlettii_A
    basilensis_B
    basilensis_D
    basilensis_E
    basilensis_F
    bataviensis_A
    BCoV/PB675/Art_lit/PAN/2011
    beibuense_A
    beijerinckii_A
    beijerinckii_D
    beijerinckii_E
    beijingensis_A
    beijingensis_B
    benefaciens_A
    benthica_A
    bentonitica_A
    benzenivorans_A
    bereziniae_A
    bermudensis_A
    bermudensis_B
    bernardetii_B
    Beta_07_inf_18
    betae_A
    betae_B
    bethesdensis_B
    bethesdensis_C
    BgL12_3S_Black
    BgL12_3S_Purple
    BgL12_3S_Red
    biceps_A
    biceps_B
    bII_SDSRA_K68
    bilis_B
    bisanense_A
    BJCX4_extB9.1
    blattaphilus_A
    BM0354_1
    BM0354_2
    Bog1183_53
    Bog1249_12
    Bog5275_51
    Bog5712_52
    Bog8989_22
    Bog9017_22
    bogorensis_A
    bogorensis_B
    bogorensis_C
    bombi_A
    boreus_A
    boreus_B
    boreus_C
    bornimense_A
    boronicumulans_A
    boronicumulans_B
    boronicumulans_C
    boroniphilus_A
    bostrichidophilus_A
    bostrichidophilus_B
    bostrichidophilus_C
    bostrichidophilus_D
    bostrichidophilus_E
    bostrichidophilus_F
    bostrichidophilus_G
    bostrichidophilus_H
    bostrichidophilus_I
    botulinum_A
    botulinum_B
    bouvetii_A
    bovienii_C
    bovigenitalium_A
    bovine/GIII.2/216_0114/2006/NOR
    bovine/GIII.2/240_0243/2005/NOR
    bovine/GIII.2/300_0250/2006/NOR
    bovine/GIII.2/312_0529/2006/NOR
    bovine/GIII.2/340_1235/2006/NOR
    bovine/GIII.2/471_0790/2005/NOR
    bovine/GIII.2/541_0448/2005/NOR
    bovine/GIII.2/584_3248/2005/NOR
    bovine/GIII.2/670_0799/2006/NOR
    bovine/GIII.2/718_0785/2006/NOR
    bovine/GIII/chimeric/107_0485/2005/NOR
    bovine/GIII/chimeric/661_1570/2006/NOR
    bovine/GIII/chimeric/752_3024/2006/NOR
    bovine/GIII/chimeric/785_0449/2006/NOR
    bovis_A
    brandeum_A
    brasilense_B
    brasilense_C
    brasilense_D
    brasilense_F
    brasilense_G
    brasiliense_A
    brasiliensis_A
    brasiliensis_B
    brasiliensis_C
    brassicacearum_A
    brassicacearum_D
    brassicacearum_E
    brassicacearum_F
    brassicacearum_G
    breve_A
    brevis_B
    brevis_C
    brevis_D
    BRH_c32
    briensis_A
    brisouii_A
    britannica_A
    bromii_A
    bromii_B
    bronchiseptica_F
    bronchisepticus_A
    bryantii_A
    bryantii_B
    bryantii_C
    bryantii_D
    BtCoV/BRA100/Car_per/BRA/2009
    BtCoV/BRA114/Car_bre/BRA/2009
    BtCoV/BRA118/Car_per/BRA/2009
    BtCoV/BRA119/Car_per/BRA/2009
    BtCoV/BRA182/Mol_ruf/BRA/2009
    BtCoV/BRA344/Car_bre/BRA/2009
    BtCoV/BRAP103/Mol_cur/BRA/2009
    BtCoV/GrNC1/Neo_cap/RSA/2012
    BtCoV/GrNC2/Neo_cap/RSA/2012
    BtCoV/GrNC3/Neo_cap/RSA/2012
    BtCoV/GrNC4/Neo_cap/RSA/2012
    BtCoV/GrNC5/Neo_cap/RSA/2012
    BtCoV/GrNC6/Neo_cap/RSA/2012
    BtCoV/GrNC7/Neo_cap/RSA/2012
    BtCoV/GrNC8/Neo_cap/RSA/2012
    BtCoV/KCR15/Pte_par/CRC/2012
    BtCoV/KCR155/Pte_par/CRC/2012
    BtCoV/KCR180/Pte_par/CRC/2012
    BtCoV/KCR216/Car_per/CRC/2010
    BtCoV/KCR22/Pte_par/CRC/2012
    BtCoV/KCR230/Pte_par/CRC/2010
    BtCoV/KCR24/Ano_geo/CRC/2010
    BtCoV/KCR252/Car_per/CRC/2010
    BtCoV/KCR253/Car_per/CRC/2010
    BtCoV/KCR260/Car_per/CRC/2012
    BtCoV/KCR289/Ano_geo/CRC/2010
    BtCoV/KCR291/Ano_geo/CRC/2010
    BtCoV/KCR293/Ano_geo/CRC/2010
    BtCoV/KCR370/Pte_par/CRC/2011
    BtCoV/KCR372/Car_per/CRC/2010
    BtCoV/KCR90/Car_per/CRC/2010
    BtCoV/KCR91/Car_per/CRC/2010
    BtCoV/KP256/Art_jam/PAN/2010
    BtCoV/KP524/Art_jam/PAN/2010
    BtCoV/KP534/Art_lit/PAN/2010
    BtCoV/KP565/Art_jam/PAN/2010
    BtCoV/KP816/Phy_dis/PAN/2011
    BtCoV/KP817/Phy_dis/PAN/2011
    BtCoV/KW2E-F53/Nyc_spec/GHA/2011
    BtCoV/KW2E-F82/Nyc_spec/GHA/2011
    BtCoV/KW2E-F93/Nyc_spec/GHA/2010
    BtCoV/MSTM2/Min_nat/RSA/2010
    BtCoV/MSTM6/Min_nat/RSA/2010
    BtCoV/NCL_MCO1/Mop_con/RSA/2012
    BtCoV/OCR11/Pte_par/CRC/2011
    BtCoV/Rhi_bla/BB89-22/BGR/2008
    BtCoV/Rhi_eur/BB89-92/BGR/2008
    BtCoV/Rhi_eur/BB89-98/BGR/2008
    BtCoV/Rhi_eur/BB99-04/BGR/2009
    BtCoV/Rhi_fer/FR0711-B11/FRA/2011
    BtCoV/Rhi_fer/FR0711-B3/FRA/2011
    BtCoV/Rhi_fer/It1/ITA/2009
    BtCoV/Rhi_fer/It13/ITA/2009
    BtCoV/Rhi_fer/It15/ITA/2009
    BtCoV/Rhi_fer/It17/ITA/2009
    BtCoV/Rhi_fer/It2/ITA/2009
    BtCoV/Rhi_hip/R07-09/SPA/2010
    BtCoV/Rhi_hip/R13-08/SPA/2010
    BtCoV/Rhi_hip/R46-03/SPA/2010
    BtCoV/Rhi_hip/R7-08/SPA/2010
    BtCoV/Rhi_hip/R77-02/SPA/2010
    BtCoV/Rhi_hip/R8-09/SPA/2010
    BtCoV/Rhi_hip/Slo48/SLO/2009
    BtCoV/Rhi_hip/Slo52/SLO/2009
    BtCoV/Rhi_hip/Slo53/SLO/2009
    BtCoV/Rhi_hip/Slo54/SLO/2009
    BtCoV/Rhi_hip/Slo57/SLO/2009
    BtCoV/Rhi_hip/Slo69/SLO/2009
    BtCoV/VH_NC2/Neo_cap/RSA/2012
    buctense_A
    buctense_B
    bullata_A
    bunya-arena-like_virus
    butyricum_A
    butyricum_B
    byssoidea_A
    C01_2wb_b10_2005-12-15
    C02_1ub_b10_2005-12-15
    C100_CLW
    C151_oct11_cons
    C18_CLW
    C39_CLW
    C47_CLW
    C49_CLW
    C60I_1
    C60I_10
    C60I_11B
    C60I_12
    C60I_14
    C60I_15
    C60I_16
    C60I_2
    C60I_4
    C60I_8
    C60I_9
    C67_CLW
    C72_CLW
    C79_CLW
    C84_CLW
    C92_CLW
    cacaoi_A
    caecigallinarius_B
    caecigallinarum_A
    caecigallinarum_B
    caecimuris_A
    caelestis_A
    caeni_A
    caeni_B
    caeruleus_A
    calcoaceticus_B
    calcoaceticus_C
    calcoaceticus_D
    calcoaceticus_E
    caldarium_A
    caldicuralii_A
    calvum_A
    cambriense_A
    cambriense_B
    camelliae_A
    camelliae_B
    campaniensis_A
    campestris_A
    campestris_B
    campestris_C
    campestris_D
    canadensis_A
    canariense_A
    canariense_C
    canariense_D
    canis_B
    canus_C
    canus_D
    canus_E
    canus_F
    capeferrum_A
    capillosus_A
    capleta_B
    capleta_C
    capleta_D
    capleta_E
    capleta_F
    capleta_G
    capleta_H
    capleta_I
    capleta_J
    capleta_K
    capleta_L
    capsici_A
    capsulatum_A
    capsulatus_B
    capsulatus_C
    cardiocondylae_A
    cardiocondylae_B
    cardiocondylae_C
    carnea_A
    carnea_B
    carnea_C
    carotovorum_I
    casei_A
    caseolyticus_A
    casseliflavus_A
    cat_SB2894
    catarrhalis_C
    catus_A
    cavernae_A
    CCSD_DF2030_Ag37_isolate_1
    CCSD_DF2030_Ag37_isolate_2
    CCSD_DF2030_Ag68_isolate_1
    CCSD_DF2030_Ag68_isolate_2
    CCSD_DF2030_Ag68_isolate_3
    CCSD_DF2030_CL68_isolate_2
    CCSD_DF2030_TE37_isolate_1
    CCSD_DF2030_TE37_isolate_2
    cecembensis_A
    cecembensis_B
    cecorum_A
    cecorum_B
    Ced_B08
    celere_A
    celerecrescens_B
    cellasea_A
    cellostaticus_A
    cellulans_A
    cellulans_D
    cellulans_E
    cellulans_F
    cellulans_G
    cellulosae_A
    cellulositrophicus_A
    cellulosum_B
    cellulosum_D
    cellulosum_E
    cellulosum_F
    cellulosum_G
    cepacia_C
    cepacia_D
    cepacia_F
    cereus_AG
    cereus_AK
    cereus_AQ
    cereus_AT
    cereus_AU
    cereus_AV
    cereus_BA
    cereus_BB
    cereus_BC
    cereus_BD
    cereus_BE
    cereus_K
    cereus_S
    cereus_U
    cerinum_A
    cerinus_A
    cervicalis_A
    cetorum_B
    cetorum_C
    CFTR_BAC01
    CFTR_BAC02
    CFTR_BAC03
    CG06_land_8_20_14_3_00_33_50
    CG06_land_8_20_14_3_00_37_11
    CG07_land_8_20_14_0_80_38_8
    CG09_39_24
    CG10_37_50
    CG10_46_32
    CG10_49_38
    CG10_big_fil_rev_8_21_14_0_10_33_18
    CG10_big_fil_rev_8_21_14_0_10_43_11
    CG17_big_fil_post_rev_8_21_14_2_50_64_8
    CG1_02_42_9
    CG2_30_31_98
    CG2_30_33_46
    CG2_30_37_16
    CG2_30_40_12
    CG2_30_54_10
    CG_4_10_14_0_2_um_filter_33_32
    CG_4_10_14_0_2_um_filter_48_144
    CG_4_10_14_0_2_um_filter_Archaea_38_6
    CG_4_10_14_0_8_um_filter_33_57
    CG_4_8_14_3_um_filter_33_28
    CG_4_8_14_3_um_filter_38_5
    CG_4_9_14_3_um_filter_33_26
    CG_4_9_14_3_um_filter_65_15
    CH30_LM136
    CH311c_11T
    CH336b_11D
    CH35_1T
    CH64b_2B
    chagasii_A
    chagasii_B
    chartreusis_D
    chelonae_A
    chengduensis_A
    chikugoensis_A
    chilikensis_A
    chitinivorans_A
    chlorellavorus_A
    chlorellavorus_B
    chlorellavorus_D
    chlorophenolicus_A
    chlororaphis_D
    chlororaphis_E
    chlororaphis_H
    chlororaphis_I
    choanae_A
    chokoriensis_B
    choshinensis_A
    chromiireducens_A
    chromiiresistens_A
    chubuense_A
    chungangensis_A
    cibaria_A
    cibarius_A
    cibarius_B
    cibarius_C
    cicadellinicola_A
    cicadellinicola_B
    cicadellinicola_C
    ciceri_B
    cichorii_B
    cichorii_D
    cichorii_E
    ciconiae_A
    cincinnatiensis_A
    cinnabarinus_B
    cinnamea_A
    cinnamoneus_A
    circulans_A
    circulans_B
    citri_A
    citronellolis_A
    clevelandensis_A
    cloacae_A
    cloacae_I
    cloacae_L
    cloacae_M
    cloacae_N
    cloacae_O
    cloacae_P
    cloacae_Q
    clostridioformis_A
    CN30_LM100
    coccineus_A
    coeruleorubidus_A
    Col_A
    coleohominis_A
    coli_A
    coli_B
    coli_E
    coli_F
    colihominis_A
    collinus_A
    colombiense_A
    colombiense_B
    colombiense_C
    columbae_A
    columbina_A
    comes_A
    commune_A
    communis_A
    communis_B
    composti_A
    composti_B
    concisus_A
    concisus_AA
    concisus_AB
    concisus_AC
    concisus_AD
    concisus_AE
    concisus_AF
    concisus_AG
    concisus_AH
    concisus_AI
    concisus_AJ
    concisus_AK
    concisus_AL
    concisus_AM
    concisus_AN
    concisus_AO
    concisus_AP
    concisus_AQ
    concisus_AR
    concisus_AS
    concisus_AT
    concisus_B
    concisus_C
    concisus_D
    concisus_E
    concisus_F
    concisus_G
    concisus_H
    concisus_I
    concisus_J
    concisus_K
    concisus_L
    concisus_M
    concisus_N
    concisus_O
    concisus_P
    concisus_Q
    concisus_R
    concisus_S
    concisus_T
    concisus_U
    concisus_V
    concisus_W
    concisus_X
    concisus_Y
    concisus_Z
    confusa_A
    confusa_B
    confusa_C
    congolensis_A
    congolensis_B
    congonensis_A
    constellatus_A
    contorta_A
    contortus_B
    contortus_C
    contractile_A
    copri_A
    copri_B
    copri_C
    copri_E
    copri_F
    copri_G
    copri_H
    copri_I
    copri_J
    coralicida_A
    coralliilyticus_A
    coronavirus/KP817/Phy_dis/PAN/2011
    corrodens_B
    corynebacterioides_A
    cosmeticum_A
    cowanii_A
    CR1_10_MR-20148
    CR1_11_MR-2014
    CR5_1_MR-2014
    CR5_2_MR-2014
    CR7_6_MR-2014
    crassostreae_C
    crateris_A
    crateris_B
    cremea_A
    cremeus_A
    cremoricolorata_A
    CRF01_AE
    CRF02_AG:08GQ032
    CRF02_AG:08GQ057
    CRF02_AG:08GQ058
    CRF02_AG:08GQ060
    CRF02_AG:08GQ061
    CRF02_AG:08GQ092
    CRF02_AG:08GQ093
    CRF02_AG:08GQ166
    CRF02_AG:08GQ200
    CRF02_AG:08GQ203
    CRF02_AG:08GQ239
    CRF02_AG:08GQ258
    CRF02_AG:08GQ269
    CRF02_AG:08GQ342
    CRF02_AG:08GQ370
    CRF02_AG:08GQ378
    CRF02_AG:08GQ385
    CRF02_AG:08GQ386
    CRF02_AG:08GQ392
    CRF02_AG:08GQ415
    CRF02_AG:08GQ433
    CRF02_AG:08GQ463
    CRF03_AB
    CRF04_cpx
    CRF06_cpx:08GQ191
    CRF08_BC
    CRF09_cpx:08GQ399
    CRF11_cpx:08GQ380
    CRF22_01A1:08GQ346
    CRF26_A5U:08GQ019
    CRF26_A5U:08GQ323
    crinochetorum_A
    cristatus_A
    cristatus_B
    cristatus_C
    cristatus_D
    cristatus_E
    cristatus_F
    cristatus_G
    cristatus_H
    cristatus_I
    cristatus_J
    cristatus_K
    cristatus_L
    cristatus_M
    crithidii_A
    crithidii_B
    crithidii_C
    CRO1006_1
    CRO1006_2
    CRO1091_1
    CRO1091_2
    CRO23_2
    CRO23_3
    CRO372_2
    CRO462_3
    CRO477_3
    CRO479_2
    CRO479_3
    CRO480_1
    CRO480_2
    CRO480_3
    CRO588_1
    CRO588_2
    CRO5_2
    CRO65_3
    CRO707_2
    CRO729_1
    CRO729_2
    CRO777_1
    CRO777_2
    CRO832_1
    CRO832_2
    CRO834_3
    croceus_A
    crocodyli_A
    Cry7_14
    Cry7_9.1
    Cry7_extC18.1
    cryaerophilus_A
    cryaerophilus_B
    cryoconitis_A
    cryoconitis_B
    cryoconitis_C
    cryoconitis_D
    cryoconitis_E
    cryocrescens_A
    crystallopoietes_A
    crystallopoietes_B
    CSeA30_26
    CSeA30_42
    CSeA30_59
    CSeA30_60
    CSu4_9
    cucumis_A
    cucumis_B
    cucumis_C
    cuenoti_A
    cuenoti_AA
    cuenoti_AB
    cuenoti_AC
    cuenoti_AD
    cuenoti_AE
    cuenoti_AF
    cuenoti_AG
    cuenoti_AH
    cuenoti_AI
    cuenoti_AJ
    cuenoti_AK
    cuenoti_AL
    cuenoti_AM
    cuenoti_AN
    cuenoti_B
    cuenoti_C
    cuenoti_D
    cuenoti_E
    cuenoti_F
    cuenoti_G
    cuenoti_H
    cuenoti_I
    cuenoti_J
    cuenoti_K
    cuenoti_L
    cuenoti_M
    cuenoti_N
    cuenoti_O
    cuenoti_P
    cuenoti_Q
    cuenoti_R
    cuenoti_S
    cuenoti_T
    cuenoti_U
    cuenoti_V
    cuenoti_W
    cuenoti_X
    cuenoti_Y
    cuenoti_Z
    culicis_A
    cuniculi_A
    curvispora_A
    cyclitrophicus_A
    cypripedii_A
    cyriacigeorgica_B
    cyriacigeorgica_C
    cyriacigeorgica_D
    cyriacigeorgica_E
    cyriacigeorgica_F
    cytotoxicus_A
    D01_9wb_b10_2005-12-15
    D02_1pb_b10_2005-12-15
    D2_1_1F-16sF
    D2_1_3F-16sF
    D3_1_1F-16sF
    D3_1_3_2F-16sF
    D_HF1_1
    D_HF1_11
    D_HF1_3
    D_HF2_144
    D_HF2_219
    D_HF2_273
    D_HF2_7
    D_HF2_82
    D_HF32_302
    D_HF32_91
    D_HF33_10
    D_HF33_14
    D_HF34_23
    D_HF34_8
    D_HF35_23
    D_HF36_426
    D_HF36_482
    D_HF38_14
    D_HF38_28
    D_HF38_35
    D_HF3_10
    D_HF3_12
    D_HF3_19
    D_HF3_26
    D_HF3_8
    D_HF4_111
    D_HF4_124
    D_HF4_1353
    D_HF4_1386
    D_HF4_144
    D_HF4_149
    D_HF4_150
    D_HF4_162
    D_HF4_1794
    D_HF4_2
    D_HF4_2562
    D_HF4_274
    D_HF4_298
    D_HF4_320
    D_HF4_329
    D_HF4_340
    D_HF4_371
    D_HF4_79
    D_HF4_80
    D_HF5_1
    D_HF5_107
    D_HF5_131
    D_HF5_137
    D_HF5_2C
    D_HF5_2R
    D_HF5_3
    D_HF5_49
    D_HF5_61
    D_HF5_75
    D_HF5_96
    D_HF6_100
    D_HF6_206
    D_HF6_322
    D_HF6_34
    D_HF6_591
    D_HF7_11
    D_HF7_3
    D_HF7_66
    D_HF7_9
    DAB_16s_R32(2)
    daeponensis_A
    danieliae_A
    Davis_19N
    DB_16s_F5
    DB_16s_F77
    DB_16s_F8
    DB_16s_R8
    debontii_A
    debontii_B
    defensa_A
    defluvii_A
    defluvii_B
    defluvii_C
    dehalogenans_B
    delafieldii_A
    delafieldii_B
    delafieldii_C
    delaneyi_A
    delbrueckii_B
    deleyi_A
    dendritiformis_A
    dendritiformis_C
    denitrificans_A
    denitrificans_B
    denitrificans_C
    denitrificans_E
    denitrificans_F
    denitrificans_G
    dentiae_A
    denticola_A
    denticola_B
    denticola_C
    denticola_D
    depolymerans_A
    deserti_A
    deserti_B
    deserti_C
    deserti_D
    desertorum_A
    desertorum_B
    desulfatans_A
    desulfuricans_B
    desulfuricans_C
    desulfuricans_D
    desulfuricans_E
    detoxificans_A
    devorans_A
    devorans_B
    dextrinosolvens_A
    dextrinosolvens_B
    diastatochromogenes_A
    diazoefficiens_A
    diazoefficiens_B
    diazoefficiens_C
    diazoefficiens_D
    diazoefficiens_E
    diazoefficiens_F
    diazoefficiens_G
    diazoefficiens_H
    difficile_A
    difficile_D
    difficile_E
    difficile_F
    difficile_G
    diffusa_B
    diminuta_A
    diminuta_B
    diminuta_C
    dinghuensis_A
    dinghuensis_B
    discipulorum_A
    discipulorum_B
    dismutans_A
    dispar_B
    disporicum_B
    distasonis_A
    distasonis_B
    distasonis_C
    distributa_B
    distributa_C
    distributa_D
    distributa_E
    divergens_A
    diversum_A
    Djaks_4
    Djaks_6
    DmelCS_103
    dog/GVI.1/HKU_Ca026F/2007/HKG
    dog/GVI.1/HKU_Ca035F/2007/HKG
    dokdonensis_B
    DOLJORAL78_63_78
    DOLJORAL78_65_58
    DOLZORAL124_38_8
    DOLZORAL124_64_63
    donghaensis_A
    donghaensis_B
    drentensis_A
    DSFBPG_3R2A
    DSFBPG_3R2B
    DSFBPG_4R3A
    DSFBPG_4R3B
    DSFBPG_4R3C
    DSFBPG_5R1B
    DSFBPG_5R3A
    DSFBPG_UR1A
    DSFBPG_UR1B
    DSFBPG_UR1C
    DSFBPG_UR1D
    DSFBPG_UR1E
    DSFBPG_UR2A
    DSFBPG_UR2B
    DSFBPG_UR2D
    DSFBPG_UR2E
    DSFBPG_UR2F
    DSFBPG_UR2G
    DSFBPG_UR3A
    DSFBPG_UR3B
    DSFBPG_UR3C
    DSFBPG_UR3D
    DSFBPG_UR3E
    DSFBPG_UR3F
    DSFBPG_URC2C
    DSFBPS_4R3A
    DSFBPS_UR1A
    DSFBPS_UR1B
    DSFBPS_UR1C
    DSFBPS_UR1D
    DSFBPS_UR1E
    DSFBPS_UR1F
    DSFBPS_UR1G
    DSFBPS_UR1H
    DSFBPS_UR2B
    DSFBPS_UR2C
    DSFBPS_UR2D
    DSFBPS_UR2E
    DSFBPS_UR2F
    DSFBPS_UR3A
    DSFBPS_UR3B
    DSFBPS_UR3C
    DSS3_MAL1
    DSS3_PM1
    DSS3_VP1
    durus_B
    E01_10wb_b10_2005-12-15
    E56_UTI
    E60_UTI
    E81_UTI
    echini_A
    echinofusca_A
    echinoides_A
    echinospora_A
    ectosymbiont_A
    ectosymbiont_B
    edaphicum_A
    edaphicum_B
    eiseniae_A
    elegans_A
    eligens_A
    elkanii_A
    elongata_B
    elongatus_A
    elongatus_B
    elsdenii_A
    EMG_100139454
    enclense_A
    enclensis_A
    endoloripes_D
    endophytica_A
    endophyticus_A
    enshiense_A
    enterocolitica_E
    entomophila_A
    entomophila_B
    entomophila_C
    entomophila_D
    entomophila_E
    entomophila_F
    entomophila_G
    enzymogenes_C
    enzymogenes_D
    Ep06i_A10B
    Ep06i_A1B
    Ep06i_A4C
    Eptesicus/13RS384_26/Italy/2012
    EqCoPV_11
    EqCoPV_8
    EqCoPV_9
    EqCoPV_A4
    equi_A
    equinus_B
    equinus_D
    equisanguinis_A
    equolifaciens_A
    equorum_A
    equorum_C
    erythraea_A
    erythrophlei_A
    erythrophlei_B
    erythrophlei_C
    erythrophlei_D
    erythropolis_D
    ES5_1357
    ES6_1418
    ES6_478
    ES_ALL_000190F
    esperanzae_A
    esteraromaticum_B
    esteraromaticum_C
    esteraromaticum_D
    estertheticum_A
    estertheticum_B
    estertheticum_C
    estertheticum_D
    estunensis_A
    ethanivorans_B
    ethanivorans_C
    etli_B
    etli_C
    euphylliae_A
    euphylliae_B
    euphylliae_C
    europaea_B
    europaeum_A
    eutropha_A
    eutropha_B
    ex4484_74
    Ex_DG74
    exalbescens_A
    excellens_B
    excentricus_A
    exercitus_A
    exile_A
    exile_C
    extorquens_A
    F01_14wb_b10_2005-12-15
    F3_an4
    F3_an5
    F3H1_a10
    FA1-2_000172F
    FA_121
    FA_130
    FA_132
    FA_139
    FA_149
    FA_151
    FA_152
    FA_183B
    FA_2CB*
    FA_86
    fabrum_A
    faecalis_A
    faecalis_D
    faecihominis_A
    faeni_A
    faeni_B
    faeni_C
    FAKO05_000032F
    FAKO27_000238F
    FAKO27_000271F
    falsenii_A
    falsenii_B
    farcinica_A
    fascians_E
    fastidiosus_A
    faulkneri_A
    Fen2266_11
    Fen418_41
    Fen4707_41
    Fen51_42
    Fen672_31
    Fen685_11
    Fen7786_21
    Fen7875_21
    Fen7895_21
    Fen7918_21
    Fen7940_21
    fermentans_A
    fermentans_B
    ferrariae_A
    ferrireducens_A
    ferrivorans_A
    ferrooxidans_B
    ferrugineus_A
    FGPV_KD09.ZAF
    fibrisolvens_A
    fibrisolvens_C
    ficellus_A
    ficellus_B
    filiformis_A
    filiformis_B
    fimi_A
    fimi_B
    fimi_C
    finisterrensis_A
    finlayi_A
    finnmarkense_A
    firmus_A
    firmus_B
    firmus_C
    fischeri_A
    fischeri_B
    fissicatena_A
    flaccumfaciens_A
    flaccumfaciens_B
    flaccumfaciens_C
    flaccumfaciens_D
    flaccumfaciens_E
    flagellatus_A
    flagelliforme_A
    flagelliforme_B
    flava_A
    flavefaciens_A
    flavefaciens_AA
    flavefaciens_AB
    flavefaciens_AC
    flavefaciens_AD
    flavefaciens_B
    flavefaciens_C
    flavefaciens_D
    flavefaciens_E
    flavefaciens_F
    flavefaciens_G
    flavefaciens_H
    flavefaciens_I
    flavefaciens_J
    flavefaciens_K
    flavefaciens_L
    flavefaciens_M
    flavefaciens_N
    flavefaciens_O
    flavefaciens_P
    flavefaciens_Q
    flavefaciens_R
    flavefaciens_S
    flavefaciens_T
    flavefaciens_U
    flavefaciens_V
    flavefaciens_W
    flavefaciens_X
    flavefaciens_Y
    flavefaciens_Z
    flaveolus_A
    flavescens_A
    flavescens_B
    flavescens_C
    flavescens_D
    flavithermus_E
    flavithermus_G
    flavofungini_A
    flavorosea_A
    flavus_A
    flour_#16
    fluorescens_A
    fluorescens_AA
    fluorescens_AJ
    fluorescens_AM
    fluorescens_AN
    fluorescens_AO
    fluorescens_AP
    fluorescens_AQ
    fluorescens_B
    fluorescens_BD
    fluorescens_BE
    fluorescens_BF
    fluorescens_BG
    fluorescens_BH
    fluorescens_BI
    fluorescens_BJ
    fluorescens_BK
    fluorescens_BM
    fluorescens_BO
    fluorescens_BP
    fluorescens_BQ
    fluorescens_BR
    fluorescens_BS
    fluorescens_BT
    fluorescens_BU
    fluorescens_BV
    fluorescens_BW
    fluorescens_BX
    fluorescens_BY
    fluorescens_BZ
    fluorescens_CA
    fluorescens_CB
    fluorescens_CC
    fluorescens_CD
    fluorescens_E
    fluorescens_G
    fluorescens_H
    fluorescens_M
    fluorescens_O
    fluorescens_Q
    fluorescens_S
    fluorescens_W
    fluorescens_X
    fluvialis_A
    fluvialis_B
    fluvialis_C
    fluvialis_D
    fluvialis_E
    fluvialis_F
    fluviatilis_A
    FMAS_PN2
    FMAS_PN3
    FMAS_PN5
    foliaceum_A
    foliorum_A
    foliorum_B
    fonticola_A
    fonticola_B
    fonticola_C
    formicae_A
    formicicum_A
    formicilis_A
    formicilis_B
    formigenes_C
    formigenes_D
    formigenes_E
    formosum_A
    forsythia_A
    fortis_A
    FR1_11-MR-2014
    FR1_16_MR-2014
    FR3_2_MR-2014
    fragariae_A
    fragi_D
    fragi_E
    fragilis_B
    frankenforstense_A
    franklandus_A
    frederiksbergense_A
    frederiksbergense_B
    frederiksbergensis_A
    frederiksbergensis_B
    frederiksbergensis_D
    frederiksbergensis_F
    frederiksbergensis_G
    frederiksbergensis_H
    frederiksbergensis_I
    frederiksbergensis_K
    fredii_A
    fredii_B
    fredii_C
    frigoris_A
    frigoritolerans_A
    frigoritolerans_B
    frisingense_B
    fugitiva_A
    fulva_A
    fulva_B
    fulva_C
    fulva_D
    fulvum_A
    fulvum_B
    fungivorans_A
    fungivorans_B
    fungorum_A
    fungorum_B
    fuscovaginae_A
    fuscovaginae_B
    fuscus_A
    fusiformis_A
    fusiformis_D
    fusiformis_E
    fusiformis_F
    G01_1mb_b10_2005-12-15
    G3_GreenLake
    G5_GreenLake
    G7_8_4BO2
    G_01
    G_02
    G_05
    G_10
    G_14
    G_20
    G_22
    G_23
    gaetbuli_A
    gaetbuli_B
    galactitolivorans_A
    galegae_A
    galegae_B
    galegae_C
    galegae_D
    gallaeciensis_A
    gallinacea_A
    gallolyticus_B
    gallolyticus_C
    gandavensis_A
    gangwonense_A
    garvieae_A
    gazogenes_A
    gelatinosus_A
    gelatinosus_B
    geoffroys_cat/URU-01/2009
    geofontis_A
    georgiana_A
    geothermicum_A
    geothermicum_B
    geothermicum_C
    gephyra_A
    GF1-2_000079F
    GI/Hu/BG/2011/GI.P3_GI.3/RV1253
    GI/Hu/BG/2011/GI.Pb_GI.6/NV2(341)
    GI/Hu/JP/2007/GI.P3_GI.3/Shimizu/KK2866
    GI/Hu/JP/2007/GI.P8_GI.8/Nagoya/KY531
    GI/Hu/JP/FukuokaP/2015/GI.3/FE49_2014
    GI/Hu/KH/2004-2006/GI.P8_GI.8/NP-451_1
    GI/Hu/KH/2004-2006/GI.Pc_GI.5/NP-093_2
    GI/PBV/bovine/BRA/PBV18_PR/2014
    gigasporarum_A
    gigasporarum_B
    GII.4/Cordoba/Sds_August09/2009/ARG
    GII.4/Cordoba/Sds_July09-2/2009/ARG
    GII.4/Cordoba/Sds_July09-3/2009/ARG
    GII.4/Cordoba/Sds_July09-4/2009/ARG
    GII.4/Cordoba/Sds_July09-5/2009/ARG
    GII.4/Cordoba/Sds_July09-6/2009/ARG
    GII.4/Cordoba/Sds_July09/2009/ARG
    GII.4/Cordoba/Sds_June09/2009/ARG
    GII.4/Cordoba/Sus_August09/2009/ARG
    GII.4/Cordoba/Sus_July09-4/2009/ARG
    GII.4/Cordoba/Sus_July09-5/2009/ARG
    GII.4/Cordoba/Sus_July09/2009/ARG
    GII.4/Cordoba/WTPi_August09/2009/ARG
    GII.4/Cordoba/WTPi_July09-1/2009/ARG
    GII.4/Cordoba/WTPi_July09-2/2009/ARG
    GII.4/Cordoba/WTPi_July09-3/2009/ARG
    GII.4/Cordoba/WTPi_July09-4/2009/ARG
    GII.4/Cordoba/WTPi_July09/2009/ARG
    GII.4/Cordoba/WTPo_August09/2009/ARG
    GII.4/Cordoba/WTPo_June09/2009/ARG
    GII.4_2006b_Minerva
    GII.4_2010_New
    GII.g/Cordoba/Sds_July09-1/2009/ARG
    GII.g/Cordoba/Sus_July09-1/2009/ARG
    GII.g/Cordoba/Sus_July09-2/2009/ARG
    GII.g/Cordoba/Sus_July09-3/2009/ARG
    GII.g/Cordoba/WTPi_June09/2009/ARG
    GII.g/Cordoba/WTPo_July09-1/2009/ARG
    GII.g/Cordoba/WTPo_July09-2/2009/ARG
    GII.g/Cordoba/WTPo_July09-3/2009/ARG
    GII.g/Cordoba/WTPo_July09/2009/ARG
    GII.P12_GII.3
    GII.Pe_GII.4
    GII/Hu/BG/2009/GII.P21_GII.3/RV638
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV672
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV703
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV737
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV746
    GII/Hu/BG/2009/GII.P4_Den_Haag_2006b/RV752
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV107
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV154
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV26
    GII/Hu/BG/2010/GII.4_Den_Haag_2006b/RV28
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV100
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV127
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV21
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV27
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV272
    GII/Hu/BG/2010/GII.4_New_Orleans_2009/RV6
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV107
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV154
    GII/Hu/BG/2010/GII.P4_Den_Haag_2006b/RV26
    GII/Hu/BG/2010/GII.P4_New_Orleans_2009/RV21
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1389
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1521
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1540
    GII/Hu/BG/2011/GII.4_New_Orleans_2009/RV1543
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1368
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1389
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1521
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1528
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1540
    GII/Hu/BG/2011/GII.P4_New_Orleans_2009/RV1543
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1223
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1251
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1256
    GII/Hu/BG/2011/GII.Pg_GII.1/RV1278
    GII/Hu/BG/2012/GII.P7_GII.6/RV84
    GII/Hu/BG/2012/GII.P7_GII.6/RV85
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju19862
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju20078
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Aracaju20341
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Itabaiana20321
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20116
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20236
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Linhares20400
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Machado20361
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Nepomuceno20170
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Recife20276
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Ribamar20310
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Salvador20137
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Salvador20148
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis19516
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20190
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20196
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20292
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Sao_Luis20302
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Tubarao19873
    GII/Hu/BRA/2011/GII.4_NewOrleans_2009/Zortea20377
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Angra_dos_Reis21429
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Blumenau21624
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Fraiburgo21677
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Mantenopolos21548
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Paco_do_Lumiar21659
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Rio_de_Janeiro21389
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Jeronimo21393
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Luis21652
    GII/Hu/BRA/2012/GII.4_Sydney_2012/Sao_Luis21666
    GII/Hu/ESP/2010/GII.P16_GII.13/SanSebastian356037
    GII/Hu/ESP/2012/GII.P7_GII.6/SanSebastian75500004
    GII/Hu/HK/2016/GII.P16_GII.2/CUHK-NS-1082
    GII/Hu/HKG/2015/GII.Pe_GII.17/CUHK-NS-682
    GII/Hu/IN/2005/GII.P4_GII.4_Hunter2004/Pune-PC09
    GII/Hu/IN/2006/GII.P4_GII.4_Hunter2004/Pune-PC12
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC11
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC13
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC16
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC18
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC19
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC20
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC21
    GII/Hu/IN/2006/GII.P4_GII.4_Yerseke2006a/Pune-PC23
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-0715450
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-0716584
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC37
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC45
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC46
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC47
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC49
    GII/Hu/IN/2007/GII.P4_GII.4_DenHaag2006b/Pune-PC56
    GII/Hu/IN/2007/GII.P4_GII.4_Hunter2004/Pune-PC31
    GII/Hu/IN/2007/GII.Pe_GII.4_Osaka2007/Pune-0713603
    GII/Hu/IN/2007/GII.Pe_GII.4_Osaka2007/Pune-0716588
    GII/Hu/IN/2008/GII.P4_GII.4_Apeldoorn2007/Pune-083366
    GII/Hu/IN/2008/GII.P4_GII.4_Apeldoorn2007/Pune-086619
    GII/Hu/IN/2008/GII.P4_GII.4_NewOrleans2009/Pune-0816948
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816943
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816955
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-0816962
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084093
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084097
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084105
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-084695
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-08496
    GII/Hu/IN/2008/GII.Pe_GII.4_Osaka2007/Pune-091056
    GII/Hu/IN/2009/GII.P4_GII.4_NewOrleans2009/Pune-093745
    GII/Hu/IN/2009/GII.Pe_GII.4_Osaka2007/Pune-091101
    GII/Hu/IN/2009/GII.Pe_GII.4_Osaka2007/Pune-092434
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1022675
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1038540
    GII/Hu/IN/2010/GII.P4_GII.4_NewOrleans2009/Pune-1038546
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-112370
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-113700
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116686
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116704
    GII/Hu/IN/2011/GII.P4_GII.4_NewOrleans2009/Pune-116710
    GII/Hu/IN/2012/GII.P4_GII.4_NewOrleans2009/Pune-124748
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-1310786
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-131742
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-1324318
    GII/Hu/IN/2013/GII.Pe_GII.4_Sydney2012/Pune-136935
    GII/Hu/JP/2000/GII.P12_GII.12/Saitama/KU16
    GII/Hu/JP/2001/GII.P12_GII.12/Saitama/T15
    GII/Hu/JP/2001/GII.P12_GII.12/Saitama/T18
    GII/Hu/JP/2001/GII.P16_GII.17/OC01031-1
    GII/Hu/JP/2001/GII.P22_GII.5/Saitama/T49
    GII/Hu/JP/2002/GII.P12_GII.13/Saitama/T80
    GII/Hu/JP/2002/GII.P16_GII.17/Saitama/T87
    GII/Hu/JP/2002/GII.P5_GII.5/Saitama/T52
    GII/Hu/JP/2007/GII.P15_GII.15/Sapporo/HK299
    GII/Hu/JP/2007/GII.P21_GII.21/Kawasaki/YO284
    GII/Hu/JP/2007/GII.P7_GII.14/Fukuoka/KK282
    GII/Hu/JP/2007/GII.P7_GII.14/Sendai/YG99
    GII/Hu/JP/2010/GII.P4_GII.4/Shimada/ASC96
    GII/Hu/JP/2010/GII.P7_GII.7/Musashimurayama/TAKAsanKimchi
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/1
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/2
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/3
    GII/Hu/JP/2012/GII.Pe_GII.4_Sydney2012/Fukuyama/5
    GII/Hu/JP/2016/GII.P16_GII.4_Sydney2012/OC16023
    GII/Hu/JP/2016/GII.P16_GII.4_Sydney2012/OH16002
    GII/Hu/KH/2004-2006/GII.P12_GII.12/NP-162_1
    GII/Hu/KH/2004-2006/GII.P12_GII.12/NP-328_1
    GII/Hu/KH/2004-2006/GII.P12_GII.13/NP-259_1
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-011_2
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-014_2
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NP-312_1
    GII/Hu/KH/2004-2006/GII.P12_GII.4_Asia_2003/NPC-350_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NP-234_3
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-239_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-268_1
    GII/Hu/KH/2004-2006/GII.P16_GII.16/NPC-269_1
    GII/Hu/KH/2004-2006/GII.P16_GII.17/NP-209_1
    GII/Hu/KH/2004-2006/GII.P21_GII.3/NP-355_1
    GII/Hu/KH/2004-2006/GII.P2_GII.2/NP-455_1
    GII/Hu/KH/2004-2006/GII.P3_GII.3/NP-122_1
    GII/Hu/KH/2004-2006/GII.P3_GII.3/NP-406_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Asia_2003/NP-593_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Asia_2003/NPC-405_2
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-328_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-330_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-353_2
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-356_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NP-422_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NPC-355_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Hunter_2004/NPC-405_1
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-544_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-561_3
    GII/Hu/KH/2004-2006/GII.P4_GII.4_Yerseke_2006a/NP-598_3
    GII/Hu/KH/2004-2006/GII.P6_GII.6/NP-544_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-255_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-318_1
    GII/Hu/KH/2004-2006/GII.P7_GII.14/NP-593_3
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-428_1
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-432_1
    GII/Hu/KH/2004-2006/GII.P7_GII.20/NP-492_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-435_5
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-483_1
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-525_3
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-526_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NP-565_2
    GII/Hu/KH/2004-2006/GII.P7_GII.6/NPC-031_1
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1006
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana108
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1087
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana124
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1331
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1363
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1367
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1383
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana1560
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana181
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana192
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana212
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2172
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2192
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2395
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2398
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2414
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2443
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2481
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2487
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2515
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2536
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2612
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2619
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2629
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2698
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2721
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2763
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2768
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana2820
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana34
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana36
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana362
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana456
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana464
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana75
    GII/Hu/SI/2014/GII.4_Sydney2012/Ljubljana795
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1025
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1026
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1096
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1314
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1399
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1447
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1457
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1537
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana1643
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana181
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana251
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana504
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana539
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana566
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana58
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana655
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana76
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana774
    GII/Hu/SI/2015/GII.4_Sydney2012/Ljubljana826
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/105
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/106
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/108
    GII/Hu/TW/2012/GII.Pe_GII.4_Sydney2012/Taipei/109
    GII/Hu/ZA/2012/GII.Pe_GII.4
    GII/Hu/ZA/2012/GII.Pg_GII.12/Bushbuckridge9306
    GII/Hu/ZA/2013/GII.P7_GII.6/Johannesburg130930
    GII/Hu/ZAF/2009/GII.P4_GII.4/Johannesburg/4175
    GII/Hu/ZAF/2011/GII.P4_GII.4/Empangeni/8501
    GII/Hu/ZAF/2011/GII.P4_GII.4/Johannesburg/7028
    GII/Hu/ZAF/2012/GII.P4_GII.4/CapeTown/10105
    GII/Hu/ZAF/2012/GII.P4_GII.4/CapeTown/9772
    GII/Hu/ZAF/2012/GII.P4_GII.4/Empangeni/8604
    GII/Hu/ZAF/2012/GII.P4_GII.4/Empangeni/9693
    GII/Hu/ZAF/2012/GII.P4_GII.4/Johannesburg/BW
    gilardii_C
    gilardii_D
    gilardii_E
    gingeri_A
    gingeri_C
    gingeri_D
    ginsengisoli_A
    ginsengisoli_B
    ginsengisoli_C
    ginsengisoli_D
    ginsengisoli_E
    ginsengisoli_F
    ginsenosidimutans_A
    glacialis_A
    glacincola_B
    glacincola_C
    gladioli_A
    gladioli_B
    glaucescens_B
    globiformis_A
    globiformis_B
    globiformis_C
    globiformis_D
    globisporus_A
    globisporus_D
    globulus_B
    gloriosae_A
    glossinidia_A
    glossinidia_B
    glossinidia_C
    glucosidilyticum_B
    glucuronolyticum_A
    glycanilyticus_A
    glycanilyticus_B
    glycinis_A
    glycyrrhizinilytica_A
    glycyrrhizinilyticus_A
    gomsonensis_A
    goodii_B
    gordonae_B
    gordonae_D
    gordonii_A
    gottheilii_A
    gracilis_A
    gracilis_B
    gracilis_C
    gracilis_D
    grahamii_A
    graminis_B
    graminis_D
    grandis_A
    grandis_C
    granulomatis_A
    griffiniae_A
    griseocarneus_A
    griseoluteus_A
    griseorubiginosus_B
    griseoviridis_A
    griseus_G
    griseus_I
    GS2_2_25
    GS2_2_29
    GS3_1_3
    GSM_I_36B_cl_10_2
    GSM_I_36B_cl_11_3
    guangdongense_A
    guariconensis_A
    guariconensis_B
    guariconensis_C
    guariconensis_D
    GW2011_AR1
    GW2011_AR10
    GW2011_AR11
    GW2011_AR13
    GW2011_AR15
    GW2011_AR16
    GW2011_AR17
    GW2011_AR18
    GW2011_AR19
    GW2011_AR20
    GW2011_AR21
    GW2011_AR3
    GW2011_AR4
    GW2011_AR5
    GW2011_AR6
    GW2011_AR9
    gwangjuense_A
    H01_1mb1_b10_2005-12-15
    H9_DGR
    H_A17_151
    H_C14_114
    H_C14_141
    H_C14_142
    H_C14_166
    H_C49_23
    H_C49_53
    H_C49_54
    H_C49_57
    H_C49_69
    H_C49_72
    H_C49_73
    haematequi_A
    haemolysans_A
    haemolysans_B
    haemolysans_C
    haemolysans_D
    haemolytica_A
    haemolyticum_A
    haemolyticus_A
    haemolyticus_G
    haemolyticus_I
    haemolyticus_J
    haemolyticus_K
    haemolyticus_L
    haemolyticus_M
    haemolyticus_N
    haemolyticus_O
    haikouensis_A
    halitosis_A
    halitosis_B
    haloalkaliphilus_A
    halocryophilus_A
    halocynthiae_B
    halodurans_A
    halophila_A
    halophilus_A
    halophilus_B
    halophilus_C
    halotolerans_A
    hampsonii_B
    hampsonii_C
    hankookensis_A
    haoranii_A
    Hare_CoV/La06_377/Lep_gra/ESP/2006
    Hare_CoV/La06_378/Lep_gra/ESP/2006
    Hare_CoV/La06_379/Lep_gra/ESP/2006
    Hare_CoV/La06_380/Lep_gra/ESP/2006
    Hare_CoV/La06_382/Lep_gra/ESP/2006
    Hare_CoV/La06_383/Lep_gra/ESP/2006
    Hare_CoV/La08_60/Lep_gra/ESP/2008
    Hare_CoV/La08_96/Lep_gra/ESP/2008
    Hare_CoV/La08_99/Lep_gra/ESP/2008
    harei_A
    harenae_A
    harundinacea_A
    harundinacea_B
    harundinacea_D
    hathewayi_A
    heliothermus_B
    helleri_A
    helvetica_A
    helveticum_A
    hengshuiensis_A
    hepatarius_A
    hepatarius_B
    heraklionense_A
    heraklionense_C
    herbae_A
    hermannii_A
    hertigii_A
    hertigii_B
    heylii_A
    higoensis_A
    hippei_A
    hippocampi_A
    hispanicum_A
    HMP_JCVI_SC0102
    HMP_JCVI_SC0110
    HMP_JCVI_SC0119
    HMP_JCVI_SC0121
    HMP_JCVI_SC0124
    HMP_JCVI_SC0125
    HMP_JCVI_SC0126
    HMP_JCVI_SC0127
    HMP_JCVI_SC0141
    HMP_JCVI_SC0143
    HMP_JCVI_SC0175
    HMP_JCVI_SC0203
    HMP_JCVI_SC0224
    HMP_JCVI_SC0234
    HMP_JCVI_SC0278
    HMP_JCVI_SC0281
    hmp_mda_pilot_jcvi_0005
    hodleri_A
    hodleri_B
    hominis_A
    hominis_B
    hominis_C
    hopiensis_A
    Hoplo_CB025_CB025_18_01
    hordeovulneris_A
    horikoshii_A
    horikoshii_B
    horikoshii_C
    horikoshii_D
    horikoshii_E
    horikoshii_F
    hormaechei_B
    hormaechei_C
    Horo_12
    Horo_15
    hortorum_B
    hortorum_C
    Hu/Caterer_Company_1_pat_1/10-04-2003/NL
    Hu/Caterer_Company_1_pat_2/10-04-2003/NL
    Hu/Caterer_Company_2_pat_1/12-04-2003/NL
    Hu/Caterer_Company_2_pat_2/12-04-2003/NL
    Hu/Caterer_personel_1/16-04-2003/NL
    Hu/Caterer_personel_2/16-04-2003/NL
    Hu/Caterer_personel_3/16-04-2003/NL
    Hu/Caterer_personel_4/16-04-2003/NL
    Hu/Caterer_personel_5/16-04-2003/NL
    Hu/Daycare-center_A/21-02-2002/NL
    Hu/Daycare-center_B/12-12-2002/NL
    Hu/Diner_Amsterdam/08-07-2002/NL
    Hu/G1/FH-0931_Kr/2010
    Hu/G1/FH-0933_Kr/2010
    Hu/G2/FH-0901_Kr/2009
    Hu/G2/FH-0902_Kr/2009
    Hu/G2/FH-0903_Kr/2009
    Hu/G2/FH-0904_Kr/2009
    Hu/G2/FH-0905_Kr/2009
    Hu/G2/FH-0906_Kr/2009
    Hu/G2/FH-0907_Kr/2009
    Hu/G2/FH-0908_Kr/2009
    Hu/G2/FH-0909_Kr/2009
    Hu/G2/FH-0910_Kr/2009
    Hu/G2/FH-0911_Kr/2009
    Hu/G2/FH-0912_Kr/2009
    Hu/G2/FH-0913_Kr/2009
    Hu/G2/FH-0914_Kr/2009
    Hu/G2/FH-0915_Kr/2009
    Hu/G2/FH-0916_Kr/2009
    Hu/G2/FH-0917_Kr/2009
    Hu/G2/FH-0918_Kr/2009
    Hu/G2/FH-0919_Kr/2009
    Hu/G2/FH-0920_Kr/2009
    Hu/G2/FH-0921_Kr/2009
    Hu/G2/FH-0922_Kr/2009
    Hu/G2/FH-0923_Kr/2009
    Hu/G2/FH-0924_Kr/2009
    Hu/G2/FH-0925_Kr/2009
    Hu/G2/FH-0926_Kr/2009
    Hu/G2/FH-0927_Kr/2009
    Hu/G2/FH-0928_Kr/2010
    Hu/G2/FH-0929_Kr/2010
    Hu/G2/FH-0930_Kr/2010
    Hu/G2/FH-0932_Kr/2010
    Hu/G2/FH-0934_Kr/2009
    Hu/GGI.2/Treated_Wastewater/Llobregat_S6a/Jan2008/SP
    Hu/GGI.2/Wastewater/Llobregat_S5/Jan2008/SP
    Hu/GGI.3/Wastewater/Llobregat_S5/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S2/Dec2008/SP
    Hu/GGI.4/River_Water/Llobregat_S2/Jan2008/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Dec2008/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Feb2009/SP
    Hu/GGI.4/River_Water/Llobregat_S3/Nov2008/SP
    Hu/GGI.4/River_Water/Llobregat_S4/Feb2009/SP
    Hu/GGI.4/River_Water/Llobregat_S7/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S9/Apr2008/SP
    Hu/GGI.4/River_Water/Llobregat_S9/Feb2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Apr2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Feb2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jan2009/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jul2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6/Jun2008/SP
    Hu/GGI.4/Treated_Wastewater/Llobregat_S6b/Jan2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Apr2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Dec2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Feb2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Jan2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Jul2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Oct2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S12/Sep2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Feb2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Feb2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Jan2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Jun2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Mar2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S5/Sep2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Jan2009/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Nov2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Oct2008/SP
    Hu/GGI.4/Wastewater/Llobregat_S8/Sep2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8/Feb2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8b/May2008/SP
    Hu/GGI.5/Wastewater/Llobregat_S8c/May2008/SP
    Hu/GGI.7/ASPB06/2010/10_11180/Jan2010/SP
    Hu/GGI.7/ASPB08/2010/10_01949/Jan2010/SP
    Hu/GGI.7/ASPB80/2010/10_31750/Dec2010/SP
    Hu/GGI.7/Wastewater/Llobregat_S8/Jun2008/SP
    Hu/GGI.8/Wastewater/Llobregat_S8a/May2008/SP
    Hu/GGI.NA2/River_Water/Llobregat_S7/Jun2008/SP
    Hu/GGI.NA2/Treated_Wastewater/Llobregat_S6/Dec2008/SP
    Hu/GGI.NA3/River_Water/Llobregat_S4/Apr2008/SP
    Hu/GGII-4/Badalona/1586/BNM_05/06/Sp
    Hu/GGII-4/Banyoles/1474/Gi_10/06/Sp
    Hu/GGII-4/Barcelona/1258/BNM_08/05
    Hu/GGII-4/Blanes/1483/Gi_22/06/Sp
    Hu/GGII-4/Cabrera/1590/RCSP_01/Sp
    Hu/GGII-4/Calonge/Gi_03/1216/05
    Hu/GGII-4/Cambrils/1105/TA_10/05
    Hu/GGII-4/Corbera/RCSP_68/1134/04
    Hu/GGII-4/Els_Pallers/1617/TA_02/06/Sp
    Hu/GGII-4/Roses/1491/Gi_23/06/Sp
    Hu/GGII-4/Sabadell/1196/RC94_04/04
    Hu/GGII-4/Sabadell/1236/RC_08/05
    Hu/GGII-4/Sabadell/1510/RC_02/06/Sp
    Hu/GGII-4/Sant_Cugat/1527/RC_06/06/Sp
    Hu/GGII-4/Sant_Quirze/1092/RC_80/04
    Hu/GGII-4/Sant_Sadurni/1087/RC_84/04
    Hu/GGII-4/Tarragona/1613/TA_01/06/Sp
    Hu/GGII-4/Tarragona/1624/TA_03/06/Sp
    Hu/GGII-4/Tarragona/1636/TA_05/06/Sp
    Hu/GGII-4/Tremp/1567/Lleida_03/06/Sp
    Hu/GGII-4/Valencia_HCUV/593/02
    Hu/GGII-4/Vall_de_Bianya/1477/Gi_19/06/Sp
    Hu/GGII.12/ASPB80/2010/10_31978/Dec2010/SP
    Hu/GGII.13/Wastewater/Llobregat_S12/Apr2008/SP
    Hu/GGII.14/River_Water/Llobregat_S3/Feb2009/SP
    Hu/GGII.21/ASPB11/2010/10_01694/Feb2010/SP
    Hu/GGII.21/River_Water/Llobregat_S4/Jan2008/SP
    Hu/GGII.21/River_Water/Llobregat_S9b/Apr2008/SP
    Hu/GGII.21/Treated_Wastewater/Llobregat_S6/Feb2008/SP
    Hu/GGII.21/Treated_Wastewater/Llobregat_S6/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Feb2009/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S12/Oct2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Feb2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S5/Jun2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S8/Jan2008/SP
    Hu/GGII.21/Wastewater/Llobregat_S8/Jan2009/SP
    Hu/GGII.3/River_Water/Llobregat_S2/Dec2008/SP
    Hu/GGII.3/River_Water/Llobregat_S3/Dec2008/SP
    Hu/GGII.3/Wastewater/Llobregat_S5b/Apr2008/SP
    Hu/GGII.4/ASPB01/2011/11_00122/Jan2011/SP
    Hu/GGII.4/ASPB02/2010/10_00563/Jan2010/SP
    Hu/GGII.4/ASPB03/2011/11_02329/Jan2011/SP
    Hu/GGII.4/ASPB05/2010/10_01034/Jan2010/SP
    Hu/GGII.4/ASPB05/2010/10_01737/Jan2010/SP
    Hu/GGII.4/ASPB05/2010/10_01739/Jan2010/SP
    Hu/GGII.4/ASPB109/2012/12_22463/Oct2012/SP
    Hu/GGII.4/ASPB11/2010/10_01644/Feb2010/SP
    Hu/GGII.4/ASPB111/2011/11_21457/Oct2011/SP
    Hu/GGII.4/ASPB111/2012/12_22947/SP
    Hu/GGII.4/ASPB14/2011/11_05091/March2011/SP
    Hu/GGII.4/ASPB15/2011/11_05033/March2011/SP
    Hu/GGII.4/ASPB15/2011/11_05034/March2011/SP
    Hu/GGII.4/ASPB25/2010/10_05477/March2010/SP
    Hu/GGII.4/ASPB25/2010/10_05657/March2010/SP
    Hu/GGII.4/ASPB74/2010/10_30425/Nov2010/SP
    Hu/GGII.4/River_Water/Llobregat_S2/Jan2008/SP
    Hu/GGII.4/River_Water/Llobregat_S3/Feb2008/SP
    Hu/GGII.4/River_Water/Llobregat_S3/Nov2008/SP
    Hu/GGII.4/River_Water/Llobregat_S4b/Jan2008/SP
    Hu/GGII.4/River_Water/Llobregat_S7/Apr2008/SP
    Hu/GGII.4/River_Water/Llobregat_S9/Feb2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S12/Mar2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S5/Dec2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S5/Oct2008/SP
    Hu/GGII.4/Wastewater/Llobregat_S8/Feb2008/SP
    Hu/GGII.6/ASPB105/2012/12_21910/Oct2012/SP
    Hu/GGII.6/ASPB108/2012/12_22457/Oct2012/SP
    Hu/GGII.6/ASPB38/2012/12_09395/May2012/SP
    Hu/GGII.6/ASPB79/2012/12_18747/Sep2012/SP
    Hu/GGII.6/River_Water/Llobregat_S3/Apr2008/SP
    Hu/GGII.6/River_Water/Llobregat_S9/Apr2008/SP
    Hu/GGII.6/Wastewater/Llobregat_S5/Apr2008/SP
    Hu/GGII.6/Wastewater/Llobregat_S5/Jan2009/SP
    Hu/GGII.6/Wastewater/Llobregat_S8/Oct2008/SP
    Hu/GGII.7/ASPB01/2011/11_00124/Jan2011/SP
    Hu/GGII.7/ASPB105/2012/12_22088/Oct2012/SP
    Hu/GGII.7/River_Water/Llobregat_S7/May2008/SP
    Hu/GI.1/CHA6A003_20091028/2009/USA
    Hu/GI.1/CHA6A003_20091031/2009/USA
    Hu/GI.1/CHA6A003_20091104/2009/USA
    Hu/GI.1/CHA9A004_20110419/2011/USA
    Hu/GI.1/CHA9A004_20110426/2011/USA
    Hu/GI.3/JKPG_881/SWE/2007
    Hu/GI.3/JKPG_882/SWE/2007
    Hu/GI.3/JKPG_883/SWE/2007
    Hu/GI.4/Muonio/11V1867_1/2011/FIN
    Hu/GI.4/Muonio/11V1869_1/2011/FIN
    Hu/GI.4/Muonio/11V1869_2/2011/FIN
    Hu/GI.6/M16_003/2016/JP
    Hu/GI.6/wastewater/Llobregat_S5/Oct2008/SP
    Hu/GI/GP1_Aug09_4102/ZAF
    Hu/GI/GP1_Feb10_4826/ZAF
    Hu/GI/GP1_Feb10_4832/ZAF
    Hu/GI/GP1_Jun09_3613/ZAF
    Hu/GI/GP1_Mar10_5068/ZAF
    Hu/GI/GP1_Mar11_7137/ZAF
    Hu/GI/GP1_May09_3474/ZAF
    Hu/GI/GP1_Oct09_4398/ZAF
    Hu/GI/GP2_Aug10_2676/ZAF
    Hu/GI/GP2_Feb08_1335/ZAF
    Hu/GI/GP2_Feb11_2905/ZAF
    Hu/GI/GP2_Jul08_1520/ZAF
    Hu/GI/GP2_Jun08_1496/ZAF
    Hu/GI/GP2_May08_1454/ZAF
    Hu/GI/InDRE2883_Tamps/2006/MEX
    Hu/GI/KZN_Dec10_6692/ZAF
    Hu/GI/MP_Aug10_6150/ZAF
    Hu/GI/MP_Dec11_8408/ZAF
    Hu/GI/MP_Jul11_3115/ZAF
    Hu/GI/MP_May10_5433/ZAF
    Hu/GI/MP_May10_5522/ZAF
    Hu/GI/MP_May11_7519/ZAF
    Hu/GI/WC_Feb11_6926/ZAF
    Hu/GI/WC_Jan11_6837/ZAF
    Hu/GI/WC_Jan11_6844/ZAF
    Hu/GII-4/2007a_ORF2-3/JP
    Hu/GII-4/2007a_ORF2/JP
    Hu/GII-4/2007b_ORF1/JP
    Hu/GII-4/2007b_ORF2-3/JP
    Hu/GII-4/2007b_ORF2/JP
    Hu/GII-4/2008a_ORF1/JP
    Hu/GII-4/2008a_ORF2-3/JP
    Hu/GII-4/2008a_ORF2/JP
    Hu/GII-4/2008b_ORF1/JP
    Hu/GII-4/2008b_ORF2-3/JP
    Hu/GII-4/2008b_ORF2/JP
    Hu/GII-4/Berga/RC69_02/926/03/Sp
    Hu/GII-4/Can_Sans/787/02/Sp
    Hu/GII-4/Manresa/81_02/914/02/Sp
    Hu/GII-4/Pineda_de_Mar/792/02/Sp
    Hu/GII-4/Res._S.Patricio/757/02/Sp
    Hu/GII-4/Res_Enbellpuig/793/02/Sp
    Hu/GII-4/Sallent/RC_73_02/921/02/Sp
    Hu/GII.13/ASPB03/2010/10_04332/March2010/SP
    Hu/GII.13/ASPB37/2010/10_15477/June2010/SP
    Hu/GII.2/Akita7_2014_JP
    Hu/GII.2/Akita8_2012_JP
    Hu/GII.2/Akita8_2014_JP
    Hu/GII.2/Aomori7_2014_JP
    Hu/GII.2/Ehime1_2012_JP
    Hu/GII.2/Ehime2_2012_JP
    Hu/GII.2/Ehime4_2014_JP
    Hu/GII.2/Ehime5_2014_JP
    Hu/GII.2/Fukui1_2012_JP
    Hu/GII.2/Fukui2_2012_JP
    Hu/GII.2/Hiroshima6_2014_JP
    Hu/GII.2/Hiroshimacity1_2012_JP
    Hu/GII.2/Hiroshimacity2_2012_JP
    Hu/GII.2/Hiroshimacity5_2014_JP
    Hu/GII.2/Hiroshimacity6_2014_JP
    Hu/GII.2/Miyagi1_2012_JP
    Hu/GII.2/Miyagi2_2014_JP
    Hu/GII.2/Niigata5_2012_JP
    Hu/GII.2/Osaka4_2014_JP
    Hu/GII.2/Osaka5_2014_JP
    Hu/GII.2/Osakacity5_2012_JP
    Hu/GII.2/Saitama4_2012_JP
    Hu/GII.2/Saitama5_2012_JP
    Hu/GII.4/Airv_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Airv_C1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R11/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R12/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R13/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R14/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R17/2012/Gothenburg/Sweden
    Hu/GII.4/Dust_A1_R18/2012/Gothenburg/Sweden
    Hu/GII.4/InDRE0644_Chis/MEX/2006
    Hu/GII.4/InDRE1307_Camp/MEX/2006
    Hu/GII.4/InDRE1401_Chis/MEX/2006
    Hu/GII.4/InDRE1697_Chis/MEX/2006
    Hu/GII.4/InDRE1839_Camp/MEX/2006
    Hu/GII.4/InDRE2115_Hgo/MEX/2006
    Hu/GII.4/InDRE2154_QRoo/MEX/2006
    Hu/GII.4/InDRE2256_QRoo/MEX/2006
    Hu/GII.4/InDRE2266_Mich/MEX/2006
    Hu/GII.4/InDRE2307_Camp/MEX/2006
    Hu/GII.4/InDRE2378_QRoo/MEX/2006
    Hu/GII.4/InDRE2384_Tamps/MEX/2006
    Hu/GII.4/InDRE2389_Tamps/MEX/2006
    Hu/GII.4/InDRE2395_Tamps/MEX/2006
    Hu/GII.4/InDRE2600_Camp/MEX/2006
    Hu/GII.4/InDRE2627_QRoo/MEX/2006
    Hu/GII.4/InDRE2628_QRoo/MEX/2006
    Hu/GII.4/InDRE2659_Camp/MEX/2006
    Hu/GII.4/InDRE2661_Camp/MEX/2006
    Hu/GII.4/InDRE2691_Chis/MEX/2006
    Hu/GII.4/InDRE2788_Tamps/MEX/2006
    Hu/GII.4/InDRE2812_Tamps/MEX/2006
    Hu/GII.4/InDRE2822_Tamps/MEX/2006
    Hu/GII.4/InDRE2835_QRoo/MEX/2006
    Hu/GII.4/InDRE2872_Chis/MEX/2006
    Hu/GII.4/InDRE2900_QRoo/MEX/2006
    Hu/GII.4/InDRE2913_QRoo/MEX/2006
    Hu/GII.4/InDRE2915_QRoo/MEX/2006
    Hu/GII.4/InDRE2917_QRoo/MEX/2006
    Hu/GII.4/InDRE2983_Tamps/MEX/2006
    Hu/GII.4/InDRE2987_Tamps/MEX/2006
    Hu/GII.4/InDRE2991_Camp/MEX/2006
    Hu/GII.4/InDRE3013_Ver/MEX/2006
    Hu/GII.4/InDRE3019_Ver/MEX/2006
    Hu/GII.4/InDRE3098_Yuc/MEX/2006
    Hu/GII.4/InDRE3101_Yuc/MEX/2006
    Hu/GII.4/InDRE3103_Yuc/MEX/2006
    Hu/GII.4/InDRE3114_Yuc/MEX/2006
    Hu/GII.4/InDRE3116_Yuc/MEX/2006
    Hu/GII.4/InDRE3129_Yuc/MEX/2006
    Hu/GII.4/InDRE3144_Tamps/MEX/2006
    Hu/GII.4/InDRE3156_Tamps/MEX/2006
    Hu/GII.4/InDRE3159_Tamps/MEX/2006
    Hu/GII.4/InDRE3186_Ver/MEX/2006
    Hu/GII.4/InDRE3203_QRoo/MEX/2006
    Hu/GII.4/InDRE3228_QRoo/MEX/2006
    Hu/GII.4/InDRE3238_Tamps/MEX/2006
    Hu/GII.4/InDRE3242_Tamps/MEX/2006
    Hu/GII.4/InDRE3246_Tamps/MEX/2006
    Hu/GII.4/InDRE3251_Tamps/MEX/2006
    Hu/GII.4/InDRE3268_Tamps/MEX/2006
    Hu/GII.4/InDRE3293_Tamps/MEX/2006
    Hu/GII.4/InDRE3296_Tamps/MEX/2006
    Hu/GII.4/InDRE3300_Tamps/MEX/2006
    Hu/GII.4/InDRE3344_Tamps/MEX/2006
    Hu/GII.4/InDRE3387_Tamps/MEX/2006
    Hu/GII.4/InDRE3400_Ver/MEX/2006
    Hu/GII.4/InDRE3402_Ver/MEX/2006
    Hu/GII.4/InDRE3437_Tamps/MEX/2006
    Hu/GII.4/InDRE3927_Mich/MEX/2006
    Hu/GII.4/InDRE3928_Mich/MEX/2006
    Hu/GII.4/InDRE3933_Mich/MEX/2006
    Hu/GII.4/InDRE3934_Mich/MEX/2006
    Hu/GII.4/P_02-2010_1/GER/2010
    Hu/GII.4/P_02-2010_10/GER/2010
    Hu/GII.4/P_02-2010_11/GER/2010
    Hu/GII.4/P_02-2010_4/GER/2010
    Hu/GII.4/P_02-2010_5/GER/2010
    Hu/GII.4/P_02-2010_6/GER/2010
    Hu/GII.4/P_02-2010_7/GER/2010
    Hu/GII.4/P_02-2010_8/GER/2010
    Hu/GII.4/P_02-2010_9/GER/2010
    Hu/GII.4/P_02-2020_3/GER/2010
    Hu/GII.4/P_04-2009_10/GER/2009
    Hu/GII.4/P_04-2009_11/GER/2009
    Hu/GII.4/P_04-2009_13/GER/2009
    Hu/GII.4/P_04-2009_3/GER/2009
    Hu/GII.4/P_04-2009_5/GER/2009
    Hu/GII.4/P_04-2009_6/GER/2009
    Hu/GII.4/P_04-2009_8/GER/2009
    Hu/GII.4/P_04-2009_9/GER/2009
    Hu/GII.4/P_09-2009_1/GER/2009
    Hu/GII.4/P_09-2009_10/GER/2009
    Hu/GII.4/P_09-2009_2/GER/2009
    Hu/GII.4/P_09-2009_3/GER/2009
    Hu/GII.4/P_09-2009_4/GER/2009
    Hu/GII.4/P_09-2009_5/GER/2009
    Hu/GII.4/P_09-2009_6/GER/2009
    Hu/GII.4/P_09-2009_7/GER/2009
    Hu/GII.4/P_09-2009_8/GER/2009
    Hu/GII.4/P_09-2009_9/GER/2009
    Hu/GII.4/patient_A+5mo/2010/USA
    Hu/GII.4/patient_A/2010/USA
    Hu/GII.4/patient_B/2010/USA
    Hu/GII.4/patient_C/2010/USA
    Hu/GII.4/PR_BRA_1008_2011
    Hu/GII.4/PR_BRA_1382_2001
    Hu/GII.4/PR_BRA_2082_2011
    Hu/GII.4/PR_BRA_2901_2011
    Hu/GII.4/PR_BRA_3074_2011
    Hu/GII.4/PR_BRA_3157_2011
    Hu/GII.4/PR_BRA_3172_2011
    Hu/GII.4/PR_BRA_3228_2011
    Hu/GII.4/PR_BRA_3270_2011
    Hu/GII.4/PR_BRA_3329_2011
    Hu/GII.4/PR_BRA_3353_2011
    Hu/GII.4/PR_BRA_3484_2011
    Hu/GII.4/PR_BRA_4287_2010
    Hu/GII.4/PR_BRA_762_2011
    Hu/GII.4/Ptab_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/Ptab_C1_R7/2012/Gothenburg/Sweden
    Hu/GII.4/R_02-2010_1/GER/2010
    Hu/GII.4/R_02-2010_11/GER/2010
    Hu/GII.4/R_02-2010_2/GER/2010
    Hu/GII.4/R_02-2010_3/GER/2010
    Hu/GII.4/R_02-2010_4/GER/2010
    Hu/GII.4/R_02-2010_5/GER/2010
    Hu/GII.4/R_02-2010_6/GER/2010
    Hu/GII.4/R_02-2010_7/GER/2010
    Hu/GII.4/R_02-2010_8/GER/2010
    Hu/GII.4/R_09-2009_1/GER/2009
    Hu/GII.4/R_09-2009_10/GER/2009
    Hu/GII.4/R_09-2009_11/GER/2009
    Hu/GII.4/R_09-2009_3/GER/2009
    Hu/GII.4/R_09-2009_5/GER/2009
    Hu/GII.4/R_09-2009_6/GER/2009
    Hu/GII.4/R_09-2009_8/GER/2009
    Hu/GII.4/R_09-2009_9/GER/2009
    Hu/GII.4/Salo/11V2359_1/2011/FIN
    Hu/GII.4/Vt_A1_R10/2012/Gothenburg/Sweden
    Hu/GII.4/Vt_A1_R11/2012/Gothenburg/Sweden
    Hu/GII.4/Vt_B2_R15/2012/Gothenburg/Sweden
    Hu/GII.4/whb_B1_R15/2012/Gothenburg/Sweden
    Hu/GII.4/whb_C1_R7/2012/Gothenburg/Sweden
    Hu/GII.4_variant
    Hu/GII.7/A_03-2006_11/GER/2006
    Hu/GII.7/A_03-2006_12/GER/2006
    Hu/GII.7/A_03-2006_2/GER/2006
    Hu/GII.7/A_03-2006_3/GER/2006
    Hu/GII.7/A_03-2006_4/GER/2006
    Hu/GII.7/A_03-2006_5/GER/2006
    Hu/GII.7/A_03-2006_6/GER/2006
    Hu/GII.7/A_03-2006_7/GER/2006
    Hu/GII.7/A_03-2006_9/GER/2006
    Hu/GII.7/A_06-2006_1/GER/2006
    Hu/GII.7/A_06-2006_2/GER/2006
    Hu/GII.7/A_06-2006_3/GER/2006
    Hu/GII.7/A_06-2006_4/GER/2006
    Hu/GII.7/A_06-2006_5/GER/2006
    Hu/GII.7/A_06-2006_7/GER/2006
    Hu/GII.7/A_06-2006_8/GER/2006
    Hu/GII.7/A_06-2006_9/GER/2006
    Hu/GII.7/GII.6/LNOV_PA16014/BRAZIL/2016
    Hu/GII.7/Muonio/11V1867_2/2011/FIN
    Hu/GII.8/GII.8/LNOV_PA16001/BRAZIL/2016
    Hu/GII.g/Salo/11V2362_1/2011/FIN
    Hu/GII.P.17/GII.17/LNOV_PA16004/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16005/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16007/BRAZIL/2016
    Hu/GII.P.17/GII.17/LNOV_PA16009/BRAZIL/2016
    Hu/GII.P16_GII.13/PR1395/2012/ITA
    Hu/GII.P16_GII.13/PR6717/2010/ITA
    Hu/GII.P16_GII.3/PR11068/2011/ITA
    Hu/GII.P16_GII.3/PR4526/2012/ITA
    Hu/GII.P17_GII.17/
    Hu/GII.P17_GII.17/Jinshan15/Shanghai/2014/CHN
    Hu/GII.P17_GII.17/NSW6016/2014/AU
    Hu/GII.P17_GII.17/PA31/2016/ITA
    Hu/GII.P17_GII.17/PA39/2016/ITA
    Hu/GII.P17_GII.17/PR668/2015/ITA
    Hu/GII.P4_GII.4/Minerva/2006/USA
    Hu/GII.P6_GII.6/1401Y107/CHN/2014
    Hu/GII.P7_GII.6/1401Y075/CHN/2014
    Hu/GII.P7_GII.6/1401Y127/CHN/2014
    Hu/GII.P7_GII.6/1403Y001/CHN/2014
    Hu/GII.P7_GII.6/1405Y017/CHN/2014
    Hu/GII.P7_GII.6/1405Y024/CHN/2014
    Hu/GII.P7_GII.6/1405Y035/CHN/2014
    Hu/GII.P7_GII.6/1406Y114/CHN/2014
    Hu/GII.P7_GII.6/1415Y002/CHN/2014
    Hu/GII.P7_GII.6/Maryland/2014/USA
    Hu/GII.P7_GII.7/1401Y082/CHN/2014
    Hu/GII.P7_GII.7/1406Y012/CHN/2014
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16002/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16003/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16010/BRAZIL/2016
    Hu/GII.Pe/GII.4-Sydney-2012/LNOV_PA16016/BRAZIL/2016
    Hu/GII.Pg/GII.1/LNOV_PA16008/BRAZIL/2016
    Hu/GII.Pg/GII.1/LNOV_PA16011/BRAZIL/2016
    Hu/GII/AUS/2015/GII.P17_GII.17/Brisbane462
    Hu/GII/CN/2013/GII.P17_GII.17/Nanjing010141
    Hu/GII/GP1_Apr11_7390/ZAF
    Hu/GII/GP1_Feb11_6939/ZAF
    Hu/GII/GP1_Mar12_8762/ZAF
    Hu/GII/GP1_Nov10_6514/ZAF
    Hu/GII/GP1_Oct10_6381/ZAF
    Hu/GII/GP1_Sep11_8120/ZAF
    Hu/GII/GP2_Feb08_1351/ZAF
    Hu/GII/GP2_Mar11_2963/ZAF
    Hu/GII/GP2_Nov08_1706/ZAF
    Hu/GII/JP/2004/GII.P12_GII.2/Tochigi-92
    Hu/GII/JP/2004/GII.P2_GII.2/Hokkaido-13
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-85
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-86
    Hu/GII/JP/2004/GII.P2_GII.2/Tochigi-87
    Hu/GII/JP/2006/GII.P2_GII.2/Hokkaido-14
    Hu/GII/JP/2008/GII.P2_GII.2/Hokkaido-15
    Hu/GII/JP/2009/GII.P16_GII.2/Kanagawa-49
    Hu/GII/JP/2010/GII.P16_GII.2/Ehime-43
    Hu/GII/JP/2010/GII.P16_GII.2/Ehime-44
    Hu/GII/JP/2010/GII.P16_GII.2/Kanagawa-50
    Hu/GII/JP/2010/GII.P16_GII.2/Kanagawa-51
    Hu/GII/JP/2010/GII.P16_GII.2/Osaka-019
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-17
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-18
    Hu/GII/JP/2010/GII.P2_GII.2/Hiroshima-19
    Hu/GII/JP/2010/GII.P2_GII.2/Hokkaido-16
    Hu/GII/JP/2011/GII.P16_GII.2/Ehime-45
    Hu/GII/JP/2011/GII.P16_GII.2/Hiroshima-26
    Hu/GII/JP/2011/GII.P16_GII.2/Hokkaido-17
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-023
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-038
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-18
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-26
    Hu/GII/JP/2011/GII.P16_GII.2/Osaka-9
    Hu/GII/JP/2011/GII.P16_GII.2/Saitama-51
    Hu/GII/JP/2011/GII.P16_GII.2/Yamaguchi-4
    Hu/GII/JP/2012/GII.P16_GII.2/Ehime-46
    Hu/GII/JP/2012/GII.P16_GII.2/Hokkaido-18
    Hu/GII/JP/2012/GII.P16_GII.2/Saitama-121
    Hu/GII/JP/2012/GII.P16_GII.2/Saitama-122
    Hu/GII/JP/2012/GII.P16_GII.2/Tochigi-26
    Hu/GII/JP/2012/GII.P16_GII.2/Tochigi-30
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-6
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-8
    Hu/GII/JP/2013/GII.P16_GII.2/Ehime-9
    Hu/GII/JP/2013/GII.P16_GII.2/Miyagi-7
    Hu/GII/JP/2013/GII.P16_GII.2/Miyagi-8
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-123
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-124
    Hu/GII/JP/2013/GII.P16_GII.2/Saitama-125
    Hu/GII/JP/2013/GII.P16_GII.2/Tochigi-46
    Hu/GII/JP/2013/GII.P17_GII.17/FE21_2013
    Hu/GII/JP/2013/GII.P17_GII.17/Saitama5203
    Hu/GII/JP/2013/GII.P17_GII.17/Saitama5309
    Hu/GII/JP/2014/GII.P16_GII.2/Hiroshima-30
    Hu/GII/JP/2014/GII.P16_GII.2/Kanagawa-52
    Hu/GII/JP/2014/GII.P16_GII.2/Osaka-225
    Hu/GII/JP/2014/GII.P16_GII.2/Osaka246
    Hu/GII/JP/2014/GII.P16_GII.2/Saitama-126
    Hu/GII/JP/2014/GII.P16_GII.2/Tochigi-17
    Hu/GII/JP/2014/GII.P17_GII.17/Kawasaki323
    Hu/GII/JP/2014/GII.P17_GII.17/Nagano7-1
    Hu/GII/JP/2014/GII.P17_GII.17/Nagano8-1
    Hu/GII/JP/2014/GII.P2_GII.2/Yamaguchi-014
    Hu/GII/JP/2014/GII.Pe_GII.2/Saitama-127
    Hu/GII/JP/2015/GII.P17_GII.17/FE20-2015
    Hu/GII/JP/2015/GII.P17_GII.17/FE44-2014
    Hu/GII/JP/2015/GII.P17_GII.17/FE47-2014
    Hu/GII/JP/2015/GII.P17_GII.17/FE58-2014
    Hu/GII/JP/2015/GII.P17_GII.17/Kawasaki308
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster1-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster2-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster3-2015
    Hu/GII/JP/2015/GII.P17_GII.17/Oyster4-2015
    Hu/GII/JP/2015/GII.P2_GII.2/Miyagi-63
    Hu/GII/JP/2015/GII.P2_GII.2/Saitama-169
    Hu/GII/JP/2015/GII.Pe_GII.4/Osaka/OSF78
    Hu/GII/JP/2016/GII.P16_GII.4_Sydney2012/Kawasaki194
    Hu/GII/JP/2016/P16_GII.2/Aichi430
    Hu/GII/JP/2016/P16_GII.2/AichiF219
    Hu/GII/JP/2016/P16_GII.2/AichiF243
    Hu/GII/JP/2017/GII.P17_GII.17/Tokyo330021
    Hu/GII/JP/2017/P16_GII.2/Aichi156
    Hu/GII/MP_Jul09_3880/ZAF
    Hu/GII/MP_Mar11_2949/ZAF
    Hu/GII/OuagadougouBF102/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF128/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF136/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF146/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF152/GII.4_Sydney2012/2012/BFA
    Hu/GII/OuagadougouBF25/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF27/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF37/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouBF38/GII.4_Sydney2012/2011/BFA
    Hu/GII/OuagadougouTB04/GII.4_Sydney2012/2011/BFA
    Hu/GII/PA/VIR_16F/BR/04-2012
    Hu/GII/PA/VIR_17F/BR/04-2012
    Hu/GII/PA/VIR_18F/BR/04-2012
    Hu/GII/PA/VIR_19F/BR/04-2012
    Hu/GII/PA/VIR_1F/BR/03-2012
    Hu/GII/PA/VIR_20F/BR/04-2012
    Hu/GII/PA/VIR_22F/BR/05-2012
    Hu/GII/PA/VIR_24F/BR/05-2012
    Hu/GII/PA/VIR_25F/BR/05-2012
    Hu/GII/PA/VIR_28F/BR/05-2012
    Hu/GII/PA/VIR_2F/BR/03-2012
    Hu/GII/PA/VIR_31F/BR/06-2012
    Hu/GII/PA/VIR_32F/BR/06-2012
    Hu/GII/PA/VIR_35F/BR/06-2012
    Hu/GII/PA/VIR_3F/BR/03-2012
    Hu/GII/PA/VIR_45F/BR/07-2012
    Hu/GII/PA/VIR_45S/BR/07-2012
    Hu/GII/PA/VIR_63F/BR/08-2012
    Hu/GII/PA/VIR_73F/BR/08-2012
    Hu/GII/PA/VIR_76F/BR/08-2012
    Hu/GII/PA/VIR_82F/BR/08-2012
    Hu/GII/PA/VIR_82S/BR/08-2012
    Hu/GII/RO/2015/GII.P17_GII.17/Dezna187865
    Hu/GII/RO/2015/GII.P17_GII.17/Sebis187601
    Hu/GII/RO/2015/GII.P17_GII.17/Sebis187602
    Hu/GII/WC_Jun10_5755/ZAF
    Hu/GIV/GP1_Jun10_5795/ZAF
    Hu/GIV/GP1_Mar10_5040/ZAF
    Hu/GIV/GP1_Mar10_5065/ZAF
    Hu/GIV/GP1_Mar11_7139/ZAF
    Hu/GIV/GP1_Sep09_4231/ZAF
    Hu/GIV/GP2_Dec08_1766/ZAF
    Hu/GIV/GP2_Jul10_2631/ZAF
    Hu/GIV/GP3_Mar08_1383/ZAF
    Hu/GIV/MP_Mar10_5036/ZAF
    Hu/GIV/MP_Mar11_8760/ZAF
    Hu/GIV/MP_May10_5510/ZAF
    Hu/GP1_Aug09_4102/ZAF
    Hu/GP1_Feb10_4826/ZAF
    Hu/GP1_Feb10_4832/ZAF
    Hu/GP1_Jun09_3613/ZAF
    Hu/GP1_Jun10_5795/ZAF
    Hu/GP1_Nov10_6514/ZAF
    Hu/GP2_Apr08_1416/ZAF
    Hu/GP2_Aug10_2676/ZAF
    Hu/GP2_Feb08_1351/ZAF
    Hu/GP2_Jun08_1496/ZAF
    Hu/GP2_Jun08_1520/ZAF
    Hu/GP2_Mar11_2949/ZAF
    Hu/Hospital_AA_PAAZ_pat_1/15-07-2002/NL
    Hu/Hospital_AA_PAAZ_personel_1/18-07-2002/NL
    Hu/Hospital_AB_A1_pat_104-02-2003/NL
    Hu/Hospital_AB_A1_pat_205-03-2003/NL
    Hu/Hospital_AB_A3_pat_1/10-02-2003/NL
    Hu/Hospital_AB_A3_personel_1/10-02-2003/NL
    Hu/Hospital_AB_A7_pat_1/06-01-2003/NL
    Hu/Hospital_AB_A8_pat_1/21-01-2003/NL
    Hu/Hospital_AB_A9_pat_1/18-12-2002/NL
    Hu/Hospital_AB_B1_pat_1/22-01-2003/NL
    Hu/Hospital_AB_B1_pat_2/28-02-2003/NL
    Hu/Hospital_AB_B1_personel_1/27-01-2003/NL
    Hu/Hospital_AB_PAAZ_pat07-02-2003/NL2.
    Hu/Hospital_AB_PAAZ_pat_3/11-02-2003/NL
    Hu/InDRE0644_Chis/MEX/2006
    Hu/InDRE1307_Camp/MEX/2006
    Hu/InDRE1401_Chis/MEX/2006
    Hu/InDRE1406_DF/MEX/2006
    Hu/InDRE1697_Chis/MEX/2006
    Hu/InDRE1835_NL/MEX/2006
    Hu/InDRE1843_Camp/MEX/2006
    Hu/InDRE2010_DF/MEX/2006
    Hu/InDRE2115_Hgo/MEX/2006
    Hu/InDRE2154_QRoo/MEX/2006
    Hu/InDRE2256_QRoo/MEX/2006
    Hu/InDRE2257_QRoo/MEX/2006
    Hu/InDRE2264_Mich/MEX/2006
    Hu/InDRE2265_Mich/MEX/2006
    Hu/InDRE2266_Mich/MEX/2006
    Hu/InDRE2283_QRoo/MEX/2006
    Hu/InDRE2289_QRoo/MEX/2006
    Hu/InDRE2295_QRoo/MEX/2006
    Hu/InDRE2389_Tamps/MEX/2006
    Hu/InDRE2395_Tamps/MEX/2006
    Hu/InDRE2627_QRoo/MEX/2006
    Hu/InDRE2628_QRoo/MEX/2006
    Hu/InDRE2641_QRoo/MEX/2006
    Hu/InDRE2659_Camp/MEX/2006
    Hu/InDRE2679_Chis/MEX/2006
    Hu/InDRE2812_Tamps/MEX/2006
    Hu/InDRE2913_QRoo/MEX/2006
    Hu/InDRE2915_QRoo/MEX/2006
    Hu/InDRE2917_QRoo/MEX/2006
    Hu/InDRE2943_QRoo/MEX/2006
    Hu/InDRE2987_Tamps/MEX/2006
    Hu/InDRE2991_Camp/MEX/2006
    Hu/InDRE3144_Tamps/MEX/2006
    Hu/InDRE3156_Tamps/MEX/2006
    Hu/InDRE3186_Ver/MEX/2006
    Hu/InDRE3228_QRoo/MEX/2006
    Hu/InDRE3238_Tamps/MEX/2006
    Hu/InDRE3242_Tamps/MEX/2006
    Hu/InDRE3244_Tamps/MEX/2006
    Hu/InDRE3246_Tamps/MEX/2006
    Hu/InDRE3344_Tamps/MEX/2006
    Hu/InDRE3358_Tamps/MEX/2006
    Hu/InDRE3919_Mich/MEX/2006
    Hu/InDRE3925_Mich/MEX/2006
    Hu/InDRE3926_Mich/MEX/2006
    Hu/InDRE3927_Mich/MEX/2006
    Hu/InDRE3928_Mich/MEX/2006
    Hu/InDRE3933_Mich/MEX/2006
    Hu/InDRE3934_Mich/MEX/2006
    Hu/KZN_Dec10_6692/ZAF
    Hu/Lelystad_Hospital/16-01-2003/NL
    Hu/Lelystad_Nursing-home/21-07-2002/NL
    Hu/Lelystad_Nursing-home/22-07-2002/NL
    Hu/MP_Dec11_8408/ZAF
    Hu/MP_Jul09_3880/ZAF
    Hu/MP_Jul11_3115/ZAF
    Hu/MP_May10_5433/ZAF
    Hu/MP_May11_7519/ZAF
    Hu/NoV/05_1/2002/SE
    Hu/NoV/05_2/2002/SE
    Hu/NoV/07_1/2005/SE
    Hu/NoV/07_2/2005/SE
    Hu/NoV/10_1/2006/SE
    Hu/NoV/10_2/2006/SE
    Hu/NoV/13_1/2006/SE
    Hu/NoV/13_2/2006/SE
    Hu/NoV/22_1/2006/SE
    Hu/NoV/22_2/2006/SE
    Hu/NoV/23_1/2006/SE
    Hu/NoV/23_2/2006/SE
    Hu/Nursing-home_A/28-12-2001/NL
    Hu/Nursing-home_B/05-05-2002/NL
    Hu/Nursing-home_C/06-06-2002/NL
    Hu/Nursing-home_D/19-11-2002/NL
    Hu/Nursing-home_E_pat_1/08-01-2003/NL/
    Hu/Nursing-home_E_pat_2/10-01-2003/NL
    Hu/Nursing-home_F/22-01-2003/NL
    Hu/Nursing-home_G/27-01-2003/NL
    Hu/Nursing-home_H/03-02-2003/NL
    Hu/PV_Voelk0017/2013/DEU
    Hu/PV_Voelk0018/2013/DEU
    Hu/River_Cruising_ship_1/16-05-2001/NL
    Hu/River_Cruising_Ship_2/08-04-2003/NL
    Hu/RU/2013/GII.4_Sydney2012/Nizhny_Novgorod4874
    Hu/RU/2014/GII.6/Nizhny_Novgorod1404
    Hu/RU/2014/GII.6/Nizhny_Novgorod1470
    Hu/RU/2014/GII.6/Nizhny_Novgorod1760
    Hu/RU/2014/GII.6/Nizhny_Novgorod18
    Hu/RU/2014/GII.6/Nizhny_Novgorod1908
    Hu/RU/2014/GII.6/Nizhny_Novgorod3125
    Hu/USA/2011/GI.P7_GI.7/CS5567
    Hu/USA/2014/GI.P7_GI.7/GA5043
    Hu/USA/2015/GII.P16_GII.4_Sydney/CA3477
    Hu/USA/2016/GI.P9_GI.9/SC6350
    Hu/WC_Feb11_6926/ZAF
    Hu/WC_Jun10_5755/ZAF
    huakuii_A
    huautlense_A
    hugenholtzii_A
    Human/GII/InDRE1839_Camp/2006/Mexico
    Human/GII/InDRE2307_Camp/2006/Mexico
    Human/GII/InDRE2378_QRoo/2006/Mexico
    Human/GII/InDRE2384_Tamps/2006/Mexico
    Human/GII/InDRE2600_Camp/2006/Mexico
    Human/GII/InDRE2661_Camp/2006/Mexico
    Human/GII/InDRE2691_Chis/2006/Mexico
    Human/GII/InDRE2788_Tamps/2006/Mexico
    Human/GII/InDRE2822_Tamps/2006/Mexico
    Human/GII/InDRE2835_QRoo/2006/Mexico
    Human/GII/InDRE2872_Chis/2006/Mexico
    Human/GII/InDRE2900_QRoo/2006/Mexico
    Human/GII/InDRE2919_QRoo/2006/Mexico
    Human/GII/InDRE2983_Tamps/2006/Mexico
    Human/GII/InDRE3013_Ver/2006/Mexico
    Human/GII/InDRE3019_Ver/2006/Mexico
    Human/GII/InDRE3098_Yuc/2006/Mexico
    Human/GII/InDRE3101_Yuc/2006/Mexico
    Human/GII/InDRE3103_Yuc/2006/Mexico
    Human/GII/InDRE3107_Yuc/2006/Mexico
    Human/GII/InDRE3114_Yuc/2006/Mexico
    Human/GII/InDRE3115_Yuc/2006/Mexico
    Human/GII/InDRE3124_Yuc/2006/Mexico
    Human/GII/InDRE3129_Yuc/2006/Mexico
    Human/GII/InDRE3159_Tamps/2006/Mexico
    Human/GII/InDRE3203_QRoo/2006/Mexico
    Human/GII/InDRE3251_Tamps/2006/Mexico
    Human/GII/InDRE3260_Tamps/2006/Mexico
    Human/GII/InDRE3268_Tamps/2006/Mexico
    Human/GII/InDRE3293_Tamps/2006/Mexico
    Human/GII/InDRE3296_Tamps/2006/Mexico
    Human/GII/InDRE3297_Tamps/2006/Mexico
    Human/GII/InDRE3300_Tamps/2006/Mexico
    Human/GII/InDRE3372_Tamps/2006/Mexico
    Human/GII/InDRE3387_Tamps/2006/Mexico
    Human/GII/InDRE3400_Ver/2006/Mexico
    Human/GII/InDRE3402_Ver/2006/Mexico
    Human/GII/InDRE3432_Tamps/2006/Mexico
    Human/GII/InDRE3437_Tamps/2006/Mexico
    humilata_A
    hungatei_A
    hungatei_B
    hwajinpoensis_A
    hwajinpoensis_B
    hwajinpoensis_C
    hwajinpoensis_D
    hydrocarbonoxydans_A
    hydrocarbonoxydans_B
    hydrocarbonoxydans_C
    hydrogenitrophicus_A
    hydrogenivorans_A
    hydrophilus_A
    hydrothermalis_A
    hygroscopicus_B
    hygroscopicus_C
    hygroscopicus_D
    hyointestinalis_A
    hypermegale_A
    I_35_MF_cl1
    I_35_MF_cl2
    idli_24
    idli_26
    idli_29
    idli_44
    idli_45
    IFC_YC12
    IHP_575_RK_Exo70_Native
    IHP_576_RK_native
    IIIL_D
    IK_MB1-800R
    IK_MB10-800R
    IK_MB11-800R
    IK_MB12-800R
    IK_MB13-800R
    IK_MB14-800R
    IK_MB15-800R
    IK_MB16-800R
    IK_MB2-800R
    IK_MB3-800R
    IK_MB4-800R
    IK_MB5-800R
    IK_MB6-800R
    IK_MB7-800R
    IK_MB8-800R
    IK_MB9-800R
    illinoisensis_A
    IM_14
    IM_15
    IM_4
    IM_4B
    IM_5
    IM_8A
    IM_9A
    IM_9B
    immobilis_A
    immobilis_B
    immobilis_C
    immobilis_D
    immobilis_E
    immobilis_F
    immobilis_G
    immobilis_H
    INBio_4054J
    INBio_4099AF
    INBio_4100K
    INBio_4507A
    INBio_4510E
    incarnatus_A
    indica_A
    indica_B
    indicum_A
    indicus_A
    indigofera_A
    indolica_A
    indologenes_A
    indologenes_B
    indologenes_E
    indologenes_F
    indologenes_G
    infantis_A
    infantis_B
    infantis_C
    infantis_E
    infantis_F
    infantis_G
    infantis_H
    infantis_I
    infantis_J
    infantis_K
    infantis_L
    infantis_M
    infantium_A
    infera_B
    infera_D
    influenzae_D
    influenzae_E
    influenzae_H
    inhae_A
    inopinata_A
    insuavis_A
    interjectum_B
    intermedia_A
    intermedia_B
    intermedia_C
    intermedius_A
    intestinalis_A
    intracellularis_B
    Ips_1
    Ips_2
    Ips_3
    Ips_4
    Ips_5
    Ips_6
    Ips_7
    iranensis_A
    iranicum_A
    iranicum_B
    iriomotense_A
    iriomotense_B
    IRQ_01
    IRQ_02
    IRQ_03
    IRQ_04
    IRQ_05
    IRQ_06
    IRQ_07
    IRQ_09
    IRQ_11
    IRQ_13
    IRQ_14
    IRQ_15
    IRQ_16
    IRQ_18
    IRQ_19
    IRQ_20
    IRQ_21
    IRQ_22
    IRQ_24
    IRQ_25
    IRQ_26
    IRQ_28
    IRQ_29
    IRQ_31
    IRQ_33
    IRQ_34
    IRQ_35
    IRQ_36
    IRQ_37
    IRQ_38
    IRQ_40
    IRQ_41
    IRQ_43
    IRQ_47
    IRQ_48
    IRQ_54
    IRQ_56
    IRQ_57
    IRQ_58
    IRQ_59
    IRQ_60
    IRQ_61
    IRQ_62
    ishigakensis_A
    ishizawai_A
    Iso11_2
    isronensis_A
    isronensis_B
    ISW_15
    ISW_2
    itersonii_A
    ixodetis_A
    jadensis_A
    japonica_A
    japonica_C
    japonicum_B
    japonicum_C
    japonicum_F
    japonicum_G
    japonicum_H
    japonicus_A
    JCVI_control_MDA_Ecoli_02
    JCVI_control_MDA_Saureus_10
    jeikeium_C
    jeikeium_D
    jejuense_A
    jejuensis_A
    jejuni_A
    jejuni_B
    jejuni_C
    jejuni_D
    jeotgali_A
    JGI_KR11_SmLR_4_I6
    JGI_KR11_SmLR_5_K8
    JGI_KR13_SmLR_4_D7
    jiangningensis_B
    jiangningensis_C
    jiangsuensis_A
    jiangxiensis_A
    jicamae_A
    jicamae_B
    JL4_S
    JN18_A7_F*
    johnsoniae_A
    johnsoniae_C
    johnsonii_A
    Jomon_CT89
    jostii_A
    JPL_7
    jumonjinensis_A
    K12_27
    K12_28b
    kashiwanohense_A
    kashmirensis_A
    katrae_A
    katrae_B
    kieliensis_A
    kilianii_A
    kilonensis_B
    kimbladii_B
    kirkii_B
    kisseleviana_A
    kitaharae_A
    kluyveri_B
    kmd_008
    kmd_043
    kmd_065
    kmd_070
    kmd_071
    kmd_072
    kmd_076
    kmd_078
    kmd_116
    kmd_123
    kmd_151
    kmd_152
    kmd_160
    kmd_217
    kmd_222
    kmd_278
    kmd_307
    kmd_323
    kmd_327
    kmd_332
    kmd_340
    kmd_360
    kobei_A
    komagatae_A
    koreensis_A
    koreensis_B
    koreensis_C
    koreensis_F
    koreensis_G
    koreensis_H
    koyamae_A
    koyamae_B
    koyamae_C
    kribbensis_B
    kroppenstedtii_B
    kujiense_A
    kunkeei_C
    kunkeei_E
    kunkeei_F
    kururiensis_A
    kushneri_A
    kyeonggiensis_A
    kyeonggiensis_B
    kyorinense_A
    kyotonensis_B
    L3_Loihi
    lablabi_A
    lablabi_C
    lablabi_D
    LaBonte_alg10
    lactamica_A
    lactamica_B
    lactatifermentans_A
    lactis_A
    lactosutens_A
    lactucae_A
    lacunata_B
    lacus_C
    lacus_D
    lacuscaerulensis_A
    lacuscaerulensis_B
    lacuscaerulensis_C
    lacustris_A
    Lad_11K
    Lad_23K
    laidlawii_A
    Lak_Megaphage_RVC_AP1_GC26
    Lak_Megaphage_RVC_AP3_GC26
    Lak_Megaphage_RVC_AP3_GC31
    Lak_Megaphage_RVC_AP4_GC26
    Lak_Megaphage_RVC_JS4_GC31
    Lak_Megaphage_Sonny
    laminariae_A
    lari_C
    lari_D
    lari_E
    larrymoorei_A
    larrymoorei_B
    lata_B
    lata_C
    latens_A
    laterosporus_D
    laterosporus_E
    lathyri_A
    lautus_A
    leadbetteri_A
    legallii_A
    leguminosarum_D
    leguminosarum_G
    leguminosarum_I
    leguminosarum_J
    leguminosarum_M
    leguminosarum_N
    leguminosarum_P
    leguminosarum_Q
    leguminosarum_R
    leguminosarum_S
    leguminosarum_T
    leguminosarum_U
    leguminosarum_V
    lekithochrous_A
    lentis_A
    lentus_A
    liaoningense_A
    liaoningense_B
    liaoningense_C
    lichenicola_A
    lichenicola_B
    lichenicola_C
    lichenicola_D
    lichenicola_E
    limicola_A
    limnetica_A
    limneticus_A
    limneticus_B
    limnophila_A
    limnophila_B
    limnophila_C
    limosum_B
    limosus_A
    limosus_C
    limosus_D
    limosus_E
    linckia_A
    linckia_B
    lini_A
    lipoferum_B
    lipoferum_C
    lipolytica_A
    lipolytica_C
    lipolyticus_A
    lipolyticus_B
    lipolyticus_C
    liquefaciens_A
    listeri_A
    lithotrophica_A
    lithotrophicus_A
    litopenaei_A
    litorale_A
    litorale_B
    litoralis_A
    litoralis_B
    litoralis_C
    litoralis_D
    litoralis_E
    litoralis_F
    litorea_A
    litorea_B
    lituseburensis_A
    Liv16S_LC_L1
    lividum_C
    lividum_D
    lividum_E
    lividum_F
    ljungdahlii_A
    longbeachae_A
    longispororuber_A
    longum_A
    longum_D
    loti_A
    loti_B
    loti_C
    loti_D
    loti_E
    loti_F
    loti_G
    lovleyi_A
    lovleyi_B
    LUC13011_P14
    lucentensis_A
    lucentensis_B
    luciferensis_B
    ludwigii_A
    lundensis_A
    lupini_A
    lupini_B
    lusitanum_A
    lusitanum_B
    lusitanum_C
    lusitanum_D
    lutea_A
    luteola_A
    luteola_B
    luteola_C
    luteola_D
    luteolum_B
    luteolum_C
    luteoviolacea_A
    luteoviolacea_B
    luteoviolacea_C
    luteoviolacea_D
    luteoviolacea_E
    luteoviolacea_F
    luteoviolacea_G
    luteoviolacea_H
    luteoviolacea_I
    luteoviolacea_J
    luteoviolacea_K
    luteum_A
    luteus_A
    luti_A
    lutilitoris_A
    lutimaris_A
    lwoffii_C
    lwoffii_D
    lwoffii_E
    lwoffii_F
    lwoffii_G
    lwoffii_H
    lwoffii_I
    lwoffii_J
    lwoffii_K
    lwoffii_L
    lwoffii_M
    lwoffii_N
    lwoffii_O
    lwoffii_P
    lydicus_C
    lyticus_A
    M12R1_907R_F06
    M13K07_deltapIII_deltaf1ori
    M32_bin.2
    M:02_AG
    M:A1_BA106
    M:A1_BA110
    M:A1_BA117
    M:A1_BA147
    M:A1_BA153
    M:A1_BA18
    M:A1_BA200
    M:A1_BA23
    M:A1_BA232
    M:A1_BA3
    M:A1_BA31
    M:A1_BA32
    M:A1_BA41
    M:A1_BA44
    M:A1_BA49
    M:A1_BA52
    M:A1_BA56
    M:A1_BA88
    M:A1_LBS024
    M:A3/U_MAU1876
    M:A3/U_MAU2044
    M:A3/U_MAU432
    M:A3_1223SC
    M:A3_125HPD
    M:A3_4769RTHALD
    M:A3_492HALD
    M:A3_K1245
    M:A3_K1339
    M:A3_L044
    M:A3_MAU159
    M:A3_MAU1838
    M:A3_MAU759
    M:A_06TG.HT032
    M:A_06TG.HT045
    M:A_06TG.HT055
    M:A_06TG.HT077
    M:A_06TG.HT085
    M:A_06TG.HT126
    M:A_06TG.HT152
    M:A_06TG.HT247
    M:A_15HALD
    M:A_171HPD
    M:A_2178/2012/KWT
    M:A_227HALD
    M:A_5HALD
    M:A_69HPD
    M:A_CHU12
    M:A_K1111
    M:A_K1113
    M:A_K1143
    M:A_K1200
    M:A_K1210
    M:A_K1212
    M:A_K1229
    M:A_K1245
    M:A_K1256
    M:A_K1259
    M:A_K1263
    M:A_K1290
    M:A_K1304
    M:A_K1305
    M:A_K1345
    M:A_K1355
    M:A_K1360
    M:A_K1363
    M:A_K1369
    M:A_K1394
    M:A_K2025
    M:A_K2032
    M:A_K2033
    M:A_KS049
    M:A_L019
    M:A_L044
    M:A_LBS024
    M:A_MP1294
    M:A_MP1392
    M:A_MP582
    M:A_MP640
    M:A_MP804
    M:A_MP812
    M:A_ORAV459
    M:A_U0434
    M:A_U1524
    M:A_U1803
    M:A_U1808
    M:A_U2226
    M:A_U3121
    M:B/A/CR14_BG_PT107
    M:B/CRF14_BG_PT176
    M:B/D_PT508
    M:B/URF_PT38
    M:B_00NOVAX03
    M:B_00NOVAX06
    M:B_00NOVAX10
    M:B_00NOVAX15
    M:B_00NOVAX20
    M:B_00NOVAX38
    M:B_00NOVAX40
    M:B_01NOVAX15
    M:B_02NOVAX30
    M:B_03NOVAX01
    M:B_03NOVAX02
    M:B_03NOVAX04
    M:B_03NOVAX06
    M:B_03NOVAX12
    M:B_03NOVAX13
    M:B_03NOVAX14
    M:B_03NOVAX15
    M:B_03NOVAX16
    M:B_03NOVAX19
    M:B_03NOVAX29
    M:B_03NOVAX40
    M:B_03NOVAX41
    M:B_04NOVAX01
    M:B_04NOVAX02
    M:B_04NOVAX12
    M:B_04NOVAX14
    M:B_04NOVAX15
    M:B_04NOVAX19
    M:B_04NOVAX20
    M:B_04NOVAX22
    M:B_04NOVAX26
    M:B_04NOVAX29
    M:B_04NOVAX36
    M:B_04NOVAX41
    M:B_04SN.MS019
    M:B_04SN.MS103
    M:B_04SN.MS482
    M:B_04SN.MS483
    M:B_04SN.MS605
    M:B_04SN.MS614
    M:B_04SN.MS619
    M:B_04SN.MS651
    M:B_04SN.MS802
    M:B_04SN.MS812
    M:B_04SN.MS830
    M:B_04SN.MS838
    M:B_04SN.MS868
    M:B_05NOVAX02
    M:B_05NOVAX23
    M:B_05NOVAX38
    M:B_06NOVAX03
    M:B_06NOVAX07
    M:B_06NOVAX10
    M:B_06NOVAX11
    M:B_06NOVAX12
    M:B_06NOVAX14
    M:B_06NOVAX19
    M:B_06NOVAX21
    M:B_06NOVAX28
    M:B_06NOVAX30
    M:B_06NOVAX32
    M:B_06NOVAX34
    M:B_06NOVAX37
    M:B_07CN.HN007
    M:B_07CN.HN009
    M:B_07CN.HN022
    M:B_07CN.HN023
    M:B_07CN.HN029
    M:B_07CN.HN030
    M:B_07CN.HN033
    M:B_07CN.HN063
    M:B_07CN.HN064
    M:B_07CN.HN079
    M:B_101Hald
    M:B_17HALD
    M:B_215HPD
    M:B_21831/2012/KWT
    M:B_21HALD
    M:B_486HALD
    M:B_513/2012/KW
    M:B_5827/2012/KWT
    M:B_7750/2012/KWT
    M:B_8830/2012/KWT
    M:B_89.6
    M:B_95NOVAX41
    M:B_96NOVAX01
    M:B_96NOVAX14
    M:B_96NOVAX23
    M:B_96NOVAX41
    M:B_9773/2012/KWT
    M:B_97NOVAX01
    M:B_97NOVAX02
    M:B_97NOVAX07
    M:B_97NOVAX13
    M:B_97NOVAX14
    M:B_97NOVAX26
    M:B_98NOVAX16
    M:B_98NOVAX19
    M:B_98NOVAX21
    M:B_98NOVAX28
    M:B_98NOVAX29
    M:B_98NOVAX32
    M:B_99NOVAX04
    M:B_99NOVAX19
    M:B_99NOVAX22
    M:B_99NOVAX37
    M:B_ARV2/SF2
    M:B_BA111
    M:B_BA205
    M:B_BM2010-1
    M:B_BM2010-2
    M:B_HXB2R
    M:B_K1043
    M:B_Lai
    M:B_MAU153
    M:B_MN
    M:B_MP051
    M:B_MP077
    M:B_PT073
    M:B_PT107
    M:B_PT114
    M:B_PT1147
    M:B_PT16
    M:B_PT176
    M:B_PT2
    M:B_PT24
    M:B_PT3013
    M:B_PT317
    M:B_PT38
    M:B_PT423
    M:B_PT471
    M:B_PT508
    M:B_PT6060
    M:B_PT695
    M:B_PT709
    M:B_PT85
    M:B_PT8731
    M:B_PT893
    M:B_PT919
    M:B_PT955
    M:B_PT983
    M:BD_472HALD
    M:BD_K1165
    M:C/U/C_MAU420-PROT
    M:C_03NOVAX25
    M:C_04NOVAX25
    M:C_04SN.MS002
    M:C_04SN.MS007
    M:C_04SN.MS010
    M:C_04SN.MS011
    M:C_04SN.MS015
    M:C_04SN.MS029
    M:C_04SN.MS034
    M:C_04SN.MS036
    M:C_04SN.MS245
    M:C_04SN.MS262
    M:C_04SN.MS281
    M:C_04SN.MS422
    M:C_04SN.MS440
    M:C_04SN.MS448
    M:C_04SN.MS456
    M:C_04SN.MS475
    M:C_04SN.MS477
    M:C_04SN.MS481
    M:C_04SN.MS487
    M:C_04SN.MS492
    M:C_04SN.MS522
    M:C_04SN.MS540
    M:C_04SN.MS700
    M:C_04SN.MS779
    M:C_04SN.MS816
    M:C_04SN.MS821
    M:C_04SN.MS835
    M:C_04SN.MS855
    M:C_04SN.MS883
    M:C_05NOVAX25
    M:C_07CN.HN003
    M:C_07CN.HN004
    M:C_10/93_KC005
    M:C_100/93_KC005
    M:C_101/93_KC005
    M:C_102HALD
    M:C_1032Hald-RT
    M:C_1083
    M:C_109/93_KC005
    M:C_11/93_KC005
    M:C_1119
    M:C_1186
    M:C_1189
    M:C_130/93_KC005
    M:C_131/93_KC005
    M:C_132/93_KC005
    M:C_138/93_KC005
    M:C_14/93_KC005
    M:C_141/93_KC005
    M:C_142/93_KC005
    M:C_143/93_KC005
    M:C_144/93_KC005
    M:C_145/93_KC005
    M:C_148/1997_KC005
    M:C_148/93_KC005
    M:C_14Fann
    M:C_15/93_KC005
    M:C_154/93_KC005
    M:C_154HALD
    M:C_16/93_KC005
    M:C_162/93_KC005
    M:C_164/93_KC005
    M:C_169/93_KC005
    M:C_176/93_KC005
    M:C_179/93_KC005
    M:C_181/93_KC005
    M:C_183/93_KC005
    M:C_19/93_KC005
    M:C_198/93_KC005
    M:C_224/1997_KC005
    M:C_23/93_KC005
    M:C_24/93_KC005
    M:C_25Fann
    M:C_260HALD
    M:C_2658HALD
    M:C_2909HALD
    M:C_2911HALD
    M:C_2931Hald
    M:C_2936HALD
    M:C_3/93_KC005
    M:C_3076HALD
    M:C_30801/2012/KWT
    M:C_32/93_KC005
    M:C_321HALD
    M:C_33/93_KC005
    M:C_34/93_KC005
    M:C_35/93_KC005
    M:C_36/93_KC005
    M:C_38/93_KC005
    M:C_39HALD
    M:C_40/93_KC005
    M:C_41/93_KC005
    M:C_4276/2011/KWT
    M:C_43/93_KC005
    M:C_44/93_KC005
    M:C_463HALD
    M:C_478HALD
    M:C_49/93_KC005
    M:C_51/93_KC005
    M:C_510HALD
    M:C_571/2012/KWT
    M:C_58/93_KC005
    M:C_59/93_KC005
    M:C_62/93_KC005
    M:C_64/93_KC005
    M:C_68/93_KC005
    M:C_6920/2011/KWT
    M:C_72/93_KC005
    M:C_77/1997_KC005
    M:C_79/93_KC005
    M:C_86/93_KC005
    M:C_8618/2012/KWT
    M:C_88/93_KC005
    M:C_9/93_KC005
    M:C_92/93_KC005
    M:C_92BR025
    M:C_94/93_KC005
    M:C_96/93_KC005
    M:C_96NOVAX25
    M:C_97/93_KC005
    M:C_99/93_KC005
    M:C_B1054
    M:C_B1085
    M:C_B1100
    M:C_BA12
    M:C_BA28
    M:C_ETH2220
    M:C_K1165
    M:C_K1335
    M:C_L065
    M:C_L160
    M:C_MAU331
    M:C_MAU420
    M:C_MP102
    M:C_MP116
    M:C_SF130
    M:C_SF431
    M:C_SF621
    M:C_U0126
    M:C_U0211
    M:C_U0221
    M:C_U0333
    M:C_U0427
    M:C_U0434
    M:C_U0435
    M:C_U0505
    M:C_U0636
    M:C_U0702
    M:C_U0705
    M:C_U0727
    M:C_U0735
    M:C_U0808
    M:C_U0812
    M:C_U0820
    M:C_U1622
    M:C_U1803
    M:C_U2132
    M:CFR02/01_04SN.MS291
    M:CRF01_03NOVAX31
    M:CRF01_04NOVAX31
    M:CRF01_97NOVAX31
    M:CRF01_AE
    M:CRF01_AE_10988/2012/KWT
    M:CRF01_AE_5717/2011/KWT
    M:CRF01_BA1
    M:CRF01_BA11
    M:CRF01_BA118
    M:CRF01_BA121
    M:CRF01_BA128
    M:CRF01_BA13
    M:CRF01_BA131
    M:CRF01_BA132
    M:CRF01_BA133
    M:CRF01_BA146
    M:CRF01_BA148
    M:CRF01_BA151
    M:CRF01_BA156
    M:CRF01_BA165
    M:CRF01_BA166
    M:CRF01_BA179
    M:CRF01_BA219
    M:CRF01_BA231
    M:CRF01_BA30
    M:CRF01_BA33
    M:CRF01_BA67
    M:CRF01_BA76
    M:CRF01_BA77
    M:CRF01_BA8
    M:CRF01_BA94
    M:CRF01_K1104
    M:CRF01_MP365
    M:CRF01_MP38
    M:CRF01_MP540
    M:CRF01_R_ADN_ge_R1
    M:CRF01_R_ADN_ge_R10
    M:CRF01_R_ADN_ge_R11
    M:CRF01_R_ADN_ge_R12
    M:CRF01_R_ADN_ge_R13
    M:CRF01_R_ADN_ge_R14
    M:CRF01_R_ADN_ge_R15
    M:CRF01_R_ADN_ge_R16
    M:CRF01_R_ADN_ge_R17
    M:CRF01_R_ADN_ge_R18
    M:CRF01_R_ADN_ge_R2
    M:CRF01_R_ADN_ge_R3
    M:CRF01_R_ADN_ge_R5
    M:CRF01_R_ADN_ge_R6
    M:CRF01_R_ADN_ge_R7
    M:CRF01_R_ADN_ge_R9
    M:CRF01_R_ARN_pl_R11
    M:CRF01_R_ARN_pl_R19
    M:CRF01_R_ARN_pl_R2
    M:CRF01_R_ARN_pl_R21
    M:CRF01_R_ARN_pl_R3
    M:CRF01_R_ARN_pl_R4
    M:CRF01_R_ARN_pl_R5
    M:CRF01_R_ARN_pl_R8
    M:CRF01_R_ARN_pl_R9
    M:CRF02-AG_MAU46
    M:CRF02/06/09_04SN.MS235
    M:CRF02/06_04SN.MS258
    M:CRF02_04SN.MS013
    M:CRF02_04SN.MS178
    M:CRF02_04SN.MS206
    M:CRF02_04SN.MS253
    M:CRF02_04SN.MS268
    M:CRF02_04SN.MS474
    M:CRF02_04SN.MS516
    M:CRF02_04SN.MS618
    M:CRF02_04SN.MS636
    M:CRF02_04SN.MS641
    M:CRF02_04SN.MS668
    M:CRF02_04SN.MS831
    M:CRF02_04SN.MS840
    M:CRF02_04SN.MS842
    M:CRF02_04SN.MS869
    M:CRF02_125HPD
    M:CRF02_163HALD
    M:CRF02_167HPD
    M:CRF02_16HALD
    M:CRF02_171HPD
    M:CRF02_18HALD
    M:CRF02_191HPD
    M:CRF02_214HPD
    M:CRF02_217HALD
    M:CRF02_227HALD
    M:CRF02_232HALD
    M:CRF02_233HALD
    M:CRF02_234HALD
    M:CRF02_265HPD
    M:CRF02_283HPD
    M:CRF02_29HALD
    M:CRF02_303HALD
    M:CRF02_304HALD
    M:CRF02_304HPD
    M:CRF02_307HPD
    M:CRF02_309HALD
    M:CRF02_311HALD
    M:CRF02_323HALD
    M:CRF02_324HALD
    M:CRF02_34HALD
    M:CRF02_383HALD
    M:CRF02_416HALD
    M:CRF02_418HALD
    M:CRF02_420HALD
    M:CRF02_422HALD
    M:CRF02_424HALD
    M:CRF02_426HALD
    M:CRF02_42HPD
    M:CRF02_455HALD
    M:CRF02_456HALD
    M:CRF02_459HALD
    M:CRF02_466HALD
    M:CRF02_469HALD
    M:CRF02_46HALD
    M:CRF02_471HALD
    M:CRF02_474HALD
    M:CRF02_477HALD
    M:CRF02_479HALD
    M:CRF02_481HALD
    M:CRF02_484HALD
    M:CRF02_492HALD
    M:CRF02_496HALD
    M:CRF02_49HALD
    M:CRF02_97HALD
    M:CRF02_A_ADN_ge_A11
    M:CRF02_A_ADN_ge_A12
    M:CRF02_A_ADN_ge_A13
    M:CRF02_A_ADN_ge_A14
    M:CRF02_A_ADN_ge_A15
    M:CRF02_A_ADN_ge_A16
    M:CRF02_A_ADN_ge_A17
    M:CRF02_A_ADN_ge_A18
    M:CRF02_A_ADN_ge_A19
    M:CRF02_A_ADN_ge_A1_2
    M:CRF02_A_ADN_ge_A20
    M:CRF02_A_ADN_ge_A2_2
    M:CRF02_A_ADN_ge_A3_1
    M:CRF02_A_ADN_ge_A3_2
    M:CRF02_A_ADN_ge_A4_1
    M:CRF02_A_ADN_ge_A4_2
    M:CRF02_A_ADN_ge_A5_1
    M:CRF02_A_ADN_ge_A8_1
    M:CRF02_A_ADN_ge_A8_2
    M:CRF02_A_ADN_ge_A9_1
    M:CRF02_A_ADN_ge_A9_2
    M:CRF02_A_ARN_ge_A1
    M:CRF02_A_ARN_ge_A10
    M:CRF02_A_ARN_ge_A11
    M:CRF02_A_ARN_ge_A12
    M:CRF02_A_ARN_ge_A13
    M:CRF02_A_ARN_ge_A16
    M:CRF02_A_ARN_ge_A17
    M:CRF02_A_ARN_ge_A18
    M:CRF02_A_ARN_ge_A19
    M:CRF02_A_ARN_ge_A3
    M:CRF02_A_ARN_ge_A6
    M:CRF02_A_ARN_ge_A7
    M:CRF02_A_ARN_pl_A10_1
    M:CRF02_A_ARN_pl_A10_2
    M:CRF02_A_ARN_pl_A12
    M:CRF02_A_ARN_pl_A16
    M:CRF02_A_ARN_pl_A18_1
    M:CRF02_A_ARN_pl_A18_2
    M:CRF02_A_ARN_pl_A2_1
    M:CRF02_A_ARN_pl_A2_2
    M:CRF02_A_ARN_pl_A3_1
    M:CRF02_A_ARN_pl_A3_2
    M:CRF02_A_ARN_pl_A4_1
    M:CRF02_A_ARN_pl_A6_1
    M:CRF02_A_ARN_pl_A6_2
    M:CRF02_A_ARN_pl_A8
    M:CRF02_AG
    M:CRF02_AG/A3/CRF02_AG_MAU216
    M:CRF02_AG/A3_8253Hald
    M:CRF02_AG/A3_MAU2140
    M:CRF02_AG/A3_MAU251
    M:CRF02_AG/URF_PT836
    M:CRF02_AG_00082500314
    M:CRF02_AG_00129346294
    M:CRF02_AG_00136644558
    M:CRF02_AG_00384364598
    M:CRF02_AG_00384760994
    M:CRF02_AG_00401150434
    M:CRF02_AG_00446856847
    M:CRF02_AG_00486811743
    M:CRF02_AG_01161893839
    M:CRF02_AG_01835871153
    M:CRF02_AG_03099751427
    M:CRF02_AG_03328130847
    M:CRF02_AG_04319750683
    M:CRF02_AG_04566394139
    M:CRF02_AG_04732291679
    M:CRF02_AG_05655055625
    M:CRF02_AG_07821754717
    M:CRF02_AG_08351618287
    M:CRF02_AG_08633910237
    M:CRF02_AG_08855259237
    M:CRF02_AG_08TG.C7003
    M:CRF02_AG_08TG.C7017
    M:CRF02_AG_08TG.C7023
    M:CRF02_AG_08TG.C7024
    M:CRF02_AG_08TG.C7028
    M:CRF02_AG_08TG.C7030
    M:CRF02_AG_08TG.E5013
    M:CRF02_AG_08TG.E5017
    M:CRF02_AG_08TG.E5025
    M:CRF02_AG_08TG.E5027
    M:CRF02_AG_08TG.E5032
    M:CRF02_AG_08TG.E5043
    M:CRF02_AG_08TG.E5044
    M:CRF02_AG_08TG.E5053
    M:CRF02_AG_08TG.E5068
    M:CRF02_AG_08TG.T3003
    M:CRF02_AG_08TG.T3008
    M:CRF02_AG_08TG.T3017
    M:CRF02_AG_08TG.T3029
    M:CRF02_AG_08TG.T3031
    M:CRF02_AG_08TG.T3043
    M:CRF02_AG_08TG.T3046
    M:CRF02_AG_08TG.T3049
    M:CRF02_AG_08TG.T3061
    M:CRF02_AG_08TG.T3069
    M:CRF02_AG_08TG.T3073
    M:CRF02_AG_09360633833
    M:CRF02_AG_09826618155
    M:CRF02_AG_1393HALD
    M:CRF02_AG_1837Hald
    M:CRF02_AG_20029892406
    M:CRF02_AG_20069834746
    M:CRF02_AG_20154094266
    M:CRF02_AG_20728228596
    M:CRF02_AG_2107Hald
    M:CRF02_AG_2340Hald
    M:CRF02_AG_2413Hald-PROT
    M:CRF02_AG_2413Hald-RT
    M:CRF02_AG_2613Hald
    M:CRF02_AG_2698Hald
    M:CRF02_AG_30111376596
    M:CRF02_AG_30220614916
    M:CRF02_AG_4039Hald
    M:CRF02_AG_4921HALD
    M:CRF02_AG_49HALD
    M:CRF02_AG_50067984846
    M:CRF02_AG_50314106694
    M:CRF02_AG_50936804498
    M:CRF02_AG_60159306954
    M:CRF02_AG_80394360234
    M:CRF02_AG_80880292104
    M:CRF02_AG_8253Hald
    M:CRF02_AG_90684486142
    M:CRF02_AG_929Hald
    M:CRF02_AG_MAU1086
    M:CRF02_AG_MAU1120
    M:CRF02_AG_MAU1138
    M:CRF02_AG_MAU1151
    M:CRF02_AG_MAU1247
    M:CRF02_AG_MAU1298
    M:CRF02_AG_MAU1339
    M:CRF02_AG_MAU1399
    M:CRF02_AG_MAU145
    M:CRF02_AG_MAU1567
    M:CRF02_AG_MAU161
    M:CRF02_AG_MAU1674
    M:CRF02_AG_MAU1743
    M:CRF02_AG_MAU1860
    M:CRF02_AG_MAU1911
    M:CRF02_AG_MAU2048
    M:CRF02_AG_MAU2059
    M:CRF02_AG_MAU2073
    M:CRF02_AG_MAU2115
    M:CRF02_AG_MAU2117
    M:CRF02_AG_MAU222
    M:CRF02_AG_MAU339
    M:CRF02_AG_MAU406
    M:CRF02_AG_MAU471
    M:CRF02_AG_MAU513
    M:CRF02_AG_MAU515
    M:CRF02_AG_MAU519
    M:CRF02_AG_MAU521
    M:CRF02_AG_MAU523
    M:CRF02_AG_MAU53
    M:CRF02_AG_MAU577
    M:CRF02_AG_MAU624
    M:CRF02_AG_MAU631
    M:CRF02_AG_MAU635
    M:CRF02_AG_MAU657
    M:CRF02_AG_MAU664
    M:CRF02_AG_MAU670
    M:CRF02_AG_MAU683
    M:CRF02_AG_MAU722
    M:CRF02_AG_MAU765
    M:CRF02_AG_MAU938
    M:CRF02_AG_MAU938-PROT
    M:CRF02_AG_PT4132
    M:CRF02_AG_PT4773
    M:CRF02_AG_PT727
    M:CRF02_AG_PT836
    M:CRF02_AG_PT9000
    M:CRF02_AG_PT988
    M:CRF02_B_ADN_ge_B1
    M:CRF02_B_ADN_ge_B10
    M:CRF02_B_ADN_ge_B11
    M:CRF02_B_ADN_ge_B12
    M:CRF02_B_ADN_ge_B13
    M:CRF02_B_ADN_ge_B15
    M:CRF02_B_ADN_ge_B16
    M:CRF02_B_ADN_ge_B17
    M:CRF02_B_ADN_ge_B18
    M:CRF02_B_ADN_ge_B19
    M:CRF02_B_ADN_ge_B20
    M:CRF02_B_ADN_ge_B21
    M:CRF02_B_ADN_ge_B24
    M:CRF02_B_ADN_ge_B25
    M:CRF02_B_ADN_ge_B3
    M:CRF02_B_ADN_ge_B4
    M:CRF02_B_ADN_ge_B5
    M:CRF02_B_ADN_ge_B6
    M:CRF02_B_ADN_ge_B8
    M:CRF02_B_ADN_ge_B9
    M:CRF02_B_ARN_ge_B1
    M:CRF02_B_ARN_ge_B10
    M:CRF02_B_ARN_ge_B11
    M:CRF02_B_ARN_ge_B12
    M:CRF02_B_ARN_ge_B13
    M:CRF02_B_ARN_ge_B15
    M:CRF02_B_ARN_ge_B16
    M:CRF02_B_ARN_ge_B17
    M:CRF02_B_ARN_ge_B18
    M:CRF02_B_ARN_ge_B19
    M:CRF02_B_ARN_ge_B2
    M:CRF02_B_ARN_ge_B3
    M:CRF02_B_ARN_ge_B4
    M:CRF02_B_ARN_ge_B5
    M:CRF02_B_ARN_ge_B6
    M:CRF02_B_ARN_ge_B7
    M:CRF02_B_ARN_ge_B8
    M:CRF02_B_ARN_ge_B9
    M:CRF02_B_ARN_pl_B10_1
    M:CRF02_B_ARN_pl_B10_2
    M:CRF02_B_ARN_pl_B11
    M:CRF02_B_ARN_pl_B14
    M:CRF02_B_ARN_pl_B15
    M:CRF02_B_ARN_pl_B16
    M:CRF02_B_ARN_pl_B1_2
    M:CRF02_B_ARN_pl_B2_1
    M:CRF02_B_ARN_pl_B2_2
    M:CRF02_B_ARN_pl_B3_1
    M:CRF02_B_ARN_pl_B3_2
    M:CRF02_B_ARN_pl_B4_1
    M:CRF02_B_ARN_pl_B4_2
    M:CRF02_B_ARN_pl_B5_1
    M:CRF02_B_ARN_pl_B6_1
    M:CRF02_B_ARN_pl_B6_2
    M:CRF02_B_ARN_pl_B7_2
    M:CRF02_B_ARN_pl_B9_1
    M:CRF02_B_ARN_pl_B9_2
    M:CRF02_BA190
    M:CRF02_BA213
    M:CRF02_BA215
    M:CRF02_BA244
    M:CRF02_BA26
    M:CRF02_BA35
    M:CRF02_BA53
    M:CRF02_BA58
    M:CRF02_BA59
    M:CRF02_BA60
    M:CRF02_H_ARN_ge_H10
    M:CRF02_H_ARN_ge_H11
    M:CRF02_K1086
    M:CRF02_K1103
    M:CRF02_K1110
    M:CRF02_K1114
    M:CRF02_K1118
    M:CRF02_K1129
    M:CRF02_K1131
    M:CRF02_K1163
    M:CRF02_K1166
    M:CRF02_K1170
    M:CRF02_K1171
    M:CRF02_K1175
    M:CRF02_K1178
    M:CRF02_K1194
    M:CRF02_K1227
    M:CRF02_K1230
    M:CRF02_K1243
    M:CRF02_K1263
    M:CRF02_K1268
    M:CRF02_K1281
    M:CRF02_K1282
    M:CRF02_K1283
    M:CRF02_K1292
    M:CRF02_K1300
    M:CRF02_K1303
    M:CRF02_K1306
    M:CRF02_K1311
    M:CRF02_K1316
    M:CRF02_K1318
    M:CRF02_K1326
    M:CRF02_K1331
    M:CRF02_K1333
    M:CRF02_K1343
    M:CRF02_K1344
    M:CRF02_K1370
    M:CRF02_K1371
    M:CRF02_K1380
    M:CRF02_K1425
    M:CRF02_K2002
    M:CRF02_K2005
    M:CRF02_K2007
    M:CRF02_K2023
    M:CRF02_K2033
    M:CRF02_K2035
    M:CRF02_K2053
    M:CRF02_K2055
    M:CRF02_K_ADN_ge_K10_1
    M:CRF02_K_ADN_ge_K10_2
    M:CRF02_K_ADN_ge_K11
    M:CRF02_K_ADN_ge_K12
    M:CRF02_K_ADN_ge_K13
    M:CRF02_K_ADN_ge_K16
    M:CRF02_K_ADN_ge_K18
    M:CRF02_K_ADN_ge_K19
    M:CRF02_K_ADN_ge_K1_1
    M:CRF02_K_ADN_ge_K1_2
    M:CRF02_K_ADN_ge_K20
    M:CRF02_K_ADN_ge_K2_1
    M:CRF02_K_ADN_ge_K2_2
    M:CRF02_K_ADN_ge_K3_2
    M:CRF02_K_ADN_ge_K4_1
    M:CRF02_K_ADN_ge_K4_2
    M:CRF02_K_ADN_ge_K5_1
    M:CRF02_K_ADN_ge_K5_2
    M:CRF02_K_ADN_ge_K6_1
    M:CRF02_K_ADN_ge_K6_2
    M:CRF02_K_ADN_ge_K7_2
    M:CRF02_K_ADN_ge_K9_1
    M:CRF02_K_ADN_ge_K9_2
    M:CRF02_K_ARN_ge_K1
    M:CRF02_K_ARN_ge_K10
    M:CRF02_K_ARN_ge_K12
    M:CRF02_K_ARN_ge_K13
    M:CRF02_K_ARN_ge_K14
    M:CRF02_K_ARN_ge_K15
    M:CRF02_K_ARN_ge_K16
    M:CRF02_K_ARN_ge_K17
    M:CRF02_K_ARN_ge_K19
    M:CRF02_K_ARN_ge_K2
    M:CRF02_K_ARN_ge_K3
    M:CRF02_K_ARN_ge_K4
    M:CRF02_K_ARN_ge_K7
    M:CRF02_K_ARN_ge_K9
    M:CRF02_K_ARN_pl_K11_1
    M:CRF02_K_ARN_pl_K12
    M:CRF02_K_ARN_pl_K13
    M:CRF02_K_ARN_pl_K17
    M:CRF02_K_ARN_pl_K18
    M:CRF02_K_ARN_pl_K19
    M:CRF02_K_ARN_pl_K1_1
    M:CRF02_K_ARN_pl_K1_2
    M:CRF02_K_ARN_pl_K2
    M:CRF02_K_ARN_pl_K20
    M:CRF02_K_ARN_pl_K5_1
    M:CRF02_K_ARN_pl_K5_2
    M:CRF02_K_ARN_pl_K6
    M:CRF02_K_ARN_pl_K7_1
    M:CRF02_K_ARN_pl_K7_2
    M:CRF02_K_ARN_pl_K8_1
    M:CRF02_K_ARN_pl_K8_2
    M:CRF02_K_ARN_pl_K9
    M:CRF02_L001
    M:CRF02_L003
    M:CRF02_L006
    M:CRF02_L009
    M:CRF02_L013
    M:CRF02_L014
    M:CRF02_L019
    M:CRF02_L035
    M:CRF02_L040
    M:CRF02_L056
    M:CRF02_L057
    M:CRF02_L074
    M:CRF02_L080
    M:CRF02_L084
    M:CRF02_L086
    M:CRF02_L091
    M:CRF02_L117
    M:CRF02_MP1071
    M:CRF02_MP1075
    M:CRF02_MP1277
    M:CRF02_MP1378
    M:CRF02_MP1539
    M:CRF02_MP1540
    M:CRF02_MP540
    M:CRF02_MP578
    M:CRF02_MP642
    M:CRF02_MP802
    M:CRF02_MP803
    M:CRF02_MP804
    M:CRF02_MP812
    M:CRF02_MP815
    M:CRF02_ORAV227
    M:CRF02_ORAV455
    M:CRF02_ORAV461
    M:CRF02_ORAV465
    M:CRF03_H_ARN_ge_H2
    M:CRF04_H_ARN_ge_H5
    M:CRF04_K2049
    M:CRF05_H_ARN_ge_H3
    M:CRF05_LBTB018
    M:CRF06_1223SC
    M:CRF06_127HPD
    M:CRF06_cpx
    M:CRF06_cpx/CRF02_AG_2401Hald
    M:CRF06_cpx_08TG.E5016
    M:CRF06_cpx_08TG.E5020
    M:CRF06_cpx_08TG.E5059
    M:CRF06_cpx_08TG.E5061
    M:CRF06_cpx_08TG.T3034
    M:CRF06_cpx_08TG.T3042
    M:CRF06_cpx_2401Hald
    M:CRF06_cpx_MAU1213
    M:CRF06_cpx_MAU1554
    M:CRF06_cpx_MAU25
    M:CRF06_cpx_MAU284
    M:CRF06_cpx_MAU60
    M:CRF06_cpx_MAU60-PROT
    M:CRF06_cpx_MAU922
    M:CRF06_H_ARN_ge_H16
    M:CRF06_K1211
    M:CRF06_L021
    M:CRF06_L058
    M:CRF06_L136
    M:CRF06_MP1075
    M:CRF06_ORAV207
    M:CRF06_ORAV208
    M:CRF06_ORAV215
    M:CRF06_ORAV216
    M:CRF06_ORAV217
    M:CRF06_ORAV224
    M:CRF06_ORAV227
    M:CRF06_ORAV405
    M:CRF06_ORAV436
    M:CRF06_ORAV452
    M:CRF06_ORAV453
    M:CRF06_ORAV455
    M:CRF06_ORAV456
    M:CRF06_ORAV459
    M:CRF06_ORAV460
    M:CRF06_ORAV462
    M:CRF06_U2929
    M:CRF07_BC
    M:CRF07_H_ARN_ge_H8
    M:CRF08_BC
    M:CRF08_H_ARN_ge_H9
    M:CRF09_04SN.MS445
    M:CRF09_04SN.MS463
    M:CRF09_04SN.MS823
    M:CRF09_16HALD
    M:CRF09_cpx
    M:CRF09_H_ARN_ge_H17
    M:CRF09_ORAV224
    M:CRF10_H_ARN_ge_H15
    M:CRF10_U1916
    M:CRF10_U1929
    M:CRF11_413HALD
    M:CRF11_BA109
    M:CRF11_BA116
    M:CRF11_BA123
    M:CRF11_BA124
    M:CRF11_BA126
    M:CRF11_BA137
    M:CRF11_BA14
    M:CRF11_BA149
    M:CRF11_BA150
    M:CRF11_BA159
    M:CRF11_BA173
    M:CRF11_BA175
    M:CRF11_BA177
    M:CRF11_BA184
    M:CRF11_BA185
    M:CRF11_BA189
    M:CRF11_BA192
    M:CRF11_BA195
    M:CRF11_BA197
    M:CRF11_BA199
    M:CRF11_BA20
    M:CRF11_BA220
    M:CRF11_BA224
    M:CRF11_BA225
    M:CRF11_BA226
    M:CRF11_BA227
    M:CRF11_BA230
    M:CRF11_BA237
    M:CRF11_BA238
    M:CRF11_BA242
    M:CRF11_BA25
    M:CRF11_BA29
    M:CRF11_BA34
    M:CRF11_BA36
    M:CRF11_BA37
    M:CRF11_BA48
    M:CRF11_BA50
    M:CRF11_BA51
    M:CRF11_BA68
    M:CRF11_BA7
    M:CRF11_BA72
    M:CRF11_BA78
    M:CRF11_BA80
    M:CRF11_BA87
    M:CRF11_BA9
    M:CRF11_C_ARN_ge_C1
    M:CRF11_C_ARN_ge_C11
    M:CRF11_C_ARN_ge_C12
    M:CRF11_C_ARN_ge_C13
    M:CRF11_C_ARN_ge_C17
    M:CRF11_C_ARN_ge_C2
    M:CRF11_C_ARN_ge_C4
    M:CRF11_C_ARN_ge_C5
    M:CRF11_C_ARN_ge_C6
    M:CRF11_C_ARN_ge_C7
    M:CRF11_C_ARN_ge_C8
    M:CRF11_C_ARN_ge_C9
    M:CRF11_C_ARN_pl_C1
    M:CRF11_C_ARN_pl_C11
    M:CRF11_C_ARN_pl_C12
    M:CRF11_C_ARN_pl_C13
    M:CRF11_C_ARN_pl_C14
    M:CRF11_C_ARN_pl_C16
    M:CRF11_C_ARN_pl_C17
    M:CRF11_C_ARN_pl_C5
    M:CRF11_C_ARN_pl_C6
    M:CRF11_C_ARN_pl_C7
    M:CRF11_C_ARN_pl_C8
    M:CRF11_cpx_MAU963
    M:CRF11_E_ADN_ge_E10
    M:CRF11_E_ADN_ge_E12
    M:CRF11_E_ADN_ge_E14
    M:CRF11_E_ADN_ge_E18
    M:CRF11_E_ADN_ge_E1_1
    M:CRF11_E_ADN_ge_E1_2
    M:CRF11_E_ADN_ge_E1_3
    M:CRF11_E_ADN_ge_E2_1
    M:CRF11_E_ADN_ge_E2_2
    M:CRF11_E_ADN_ge_E2_3
    M:CRF11_E_ADN_ge_E3_1
    M:CRF11_E_ADN_ge_E3_2
    M:CRF11_E_ADN_ge_E4_1
    M:CRF11_E_ADN_ge_E4_3
    M:CRF11_E_ADN_ge_E5_1
    M:CRF11_E_ADN_ge_E5_2
    M:CRF11_E_ADN_ge_E6_1
    M:CRF11_E_ADN_ge_E6_2
    M:CRF11_E_ADN_ge_E6_3
    M:CRF11_E_ADN_ge_E7_1
    M:CRF11_E_ADN_ge_E7_2
    M:CRF11_E_ADN_ge_E8_1
    M:CRF11_E_ADN_ge_E8_2
    M:CRF11_E_ADN_ge_E9_1
    M:CRF11_E_ADN_ge_E9_2
    M:CRF11_E_ARN_ge_E1
    M:CRF11_E_ARN_ge_E10
    M:CRF11_E_ARN_ge_E13
    M:CRF11_E_ARN_ge_E15
    M:CRF11_E_ARN_ge_E16
    M:CRF11_E_ARN_ge_E2
    M:CRF11_E_ARN_ge_E20
    M:CRF11_E_ARN_ge_E3
    M:CRF11_E_ARN_ge_E7
    M:CRF11_E_ARN_ge_E8
    M:CRF11_E_ARN_pl_E1
    M:CRF11_E_ARN_pl_E10_1
    M:CRF11_E_ARN_pl_E10_2
    M:CRF11_E_ARN_pl_E11
    M:CRF11_E_ARN_pl_E12
    M:CRF11_E_ARN_pl_E14
    M:CRF11_E_ARN_pl_E15
    M:CRF11_E_ARN_pl_E16
    M:CRF11_E_ARN_pl_E17
    M:CRF11_E_ARN_pl_E19
    M:CRF11_E_ARN_pl_E2_2
    M:CRF11_E_ARN_pl_E3_1
    M:CRF11_E_ARN_pl_E3_2
    M:CRF11_E_ARN_pl_E4_1
    M:CRF11_E_ARN_pl_E4_2
    M:CRF11_E_ARN_pl_E5_1
    M:CRF11_E_ARN_pl_E6_1
    M:CRF11_E_ARN_pl_E6_2
    M:CRF11_E_ARN_pl_E7_1
    M:CRF11_E_ARN_pl_E7_2
    M:CRF11_E_ARN_pl_E8_1
    M:CRF11_E_ARN_pl_E8_2
    M:CRF11_E_ARN_pl_E8_3
    M:CRF11_E_ARN_pl_E9_1
    M:CRF11_E_ARN_pl_E9_2
    M:CRF11_F_ARN_ge_F1
    M:CRF11_F_ARN_ge_F13
    M:CRF11_F_ARN_ge_F14
    M:CRF11_F_ARN_ge_F16
    M:CRF11_F_ARN_ge_F19
    M:CRF11_F_ARN_ge_F2
    M:CRF11_F_ARN_ge_F4
    M:CRF11_F_ARN_pl_F1
    M:CRF11_F_ARN_pl_F12
    M:CRF11_F_ARN_pl_F13
    M:CRF11_F_ARN_pl_F14
    M:CRF11_F_ARN_pl_F15
    M:CRF11_F_ARN_pl_F16
    M:CRF11_F_ARN_pl_F3_1
    M:CRF11_F_ARN_pl_F4_1
    M:CRF11_F_ARN_pl_F4_2
    M:CRF11_F_ARN_pl_F5
    M:CRF11_F_ARN_pl_F6
    M:CRF11_F_ARN_pl_F7_1
    M:CRF11_F_ARN_pl_F9
    M:CRF11_G_ADN_ge_G1
    M:CRF11_G_ADN_ge_G10
    M:CRF11_G_ADN_ge_G11
    M:CRF11_G_ADN_ge_G12
    M:CRF11_G_ADN_ge_G13
    M:CRF11_G_ADN_ge_G14
    M:CRF11_G_ADN_ge_G15
    M:CRF11_G_ADN_ge_G16
    M:CRF11_G_ADN_ge_G17
    M:CRF11_G_ADN_ge_G18
    M:CRF11_G_ADN_ge_G19
    M:CRF11_G_ADN_ge_G2
    M:CRF11_G_ADN_ge_G20
    M:CRF11_G_ADN_ge_G3
    M:CRF11_G_ADN_ge_G4
    M:CRF11_G_ADN_ge_G5
    M:CRF11_G_ADN_ge_G7
    M:CRF11_G_ADN_ge_G8
    M:CRF11_G_ADN_ge_G9
    M:CRF11_G_ARN_ge_G10
    M:CRF11_G_ARN_ge_G11
    M:CRF11_G_ARN_ge_G14
    M:CRF11_G_ARN_ge_G15
    M:CRF11_G_ARN_ge_G16
    M:CRF11_G_ARN_ge_G17
    M:CRF11_G_ARN_ge_G18
    M:CRF11_G_ARN_ge_G3
    M:CRF11_G_ARN_ge_G4
    M:CRF11_G_ARN_ge_G5
    M:CRF11_G_ARN_ge_G6
    M:CRF11_G_ARN_ge_G7
    M:CRF11_G_ARN_ge_G9
    M:CRF11_G_ARN_pl_G1
    M:CRF11_G_ARN_pl_G10
    M:CRF11_G_ARN_pl_G11
    M:CRF11_G_ARN_pl_G12
    M:CRF11_G_ARN_pl_G13
    M:CRF11_G_ARN_pl_G14
    M:CRF11_G_ARN_pl_G15
    M:CRF11_G_ARN_pl_G16
    M:CRF11_G_ARN_pl_G18
    M:CRF11_G_ARN_pl_G2
    M:CRF11_G_ARN_pl_G3
    M:CRF11_G_ARN_pl_G4
    M:CRF11_G_ARN_pl_G7
    M:CRF11_G_ARN_pl_G8
    M:CRF11_G_ARN_pl_G9
    M:CRF11_H_ARN_ge_H4
    M:CRF11_K1140
    M:CRF11_K1186
    M:CRF11_K1213
    M:CRF11_K1244
    M:CRF11_K1288
    M:CRF11_K1356
    M:CRF11_KS049
    M:CRF11_L018
    M:CRF11_L_ADN_ge_L10_1
    M:CRF11_L_ADN_ge_L10_2
    M:CRF11_L_ADN_ge_L1_1
    M:CRF11_L_ADN_ge_L1_3
    M:CRF11_L_ADN_ge_L2_2
    M:CRF11_L_ADN_ge_L3_1
    M:CRF11_L_ADN_ge_L3_2
    M:CRF11_L_ADN_ge_L3_3
    M:CRF11_L_ADN_ge_L4_1
    M:CRF11_L_ADN_ge_L4_2
    M:CRF11_L_ADN_ge_L5_2
    M:CRF11_L_ADN_ge_L5_3
    M:CRF11_L_ADN_ge_L6_1
    M:CRF11_L_ADN_ge_L6_2
    M:CRF11_L_ADN_ge_L6_3
    M:CRF11_L_ADN_ge_L7_1
    M:CRF11_L_ADN_ge_L7_2
    M:CRF11_L_ADN_ge_L7_3
    M:CRF11_L_ADN_ge_L8_2
    M:CRF11_L_ADN_ge_L8_3
    M:CRF11_L_ADN_ge_L9_1
    M:CRF11_L_ADN_ge_L9_2
    M:CRF11_L_ADN_ge_L9_3
    M:CRF11_L_ARN_ge_L1
    M:CRF11_L_ARN_ge_L11
    M:CRF11_L_ARN_ge_L12
    M:CRF11_L_ARN_ge_L13
    M:CRF11_L_ARN_ge_L14
    M:CRF11_L_ARN_ge_L15
    M:CRF11_L_ARN_ge_L16
    M:CRF11_L_ARN_ge_L17
    M:CRF11_L_ARN_ge_L18
    M:CRF11_L_ARN_ge_L19
    M:CRF11_L_ARN_ge_L2
    M:CRF11_L_ARN_ge_L3
    M:CRF11_L_ARN_ge_L4
    M:CRF11_L_ARN_ge_L5
    M:CRF11_L_ARN_ge_L6
    M:CRF11_L_ARN_ge_L7
    M:CRF11_L_ARN_ge_L8
    M:CRF11_L_ARN_pl_L10_2
    M:CRF11_L_ARN_pl_L11
    M:CRF11_L_ARN_pl_L17
    M:CRF11_L_ARN_pl_L19
    M:CRF11_L_ARN_pl_L1_1
    M:CRF11_L_ARN_pl_L1_2
    M:CRF11_L_ARN_pl_L20
    M:CRF11_L_ARN_pl_L2_2
    M:CRF11_L_ARN_pl_L3_1
    M:CRF11_L_ARN_pl_L3_2
    M:CRF11_L_ARN_pl_L4
    M:CRF11_L_ARN_pl_L6
    M:CRF11_L_ARN_pl_L7_1
    M:CRF11_L_ARN_pl_L7_2
    M:CRF11_MP703
    M:CRF11_MP809
    M:CRF11_MP818
    M:CRF12_H_ARN_ge_H14
    M:CRF13_BA182
    M:CRF13_BA40
    M:CRF13_BA64
    M:CRF13_cpx_2829Hald
    M:CRF13_H_ARN_ge_H1
    M:CRF13_K1209
    M:CRF13_K1212
    M:CRF13_MP1034
    M:CRF13_MP816
    M:CRF13_N_ADN_ge_N10_1
    M:CRF13_N_ADN_ge_N10_2
    M:CRF13_N_ADN_ge_N11
    M:CRF13_N_ADN_ge_N12
    M:CRF13_N_ADN_ge_N13_1
    M:CRF13_N_ADN_ge_N13_2
    M:CRF13_N_ADN_ge_N14
    M:CRF13_N_ADN_ge_N15
    M:CRF13_N_ADN_ge_N16
    M:CRF13_N_ADN_ge_N17
    M:CRF13_N_ADN_ge_N18
    M:CRF13_N_ADN_ge_N19
    M:CRF13_N_ADN_ge_N1_2
    M:CRF13_N_ADN_ge_N20
    M:CRF13_N_ADN_ge_N2_1
    M:CRF13_N_ADN_ge_N3_2
    M:CRF13_N_ADN_ge_N4_1
    M:CRF13_N_ADN_ge_N4_2
    M:CRF13_N_ADN_ge_N5_1
    M:CRF13_N_ADN_ge_N5_2
    M:CRF13_N_ADN_ge_N6_1
    M:CRF13_N_ADN_ge_N6_2
    M:CRF13_N_ADN_ge_N7_1
    M:CRF13_N_ADN_ge_N7_2
    M:CRF13_N_ADN_ge_N8_1
    M:CRF13_N_ADN_ge_N8_2
    M:CRF13_N_ADN_ge_N9_1
    M:CRF13_N_ADN_ge_N9_2
    M:CRF13_N_ARN_ge_N1
    M:CRF13_N_ARN_ge_N10
    M:CRF13_N_ARN_ge_N11
    M:CRF13_N_ARN_ge_N12
    M:CRF13_N_ARN_ge_N13
    M:CRF13_N_ARN_ge_N14
    M:CRF13_N_ARN_ge_N15
    M:CRF13_N_ARN_ge_N16
    M:CRF13_N_ARN_ge_N17
    M:CRF13_N_ARN_ge_N18
    M:CRF13_N_ARN_ge_N19
    M:CRF13_N_ARN_ge_N1_2
    M:CRF13_N_ARN_ge_N2
    M:CRF13_N_ARN_ge_N2_2
    M:CRF13_N_ARN_ge_N3
    M:CRF13_N_ARN_ge_N3_1
    M:CRF13_N_ARN_ge_N4
    M:CRF13_N_ARN_ge_N5
    M:CRF13_N_ARN_ge_N6
    M:CRF13_N_ARN_ge_N7
    M:CRF13_N_ARN_ge_N8
    M:CRF13_N_ARN_ge_N9
    M:CRF13_N_ARN_pl_N11
    M:CRF13_N_ARN_pl_N13
    M:CRF13_N_ARN_pl_N14
    M:CRF13_N_ARN_pl_N15
    M:CRF13_N_ARN_pl_N19
    M:CRF13_N_ARN_pl_N1_2
    M:CRF13_N_ARN_pl_N20
    M:CRF13_N_ARN_pl_N2_2
    M:CRF13_N_ARN_pl_N3_2
    M:CRF13_N_ARN_pl_N4_2
    M:CRF13_N_ARN_pl_N6_1
    M:CRF13_N_ARN_pl_N7
    M:CRF13_N_ARN_pl_N8_1
    M:CRF13_N_ARN_pl_N8_2
    M:CRF14_H_ARN_pl_H2
    M:CRF15_BA162
    M:CRF15_BA219
    M:CRF15_BA49
    M:CRF15_H_ARN_pl_H9
    M:CRF16_H_ARN_pl_H4_2
    M:CRF17_H_ARN_pl_H5_1
    M:CRF18_H_ARN_pl_H4_1
    M:CRF18_K1111
    M:CRF18_MP1294
    M:CRF18_MP1335
    M:CRF19_H_ARN_pl_H6_2
    M:CRF20_H_ARN_pl_H8
    M:CRF21_H_ARN_pl_H13
    M:CRF22_01A1
    M:CRF22_316HALD
    M:CRF22_413HALD
    M:CRF22_H_ARN_pl_H6_1
    M:CRF22_K1116
    M:CRF22_K1135
    M:CRF22_K1161
    M:CRF22_K1166
    M:CRF22_K1179
    M:CRF22_K1182
    M:CRF22_K1183
    M:CRF22_K1214
    M:CRF22_K1223
    M:CRF22_K1274
    M:CRF22_K1301
    M:CRF22_K1314
    M:CRF22_K1369
    M:CRF22_K2006
    M:CRF22_K2030
    M:CRF22_K2031
    M:CRF22_K2049
    M:CRF22_K2051
    M:CRF22_MP1036
    M:CRF22_MP1073
    M:CRF22_MP1237
    M:CRF22_MP512
    M:CRF22_MP811
    M:CRF22_U1617
    M:CRF22_U2929
    M:CRF23_H_ARN_pl_H7_2
    M:CRF24_H_ARN_pl_H7_1
    M:CRF24_K1254
    M:CRF24_MP1416
    M:CRF25_10HALD
    M:CRF25_H_ARN_pl_H15
    M:CRF25_K1123
    M:CRF26_H_ARN_pl_H14
    M:CRF37_cpx/U_MAU969
    M:CRF37_K1163
    M:CRF43_K1348
    M:CRF43_K1349
    M:CRF45_316HALD
    M:CRF45_413HALD
    M:CRF45_cpx/U_1523Hald
    M:CRF45_K2023
    M:CRF49_cpx
    M:CRF_43
    M:D/A1_MAU2002
    M:D/B_PT317
    M:D_03NOVAX33
    M:D_305HALD
    M:D_3259Hald
    M:D_472HALD
    M:D_97NOVAX33
    M:D_AN010
    M:D_AN019
    M:D_AN034
    M:D_AN113
    M:D_AN174
    M:D_AN175
    M:D_AN176
    M:D_AN177
    M:D_ARV32
    M:D_BA160
    M:D_BA66
    M:D_BA69
    M:D_BA85
    M:D_BR014
    M:D_BR015
    M:D_BR019
    M:D_CA035
    M:D_CA048
    M:D_CA067
    M:D_CA080
    M:D_CHU07
    M:D_CHU19
    M:D_CHU27
    M:D_CHU39
    M:D_CHU53
    M:D_CP03
    M:D_K1176
    M:D_K1185
    M:D_K1186
    M:D_K1192
    M:D_K1291
    M:D_K1299
    M:D_K1300
    M:D_K1371
    M:D_K1417
    M:D_K2055
    M:D_L016
    M:D_MBFE252
    M:D_MP1017
    M:D_NA113
    M:D_NA125
    M:D_NA126
    M:D_NA127
    M:D_NA128
    M:D_NA129
    M:D_NA130
    M:D_NA131
    M:D_NA132
    M:D_NA133
    M:D_NA134
    M:D_NA135
    M:D_PO011
    M:D_PO012
    M:D_PO022
    M:D_PT317
    M:D_RE056
    M:D_RE083
    M:D_RE142
    M:D_RE207
    M:D_RE236
    M:D_RE237
    M:D_RE238
    M:D_RO001
    M:D_RO002
    M:D_RO003
    M:D_RO004
    M:D_RO005
    M:D_RO006
    M:D_U0106
    M:D_U1916
    M:F1_93BR020
    M:F1_BA216
    M:F1_BA93
    M:F1_BA97
    M:F1_KBS154
    M:F1_MP1035
    M:F1_MP1436
    M:F1_MP411
    M:F1_MP577
    M:F1_PT658
    M:F1_VI850
    M:F2_K1125
    M:F2_K1130
    M:F2_K1135
    M:F2_K1191
    M:F2_K1240
    M:F2_K1308
    M:F2_K1346
    M:F2_K1361
    M:F2_K2001
    M:F2_K2006
    M:F2_K2009
    M:F2_MP1017
    M:F2_MP1031
    M:F2_MP1035
    M:F2_MP1236
    M:F2_MP1367
    M:F2_MP255C
    M:F2_MP257C
    M:F2_MP576
    M:F2_MP577
    M:F_CHU51
    M:G/CRF02_04SN.MS642
    M:G_04SN.MS209
    M:G_04SN.MS210
    M:G_04SN.MS279
    M:G_04SN.MS297
    M:G_04SN.MS486
    M:G_06TG.HT023
    M:G_06TG.HT049
    M:G_06TG.HT056
    M:G_06TG.HT065
    M:G_06TG.HT068
    M:G_06TG.HT072
    M:G_06TG.HT081
    M:G_06TG.HT082
    M:G_06TG.HT086
    M:G_06TG.HT087
    M:G_06TG.HT116
    M:G_06TG.HT139
    M:G_06TG.HT149
    M:G_08TG.E5011
    M:G_92NG083
    M:G_ARV93
    M:G_BA125
    M:G_BA134
    M:G_BA152
    M:G_BA157
    M:G_BA172
    M:G_BA202
    M:G_BA233
    M:G_BA234
    M:G_BA240
    M:G_CHU06
    M:G_CHU11
    M:G_CHU16
    M:G_CHU38
    M:G_CHU46
    M:G_CHU56
    M:G_CP06
    M:G_GB03
    M:G_K1167
    M:G_K1184
    M:G_K1243
    M:G_K1247
    M:G_K1254
    M:G_K1256
    M:G_K1291
    M:G_K1293
    M:G_K1295
    M:G_K1316
    M:G_K2005
    M:G_L136
    M:G_MP1033
    M:G_MP1367
    M:G_MP1369
    M:G_MP1372
    M:G_MP1397
    M:G_MP1416
    M:G_MP1417
    M:G_MP694
    M:G_MP801
    M:G_MP962
    M:G_PT114
    M:G_PT13
    M:G_PT1544
    M:G_PT1631
    M:G_PT176
    M:G_PT227
    M:G_PT251
    M:G_PT271
    M:G_PT3
    M:G_PT4132
    M:G_PT471
    M:G_PT4773
    M:G_PT482
    M:G_PT5236
    M:G_PT6060
    M:G_PT658
    M:G_PT709
    M:G_PT7139
    M:G_PT72
    M:G_PT727
    M:G_PT806
    M:G_PT836
    M:G_PT85
    M:G_PT9000
    M:G_PT91
    M:G_PT9108
    M:G_PT919
    M:G_PT9456
    M:G_PT955
    M:G_PT98
    M:G_PT988
    M:G_PTR251
    M:G_SE6165
    M:G_U1720
    M:H_316HALD
    M:H_90CF056
    M:H_BA16
    M:H_BA201
    M:H_BA208
    M:H_CP07
    M:H_KTB238
    M:H_MAU757
    M:H_MP964
    M:H_U3121
    M:H_VI991
    M:J_CHU29
    M:J_CP14
    M:J_KTB238
    M:J_SE9173
    M:J_SE9280
    M:K_96CM-MP535
    M:K_97ZR-EQTB11
    M:K_U2226
    M:RecCpx_D_ADN_ge_D1
    M:RecCpx_D_ADN_ge_D10
    M:RecCpx_D_ADN_ge_D11
    M:RecCpx_D_ADN_ge_D12
    M:RecCpx_D_ADN_ge_D13
    M:RecCpx_D_ADN_ge_D14
    M:RecCpx_D_ADN_ge_D15
    M:RecCpx_D_ADN_ge_D16
    M:RecCpx_D_ADN_ge_D18
    M:RecCpx_D_ADN_ge_D19
    M:RecCpx_D_ADN_ge_D2
    M:RecCpx_D_ADN_ge_D3
    M:RecCpx_D_ADN_ge_D4
    M:RecCpx_D_ADN_ge_D5
    M:RecCpx_D_ADN_ge_D6
    M:RecCpx_D_ADN_ge_D7
    M:RecCpx_D_ADN_ge_D8
    M:RecCpx_D_ADN_ge_D9
    M:RecCpx_D_ARN_ge_D1
    M:RecCpx_D_ARN_ge_D10
    M:RecCpx_D_ARN_ge_D11
    M:RecCpx_D_ARN_ge_D12
    M:RecCpx_D_ARN_ge_D13
    M:RecCpx_D_ARN_ge_D14
    M:RecCpx_D_ARN_ge_D15
    M:RecCpx_D_ARN_ge_D16
    M:RecCpx_D_ARN_ge_D17
    M:RecCpx_D_ARN_ge_D18
    M:RecCpx_D_ARN_ge_D19
    M:RecCpx_D_ARN_ge_D2
    M:RecCpx_D_ARN_ge_D3
    M:RecCpx_D_ARN_ge_D4
    M:RecCpx_D_ARN_ge_D5
    M:RecCpx_D_ARN_ge_D6
    M:RecCpx_D_ARN_ge_D7
    M:RecCpx_D_ARN_ge_D8
    M:RecCpx_D_ARN_ge_D9
    M:RecCpx_D_ARN_pl_D1
    M:RecCpx_D_ARN_pl_D13
    M:RecCpx_D_ARN_pl_D14
    M:RecCpx_D_ARN_pl_D15
    M:RecCpx_D_ARN_pl_D16
    M:RecCpx_D_ARN_pl_D17
    M:RecCpx_D_ARN_pl_D18
    M:RecCpx_D_ARN_pl_D19
    M:RecCpx_D_ARN_pl_D2
    M:RecCpx_D_ARN_pl_D4
    M:RecCpx_D_ARN_pl_D5
    M:RecCpx_D_ARN_pl_D6
    M:RecCpx_D_ARN_pl_D7
    M:RecCpx_D_ARN_pl_D9
    M:RecCpx_M_ADN_ge_M10
    M:RecCpx_M_ADN_ge_M12
    M:RecCpx_M_ADN_ge_M13
    M:RecCpx_M_ADN_ge_M14
    M:RecCpx_M_ADN_ge_M15
    M:RecCpx_M_ADN_ge_M17
    M:RecCpx_M_ADN_ge_M1_1
    M:RecCpx_M_ADN_ge_M1_2
    M:RecCpx_M_ADN_ge_M2_1
    M:RecCpx_M_ADN_ge_M2_2
    M:RecCpx_M_ADN_ge_M3_1
    M:RecCpx_M_ADN_ge_M3_2
    M:RecCpx_M_ADN_ge_M4_1
    M:RecCpx_M_ADN_ge_M4_2
    M:RecCpx_M_ADN_ge_M5_1
    M:RecCpx_M_ADN_ge_M6_1
    M:RecCpx_M_ADN_ge_M6_2
    M:RecCpx_M_ADN_ge_M8
    M:RecCpx_M_ADN_ge_M9_1
    M:RecCpx_M_ADN_ge_M9_2
    M:RecCpx_M_ARN_ge_M1
    M:RecCpx_M_ARN_ge_M10
    M:RecCpx_M_ARN_ge_M11
    M:RecCpx_M_ARN_ge_M12
    M:RecCpx_M_ARN_ge_M13
    M:RecCpx_M_ARN_ge_M14
    M:RecCpx_M_ARN_ge_M15
    M:RecCpx_M_ARN_ge_M16
    M:RecCpx_M_ARN_ge_M17
    M:RecCpx_M_ARN_ge_M18
    M:RecCpx_M_ARN_ge_M19
    M:RecCpx_M_ARN_ge_M1_2
    M:RecCpx_M_ARN_ge_M2
    M:RecCpx_M_ARN_ge_M2(2)
    M:RecCpx_M_ARN_ge_M3
    M:RecCpx_M_ARN_ge_M3_2
    M:RecCpx_M_ARN_ge_M4
    M:RecCpx_M_ARN_ge_M5
    M:RecCpx_M_ARN_ge_M6
    M:RecCpx_M_ARN_ge_M7
    M:RecCpx_M_ARN_ge_M8
    M:RecCpx_M_ARN_ge_M9
    M:RecCpx_M_ARN_pl_M10_1
    M:RecCpx_M_ARN_pl_M10_2
    M:RecCpx_M_ARN_pl_M12
    M:RecCpx_M_ARN_pl_M13
    M:RecCpx_M_ARN_pl_M14
    M:RecCpx_M_ARN_pl_M16
    M:RecCpx_M_ARN_pl_M17
    M:RecCpx_M_ARN_pl_M18
    M:RecCpx_M_ARN_pl_M19
    M:RecCpx_M_ARN_pl_M1_1
    M:RecCpx_M_ARN_pl_M2
    M:RecCpx_M_ARN_pl_M20
    M:RecCpx_M_ARN_pl_M3_2
    M:RecCpx_M_ARN_pl_M4
    M:RecCpx_M_ARN_pl_M7_1
    M:RecCpx_M_ARN_pl_M7_2
    M:RecCpx_M_ARN_pl_M8_1
    M:RecCpx_M_ARN_pl_M8_2
    M:RecCpx_O_ARN_ge_O1
    M:RecCpx_O_ARN_ge_O10
    M:RecCpx_O_ARN_ge_O11
    M:RecCpx_O_ARN_ge_O12
    M:RecCpx_O_ARN_ge_O13
    M:RecCpx_O_ARN_ge_O14
    M:RecCpx_O_ARN_ge_O16
    M:RecCpx_O_ARN_ge_O18
    M:RecCpx_O_ARN_ge_O1_1
    M:RecCpx_O_ARN_ge_O2
    M:RecCpx_O_ARN_ge_O20
    M:RecCpx_O_ARN_ge_O2_2
    M:RecCpx_O_ARN_ge_O3
    M:RecCpx_O_ARN_ge_O3_2
    M:RecCpx_O_ARN_ge_O4
    M:RecCpx_O_ARN_ge_O5
    M:RecCpx_O_ARN_ge_O6
    M:RecCpx_O_ARN_ge_O7
    M:RecCpx_O_ARN_ge_O8
    M:RecCpx_O_ARN_ge_O9
    M:RecCpx_O_ARN_pl_O10_1
    M:RecCpx_O_ARN_pl_O12
    M:RecCpx_O_ARN_pl_O13
    M:RecCpx_O_ARN_pl_O14
    M:RecCpx_O_ARN_pl_O15
    M:RecCpx_O_ARN_pl_O1_1
    M:RecCpx_O_ARN_pl_O1_2
    M:RecCpx_O_ARN_pl_O2
    M:RecCpx_O_ARN_pl_O3_1
    M:RecCpx_O_ARN_pl_O3_2
    M:RecCpx_O_ARN_pl_O4_1
    M:RecCpx_O_ARN_pl_O4_2
    M:RecCpx_O_ARN_pl_O5_1
    M:RecCpx_O_ARN_pl_O5_2
    M:RecCpx_O_ARN_pl_O6_1
    M:RecCpx_O_ARN_pl_O6_2
    M:RecCpx_O_ARN_pl_O7_1
    M:RecCpx_O_ARN_pl_O8_1
    M:RecCpx_O_ARN_pl_O8_2
    M:RecCpx_O_ARN_pl_O9_1
    M:RecCpx_O_ARN_pl_O9_2
    M:U/A3_4769PROTHALD
    M:U/CRF01_AE_MAU509
    M:U/CRF45_cpx_1523Hald
    M:U/G_04SN.MS285
    M:U_225HALD
    M:U_29HALD
    M:U_305HALD
    M:U_483HALD
    M:U_K1200
    M:U_K1273
    M:U_K1381
    M:U_MP1069
    M:U_MP816
    M:U_ORAV208
    M:U_U1720
    M:URF(A/CRF02_AG)
    M:URF(A3/CRF02_AG)
    M:URF(C/CRF02_AG)
    M:URF(CRF02/A3)_08TG.T3039
    M:URF(CRF02/A3)_08TG.T3041
    M:URF(CRF02/CRF06/CRF02)_08TG.E5009
    M:URF(CRF02/U/CRF02)_08TG.T3047
    M:URF(CRF02_AG/A3)
    M:URF(CRF02_AG/A3/CRF02_AG)
    M:URF(CRF02_AG/A3/CRF02_AG/A3)
    M:URF(CRF02_AG/C)
    M:URF(CRF02_AG/CRF06_cpx)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG/CRF06_cpx)
    M:URF(CRF02_AG/CRF06_cpx/CRF02_AG/CRF06_cpx/U)
    M:URF(CRF02_AG/CRF09_cpx/CRF02_AG)
    M:URF(CRF02_AG/G)
    M:URF(CRF02_AG/U)
    M:URF(CRF02_AG/U/A3)
    M:URF(CRF06/G)_08TG.E5047
    M:URF(CRF06_cpx/CRF02_AG/A)
    M:URF(CRF06_cpx/CRF02_AG/CRF06_cpx)
    M:URF(CRF37_cpx/A/CRF37_cpx)
    M:URF(CRF45_cpx/C)
    M:URF(D/CRF02_AG/D)
    M:URF(G/A)_08TG.T3067
    M:URF(G/A/CRF02)_08TG.E5024
    M:URF(G/A3)_08TG.T3071
    M:URF(G/CRF02/G/CRF02)_08TG.C7044
    M:URF(G/CRF02/G/CRF02/G)_08TG.C7022
    M:URF(G/CRF06_cpx)
    M:URF(J/U/K/J)_08TG.C7012
    M:URF(U/CRF02_AG)_08TG.T3013
    M:URF(U/CRF06/CRF02/CRF06/U)_08TG.C7004
    M:URF/D_PT317
    M:URF_03092610735
    M:URF_06526755339
    M:URF_07415052079
    M:URF_09504133825
    M:URF_30631572644
    M_02CD.KS069
    M_02CD.LBTB032
    M_02CD.LBTB084
    M_02CD.MBTB047
    M_97CD.KFE267
    M_97CD.KMST91
    M_97CD.KTB119
    M_97CD.MBFE250
    M_Sw_oHS_04/11_10_3(2)
    M_Sw_oHS_08/11_6_2(1)
    MA_11
    MA_26
    macerans_A
    macleodii_A
    macrogoltabida_A
    macrogoltabida_B
    macrosporangiidus_B
    macrosporus_A
    macuxiense_A
    magna_F
    magna_H
    magna_I
    magna_J
    magnetica_A
    magneticus_A
    magneticus_B
    magnusonii_B
    magnusonii_C
    mali_A
    mallensis_A
    malorum_A
    maltophilia_A
    maltophilia_AA
    maltophilia_AB
    maltophilia_AC
    maltophilia_AD
    maltophilia_AE
    maltophilia_AG
    maltophilia_AH
    maltophilia_AJ
    maltophilia_AL
    maltophilia_AM
    maltophilia_AN
    maltophilia_AP
    maltophilia_AQ
    maltophilia_AR
    maltophilia_AS
    maltophilia_AT
    maltophilia_AU
    maltophilia_AV
    maltophilia_AW
    maltophilia_AX
    maltophilia_AY
    maltophilia_G
    maltophilia_J
    maltophilia_K
    maltophilia_L
    maltophilia_N
    maltophilia_O
    maltophilia_P
    maltophilia_Q
    maltophilia_R
    maltophilia_S
    maltophilia_Z
    manausense_A
    manausense_B
    manausense_C
    manausense_D
    mandelii_B
    mandelii_C
    mandelii_E
    mandelii_F
    mangenotii_A
    mangrovi_A
    mantenii_A
    MAR_274_D06_2010-01-20
    marcescens_A
    marcescens_F
    marcescens_J
    marcescens_K
    marcescens_L
    marginalis_D
    marina_A
    marina_B
    marincola_A
    marinum_A
    marinus_A
    marinus_AA
    marinus_AB
    marinus_AC
    marinus_AD
    marinus_AE
    marinus_AF
    marinus_AG
    marinus_AH
    marinus_B
    marinus_C
    marinus_D
    marinus_E
    marinus_F
    marinus_G
    marinus_H
    marinus_I
    marinus_J
    marinus_K
    marinus_L
    marinus_M
    marinus_N
    marinus_O
    marinus_P
    marinus_Q
    marinus_R
    marinus_T
    marinus_U
    marinus_V
    marinus_W
    marinus_X
    marinus_Y
    marinus_Z
    maripaludis_A
    maripaludis_D
    maris_A
    marisflavi_A
    marisflavi_B
    marisnigri_B
    maritima_A
    maritimus_A
    maritimus_B
    maritimus_C
    maritypicum_A
    marmoris_A
    massiliana_A
    massiliensis_A
    massiliensis_B
    massiliensis_C
    massiliensis_D
    mayombei_A
    MC1_11
    MC1_16
    MC1_2
    MC1_26
    MC2_24_MR-2014
    MC3_11
    mccartyi_A
    mccartyi_B
    mediolanus_A
    mediterranea_A
    mediterranea_B
    mediterranea_C
    mediterraneensis_A
    mediterraneus_A
    megaterium_B
    megaterium_C
    megaterium_D
    melaninogenica_A
    melaninogenica_B
    melaninogenica_C
    melaninogenica_D
    meliloti_A
    mellis_A
    melonis_A
    meluiroseus_A
    meluiroseus_B
    meluiroseus_C
    meluiroseus_D
    meluiroseus_E
    meluiroseus_F
    meluiroseus_G
    meluiroseus_H
    mendocina_C
    mendocina_D
    meningitidis_C
    MEP_002
    MEP_003
    MEP_007
    MEP_009
    MEP_010
    MEP_014
    merdigallinarum_A
    mesenteroides_B
    mesonae_A
    mesophila_A
    mesophilicum_A
    metallireducens_B
    methanica_A
    methanica_B
    methanolica_A
    methanolicus_A
    methanolicus_B
    methylutens_A
    metoecus_B
    Mex_CoV-1
    Mex_CoV-10
    Mex_CoV-11a
    Mex_CoV-11b
    Mex_CoV-2
    Mex_CoV-3
    Mex_CoV-4
    Mex_CoV-5a
    Mex_CoV-5b
    Mex_CoV-6
    Mex_CoV-7
    Mex_CoV-8
    Mex_CoV-9
    mexicana_A
    mexicanum_A
    MF1_21
    MF1_22_MR-2014
    MF_isolate_1
    MF_isolate_10
    MF_isolate_11
    MF_isolate_13
    MF_isolate_14
    MF_isolate_15
    MF_isolate_16
    MF_isolate_17
    MF_isolate_18
    MF_isolate_22
    MF_isolate_26
    MF_isolate_28
    MF_isolate_29
    MF_isolate_31
    MF_isolate_33
    MF_isolate_35
    MF_isolate_4
    MF_isolate_41
    MF_isolate_42
    MF_isolate_43
    MF_isolate_8
    michiganensis_A
    michiganensis_C
    michiganensis_K
    michiganensis_O
    microaerophila_A
    migulae_B
    milii_A
    millerae_A
    millerae_B
    mink/NLD/2020/NT_4
    minutissimum_A
    minutissimum_B
    minutum_A
    minutum_B
    minutum_C
    mirabilis_B
    misionensis_A
    mitis_A
    mitis_AC
    mitis_AD
    mitis_AF
    mitis_AH
    mitis_AI
    mitis_AK
    mitis_AP
    mitis_AQ
    mitis_AR
    mitis_AT
    mitis_AV
    mitis_AW
    mitis_AX
    mitis_AY
    mitis_AZ
    mitis_BA
    mitis_BB
    mitis_BC
    mitis_BD
    mitis_BE
    mitis_BF
    mitis_BG
    mitis_BH
    mitis_BI
    mitis_BJ
    mitis_BK
    mitis_BL
    mitis_BM
    mitis_BN
    mitis_BO
    mitis_BP
    mitis_BQ
    mitis_BR
    mitis_BS
    mitis_BT
    mitis_BU
    mitis_BV
    mitis_BW
    mitis_BX
    mitis_BY
    mitis_BZ
    mitis_C
    mitis_CA
    mitis_CB
    mitis_CC
    mitis_CD
    mitis_CE
    mitis_CF
    mitis_CG
    mitis_CH
    mitis_CI
    mitis_CJ
    mitis_CK
    mitis_CL
    mitis_CM
    mitis_CN
    mitis_CO
    mitis_CP
    mitis_CQ
    mitis_CR
    mitis_CS
    mitis_CT
    mitis_CU
    mitis_CV
    mitis_CW
    mitis_CX
    mitis_CY
    mitis_CZ
    mitis_D
    mitis_DA
    mitis_DB
    mitis_DC
    mitis_DD
    mitis_DE
    mitis_DF
    mitis_DG
    mitis_E
    mitis_F
    mitis_H
    mitis_I
    mitis_J
    mitis_K
    mitis_L
    mitis_M
    mitis_N
    mitis_O
    mitis_P
    mitis_Q
    mitis_W
    mitis_Y
    mixtus_A
    mobilis_A
    mobilis_B
    monocytogenes_B
    monocytogenes_C
    monocytogenes_D
    montana_A
    montanisoli_A
    monteilii_A
    moraviensis_A
    morganii_A
    morganii_B
    morganii_C
    morganii_D
    morganii_E
    mori_A
    moriokaense_A
    moriokaense_B
    mortivallis_A
    mosselii_B
    mosselii_C
    MPB_DPBA28
    MPB_DPFF10B
    MPB_DPFF1a
    MPB_DPFF2
    MPB_DPFF3a
    MPB_DPFF9B
    MPB_DPMF2
    MPB_JP_a
    MPB_JP_b
    MPB_LP_a
    MPB_LP_b
    MPB_LP_c
    MPB_LP_d
    MPB_LP_e
    MPB_LPxJP_a
    MPB_LPxJP_b
    MPB_LPxJP_c
    MPB_LPxJP_d
    MPB_LPxJP_e
    MPB_LPxJP_f
    MPB_LPxJP_g
    MPB_LPxJP_h
    MPB_OSA52A
    MPB_OSA52B
    MSD3_H10
    MSK17_88
    MSK18_59
    MSK20_81
    MSX_63619
    mucilaginosa_A
    mucilaginosa_B
    mucilaginosa_C
    mucilaginosus_A
    muciniphila_A
    muciniphila_C
    muciniphila_D
    muciniphila_E
    muciniphila_F
    muciniphilus_A
    muciniphilus_B
    muciniphilus_C
    muciniphilus_D
    muciniphilus_E
    mucogenicum_B
    mucor_C
    mucosa_A
    mucosus_A
    muelleri_A
    mulieris_A
    multiformis_A
    multiformis_B
    multiformis_D
    multiformis_E
    multiformis_F
    multipara_A
    multivorans_A
    multocida_A
    mundtii_B
    mundtii_C
    murale_A
    muralis_A
    muris_A
    muris_B
    muscicola_A
    muscorum_A
    mycoides_B
    mycoides_C
    Myotis/13rs384_31/Italy/2012
    myrsinacearum_A
    myxococcoides_A
    N118_L5885_C67.5
    N11_011-8
    N127_L5035_C77.3
    N159_L15121_C59.1
    N55_L8635_C17.5
    N88_L5317_C7.5
    N_OV_001
    N_OV_003
    N_OV_008
    N_OV_011
    N_OV_013
    N_OV_039
    N_OV_064
    N_OV_080
    N_OV_137
    N_YBF106
    N_YBF30
    naejangsanensis_A
    naejangsanensis_B
    namurensis_A
    nanhaiensis_A
    nanhaiensis_B
    nanhaiensis_C
    nanhaiensis_D
    nanshensis_A
    naphthalenivorans_A
    nasicola_B
    nasicola_C
    nasimurium_B
    nasimurium_D
    nasimurium_E
    nasimurium_F
    nasimurium_G
    nasimurium_H
    natronolimnaea_A
    neapolitanus_A
    neapolitanus_B
    necator_A
    necator_B
    necator_C
    necator_D
    necessarius_E
    necessarius_F
    necessarius_G
    necessarius_H
    necessarius_I
    necessarius_J
    necessarius_K
    nectariphila_A
    negativus_A
    negevensis_A
    nelsonii_A
    nelsonii_B
    neoaurum_A
    neocaledoniensis_A
    neohaematopini_A
    neohaematopini_B
    neohaematopini_C
    neohaematopini_D
    neohaematopini_E
    neohaematopini_F
    neohaematopini_G
    neptunia_A
    neteri_A
    neteri_B
    neteri_C
    nevei_A
    NG_camel27
    NG_camel37
    NG_camel40
    NG_chicken
    NG_chicken11
    NG_chicken12
    NG_chicken13
    NG_chicken14
    NG_chicken15
    NG_chicken16
    NG_chicken17
    NG_chicken18
    NG_chicken19
    NG_chicken2
    NG_chicken20
    NG_chicken21
    NG_chicken22
    NG_chicken23
    NG_chicken24
    NG_chicken25
    NG_chicken26
    NG_chicken3
    NG_chicken31
    NG_chicken32
    NG_chicken33
    NG_chicken36
    NG_chicken37
    NG_chicken38
    NG_chicken39
    NG_chicken5
    NG_chicken6
    NG_chicken7
    NG_chicken8
    NG_chicken9
    NG_cow12
    NG_cow2
    NG_cow23
    NG_sheep50
    niabensis_A
    niabensis_B
    niacini_A
    niacini_B
    niacini_C
    niacini_D
    niacini_E
    niacini_F
    niacini_G
    nicotinovorans_A
    nicotinovorans_B
    nimipressuralis_A
    NIO_00081
    nitratireducens_A
    nitritireducens_A
    nitroguajacolicus_A
    nitroguajacolicus_B
    nitroreducens_A
    nitroreducens_B
    nitroreducens_C
    nivea_A
    niveus_A
    nivimaris_A
    nivimaris_B
    nivis_A
    nodosa_A
    norvegicus_A
    norvegicus_B
    nosocomialis_A
    nova_A
    nova_B
    novellum_A
    NQ1K_cl_15_1d
    NQ1K_cl_15_2b
    NQ1K_cl_15_3b
    NQ1K_cl_15_5d
    NQ2B_9
    NSA_12
    NSA_23
    nucleatum_D
    nucleatum_J
    numerosa_A
    numerosa_B
    O_ANT70
    O_MVP5180
    obscurus_A
    obscurus_B
    oceani_A
    oceani_B
    oceanisediminis_A
    oceanisediminis_C
    oceanisediminis_D
    oculi_A
    odensis_A
    odontolytica_A
    odontolytica_B
    odontolytica_C
    odontolytica_D
    odontotermitis_A
    odontotermitis_B
    odontotermitis_C
    odoratimimum_A
    odoratum_A
    odorimutans_A
    odysseyi_A
    okenii_A
    okinawensis_A
    okinawensis_B
    okinawensis_C
    okinawensis_D
    olei_A
    oleivorans_A
    oleivorans_B
    oleivorans_C
    oleivorans_D
    oleovorans_B
    oligotropha_A
    oligotropha_B
    oligotrophus_A
    olivaria_A
    olivochromogenes_A
    olleyae_A
    omphalii_A
    omphalii_B
    oncorhynchi_A
    onubensis_A
    onubensis_B
    onubensis_C
    opacus_C
    opacus_D
    ophiovirus_agro
    ophiovirus_brac
    ophiovirus_den
    ophiovirus_indi
    ophiovirus_macra
    ophiovirus_mori
    ophiovirus_odo
    ophiovirus_pli
    ophiovirus_poa
    ophiovirus_sela
    ophiovirus_tri
    ophiovirus_visco
    oralis_A
    oralis_AA
    oralis_AB
    oralis_AC
    oralis_AD
    oralis_B
    oralis_BA
    oralis_BC
    oralis_BD
    oralis_BE
    oralis_BF
    oralis_BG
    oralis_BH
    oralis_BI
    oralis_BJ
    oralis_BK
    oralis_BL
    oralis_BM
    oralis_BN
    oralis_BO
    oralis_BP
    oralis_BQ
    oralis_BR
    oralis_BS
    oralis_BT
    oralis_BU
    oralis_BV
    oralis_BW
    oralis_BX
    oralis_BY
    oralis_BZ
    oralis_C
    oralis_CA
    oralis_CB
    oralis_CC
    oralis_CD
    oralis_CE
    oralis_CF
    oralis_CG
    oralis_CH
    oralis_CI
    oralis_CJ
    oralis_CK
    oralis_CL
    oralis_CM
    oralis_CN
    oralis_CO
    oralis_D
    oralis_E
    oralis_F
    oralis_G
    oralis_H
    oralis_I
    oralis_J
    oralis_K
    oralis_L
    oralis_M
    oralis_N
    oralis_O
    oralis_Q
    oralis_R
    oralis_S
    oralis_T
    oralis_U
    oralis_V
    oralis_W
    oralis_X
    oralis_Y
    oralis_Z
    oranimense_A
    orientale_A
    orientalis_A
    orientalis_B
    orientalis_D
    oris_A
    oris_B
    oris_C
    oris_D
    oris_E
    oryzae_A
    oryzae_B
    oryzae_C
    oryzihabitans_B
    oryzihabitans_D
    oryzihabitans_E
    oryzisoli_A
    oryzisoli_B
    oryzisoli_C
    ostreistagni_A
    OSW_10
    ottawaense_A
    OU_20
    ovatus_A
    ovipneumoniae_A
    ovipneumoniae_B
    ovipneumoniae_C
    ovipneumoniae_D
    ovipneumoniae_E
    OX_B
    OX_D
    OX_LEAF1
    OX_LEAF2
    OX_LEAF3
    OX_LEAF4
    OX_LONAR
    OX_M
    OX_R
    OX_T1
    OX_T2
    OX_T3
    OX_V1
    OX_V2
    OX_Y
    oxalaticus_A
    oxydans_A
    oxydans_B
    oxydans_C
    oxydans_D
    oxyfera_C
    oxyfera_D
    oxytoca_C
    p5_H12
    p62_G06
    p8_G05
    P_269_FSTHT
    P_270_FSTHT
    P_278_FSTHT
    P_321_FSTCO
    pAAV_pCAG_eGFP_2A_Halo_HAPLN1
    pAAV_pCAG_mGFP_2A_Halo_HAPLN1
    pAAV_pCAG_mGFP_2A_Halo_HAPLN1_del68AA
    pAAV_pCAG_mGFP_2A_Halo_HAPLN1Z307-313A
    pAAV_pSyn_eGFP_2A_Halo_HAPLN1
    pAAV_pSyn_eGFP_HAPLN1
    pAAV_pSyn_mGFP_2A_Halo_HAPLN1
    pAAVsc_gLDLR
    pAAVsc_gPCSK9
    pAAVsc_gSerpina1a
    pAAVsc_gSerpina1b
    pAAVsc_gSerpina1c
    pAAVsc_gSerpina1d
    pAAVsc_gSerpina1e
    pAAVsc_LP1-Cre
    pabuli_B
    pacifica_A
    pacificus_A
    pacificus_B
    pactum_A
    pAH05_PsodB-100-GFPova_RhyB2-10-BFP_mCh
    paludicola_A
    paludicola_B
    paludicola_C
    palustris_A
    palustris_C
    palustris_F
    palustris_G
    palustris_H
    palustris_I
    palustris_J
    palustris_K
    palustris_L
    panacihumi_A
    panaciterrae_A
    panipatense_A
    panis_A
    panniformis_A
    papyrosolvens_A
    paradoxus_A
    paradoxus_C
    paradoxus_E
    paradoxus_F
    paradoxus_G
    paradoxus_H
    paradoxus_I
    paradoxus_J
    paradoxus_K
    paradoxus_L
    paradoxus_M
    paradoxus_N
    paradoxus_O
    paraffinicum_A
    parafulva_A
    parafulva_B
    parahaemolyticus_A
    parahaemolyticus_B
    parainfluenzae_A
    parainfluenzae_AA
    parainfluenzae_AB
    parainfluenzae_AC
    parainfluenzae_J
    parainfluenzae_K
    parainfluenzae_L
    parainfluenzae_M
    parainfluenzae_N
    parainfluenzae_O
    parainfluenzae_P
    parainfluenzae_Q
    parainfluenzae_R
    parainfluenzae_S
    parainfluenzae_T
    parainfluenzae_U
    parainfluenzae_V
    parainfluenzae_W
    parainfluenzae_X
    parainfluenzae_Y
    parainfluenzae_Z
    paralvinellae_A
    paraoxydans_B
    paraoxydans_C
    parapertussis_A
    parapertussis_B
    parapertussis_C
    parapertussis_D
    parapertussis_E
    parapertussis_F
    parapertussis_G
    parapertussis_H
    parapertussis_I
    parapertussis_J
    parapertussis_K
    paraphrohaemolyticus_A
    parasanguinis_A
    parasanguinis_B
    parasanguinis_C
    parasanguinis_D
    parasanguinis_F
    parasanguinis_G
    parasanguinis_H
    parasanguinis_I
    parasanguinis_J
    parasanguinis_K
    parasanguinis_L
    parasanguinis_M
    parasanguinis_N
    parasanguinis_O
    parasanguinis_P
    parasanguinis_Q
    parasanguinis_R
    parasanguinis_S
    parasanguinis_T
    parasanguinis_U
    parvula_A
    parvula_B
    parvula_C
    parvum_A
    parvum_B
    PAS_122122
    pasadenensis_A
    pascens_A
    pascens_B
    pasteuri_A
    pasteurianum_A
    paucimobilis_A
    pauculus_B
    pauculus_C
    pauculus_D
    pauculus_E
    paurometabola_A
    pBADamp_npmC
    pBbB1c-CBGA_PlacUV5
    pBbB1c-CBGA_Ptrc
    pBbB2c-CBGA_J23100
    pBbB2c-CBGA_J23105
    pBbB2c-CBGA_J23116
    pBbB2c-CBGA_J23119
    pBbB2c-CBGA_J23150
    pBbB2c-CBGA_pBAD
    pBbB5c-CBGA_PlacUV5
    pBbB5c-CBGA_Ptrc
    pBSY3Z-DAS2-NcAA9C_nat
    pCDFDuet-1-ectD_Ac
    pCDFDuet-1-ectD_He
    pCDFDuet-1-ectD_Ps
    pcDNA3.1_AU1_Shble10_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble10mut_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble11_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble12_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble13_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble13mut_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble1_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble2_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble3_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble4_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble4mut_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble5_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble6_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble6mut_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble7_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble7mut_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble8_FLAG_STOP_eGFP
    pcDNA3.1_AU1_Shble9_FLAG_STOP_eGFP
    pCH13_HG8_pta_KO
    pCH16_HG8_ldh_KO
    pCH19_HG8_lctD_KO
    pCH22_HG8_rnf_KO
    pCH24_HG8_hfsB_KO
    pCH50_HG8_MG_reductase_KO
    pCH51_HG8_MG_synthase_KO
    pCH53_HG8_GlycerolDH_KO
    pCH_pglB_SV
    pCOMB3xss_dsbA_Li005_LPS
    pComb3XXT_dSBA_Nb_C20
    PE_BacIPD31B
    PE_IP16
    PE_IP5a
    PE_IPC15A
    PE_IPFFA5
    PE_IPFFA6
    PE_IPFFB6
    PE_IPFFB7
    PE_IPFFB9
    PE_IPFFC12
    PE_IPFFC13
    PE_IPFFC14
    PE_IPFFC15-B
    PE_IPFFC18
    PE_IPFFC19
    PE_IPFFD20
    PE_IPFFD21
    PE_IPMFB27
    PE_IPMFB28
    pEAQ-HT_OsD27-myc
    pEAQ-HT_OsD27deltaDIS-myc
    pEBO_BDBV-GP
    pectinoschiza_A
    pectinoschiza_B
    pectinovorum_A
    pectinovorum_B
    pecuniae_A
    pEG01_128_pLux-TruD
    pEG02_192_RSTN-PapoA
    pEG02_194_RSTN-BamA
    pEG02_195_RSTN-EpiA
    pEG02_199_RSTN-HalA1
    pEG02_200_RSTN-HalA2
    pEG02_312_RSTN-papa-tev
    pEG02_575_RSTN-PsnA2-tev
    pEG03_017_T7A1-MBP-TruE
    pEG03_045_PP37_His-MdnA
    pEG03_046_His-BmbC
    pEG03_047_His-StrA
    pEG03_048_His-pqqA
    pEG03_049_His-sboA
    pEG03_051_His-TfxA
    pEG03_052_His-ProcA
    pEG03_053_His-TbtA
    pEG03_055_His-Pgm2
    pEG03_057_RSTN-TruE
    pEG03_058_RSTN-MdnA
    pEG03_059_RSTN-SboA
    pEG03_060_RSTN-PqqA
    pEG03_061_RSTN-StrA
    pEG03_062_RSTN-BmbC
    pEG03_063_RSTN-TfxA
    pEG03_064_RSTN-ProcA
    pEG03_065_RSTN-TbtA
    pEG03_067_RSTN-Pgm2
    pEG03_121_RSTN-mdnA
    pEG03_128_RSTN-proc
    pEG03_132_RSTN-paaP
    pEG03_157_RSTN-MibA
    pEG03_161_RSTN-PlpA1
    pEG03_162_RSTN-PlpA2
    pEG03_165_RSTN-PbtA
    pEG03_172_RSTN-LtnA1
    pEG03_173_RSTN-LtnA2
    pEG03_174_RSTN-CrnA1
    pEG03_175_RSTN-CrnA2
    pEG03_176_RSTN-BsjA2
    pEG03_177_RSTN-BsjA3
    pEG03_178_RSTN-CinA
    pEG03_180_RSTN-LasA
    pEG03_181_RSTN-AlbsA
    pEG03_182_RSTN-McbA
    pEG03_194_RSTN-PsnA2
    pEG03_197_RSTN-AMdnA
    pEG03_212_RSTC-CapA
    pEG03_213_RSTC-LasA
    pEG03_214_RSTC-AlbsA
    pEG03_215_RSTC-AtxA1
    pEG03_248_RSTN-sboA
    pEG03_283_RSTN-PapA
    pEG03_286_RSTN-PcpA
    pEG03_553_RSTC-Cln1A1
    pEG03_554_RSTC-Cln1A2
    pEG03_555_RSTC-Cln2A1
    pEG03_556_RSTC-Cln2A2
    pEG03_557_RSTC-Cln3A1
    pEG03_558_RSTC-Cln3A2
    pEG03_559_RSTC-Cln3A3
    pEG03_560_RSTC-CsegA1
    pEG03_561_RSTC-CsegA2
    pEG03_562_RSTC-CsegA3
    pEG03_563_RSTN-PadeA
    pEG03_564_RSTN-ThcoA
    pEG03_565_RSTN-StspA
    pEG03_567_RSTN-LcnA
    pEG03_568_RSTN-PalA
    pEG03_570_RSTN-RaxX
    pEG03_571_RSTN-ComX
    pEG03_572_RSTN-KgpE
    pEG03_574_RSTN-TgnA*
    pEG03_871_RSTN-SgbA
    pEG03_905_RSTN-truE*
    pEG07_034_pCym-TruD
    pEG07_035_pCym-AlbA
    pEG07_037_pCym-MdnC
    pEG07_043_pCym-ProcM
    pEG07_047_pCym-MibHS
    pEG07_048_pCym-MibD
    pEG07_056_pCym-PlpXY
    pEG07_058_pCym-PbtO
    pEG07_059_pCym-PbtM1
    pEG07_060_pCym-PaaA
    pEG07_066_pCym-CinX
    pEG07_067_pCym-CapBC
    pEG07_068_pCym-LasBCD
    pEG07_069_pCym-LasF
    pEG07_070_pCym-AlbsBC
    pEG07_071_pCym-AlbsT
    pEG07_073_pCym-McbCD
    pEG07_074_pCym-MibO
    pEG07_076_pCym-LtnM1
    pEG07_077_pCym-LtnM2
    pEG07_078_pCym-CrnM
    pEG07_079_pCym-BsjM
    pEG07_127_pCym-PsnB
    pEG07_130_pCym-AMdnC
    pEG07_132_pCym-AtxBC
    pEG07_133_pCym-Cln1BC
    pEG07_134_pCym-Cln2BC
    pEG07_135_pCym-Cln3BC
    pEG07_136_pCym-CsegBC
    pEG07_137_pCym-PadeK
    pEG07_138_pCym-ThcoK
    pEG07_139_pCym-StspM
    pEG07_141_pCym-LcnG
    pEG07_142_pCym-PalS
    pEG07_143_pCym-SgbL
    pEG07_144_pCym-RaxST
    pEG07_145_pCym-ComQ
    pEG07_146_pCym-KgpF
    pEG07_147_pCym-TgnB
    pEG07_149_pCym-PapB
    pEG07_152_pCym-PcpXY
    pEG07_160_pCym-LynD
    pEG07_166_pCym-PapoK
    pEG07_169_pCym-EpiD
    pEG07_171_pCym-BamB
    pEG07_172_pCym-HalM1
    pEG07_173_pCym-HalM2
    pEGY_EC_13655
    pelagia_A
    pelagica_B
    pelagius_A
    penaeicida_A
    penaeicida_B
    pennivorans_A
    pentastirinorum_A
    peptidovorans_A
    PEPV_San92
    peregrinum_A
    peregrinum_B
    perflava_A
    perflava_B
    perfringens_A
    periodonticum_B
    periodonticum_D
    permense_A
    persicina_A
    persicus_A
    pestifer_A
    pET19b_DAD2cpGFP(G166)
    pET19b_DAD2cpGFP(G166)LSSmOrange
    pET19b_HisDAD2
    pET19b_HisDAD2cpGFP(G166)
    pET19b_ShHTL7cpGFP(L166)LSSmOrange
    petrii_C
    petrii_D
    petrii_E
    peucetius_A
    pFR015_pBAD_retron-RT_araC_bsaI
    pFR156_pBAD_CspRecT_MutL-E32K_tracRNA
    Pgap_Dual_Kana
    pGE_curli_3R-A
    pGE_curli_3R-I
    pGE_curli_3R-L
    pGE_curli_3R-X
    pGE_curli_3R-X2
    pGE_curli_3R-X3
    pGE_curli_3R-Xlon
    pGEC_delta-flhDC
    pGEC_TetR_Ptet-flhDC
    pGECfimEcN_gfpTCD
    pGECfimEcN_mKATE2
    pGECfluEcN_gfpTCD
    pGECmatEcN_lux
    PGPV10K_LD1
    PGPV10K_RP1
    PGPV10K_SP1
    PGPV11K_FP1
    PGPV11K_FP2
    PGPV11K_LD2
    PGPV11K_Pi5
    PGPV93K_RP2
    PGPVO_Pi1
    PGPVO_Pi2
    PGPVO_Pi4
    Ph1_Loihi
    Ph2_Loihi
    phaeobacteroides_A
    phaeum_A
    phage_Deep-GF0-KM16-C193
    phage_Deep1-GF2-KM23-C739
    phage_EC_HWW_001
    phage_EF62phi
    phage_MedDCM-OCT-S28-C10
    phage_MedDCM-OCT-S28-C3
    phage_MedDCM-OCT-S30-C28
    phage_MedDCM-OCT-S31-C1
    phage_MedDCM-OCT-S35-C6
    phage_MedDCM-OCT-S37-C6
    phage_MedDCM-OCT-S38-C3
    phage_MedDCM-OCT-S39-C11
    phage_MedDCM-OCT-S42-C7
    phage_MedDCM-OCT-S45-C18
    phage_MedDCM-OCT-S45-C4
    phage_vB_CfrD_Devorator
    phage_vB_EcoD_Punny
    phagocytophilum_A
    phaseoli_A
    pHBA008_BASIS-components_rK_3-prime
    pHBA010_BASIS-components_pH_3-prime
    pHBA031_BASIS-LS-rK-BAC266-PH-3
    pHC_pUC_ori
    phenacola_C
    phenacola_D
    phenanthrenivorans_B
    phenazinium_A
    phenazinium_B
    phenolica_A
    phenolica_B
    phenoliruptrix_A
    philomiragia_B
    phocae_A
    phocaeensis_A
    phocaeensis_B
    phoceense_A
    phoeniceum_A
    phosalacinea_A
    phosalacinea_C
    phosphatis_A
    phosphatis_G
    phosphorivorans_A
    phosphorivorans_B
    phosphorivorans_C
    phosphorivorans_D
    phototrophica_A
    phyllosphaerae_B
    phyllosphaerae_C
    phytofermentans_A
    PI_GH1.1.B5
    PI_GH2.1.C2
    pickettii_A
    pickettii_B
    pickettii_D
    pictum_A
    piezotolerans_A
    pig/P15_2/2004/SVN
    pig/sw-90_1/SWE/2010
    pig/sw-92_2/SWE/2010
    pig/Swebo_1/SWE
    piger_A
    PII_GH2.2.C11
    Pili_CB159_MCB012_18_01
    pilosicoli_A
    pilosicoli_B
    pinensis_A
    pinensis_B
    pinensis_C
    pinensis_D
    pinensis_E
    pinensis_F
    pinensis_G
    Pip1_Cr_FR_2014
    Pip2_Cr_FR_2014
    Pip3_M_FR_2014
    pipientis_A
    pipientis_B
    pipientis_C
    pipientis_D
    piscatorii_A
    piscicida_A
    piscicida_B
    piscicola_A
    piscisalsi_A
    piscolens_A
    pittii_E
    pittii_H
    pittii_I
    pIVTRA_hADAR1
    pJM_CO_001
    pJM_CO_002
    pJM_CO_003
    pJM_CO_004
    pJM_CO_005
    pJM_CO_006
    pJM_CO_007
    pJM_CO_008
    pJM_CO_009
    pJM_CO_010
    pJM_CO_011
    pJM_CO_012
    pJM_CO_013
    pJM_CO_014
    pJM_CO_015
    pJM_CO_016
    pJM_CO_017
    pJM_CO_018
    pJM_CO_019
    pJM_CO_020
    pKW20_CDFtet_pAraRedCas9_TracrRNA
    pKW3_MB1amp_tracr_Universal1
    pKW3_MB1amp_tracr_Universal2
    plakortidis_A
    planktonicus_A
    plantarum_A
    pLC_pBBR1_ori
    plebeius_A
    plebeius_B
    plebeius_C
    pLF118_Gm_pAraRed(rec)_TracrRNA
    pLSU1/t35s(TMV)_mNG-CI(AGO198)_NosT
    pLSU1/t35s(TMV)_mNG-I(AGO198)_NosT
    pLSU1/t35s(TMV)_mNG-I(PIV198)_NosT
    plymuthica_B
    PM58B_RA
    pMETE_M
    pMETE_N
    pMis1_1B
    pMU2105_M0795_tdk_KO
    pMVA_NR2
    pneumoniae_A
    pneumoniae_B
    pneumoniae_C
    pneumoniae_E
    pNL4-3_gag-pol(Delta1443-4553)_EGFP
    poae_A
    poae_C
    polaris_A
    polyisoprenivorans_A
    polymorphus_A
    polymyxa_B
    polymyxa_C
    polymyxa_D
    polymyxa_E
    polysaccharea_A
    polysaccharea_B
    polysaccharolyticus_A
    polyspora_A
    polyspora_B
    polytropus_A
    pontis_A
    populi_A
    populi_B
    porcinum_A
    porcinus_A
    porphyrae_A
    portucalensis_A
    postgatei_A
    postgatei_B
    pPGG_tRNAarI_sgRNA_Cas9
    pratensis_A
    prausnitzii_A
    prausnitzii_D
    prausnitzii_E
    prausnitzii_F
    prausnitzii_I
    prausnitzii_J
    prausnitzii_L
    prausnitzii_M
    PREDICT_AdV-12
    PREDICT_AdV-13
    PREDICT_AdV-14
    PREDICT_AdV-15
    PREDICT_AdV-16
    PREDICT_AdV-17
    PREDICT_AdV-18
    PREDICT_AdV-19
    PREDICT_AdV-2
    PREDICT_AdV-20
    PREDICT_AdV-21
    PREDICT_AdV-22
    PREDICT_AdV-23
    PREDICT_AdV-24
    PREDICT_AdV-25
    PREDICT_AdV-26
    PREDICT_AdV-27
    PREDICT_AdV-28
    PREDICT_AdV-29
    PREDICT_AdV-3
    PREDICT_AdV-30
    PREDICT_AdV-31
    PREDICT_AdV-32
    PREDICT_AdV-33
    PREDICT_AdV-34
    PREDICT_AdV-35
    PREDICT_AdV-36
    PREDICT_AdV-37
    PREDICT_AdV-38
    PREDICT_AdV-39
    PREDICT_AdV-40
    PREDICT_AdV-41
    PREDICT_AdV-42
    PREDICT_AdV-43
    PREDICT_AdV-44
    PREDICT_AdV-45
    PREDICT_AdV-46
    PREDICT_AdV-47
    PREDICT_AdV-48
    PREDICT_AdV-49
    PREDICT_AdV-5
    PREDICT_AdV-50
    PREDICT_AdV-51
    PREDICT_AdV-52
    PREDICT_AdV-53
    PREDICT_AdV-54
    PREDICT_AdV-55
    PREDICT_AdV-56
    PREDICT_AdV-57
    PREDICT_AdV-58
    PREDICT_AdV-59
    PREDICT_AdV-60
    PREDICT_AdV-61
    PREDICT_AdV-62
    PREDICT_AdV-63
    PREDICT_AdV-64
    PREDICT_AdV-65
    PREDICT_AdV-66
    PREDICT_AdV-67
    PREDICT_AdV-68
    PREDICT_AdV-69
    PREDICT_AdV-7
    PREDICT_AdV-70
    PREDICT_AdV-71
    PREDICT_AdV-72
    PREDICT_AdV-73
    PREDICT_AdV-74
    PREDICT_AdV-75
    PREDICT_AdV-77
    PREDICT_AdV-78
    PREDICT_AdV-79
    PREDICT_AdV-8
    PREDICT_AdV-80
    PREDICT_AdV-81
    PREDICT_AdV-87
    PREDICT_AdV-90
    PREDICT_AdV-91
    PREDICT_CoV-100
    PREDICT_CoV-101
    PREDICT_CoV-103
    PREDICT_CoV-105
    PREDICT_CoV-106
    PREDICT_CoV-109
    PREDICT_CoV-110
    PREDICT_CoV-112
    PREDICT_CoV-113
    PREDICT_CoV-114
    PREDICT_CoV-115
    PREDICT_CoV-116
    PREDICT_CoV-117
    PREDICT_CoV-118
    PREDICT_CoV-119
    PREDICT_CoV-16
    PREDICT_CoV-17
    PREDICT_CoV-27
    PREDICT_CoV-35
    PREDICT_CoV-44
    PREDICT_CoV-47
    PREDICT_CoV-49
    PREDICT_CoV-50
    PREDICT_CoV-52
    PREDICT_CoV-54
    PREDICT_CoV-56
    PREDICT_CoV-63
    PREDICT_CoV-64
    PREDICT_CoV-65
    PREDICT_CoV-70
    PREDICT_CoV-72
    PREDICT_CoV-73
    PREDICT_CoV-74
    PREDICT_CoV-75
    PREDICT_CoV-76
    PREDICT_CoV-77
    PREDICT_CoV-78
    PREDICT_CoV-81
    PREDICT_CoV-82
    PREDICT_CoV-85
    PREDICT_CoV-86
    PREDICT_CoV-88
    PREDICT_CoV-89
    PREDICT_CoV-90
    PREDICT_CoV-91
    PREDICT_CoV-92
    PREDICT_CoV-93
    PREDICT_CoV-94
    PREDICT_CoV-96
    PREDICT_CoV-97
    PREDICT_CoV-98
    PREDICT_CoV-99
    PREDICT_Flavi-4
    PREDICT_Flavi-5
    PREDICT_Flavi-6
    PREDICT_MAstV-135
    PREDICT_Orbi-6
    PREDICT_PapV-1
    PREDICT_PbV-1
    PREDICT_PbV-10
    PREDICT_PbV-100
    PREDICT_PbV-101
    PREDICT_PbV-102
    PREDICT_PbV-103
    PREDICT_PbV-104
    PREDICT_PbV-105
    PREDICT_PbV-106
    PREDICT_PbV-107
    PREDICT_PbV-108
    PREDICT_PbV-109
    PREDICT_PbV-11
    PREDICT_PbV-110
    PREDICT_PbV-111
    PREDICT_PbV-112
    PREDICT_PbV-113
    PREDICT_PbV-114
    PREDICT_PbV-115
    PREDICT_PbV-116
    PREDICT_PbV-117
    PREDICT_PbV-118
    PREDICT_PbV-119
    PREDICT_PbV-12
    PREDICT_PbV-120
    PREDICT_PbV-121
    PREDICT_PbV-13
    PREDICT_PbV-14
    PREDICT_PbV-15
    PREDICT_PbV-16
    PREDICT_PbV-17
    PREDICT_PbV-18
    PREDICT_PbV-19
    PREDICT_PbV-2
    PREDICT_PbV-20
    PREDICT_PbV-21
    PREDICT_PbV-22
    PREDICT_PbV-23
    PREDICT_PbV-24
    PREDICT_PbV-25
    PREDICT_PbV-26
    PREDICT_PbV-27
    PREDICT_PbV-28
    PREDICT_PbV-29
    PREDICT_PbV-3
    PREDICT_PbV-30
    PREDICT_PbV-31
    PREDICT_PbV-32
    PREDICT_PbV-33
    PREDICT_PbV-34
    PREDICT_PbV-35
    PREDICT_PbV-36
    PREDICT_PbV-37
    PREDICT_PbV-38
    PREDICT_PbV-39
    PREDICT_PbV-4
    PREDICT_PbV-40
    PREDICT_PbV-41
    PREDICT_PbV-42
    PREDICT_PbV-43
    PREDICT_PbV-44
    PREDICT_PbV-45
    PREDICT_PbV-46
    PREDICT_PbV-47
    PREDICT_PbV-48
    PREDICT_PbV-49
    PREDICT_PbV-5
    PREDICT_PbV-50
    PREDICT_PbV-51
    PREDICT_PbV-52
    PREDICT_PbV-53
    PREDICT_PbV-54
    PREDICT_PbV-55
    PREDICT_PbV-57
    PREDICT_PbV-58
    PREDICT_PbV-59
    PREDICT_PbV-6
    PREDICT_PbV-60
    PREDICT_PbV-61
    PREDICT_PbV-62
    PREDICT_PbV-63
    PREDICT_PbV-64
    PREDICT_PbV-65
    PREDICT_PbV-66
    PREDICT_PbV-67
    PREDICT_PbV-68
    PREDICT_PbV-69
    PREDICT_PbV-7
    PREDICT_PbV-70
    PREDICT_PbV-71
    PREDICT_PbV-72
    PREDICT_PbV-73
    PREDICT_PbV-74
    PREDICT_PbV-75
    PREDICT_PbV-76
    PREDICT_PbV-77
    PREDICT_PbV-78
    PREDICT_PbV-79
    PREDICT_PbV-8
    PREDICT_PbV-80
    PREDICT_PbV-81
    PREDICT_PbV-82
    PREDICT_PbV-83
    PREDICT_PbV-84
    PREDICT_PbV-85
    PREDICT_PbV-86
    PREDICT_PbV-87
    PREDICT_PbV-88
    PREDICT_PbV-89
    PREDICT_PbV-9
    PREDICT_PbV-90
    PREDICT_PbV-91
    PREDICT_PbV-92
    PREDICT_PbV-93
    PREDICT_PbV-94
    PREDICT_PbV-95
    PREDICT_PbV-96
    PREDICT_PbV-97
    PREDICT_PbV-98
    PREDICT_PbV-99
    PREDICT_PicornaV-1
    PREDICT_PicornaV-2
    PREDICT_PicornaV-3
    PREDICT_PicornaV-4
    PREDICT_PMV-10
    PREDICT_PMV-100
    PREDICT_PMV-101
    PREDICT_PMV-102
    PREDICT_PMV-103
    PREDICT_PMV-104
    PREDICT_PMV-105
    PREDICT_PMV-106
    PREDICT_PMV-107
    PREDICT_PMV-108
    PREDICT_PMV-109
    PREDICT_PMV-11
    PREDICT_PMV-110
    PREDICT_PMV-111
    PREDICT_PMV-112
    PREDICT_PMV-113
    PREDICT_PMV-114
    PREDICT_PMV-115
    PREDICT_PMV-116
    PREDICT_PMV-117
    PREDICT_PMV-118
    PREDICT_PMV-119
    PREDICT_PMV-12
    PREDICT_PMV-120
    PREDICT_PMV-121
    PREDICT_PMV-122
    PREDICT_PMV-124
    PREDICT_PMV-125
    PREDICT_PMV-126
    PREDICT_PMV-127
    PREDICT_PMV-128
    PREDICT_PMV-13
    PREDICT_PMV-131
    PREDICT_PMV-132
    PREDICT_PMV-133
    PREDICT_PMV-137
    PREDICT_PMV-138
    PREDICT_PMV-139
    PREDICT_PMV-14
    PREDICT_PMV-140
    PREDICT_PMV-141
    PREDICT_PMV-142
    PREDICT_PMV-143
    PREDICT_PMV-144
    PREDICT_PMV-145
    PREDICT_PMV-146
    PREDICT_PMV-147
    PREDICT_PMV-148
    PREDICT_PMV-149
    PREDICT_PMV-15
    PREDICT_PMV-150
    PREDICT_PMV-151
    PREDICT_PMV-152
    PREDICT_PMV-153
    PREDICT_PMV-154
    PREDICT_PMV-155
    PREDICT_PMV-156
    PREDICT_PMV-159
    PREDICT_PMV-16
    PREDICT_PMV-160
    PREDICT_PMV-161
    PREDICT_PMV-163
    PREDICT_PMV-167
    PREDICT_PMV-168
    PREDICT_PMV-169
    PREDICT_PMV-17
    PREDICT_PMV-170
    PREDICT_PMV-171
    PREDICT_PMV-172
    PREDICT_PMV-173
    PREDICT_PMV-174
    PREDICT_PMV-175
    PREDICT_PMV-176
    PREDICT_PMV-177
    PREDICT_PMV-178
    PREDICT_PMV-179
    PREDICT_PMV-18
    PREDICT_PMV-180
    PREDICT_PMV-181
    PREDICT_PMV-182
    PREDICT_PMV-183
    PREDICT_PMV-184
    PREDICT_PMV-185
    PREDICT_PMV-186
    PREDICT_PMV-187
    PREDICT_PMV-188
    PREDICT_PMV-189
    PREDICT_PMV-19
    PREDICT_PMV-190
    PREDICT_PMV-191
    PREDICT_PMV-192
    PREDICT_PMV-193
    PREDICT_PMV-194
    PREDICT_PMV-2
    PREDICT_PMV-20
    PREDICT_PMV-21
    PREDICT_PMV-22
    PREDICT_PMV-23
    PREDICT_PMV-24
    PREDICT_PMV-25
    PREDICT_PMV-26
    PREDICT_PMV-27
    PREDICT_PMV-28
    PREDICT_PMV-29
    PREDICT_PMV-3
    PREDICT_PMV-30
    PREDICT_PMV-31
    PREDICT_PMV-32
    PREDICT_PMV-33
    PREDICT_PMV-34
    PREDICT_PMV-35
    PREDICT_PMV-36
    PREDICT_PMV-37
    PREDICT_PMV-38
    PREDICT_PMV-39
    PREDICT_PMV-4
    PREDICT_PMV-40
    PREDICT_PMV-41
    PREDICT_PMV-42
    PREDICT_PMV-43
    PREDICT_PMV-44
    PREDICT_PMV-45
    PREDICT_PMV-46
    PREDICT_PMV-47
    PREDICT_PMV-48
    PREDICT_PMV-49
    PREDICT_PMV-5
    PREDICT_PMV-50
    PREDICT_PMV-51
    PREDICT_PMV-52
    PREDICT_PMV-53
    PREDICT_PMV-54
    PREDICT_PMV-55
    PREDICT_PMV-56
    PREDICT_PMV-57
    PREDICT_PMV-58
    PREDICT_PMV-59
    PREDICT_PMV-6
    PREDICT_PMV-60
    PREDICT_PMV-62
    PREDICT_PMV-63
    PREDICT_PMV-64
    PREDICT_PMV-65
    PREDICT_PMV-66
    PREDICT_PMV-67
    PREDICT_PMV-68
    PREDICT_PMV-69
    PREDICT_PMV-70
    PREDICT_PMV-71
    PREDICT_PMV-72
    PREDICT_PMV-73
    PREDICT_PMV-74
    PREDICT_PMV-75
    PREDICT_PMV-76
    PREDICT_PMV-77
    PREDICT_PMV-78
    PREDICT_PMV-79
    PREDICT_PMV-80
    PREDICT_PMV-81
    PREDICT_PMV-82
    PREDICT_PMV-83
    PREDICT_PMV-85
    PREDICT_PMV-86
    PREDICT_PMV-91
    PREDICT_PMV-92
    PREDICT_PMV-93
    PREDICT_PMV-94
    PREDICT_PMV-95
    PREDICT_PMV-96
    PREDICT_PMV-97
    PREDICT_PMV-98
    PREDICT_PMV-99
    PREDICT_PyV-14
    PREDICT_PyV-15
    PREDICT_PyV-22
    PREDICT_PyV-23
    PREDICT_PyV-24
    PREDICT_PyV-25
    PREDICT_PyV-28
    PREDICT_PyV-30
    PREDICT_PyV-31
    PREDICT_PyV-32
    PREDICT_PyV-34
    PREDICT_PyV-5
    PREDICT_PyV-6
    PREDICT_PyV-7
    PREDICT_PyV-9
    PREDICT_RbdV-14
    PREDICT_RbdV-15
    PREDICT_RbdV-21
    PREDICT_RbdV-27
    PREDICT_RbdV-28
    PREDICT_RbdV-30
    PREDICT_RbdV-31
    PREDICT_RbdV-32
    PREDICT_RbdV-33
    PREDICT_RbdV-34
    PREDICT_RbdV-35
    pretoriensis_A
    priestleyi_A
    priestleyi_B
    primitia_A
    producens_A
    profundi_A
    profundicola_A
    profundimaris_B
    profundimaris_C
    profundimaris_D
    profundum_A
    profundus_A
    profusus_A
    prolificus_A
    proomii_A
    proomii_B
    propionica_A
    propionica_B
    propionica_C
    propionicus_A
    propionicus_B
    propionicus_C
    protaetiae_A
    proteamaculans_D
    proteamaculans_E
    protegens_A
    protegens_B
    proteoclasticus_A
    proteoclasticus_B
    proteolyticum_A
    proteolyticus_A
    provencensis_A
    pRS315_AnSuc1:pep1
    pRS315_Ppdc_xyr1V821F:ace1
    pRS315_Pxyn1_ReCel3a:slp1
    pRS315_res1
    pRS425_AtD27
    pRS425_AtD27_MAX3_MAX4
    pRS425_DbD27
    pRS425_OsD27
    pRS425_YPRCtau3-BTS1-CrtI-CrtYB
    pRS425_YPRCtau3-CrtE-CrtI-CrtYB
    pRSFDuet-ectA-ectB(E407D)-ectC_Hv
    pRSFDuet-ectABC_Hv
    prydzensis_A
    pSC101_rK-sacB_selection-construct
    pSC101_sC-rpsL_selection-construct
    pseudoflava_A
    pseudolongum_C
    pseudomultivorans_A
    pseudomycoides_B
    pseudomycoides_C
    pseudonitzschiae_A
    pseudopneumoniae_A
    pseudopneumoniae_C
    pseudopneumoniae_D
    pseudopneumoniae_E
    pseudopneumoniae_G
    pseudopneumoniae_H
    pseudopneumoniae_J
    pseudopneumoniae_L
    pseudopneumoniae_M
    pseudopneumoniae_N
    pseudopneumoniae_P
    pseudopneumoniae_Q
    pseudopneumoniae_R
    pseudotrichonymphae_A
    pseudovenezuelae_A
    pSGK_ChoD
    pSGK_P45_F3213
    pSP43_pKW3spec(rec)_SapI_insert_gRNA
    pStA0.BBa_J23100
    pStA0.BBa_J23102
    pStA0.BBa_J23107
    pStA0.BBa_J23113
    pStA0.BBa_J23116
    pStA0.BBa_J23118
    psychrerythraea_A
    psychrerythraea_B
    psychrerythraea_C
    psychrerythraea_D
    psychrophenolicus_A
    psychrophila_A
    psychrophila_B
    psychrophilum_A
    psychrophilum_B
    psychrophilum_C
    psychrophilum_D
    psychrophilum_E
    psychrophilum_F
    psychrophilum_G
    psychrophilum_H
    psychrophilum_I
    psychrophilum_J
    psychrophilum_K
    psychrophilum_L
    psychrophilum_M
    psychrophilum_N
    psychrophilum_O
    psychrophilum_P
    psychrophilum_Q
    psychrophilum_R
    psychrophilum_S
    psychrophilum_T
    psychrotolerans_A
    psychrotolerans_B
    psychrotolerans_C
    pT7SA11NSP1_3xFL_UnaG
    Ptet_Dual_Spe
    pTLAB21_Bs314EBE_avrGf2
    pTTC_FusionRed_eGFP_3UTR
    pTU2-b_TrxD27-TrxCCD7-MBPCCD8
    ptyseos_A
    pullicaecorum_A
    pullicaecorum_B
    pullicaecorum_C
    pullorum_B
    pullorum_C
    pulmonis_A
    pumilus_G
    pumilus_M
    pumilus_N
    pumilus_O
    pumilus_P
    punctiforme_A
    pusillum_A
    pusillum_B
    pusillum_C
    putida_A
    putida_AA
    putida_AB
    putida_AC
    putida_AD
    putida_AE
    putida_AF
    putida_AG
    putida_B
    putida_E
    putida_F
    putida_H
    putida_J
    putida_K
    putida_L
    putida_N
    putida_P
    putida_Q
    putida_R
    putida_S
    putida_T
    putida_U
    putida_V
    putida_W
    putida_X
    putida_Y
    putida_Z
    putrefaciens_C
    pVMG0266_nanos-Cas9_Ice2
    pVMG0278_ebony_gRNA1
    pVMG0288_LbCpf1_D156R
    pVMG0295_Cquin_white-sgRNA-6_recoded
    pVMG0313_ebony_gRNA4
    pVMG090_White_gRNA2
    pylori_A
    pylori_BU
    pylori_C
    pylori_CG
    pylori_CH
    pylori_CI
    pylori_CJ
    pylori_CK
    pylori_CL
    pylori_CM
    pylori_CN
    pylori_CO
    pylori_CP
    pylori_CQ
    pylori_CR
    pylori_CS
    pyogenes_A
    pyrrocinia_B
    pyrrocinia_C
    pyrrocinia_D
    pyrrocinia_E
    pYTNB01K_1G7
    pYTRW07K_0G5
    pYTRW08K_0C5
    pYTRW09K_0T5
    pYTRW10K_0x5
    pYTRW11K_0S5
    pYTRW13K_3G5
    pYTRW14K_7G5
    pYTRW15K_2G5
    pYTRW16K_1G5
    pYTRW17K_6G5
    pYTRW18K_3T5
    pYTRW20K_0Ti1
    pYTRW21K_1Ti1
    pYTRW22K_7Ti1
    pYTRW26K_1Ti1
    pYTRW27K_7Ti1
    pYTRW28K_0Ti1
    pYTSK00K_0x7
    pYTSK02A_0G7
    pYTSK31K_1G7
    pYTSK54K_7G7
    pYTSK55K_2G7
    pYTSK56K_3G7
    pYTSK58K_6G7
    pYTSK65K_8G7
    Q_sR_OV_011
    Q_sR_OV_019
    Q_sR_OV_020
    Q_sR_OV_022
    Q_sR_OV_023
    Q_sR_OV_025
    Q_sR_OV_036
    Q_sR_OV_041
    Q_sR_OV_042
    qingshengii_A
    qingshengii_B
    queenslandensis_A
    queenslandensis_B
    quercina_D
    quinivorans_A
    quinlivanii_A
    R1_CDLR_pACNR-FL-POWV-DTV
    R1_CpG-H-M1_pACNR-FL-POWV-DTV
    R1_CpG-H-M2_pACNR-FL-POWV-DTV
    R1_UpA-H-M1_pACNR-FL-POWV-DTV
    R1_UpA-H-M2_pACNR-FL-POWV-DTV
    R2_11
    R2_8
    R2_CDLR_pACNR-FL-POWV-DTV
    R2_CpG-H-M1_pACNR-FL-POWV-DTV
    R2_CpG-H-M2_pACNR-FL-POWV-DTV
    R2_UpA-H-M1_pACNR-FL-POWV-DTV
    R2_UpA-H-M2_pACNR-FL-POWV-DTV
    Rabbit_CoV/La06_256/Ory_cun/ESP/2006
    Rabbit_CoV/La09_24/Ory_cun/ESP/2009
    Rabbit_CoV/La09_31/Ory_cun/ESP/2009
    Rabbit_CoV/La09_32/Ory_cun/ESP/2009
    Rabbit_CoV/La09_48/Ory_cun/ESP/2009
    radicis_A
    radingae_A
    radiosa_A
    radiosum_A
    raffinosus_A
    ramosus_A
    ramulus_A
    RBC_A_MPN10_2
    RBC_B_mpn_10_2_GM
    RBG_16_50_20
    RCC_150
    RCoV/Bor473/Rat_rat/MYS/2012
    RCoV/D_RMU10_1055/Myo_gla/GER/2010
    RCoV/D_RMU10_1087/Apo_fla/GER/2010
    RCoV/D_RMU10_1182/Mic_agr/GER/2010
    RCoV/D_RMU10_1222/Mus_dom/GER/2010
    RCoV/D_RMU10_1226/Myo_gla/GER/2010
    RCoV/D_RMU10_1228/Mus_dom/GER/2010
    RCoV/D_RMU10_1919/Myo_gla/GER/2010
    RCoV/D_RMU10_1940/Myo_gla/GER/2010
    RCoV/D_RMU10_1972/Myo_gla/GER/2010
    RCoV/D_RMU10_1974/Myo_gla/GER/2010
    RCoV/D_RMU10_1985/Myo_gla/GER/2010
    RCoV/D_RMU10_1989/Myo_gla/GER/2010
    RCoV/D_RMU10_1993/Myo_gla/GER/2010
    RCoV/D_RMU10_2065/Myo_gla/GER/2010
    RCoV/D_RMU10_2066/Myo_gla/GER/2010
    RCoV/D_RMU10_2070/Myo_gla/GER/2010
    RCoV/D_RMU10_2089/Myo_gla/GER/2010
    RCoV/D_RMU10_590/Mic_arv/GER/2010
    RCoV/D_RMU10_592/Mic_arv/GER/2010
    RCoV/D_RMU10_974/Myo_gla/GER/2010
    RCoV/D_RMU10_986/Myo_gla/GER/2010
    RCoV/KS10_2636/Mic_arv/GER/2010
    RCoV/KS10_2743/Mic_arv/GER/2010
    RCoV/KS10_2810/Mic_arv/GER/2010
    RCoV/KS10_2864/Myo_glar/GER/2010
    RCoV/KS11_0514/Mic_arv/GER/2011
    RCoV/KS11_0997/Apo_agr/GER/2011
    RCoV/LEO/Rat_sik/CHN/2007
    RCoV/Lmg115/Sig_spec/MEX/2011
    RCoV/NL_69/Mic_arv/NED/2007
    RCoV/NSP32/Rha_pum/RSA/2008
    RCoV/NSP63/Rha_pum/RSA/2008
    RCoV/NSP65/Rha_pum/RSA/2008
    RCoV/RMU10_2010/Myo_gla/GER/2010
    RCoV/RMU10_3052/Apo_fla/GER/2010
    RCoV/RMU10_3147/Apo_fla/GER/2010
    RCoV/RMU10_3212/Myo_gla/GER/2010
    RCoV/RMU10_620/Apo_agr/GER/2010
    RCoV/SPB40/Liomys_spec/MEX/2011
    RCoV/TP3_22/Ban_ind/THA/2005
    RCoV/TP3_26/Rat_rat/THA/2005
    resinovorans_A
    resinovorans_C
    resinovorans_D
    resinovorum_A
    restrictus_A
    rettgeri_A
    rettgeri_D
    rettgeri_E
    rettgeri_F
    reuszeri_A
    reuteri_D
    reuteri_E
    reuteri_F
    reuteri_G
    reuteri_H
    rhinotracheale_A
    rhinotracheale_B
    rhizogenes_A
    rhizogenes_C
    rhizogenes_D
    rhizomae_A
    rhizophila_A
    rhizophila_B
    rhizophila_C
    rhizophilum_A
    rhizovicinus_B
    rhodesiae_A
    rhodesiae_B
    Ri1Ps_4703
    Ri1Ps_4749
    Ri1Pw_5477
    Ri1Pw_5610
    Ri1Pw_5615
    Ri2Ps_4855
    Ri2Ps_4998
    Ri2Ps_5096
    Ri2Ps_5130
    Ri2Pt_4001
    Ri2Pt_4081
    Ri2Pw_6101
    Ri3Ps_5313
    Ri3Pw_6170
    Ri3Pw_6328
    rigui_A
    rimosus_A
    rishiriensis_A
    RK10_tank
    RK10a_tank
    RK11_tank
    RK13_tank
    RK14_tank
    RK16_1_tank
    RK17_tank
    RK18_tank
    RK1_tank
    RK20_tank
    RK23_tank
    RK24_tank
    RK25_tank
    RK27_tank
    RK28_tank
    RK2a_tank
    RK31_tank
    RK31a_tank
    RK39_tank
    RK4_tank
    RK5_tank
    RK5a2_tank
    RK5a_tank
    RK65_tank
    RK66_tank
    RK67_tank
    RK69_tank
    RK70_tank
    RK73_tank
    RK77_tank
    RK78_tank
    RK79_tank
    RK7_tank
    RK80_tank
    RK81_tank
    RK84_tank
    RK85_tank
    RK88_tank
    RK89_tank
    RK8_tank
    RK91_tank
    RK92_tank
    RK97_tank
    RK9_tank
    RM12EL_21F
    Rn/GV/HKU_CT2/HKG/2011
    Rn/GV/HKU_KT/HKG/2012
    RO_4
    RO_isolate_1
    RO_isolate_11
    RO_isolate_12
    RO_isolate_14
    RO_isolate_17
    RO_isolate_18
    RO_isolate_19
    RO_isolate_2
    RO_isolate_23
    RO_isolate_26
    RO_isolate_29
    RO_isolate_3
    RO_isolate_32
    RO_isolate_6
    RO_isolate_7
    robiniae_A
    robusta_A
    rosea_A
    roseus_A
    roseus_B
    rossiae_B
    ruber_A
    rubeus_A
    rubi_A
    rubra_A
    rubra_B
    rubra_D
    rubra_E
    rubra_F
    rubra_G
    rudis_A
    ruminantium_A
    ruminantium_B
    ruminantium_C
    ruminantium_D
    ruminantium_E
    ruminantium_G
    ruminantium_I
    ruminantium_J
    ruminantium_K
    ruminantium_L
    ruminantium_M
    ruminantium_N
    ruminantium_O
    ruminicola_A
    ruminicola_B
    ruminicola_C
    ruminicola_D
    ruminis_A
    ruminis_B
    ruminis_C
    ruminis_D
    ruminis_E
    ruminis_F
    rumoiensis_A
    russellii_A
    rwandensis_B
    S-RIM12_Np_14_0310
    S-RIM12_Np_15_0310
    S-RIM12_Np_22_1112
    S-RIM12_RW_04_0310
    S-RIM12_RW_04_0709
    S-RIM12_RW_07_1112
    S-RIM12_RW_14_0101
    S-RIM12_RW_22_0110
    S-RIM12_RW_25_0210
    S-RIM12_RW_28_1109
    S-RIM12_RW_29_1109
    S-RIM12_Sn_07_0910
    S-RIM12_Sn_31_0910
    S-RIM12_W1_12_0610
    S-RIM12_W1_24_0910
    S-RIM12_WH_05_0310
    S-RIM12_WH_07_0310
    S1_MeH
    S2_MeH
    S8A-217_16S
    S_A17_125
    S_A17_144
    S_A17_145
    S_C14_118
    S_C14_131
    S_C14_133
    S_C14_134
    S_C14_135
    S_C14_136
    S_C49_155
    S_C49_24
    S_C49_26
    S_C49_70
    S_C49_74
    S_D75_1
    S_D75_109
    S_D75_157
    S_D75_164
    S_D75_168
    S_D75_169
    S_D75_33
    S_D75_81
    S_D75_83
    S_D75_84
    S_D75_94
    SA08_00821
    SA_07_02434
    SA_0800821
    sabuli_A
    sabulinigri_A
    sacchari_B
    sacchari_C
    saccharolyticum_A
    saccharolyticum_B
    saccharolyticum_C
    saccharophilus_A
    saccharophilus_B
    safensis_B
    sakei_A
    SAL_1
    SAL_18
    SAL_7
    salarius_A
    salexigens_A
    salicanalis_A
    salicis_cluster2
    salicis_cluster3
    salina_A
    salina_B
    salina_C
    saliphilus_A
    salivarius_D
    salmonicolor_A
    sandarakina_A
    sandarakinum_A
    sanguineus_A
    sanguinis_A
    sanguinis_B
    sanguinis_C
    sanguinis_D
    sanguinis_E
    sanguinis_F
    sanguinis_G
    sanguinis_H
    sanguinis_I
    sanguinis_J
    sanguinis_K
    sanguinis_L
    sanguinis_M
    sanguinis_N
    sanguinis_O
    sanguinis_P
    sanguinis_Q
    sanguinis_R
    sanguinis_S
    sanguinis_T
    sanguinis_U
    sanguinis_V
    sanxanigenens_A
    sanyensis_A
    saphenum_A
    sartagoforme_A
    sartagoforme_B
    satyanarayanai_B
    SAV/Human/BRA/IAL01_42160/2010
    SAV/Human/BRA/IAL03_21214/2010
    SAV/Human/BRA/IAL03_21218/2010
    SAV/Human/BRA/IAL03_21220/2010
    SAV/Human/BRA/IAL11_10818/2010
    SAV/Human/BRA/IAL11_9928/2010
    saxobsidens_A
    saxobsidens_B
    SB_Bac1
    SB_Bac10
    SB_Bac11
    SB_Bac2
    SB_Bac4
    SB_Bac5
    SB_Bac7
    SB_Bac9
    SB_Bact3
    SBN1_DGR
    SBOS_U
    SC_4_H6H8_2017
    scabrispora_A
    schaalii_A
    schaalii_B
    schaalii_C
    schaedleri_A
    schinkii_B
    schleiferi_A
    schmidteae_A
    schmidteae_B
    Schreyahn_AOB_Sali_Kultur_19
    Schreyahn_AOB_Sali_Kultur_8
    Schreyahn_AOB_SSU_Aster_6
    Schreyahn_AOB_SSU_Aster_8
    Schreyahn_K10.Sali
    Schreyahn_K9.Sali
    schumannianae_A
    sciuri_A
    sclerotialus_B
    scrofulaceum_C
    SCS_04Z1_1
    SCS_04Z1_26_2
    SCS_04Z1_29
    SCS_04Z1_3
    SCS_04Z1_41_2
    SCS_04Z1_41_3
    SCS_04Z1_42_2
    SCS_04Z1_47_3
    SCS_04Z1_60_1
    SCS_04Z1_60_2
    SCS_04Z1_62
    SCS_04Z1_68_2
    SCS_30Z0_1
    SCS_30Z0_10_3
    SCS_30Z0_12
    SCS_30Z0_20_2
    SCS_30Z0_23
    SCS_30Z0_37_2
    SCS_30Z0_4
    SCS_30Z0_45_2
    SCS_30Z0_49_3
    SCS_30Z0_6_2
    SCS_30Z1_11
    SCS_30Z1_12_2
    SCS_30Z1_14_2
    SCS_30Z1_20_3
    SCS_30Z1_24
    SCS_30Z1_28
    SCS_30Z1_37_2
    SCS_30Z1_39_2
    SCS_30Z1_4_2
    SCS_30Z1_8
    Se1Ps_1969
    Se1Pw_3014
    Se1Pw_3206
    Se2Ps_2218
    Se2Ps_2269
    Se2Ps_2350
    Se2Ps_2423
    Se2Ps_2430
    Se2Ps_2446
    Se2Ps_2490
    Se2Ps_640
    Se3Ps_2605
    Se3Ps_2679
    Se3Ps_2758
    Se3Pw_3619
    seckii_A
    sediminis_A
    sediminum_A
    segnicrescens_A
    segnis_A
    segnis_B
    segnis_C
    segnis_D
    sejongensis_A
    selenatarsenatis_A
    selenatireducens_A
    selenitireducens_A
    senegalense_A
    senegalensis_A
    senegalensis_B
    senioris_A
    seoulensis_A
    SH_25-50_Exo70_native
    shenzhenensis_A
    shigense_A
    shinshuensis_A
    shinshuensis_B
    shioyasakiensis_A
    shioyasakiensis_B
    showae_A
    showae_B
    showae_D
    showae_E
    sibiricum_A
    sicca_A
    sicca_B
    sicca_C
    sicca_E
    siccitolerans_A
    sifiae_A
    silvestris_A
    simiae_A
    simplex_A
    simplex_B
    simplex_D
    simplex_E
    simulans_A
    simulans_B
    sinaloensis_C
    sinaloensis_D
    sinense_A
    sinense_B
    sinense_C
    singularis_A
    singularis_B
    sinuspersici_A
    sinuspersici_B
    SM23_31
    SM23_57
    SM_12
    SM_3
    smithii_A
    SNISO_A2
    SNISO_B12
    SNISO_C1
    SNISO_C6
    SNISO_D1
    SNISO_E1
    SNISO_E12
    SNISO_E3
    SNISO_F2
    SNISO_F3
    SNISO_F4
    SNISO_I12
    SNISO_J2
    SNISO_J4
    So1Ps_682
    So1Pw_1092
    So1Pw_968
    So1Pw_972
    So1Pw_974
    So3Ps_1722
    So3Ps_1723
    So3Ps_1725
    So3Pw_1027
    So3Pw_2867
    SOB_395
    sobria_A
    sojae_A
    soli_A
    somerae_A
    sonorensis_A
    sordidicola_A
    sordidicola_C
    sp.48_UnivF-1183
    sp.CRBC_G2051
    sp.CRBC_G2061
    sp.CRBC_G2073
    sp.CRBC_G2107
    sp.CRBC_G2174
    sp.CRBC_G2230
    sp.CRBC_G2377
    sp.CRBC_G2430
    sp.CRBC_G2451
    sp.R0101_021
    sp_1_GP
    sp_2_GP
    sp_AL
    spadix_A
    spanius_B
    spanius_C
    spanius_D
    spanius_E
    spartinae_A
    Spd2B_4
    sphaericus_A
    sphaericus_B
    sphaericus_D
    sphaericus_E
    sphaeroides_A
    sphenoides_A
    spinoverrucosus_A
    spiritivorum_A
    spiroformis_A
    splendidus_F
    splendidus_I
    splendidus_J
    splendidus_K
    spongiarum_B
    spongiarum_C
    spongiarum_E
    spongiarum_F
    sporothermodurans_A
    sputorum_B
    SRC_DS3
    SRC_DS3a
    SRC_DSE1
    SRC_DSE11
    SRC_DSE3
    SRC_DSE5
    SRC_DSE6
    SRC_DSE9
    SRC_DSI1
    SRC_DSI2
    SRU_1
    SRU_17
    SRU_35
    stadtmanae_A
    stadtmanae_B
    stadtmanae_C
    staleyi_A
    staleyi_B
    staleyi_C
    stanieri_B
    stearothermophilus_A
    stenotrophus_A
    stenotrophus_B
    stenotrophus_C
    stenotrophus_D
    stercorihominis_A
    stercoris_A
    stetteri_A
    steynii_A
    stimsonii_A
    stipitatus_A
    stratosphericus_B
    stuttgartiensis_A
    stutzeri_A
    stutzeri_AA
    stutzeri_AB
    stutzeri_AC
    stutzeri_AD
    stutzeri_AE
    stutzeri_AF
    stutzeri_AH
    stutzeri_AI
    stutzeri_AK
    stutzeri_AN
    stutzeri_AO
    stutzeri_AP
    stutzeri_AQ
    stutzeri_AR
    stutzeri_B
    stutzeri_C
    stutzeri_D
    stutzeri_E
    stutzeri_G
    stutzeri_H
    stutzeri_N
    stutzeri_O
    stutzeri_P
    stutzeri_Q
    stutzeri_R
    stutzeri_S
    stutzeri_T
    stutzeri_U
    subflava_A
    subflava_B
    subflava_D
    subflava_E
    subglaciescola_A
    subrutilus_A
    subsalsa_A
    subterranea_A
    subterraneum_A
    subterraneum_E
    subterraneus_B
    subterraneus_C
    subtilis_G
    subvibrioides_B
    subvibrioides_C
    subvibrioides_D
    succinatutens_A
    succinatutens_B
    succinicans_A
    succinicivorans_A
    succinogenes_A
    succinogenes_B
    succinogenes_C
    succinogenes_D
    succinogenes_E
    succinogenes_F
    succinogenes_G
    succinogenes_H
    succinogenes_I
    succinogenes_J
    succinogenes_K
    succinogenes_L
    succinogenes_M
    succinogenes_N
    succinogenes_O
    succinogenes_P
    succinogenes_Q
    suis_A
    suis_AA
    suis_AB
    suis_AC
    suis_AD
    suis_AE
    suis_AF
    suis_AG
    suis_AH
    suis_B
    suis_F
    suis_I
    suis_L
    suis_O
    suis_P
    suis_R
    suis_S
    suis_T
    suis_U
    suis_V
    suis_W
    suis_X
    suis_Y
    suis_Z
    sukumoe_B
    sulfexigens_A
    sulfexigens_B
    sulfidiphilus_A
    sulfonivorans_A
    sulfonivorans_B
    SUND_HBM1
    suwonensis_A
    suwonensis_B
    suwonensis_C
    SV72_D
    symbiosa_A
    symbiotica_A
    symbiotica_D
    symbiotica_E
    symbiotica_F
    synxantha_A
    synxantha_B
    syringae_A
    syringae_F
    syringae_J
    syringae_M
    syringae_P
    syringae_Q
    syringae_R
    syringae_S
    syringae_T
    taeanensis_A
    taeanensis_B
    taichungensis_A
    taiwanensis_B
    taiwanensis_C
    taiwanensis_D
    tanakaei_A
    tanakaei_B
    tanakaei_C
    tannerae_A
    tannerae_B
    tasmaniensis_A
    tataouinensis_A
    tataouinensis_B
    tataouinensis_C
    taxi_A
    taxi_B
    taylorii_A
    TC1_29
    TC1_4
    TC2_12
    TC2_13
    TC3_16_MR-2014
    TC3_23_MR-2014
    TC3_29
    TC3_7_MR-2014
    tenebrarum_A
    tengchongensis_A
    tepidamans_A
    tepidum_A
    termitidis_A
    terrae_A
    terrae_B
    terrae_C
    terrain_en_construction
    tertiaricarbonis_A
    tertiaricarbonis_B
    tessellarius_A
    testaceum_A
    testaceum_B
    testaceum_C
    testaceum_E
    testaceum_F
    testaceum_G
    testaceum_H
    testosteroni_B
    testosteroni_C
    testosteroni_D
    testosteroni_E
    testosteroni_F
    testosteroni_G
    testosteroni_H
    TF4_16
    TF4_17
    TF4_17_MR-2014
    TF5_22_MR-2014
    thalassa_A
    thalpophilum_A
    thalpophilum_B
    thaueri_A
    thaueri_B
    thaueri_C
    thermautotrophicus_A
    thermoacetica_A
    thermoglucosidasius_A
    thermohalophila_A
    thermoleovorans_A
    thermophila_A
    thermophilum_A
    thermophilus_C
    thermosulfidooxidans_A
    thermotolerans_A
    thetaiotaomicron_A
    thiaminolyticus_A
    thiocyanaticus_A
    thiodismutans_A
    thiooxidans_A
    thiooxydans_A
    thiotaurini_A
    thuringiensis_AA
    thuringiensis_AB
    thuringiensis_K
    thuringiensis_N
    thuringiensis_S
    tibiigranuli_A
    tiedjei_A
    timonense_A
    timonensis_A
    timorensis_A
    tombus-like_virus
    towneri_A
    toxinivorans_A
    TPMV_P-EGFP
    TPMV_P-EGFP_H-His6
    TPMV_P-EGFP_HaCD20
    TPMV_P-EGFP_HaEGFR
    trachydisci_A
    translucens_C
    transvalensis_A
    trapanicum_A
    triatomae_A
    tribocorum_A
    tribocorum_B
    trichothecenolyticum_B
    triculaminicus_A
    tropica_A
    tropici_A
    tropici_B
    tropici_C
    tropiciagri_A
    tropiciagri_B
    TsCyV-1_JP-NUBS-2014
    tsukubensis_B
    tsuruhatensis_A
    tsutsugamushi_C
    tuberculostearicum_C
    tuberculostearicum_D
    tuberculostearicum_E
    tubiashii_A
    tubiashii_B
    tuirus_A
    tumefaciens_B
    tumefaciens_C
    tumefaciens_I
    tundrae_A
    tundripaludum_A
    turgidiscabies_A
    turicensis_A
    turicensis_B
    turnerae_B
    tusciae_A
    typhi_A
    typhi_B
    typicum_A
    tyrfis_A
    tyrfis_B
    tyrobutyricum_A
    uberis_A
    ubique_A
    ubique_B
    ubique_D
    ubique_E
    ubique_F
    ubique_G
    ubique_J
    ubique_K
    ubique_N
    ubique_O
    ubique_P
    ubique_Q
    ubique_S
    ubique_T
    ubonensis_B
    UBP10_UBA1149
    UBP10_UBA1160
    UBP11_UBA4055
    UBP12_UBA5184
    UBP13_UBA5359
    UBP14_UBA6098
    UBP15_UBA6099
    UBP16_UBA6123
    UBP17_UBA6191
    UBP18_UBA7526
    UBP1_UBA2172
    UBP1_UBA4771
    UBP1_UBA4783
    UBP2_UBA2226
    UBP2_UBA2255
    UBP2_UBA2291
    UBP2_UBA4780
    UBP2_UBA5202
    UBP2_UBA5599
    UBP3_UBA1247
    UBP3_UBA1439
    UBP3_UBA3054
    UBP3_UBA6266
    UBP4_UBA4092
    UBP4_UBA6127
    UBP5_UBA1551
    UBP5_UBA1559
    UBP6_UBA1177
    UBP6_UBA1196
    UBP6_UBA1209
    UBP6_UBA3056
    UBP7_UBA6624
    UBP7_UBA7824
    UBP8_UBA6595
    UBP8_UBA7324
    UBP8_UBA7816
    UBP8_UBA817
    UBP8_UBA827
    UBP9_UBA1085
    UBP9_UBA11836
    UBP9_UBA1432
    UBP9_UBA4627
    UBP9_UBA4705
    UBP9_UBA5552
    UBP9_UBA6111
    UBP9_UBA6348
    UBP9_UBA6438
    Udayana_MDPVA
    Udayana_MDPVB
    uenonis_A
    UK_ancient_CT89
    ulcerans_A
    uli_A
    uli_B
    uliginosa_B
    ulvae_A
    ulvanivorans_A
    umbonata_A
    umbrinus_A
    undulata_A
    universalis_A
    universalis_B
    uraniireducens_A
    urativorans_A
    urativorans_B
    urativorans_C
    ureae_A
    ureae_B
    ureae_C
    ureae_D
    urealyticum_A
    ureilytica_A
    ureisolvens_A
    ureolyticus_A
    urinae_A
    urinae_B
    urinae_C
    urinae_D
    urinae_E
    usitatum_A
    utahensis_B
    vadi_A
    vagans_C
    vaginale_A
    vaginale_B
    vaginale_C
    vaginale_D
    vaginale_E
    vaginale_F
    vaginale_H
    vaginale_I
    vaginalis_B
    vaginalis_C
    vaginatus_A
    vaginatus_B
    valericigenes_A
    valericigenes_B
    vanbaalenii_A
    vancanneytii_A
    variabilis_A
    variabilis_B
    variabilis_C
    varians_A
    varians_B
    varium_A
    varium_C
    vaviloviae_B
    vB_EcoM-p111
    vB_KsaM-C1
    vB_LmoS_P7
    vB_PaeM_Sem1
    vB_RanS_PJN03
    vB_SabS_Sds2
    vB_SauM_SDQ
    venezuelae_B
    venezuelae_C
    venezuelae_D
    venezuelae_E
    venezuelae_F
    venezuelae_G
    ventosae_B
    ventosae_C
    vermicola_A
    veronii_A
    versatilis_A
    versatilis_B
    versiforme_A
    vesicularis_B
    vesicularis_C
    vesicularis_D
    vesicularis_E
    vestfoldensis_A
    vestfoldensis_B
    vibrioides_A
    vibrioides_C
    vibrioides_D
    vibrioides_E
    vicinus_A
    vietnamensis_A
    vietnamensis_B
    violaceoruber_B
    violaceum_A
    violaceum_B
    violaceus_A
    violaceusniger_A
    violascens_A
    virginiae_A
    viridans_A
    viridans_B
    viridans_C
    viridarii_A
    viridescens_A
    viridiflava_C
    viridiflava_D
    viridochromogenes_A
    viridochromogenes_B
    viscerum_A
    vitaeruminis_A
    vitis_B
    vitis_D
    vitis_E
    vitis_F
    vitis_G
    vitis_H
    Vk*MYCDLoxP_MYC_FULL_LENGTH_ONLY
    Vk*MYCDLoxP_MYCS
    volcania_A
    voltae_C
    voltae_D
    volucris_A
    Vph_2K
    VPXV_CA85
    vulcani_A
    vulcania_A
    vulcanus_A
    vulgaris_A
    vulgaris_B
    vulgaris_C
    vulnificus_B
    W13C10_B
    W13C11_B
    W13C12_B
    W13C14_B
    W13C15_B
    W13C16_B
    W13C17_B
    W13C18_B
    W13C21_B
    W13C22_B
    W13C23_B
    W13C24_B
    W13C25_B
    W13C27_B
    W13C28_B
    W13C2_B
    W13C30_B
    W13C32_B
    W13C33_B
    W13C34_B
    W13C5_B
    W13C7_B
    W13M2_F
    W13M4_B
    W13M60_B
    W13M61a_B
    W13M62b_B
    W13M7_B
    W13S12_B
    W13S13_B
    W13S16_B
    W13S18_B
    W13S21_B
    W13S22_B
    W13S23_B
    W13S26_B
    W13S28_F
    W13S29_B
    W13S2_B
    W13S30_B
    W13S31_B
    W13S32_B
    W13S34_B
    W13S46_B
    W13S49_B
    W13S4_B
    W13S51_B
    W13S54_B
    W13S8_B
    W15C15_F
    W15C16_F
    W15C18a_F
    W15C28_F
    W15C2_F
    W15M16_F
    W15M1a_F
    W15M34_F
    W15S10_F
    W15S11_F
    W15S14_F
    W15S18_F
    W15S24b_F
    W15S32_F
    W15S58a_F
    W15S67_F
    W15S83_F
    W15S84a_F
    W15S84b_F
    W15S87a_F
    wadei_A
    wadsworthensis_A
    wallacei_A
    wallacei_B
    walsbyi_A
    walsbyi_B
    warneri_A
    wastewater/GI.1/GP_Dec11_1395_1/2011/ZAF
    wastewater/GI.1/GP_Oct11_1145_3/2011/ZAF
    wastewater/GI.1/MP_Aug10_992/2010/ZAF
    wastewater/GI.1/MP_Nov10_1391/2010/ZAF
    wastewater/GI.1/MP_Oct11_1168/2011/ZAF
    wastewater/GI.1/MP_Sep10_1108/2010/ZAF
    wastewater/GI.3/GP_Oct11_1145_1/2011/ZAF
    wastewater/GI.3/GP_Oct11_1145_5/2011/ZAF
    wastewater/GI.3/NW_Nov11_1266/2011/ZAF
    wastewater/GI.4/GP_Sep11_1051/2011/ZAF
    wastewater/GI.8/NW_Aug11_957/2011/ZAF
    wastewater/GI/GP_Dec11_1395_2/2011/ZAF
    wastewater/GI/GP_Oct11_1085/2011/ZAF
    wastewater/GI/GP_Oct11_1145_2/2011/ZAF
    wastewater/GI/NW_Dec11_1394/2011/ZAF
    wastewater/GI/NW_Jul11_759/2011/ZAF
    wastewater/GI/NW_Nov11_1268/2011/ZAF
    wastewater/GII.12/NW_Aug11_957_1/2011/ZAF
    wastewater/GII.13/MP_Aug10_1012_3/2010/ZAF
    wastewater/GII.13/MP_Oct11_1086_5/2011/ZAF
    wastewater/GII.13/MP_Oct11_1168_2/2011/ZAF
    wastewater/GII.17/GP_Dec11_1395_1/2011/ZAF
    wastewater/GII.17/MP_Aug10_1012_5/2010/ZAF
    wastewater/GII.17/MP_Jul11_862_1/2011/ZAF
    wastewater/GII.17/MP_Sep10_1108/2010/ZAF
    wastewater/GII.17/NW_Aug11_1035_1/2011/ZAF
    wastewater/GII.17/NW_Aug11_908/2011/ZAF
    wastewater/GII.17/NW_Jul11_759/2011/ZAF
    wastewater/GII.17/NW_Jul11_874_1/2011/ZAF
    wastewater/GII.17/NW_Nov11_1268/2011/ZAF
    wastewater/GII.2/GP_Oct11_1083/2011/ZAF
    wastewater/GII.2/GP_Oct11_1145/2011/ZAF
    wastewater/GII.2/GP_Oct11_1147/2011/ZAF
    wastewater/GII.2/GP_Sep11_1051/2011/ZAF
    wastewater/GII.2/MP_Aug10_985/2010/ZAF
    wastewater/GII.2/MP_Oct11_1168_3/2011/ZAF
    wastewater/GII.2/MP_Sep10_1124/2010/ZAF
    wastewater/GII.2/NW_Aug11_909_3/2011/ZAF
    wastewater/GII.2/NW_Dec11_1394/2011/ZAF
    wastewater/GII.2/NW_Nov11_1266_1/2011/ZAF
    wastewater/GII.2/NW_Oct11_1077_1/2011/ZAF
    wastewater/GII.2/NW_Oct11_1077_2/2011/ZAF
    wastewater/GII.3/MP_Aug10_1010_4/2010/ZAF
    wastewater/GII.3/MP_Aug10_1012_4/2010/ZAF
    wastewater/GII.3/MP_Dec10_1493/2010/ZAF
    wastewater/GII.3/MP_Dec10_1521_6/2010/ZAF
    wastewater/GII.3/MP_Oct10_1229/2010/ZAF
    wastewater/GII.4/GP_Dec11_1395_2/2011/ZAF
    wastewater/GII.4/GP_Dec11_1395_3/2011/ZAF
    wastewater/GII.4/KZN_Nov11_1306/2011/ZAF
    wastewater/GII.4/KZN_Sep10_1052/2010/ZAF
    wastewater/GII.4/MP_Aug10_1010_2/2010/ZAF
    wastewater/GII.4/MP_Aug10_1010_3/2010/ZAF
    wastewater/GII.4/MP_Aug10_1011_1/2010/ZAF
    wastewater/GII.4/MP_Aug10_1011_2/2010/ZAF
    wastewater/GII.4/MP_Oct11_1086_1/2011/ZAF
    wastewater/GII.4/MP_Oct11_1168_1/2011/ZAF
    wastewater/GII.4/NW_Aug11_957_4/2011/ZAF
    wastewater/GII.4/NW_Jul11_757_1/2011/ZAF
    wastewater/GII.4/NW_Nov11_1266_3/2011/ZAF
    wastewater/GII.6/MP_Aug10_1011_3/2010/ZAF
    wastewater/GII.6/MP_Aug10_995/2010/ZAF
    wastewater/GII.6/MP_Dec10_1521_3/2010/ZAF
    wastewater/GII.6/NW_Aug11_909_1/2011/ZAF
    wastewater/GII.6/NW_Aug11_909_2/2011/ZAF
    wastewater/GII.6/NW_Jul11_757_4/2011/ZAF
    wastewater/GII.6/NW_Nov11_1266_2/2011/ZAF
    wastewater/GII.7/MP_Aug10_1012_1/2010/ZAF
    wastewater/GII.7/MP_Dec10_1521_4/2010/ZAF
    wastewater/GP_Dec11_1395/ZAF
    wastewater/GP_Oct11_1145/ZAF
    wastewater/MP_Aug10_1010/ZAF
    wastewater/MP_Aug10_1011/ZAF
    wastewater/MP_Aug10_985/ZAF
    wastewater/MP_Aug10_992/ZAF
    wastewater/MP_Aug10_995/ZAF
    wastewater/MP_Jul11_862/ZAF
    wastewater/MP_Oct10_1229/ZAF
    wastewater/MP_Oct11_1086/ZAF
    wastewater/MP_Sep10_1102/ZAF
    wastewater/MP_Sep10_1124/ZAF
    wastewater/NW_Aug11_909/ZAF
    wastewater/NW_Dec11_1396/ZAF
    wastewater/NW_Jul11_759/ZAF
    wastewater/NW_Nov11_1268/ZAF
    water/LP_Jan12_1/ZAF
    water/LP_Jan12_23/ZAF
    water/LP_Jan12_5/ZAF
    water/LP_Jan12_7/ZAF
    water/LP_Mar12_7/ZAF
    water/LP_Mar12_PVT/ZAF
    wautersii_A
    WBUT_C01
    weilii_A
    wenxiniae_A
    wenyonii_C
    wesenbergii_A
    wexlerae_B
    wexlerae_C
    wieringae_A
    willei_A
    winogradskyi_A
    WM06i_A10G
    WM06i_A11B
    WM06i_A1D
    WM06i_A5D
    WM06i_B11A
    WM06i_B11H
    WM06i_B1G
    WM06i_B6A
    WM06i_B8F
    WM06i_B9F
    woesei_A
    woluwensis_A
    woodii_A
    woodruffii_A
    woosongensis_A
    WOR_2
    wratislaviensis_A
    WS01_1416
    WS01_480
    WS06_481
    WS08_522
    WS10_380
    WS12_472
    WS13_484
    WS18_483
    WS19_479
    wynnii_A
    XAN_1
    XAN_14
    XAN_8
    xanthomarina_A
    xanthopallidus_A
    xanthophaeus_A
    xanthus_A
    xanthus_B
    xiamenensis_A
    xiamenensis_B
    xinganensis_A
    xinjiangensis_A
    xylanexedens_A
    xylanexedens_B
    xylanexedens_C
    xylanilyticus_A
    xylanivorans_A
    xylanolytica_B
    xylanolytica_C
    xylanophilus_A
    xyli_A
    xyli_C
    xylinus_C
    xylinus_D
    xylinus_E
    xylosoxidans_B
    xylosoxidans_C
    xylosus_B
    XZ110_XZ_CHN_2017
    XZ111_XZ_CHN_2017
    XZ112_XZ_CHN_2017
    XZ114_XZ_CHN_2017
    XZ115_XZ_CHN_2017
    XZ93_XZ_CHN_2017
    XZ97_XZ_CHN_2017
    XZ98_XZ_CHN_2017
    XZ99_XZ_CHN_2017
    Y2H-GFP_URA3
    Y4_579_1
    yabuuchiae_A
    yamanorum_A
    yanchengensis_A
    yannicii_A
    yanoikuyae_A
    yeojuensis_A
    yomogidense_A
    YS1-2_2434
    YS1-2_2437
    yuanmingense_A
    zalophi_A
    zeae_A
    zhangzhouensis_A
    zhoushanensis_A
    zoogloeoides_A
    zoogloeoides_B
    zoogloeoides_C
    zucineum_B
    zundukense_A
    );

my %is_hyphenated  = map { $_ => 1 } @GENERA;
my %is_underscored = map { $_ => 1 } @SPECIES;

sub _set_gca_and_or_taxon_id {
    my $self   = shift;
    my $strain = shift;

    # Note: we use named captures here (?<capture name> ... )

    ## no critic (ProhibitUnusedCapture)

    # handle GCA/GCF acting as taxon_ids
    if ($strain =~ m/\A (:?(?<strain>.*)_)? (?<gca>$NCBIGCA) \z/xms) {

        # truncate strain to optional non-GCA/GCF part (excl. underscore)...
        # ... and set taxon_id to full GCA/GCF
        $self->_set_strain($+{strain});
        my $gca = $+{gca};
        $self->_set_taxon_id($gca);
        $self->_set_gca(     $gca);     # alias

        # further store GCA/GCF components for easier comparison
        my ($gca_novers, $gca_vers) = split /\./xms, $gca;
        $self->_set_gca_novers($gca_novers);
        $self->_set_gca_vers(  $gca_vers  );
        my ($gca_prefix, $gca_number) = split /_/xms, $gca_novers;
        $self->_set_gca_prefix($gca_prefix);
        $self->_set_gca_number($gca_number);
    }

    # handle regular taxon_ids
    elsif ($strain =~ m/\A (:?(?<strain>.*)_)? (?<taxon_id>$NCBIPKEY) \z/xms) {
        # truncate strain to optional non-taxon_id part (excl. underscore)...
        # ... and set taxon_id
        $self->_set_strain(  $+{strain  });
        $self->_set_taxon_id($+{taxon_id});
    }

    ## use critic

    # ignore legacy strains
    return;
}

sub BUILD {
    my $self = shift;

    # possibly remove surrounding quotes (before anything else is done)
    if ($self->full_id =~ m/\A $RE{delimited}{-delim=>q{'"}}{-keep} \z/xms) {
        $self->_set_full_id($3);
    }

    # warn of trailing spaces as they harm taxonomic analysis
    carp '[BMC] Warning: "' . $self->full_id . '" has trailing spaces;'
        . ' cannot parse sequence id!' if $self->full_id =~ m/\s+\z/xms;

    # parse potential taxonomy-aware abbreviated id
    # Note: such ids are still considered as foreign_ids
    {
        my ($strain, $acc) = $self->full_id =~ $TAXABBR_ID;
        if (defined $strain) {
            $self->_set_gca_and_or_taxon_id($strain);
            $self->_set_accession($acc);
            $self->_set_foreign;
            return;
        }
    }

    # parse potential NCBI FASTA-style GI id
    # Note: such ids are still considered as foreign_ids
    {
        my ($gi, $acc) = $self->full_id =~ $NCBIGI_ID;
        if (defined $gi) {
            $self->_set_gi($gi);
            $self->_set_accession($acc);
            $self->_set_foreign;
            return;
        }
    }

    # parse potential NCBI FASTA-style GNL id
    # Note: such ids are still considered as foreign_ids
    {
        my ($match) = $self->full_id =~ $NCBIGNL_ID;
        if ($match) {
            $self->_set_parts( [ split /\|/xms, $match ] );
            $self->_set_database(   $self->get_part(1) );   # official
            $self->_set_identifier( $self->get_part(2) );   # aliases
            $self->_set_foreign;
            return;
        }
    }

    # skip RiboDB ids that look like weird full_ids
    if ($self->full_id =~ m/~$NCBIGCA/xms) {
        $self->_set_foreign;
        return;
    }

    # check full_id validity
    my  ($family, $tag, $genus, $species, $strain, $acc, $tail, $new)
        = $self->full_id =~ $FULL_ID;
    unless (defined $genus) {

        # First try to coerce foreign full_id by replacing 1st '_' by ' '. If
        # this does not work, keep the original full_id and flag it as foreign.
        # This approach allows the transparent conversion of valid full_ids
        # from foreign software able to handle unlimited gap-free ids.
        # Note: This will fails if the optional family part contains an '_'.

        my $cand_id = $self->full_id =~ s{_}{ }xmsr;
        ($family, $tag, $genus, $species, $strain, $acc, $tail, $new)
            = $cand_id =~ $FULL_ID;
        unless (defined $genus) {
            $self->_set_foreign;
            return;
        }
        $self->_set_full_id($cand_id);
    }

    # handle hyphenated genera that could interfere with family definition
    if (defined $family) {
        my $hyphenated = $family . '-' . $genus;
        if (defined $is_hyphenated{$hyphenated}) {
            $family = undef;
            $genus = $hyphenated;
        }
    }

    # handle underscored species that could interfere with species definition
    if (defined $strain) {
        my $underscored = $species . '_' . $strain;
        if (defined $is_underscored{$underscored}) {
            $strain = undef;
            $species = $underscored;
        }
    }

    # populate legacy components from full_id (and tail)
    $self->_set_family($family);
    $self->_set_tag($tag);
    $self->_set_genus($genus);
    $self->_set_species($species);
    $self->_set_strain($strain);
    $self->_set_accession($acc);
    $self->_set_tail($tail);

    # populate modern components if available
    $self->_set_gca_and_or_taxon_id($strain) if defined $strain;
    $self->_set_gi($acc)                     if $acc =~ $PKEYONLY;

    # set new flag if needed
    $self->_set_new if $new;

    return;
}

}   # end of BEGIN block


around qr{ is_new | is_genus_only | is_doubtful | org $  }xms => sub {
    my $method = shift;
    my $self   = shift;

    # Note: we return an explicit undef to emulate other accessor behavior
    return undef if $self->is_foreign;      ## no critic (ProhibitExplicitReturnUndef)
    return $self->$method(@_);
};



sub is_genus_only {
    my $self = shift;
    return 1 if $self->species =~ $GENUSONLY;
    return 0;
}


sub is_doubtful {
    my $self = shift;
    return 1 if defined $self->tag && $self->tag =~ $DOUBTFUL;
    return 0;
}


sub org {
    my $self = shift;
    return $self->genus . q{ } . $self->species;
}


sub abbr_org {
    my $self = shift;
    return substr($self->genus, 0, 1) . q{. } . $self->species;
}


sub full_org {
    my $self = shift;
    my $sep  = shift // '_';        # default to underscore

    my $org = $self->org;
       $org .= $sep . $self->strain   if $self->strain;
       $org .= $sep . $self->taxon_id if $self->taxon_id;
                                    # also works with GCA/GCF!
    return $org;
}


sub family_then_full_org {                  ## no critic (RequireArgUnpacking)
    my $self = shift;

    my $family = $self->family // q{};
    $family .= '-' if $family;

    return $family . $self->full_org(@_);   # note the currying
}


sub contam_org {
    my $self = shift;

    return undef unless $self->tail;        ## no critic (ProhibitExplicitReturnUndef)
    # Note: we return an explicit undef to emulate other accessor behavior

    # only replace the first underscore by a space (for consistency)
    return $self->tail =~ s{_}{ }xmsr;
}


sub foreign_id {
    my $self = shift;

    # do not alter full_id if already foreign!
    return $self->full_id if $self->is_foreign;

    # otherwise, only replace the first space by an underscore
    return $self->full_id =~ s{\ }{_}xmsr;
}


sub nexus_id {
    my $self = shift;

    # quoted named are not escaped by Bio::Phylo...
    # ... but cannot contain single quotes

    my $full_id = $self->full_id =~ tr/'//dr;       # remove quotes...
    return qq{'$full_id'};                          # ... and requote
}


sub abbr_with_regex {
    my $self   = shift;
    my $prefix = shift // q{};
    my $regex  = shift // $DEF_ID;

    my @ids = $self->full_id =~ $regex;       # capture original id(s)
    s{$NOID_CHARS}{_}xmsg for @ids;           # substitute forbidden chars
    my $abbr_id = join q{}, $prefix, @ids;

    return $abbr_id;
}

# from Bio::Phylo::PhyloRole

# =item get_nexus_name()
#
# Gets invocant's name, modified to be safely used in nexus files. This means that:
#
# =item names with spaces in them that aren't 'single quoted' have their spaces replaced
# with underscores
#
# =item names with any of the following characters in them are single quoted:
#  -^*(){}[]+=;:"\<>/,
#
# =item names with single quotes inside them (i.e. not around them) are "double quoted"
#
#  Type    : Accessor
#  Title   : get_nexus_name
#  Usage   : my $name = $obj->get_nexus_name;
#  Function: Returns the object's name.
#  Returns : A string
#  Args    : None
#
# =cut
#
#     sub get_nexus_name {
#         my $self = shift;
#         my $name = $self->get_internal_name;
#         if ( $name =~ /\s/ && $name !~ /^'.+'$/ ) {
#             $name =~ s/\s/_/g;
#         }
#         if (   $name =~ /(?:\-|\^|\*|\(|\)|{|}|\[|\]|\+|=|;|:|"|\\|<|>|\/|,)/
#             && $name !~ /^'.+'$/ )
#         {
#             $name = "'${name}'";
#         }
#         if ( $name =~ /'/ && $name !~ /^".+"$/ && $name !~ /^'.+'$/ ) {
#             $name = "\"${name}\"";
#         }
#         return $name;
#     }


# memoized constructor derived from MooseX::Role::Flyweight


# cache for created SeqId objects (even if out of scope)
my %instance_for;

sub instance {
    my ($class, %args) = @_;

    # TODO: check if this optimized way always work for us
    my $key = $args{full_id};

    # return the existing instance
    return $instance_for{$key} if defined $instance_for{$key};

    # create a new instance
    my $instance = $class->new(%args);
    $instance_for{$key} = $instance;
    # Note: do not weaken reference or this will defeat the purpose.
    # However, this could lead to memory leak in some extreme cases.
    # Scalar::Util::weaken $instance_for{$key};

    return $instance;
}


# class methods to build modern MUST-compliant id from NCBI components


sub new_with {                              ## no critic (RequireArgUnpacking)
    my $class = shift;
    my %args  = @_;                         # TODO: handle HashRef?

    my ($org, $taxon_id, $accession, $keep_strain)
        = @args{ qw(org taxon_id accession keep_strain) };
    $accession //= $args{gi};       # fall back to legacy argument name

    # extract components from organism name
    my ($genus, $species, $strain) = $class->parse_ncbi_name($org);
    # Note: genus, species and strain will have been cleaned-up at this stage

    # truncate name to Genus species (or sp. if none)
    # append strain (if asked to do so)
    # append NCBI taxon_id or GCA/GCF as pseudo-strain
    # append accession number (if provided)
    my $full_id
        = $genus . ' '
        . ($species   ? ( $species                               ) : 'sp.')
        . ($strain    ? ( $keep_strain ? ('_' . $strain  ) : q{} ) :  q{} )
        . ($taxon_id  ? ('_' . $taxon_id                         ) :  q{} )
        . ($accession ? ('@' . $accession                        ) :  q{} )
    ;

    return $class->new( full_id => $full_id );
}



sub parse_ncbi_name {
    my $class = shift;
    my $org   = shift;

    # clean org name
    $org = $class->clean_ncbi_name($org);

    # split org name into max 3 components: genus, species and strain
    # strain is a greedy component for trailing information
    my ($genus, $species, $strain) = split /\s+/xms, $org, 3;

    # clean strain of unwanted prefices and characters (if any)
    $strain = $class->clean_strain($strain) if $strain;

    return ($genus, $species, $strain);
}


sub clean_ncbi_name {
    my $class = shift;
    my $org   = shift;

    # remove unwanted prefices
    $org =~ s{uncultured \s+ candidatus \s+}{}xmsgi;
    $org =~ s{candidatus \s+}{}xmsgi;

    # remove cf. and aff.
    $org =~ s{\b (:? cf|aff)\.? \s+ (?= \D+$ | \S+\s+)}{}xmsgi;
    # Note: delete only if followed by a word without digits (not a strain)
    # hence when 'cf' or 'aff' does not stand for a species itself

    # remove unwanted characters
    $org =~ s{[\[\]\']}{}xmsgi;

    return $org;
}


sub clean_strain {
    my $class  = shift;
    my $strain = shift;

    # remove unwanted prefices and characters (if any)
    $strain =~ s{\b substr \b}{}xmsgi;
    $strain =~ s{\b strain \b}{}xmsgi;
    $strain =~ s{\b subsp  \b}{}xmsgi;
    $strain =~ s{\b ssp\.    }{}xmsgi;
    $strain =~ s{\b str    \b}{}xmsgi;
    $strain =~ tr/A-Za-z0-9//cd;        # delete non-alphanumeric chars

    return $strain;
}


sub tag_filter {
    my $self = shift;
    my $list = shift;

    return Filter->new( _specs => $list, component => 'tag' );
}


sub family_filter {
    my $self = shift;
    my $list = shift;

    return Filter->new( _specs => $list, component => 'family' );
}


sub genus_filter {
    my $self = shift;
    my $list = shift;

    return Filter->new( _specs => $list, component => 'genus' );
}


sub species_filter {
    my $self = shift;
    my $list = shift;

    return Filter->new( _specs => $list, component => 'species' );
}


__PACKAGE__->meta->make_immutable;
1;

__END__

=pod

=head1 NAME

Bio::MUST::Core::SeqId - Modern and legacy MUST-compliant sequence id

=head1 VERSION

version 0.251140

=head1 SYNOPSIS

    # TODO

=head1 DESCRIPTION

    # TODO

=head1 METHODS

=head2 is_genus_only

=head2 is_doubtful

=head2 org

=head2 abbr_org

=head2 full_org

=head2 family_then_full_org

=head2 contam_org

=head2 foreign_id

=head2 nexus_id

=head2 abbr_with_regex

=head2 instance

=head2 new_with

=head2 parse_ncbi_name

=head2 clean_ncbi_name

=head2 clean_strain

=head2 tag_filter

=head2 family_filter

=head2 genus_filter

=head2 species_filter

=head1 AUTHOR

Denis BAURAIN <denis.baurain@uliege.be>

=head1 CONTRIBUTOR

=for stopwords Mick VAN VLIERBERGHE

Mick VAN VLIERBERGHE <mvanvlierberghe@doct.uliege.be>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2013 by University of Liege / Unit of Eukaryotic Phylogenomics / Denis BAURAIN.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
