## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(stbl)

## -----------------------------------------------------------------------------
register_user <- function(username,
                          email_address,
                          age,
                          is_premium_member,
                          interests) {
  # Imagine this is a slow API call
  list(
    username = username,
    email_address = email_address,
    age = age,
    is_premium_member = is_premium_member,
    interests = interests
  )
}

## -----------------------------------------------------------------------------
register_user <- function(username,
                          email_address,
                          age,
                          is_premium_member,
                          interests) {
  interests <- to_chr(interests)

  list(
    username = username,
    email_address = email_address,
    age = age,
    is_premium_member = is_premium_member,
    interests = interests
  )
}

# It works with a character vector or a list of characters. We use `str()` to 
# make the output easier to see.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
) |> str()
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = list("R", "hiking")
) |> str()

## ----error = TRUE-------------------------------------------------------------
try({
# Fails because the list contains a function, which is not character-like.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = list("R", mean)
)
})

## -----------------------------------------------------------------------------
register_user <- function(username,
                          email_address,
                          age,
                          is_premium_member,
                          interests) {
  interests <- to_chr(interests)
  age <- to_int_scalar(age)
  is_premium_member <- to_lgl_scalar(is_premium_member)

  list(
    username = username,
    email_address = email_address,
    age = age,
    is_premium_member = is_premium_member,
    interests = interests
  )
}

# This works.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = "42", 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
) |> str()

## ----error = TRUE-------------------------------------------------------------
try({
# Fails because age is not a single value.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = c(30, 31), 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
)

# Fails because "forty-two" cannot be converted to an integer.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = "forty-two", 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
)
})

## -----------------------------------------------------------------------------
register_user <- function(username,
                          email_address,
                          age,
                          is_premium_member,
                          interests) {
  interests <- to_chr(interests)
  age <- to_int_scalar(age)
  is_premium_member <- to_lgl_scalar(is_premium_member)

  space_regex <- c("must not contain spaces" = "\\s")
  attr(space_regex, "negate") <- TRUE
  username <- stabilize_chr_scalar(
    username,
    regex = space_regex
  )

  email_regex <- "^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\\.[a-zA-Z]{2,}$"
  email_address <- stabilize_chr_scalar(
    email_address,
    regex = c("must be a valid email address" = email_regex)
  )

  list(
    username = username,
    email_address = email_address,
    age = age,
    is_premium_member = is_premium_member,
    interests = interests
  )
}

# A successful call.
register_user(
  username = "test_user", 
  email_address = "test@example.com", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
) |> str()

## ----error = TRUE-------------------------------------------------------------
try({
# Fails because the username has a space.
register_user(
  username = "test user", 
  email_address = "test@example.com", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
)

# Fails because the email address is invalid.
register_user(
  username = "test_user", 
  email_address = "not-a-valid-email", 
  age = 42, 
  is_premium_member = TRUE, 
  interests = c("R", "hiking")
)
})

