## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(tidy = FALSE)

## ----echo = FALSE-------------------------------------------------------------
library(YEAB)

## -----------------------------------------------------------------------------
data("DD_data")
norm_sv <- DD_data$norm_sv
delays <- DD_data$Delay

DD_data

## -----------------------------------------------------------------------------
# first, fit a linear model
lineal_m <- lm(norm_sv ~ delays)
# hyperbolic model
hyp_m <- hyperbolic_fit(norm_sv, delays, 0.1)
# exponential model
exp_m <- exp_fit(norm_sv, delays, 0.1)
AIC(lineal_m, hyp_m, exp_m)

## -----------------------------------------------------------------------------
k_hyp <- coef(hyp_m)
k_exp <- coef(exp_m)
k_lin <- coef(lineal_m)

## ----echo = FALSE-------------------------------------------------------------
paste("K_hyp: ", k_hyp)
paste("K_exp: ", k_exp)
paste("K_lin: ", k_lin)

## -----------------------------------------------------------------------------
delay_norm <- delays / max(delays) # It is important to normalize the delay values first in order to get a coherent AUC.
AUC_value <- trapezoid_auc(delay_norm, norm_sv)

## ----echo = FALSE-------------------------------------------------------------
y_data <- seq(0, max(delays), len = 200) # For plotting the curves.

plot(
  delays,
  norm_sv,
  ylim = c(0, 1),
  pch = 21,
  ylab = "Subjective Values",
  xlab = "Delay",
  bg = "orange",
  col = "black"
)
lines(
  y_data,
  eq_hyp(k = k_hyp, y_data),
  col = "green4",
  lwd = 2
)
lines(
  y_data,
  exp(-k_exp * y_data),
  col = "steelblue",
  lwd = 2
)
abline(lineal_m, lty = 2, lwd = 2)

legend(
  "topright",
  legend = c("data", "exp fit", "hyp fit", "linear fit", paste("AUC=", AUC_value)),
  pch = c(21, NA, NA, NA, NA),
  lty = c(NA, 1, 1, 2, NA),
  pt.bg = c("orange", NA, NA, NA, NA),
  col = c(1, "steelblue", "green4", 1),
)

## -----------------------------------------------------------------------------
y_data <- seq(0, max(delays), len = 200)
x_data <- eq_hyp(k = k_hyp, y_data)

## -----------------------------------------------------------------------------
x_data <- exp(-k_exp * y_data)

