## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.align = 'center'
)

## ----echo = FALSE, out.width = "60%", fig.cap = "Fig 1: Caco-2 experimental set up", fig.topcaption = TRUE, fig.align = "center"----
knitr::include_graphics("img/Caco2_diagram.png")

## ----setup, message = FALSE, warning = FALSE----------------------------------
# Primary package 
library(invitroTKstats)
# Data manipulation package 
library(dplyr)
# Table formatting package
library(flextable)

## ----Load example data--------------------------------------------------------
# Load example caco2 data 
data("Caco2-example")

## ----echo = FALSE, tab = 1----------------------------------------------------
head(caco2_L0, n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(
      )
      )
  ) %>% 
  set_caption(caption = "Table 1: Level 0 data",
              align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----required cols, echo = FALSE----------------------------------------------
# Create a table of required arguments for Level 1

req_cols <- data.frame(matrix(nrow = 34, ncol = 5))
vars <- c("Argument", "Default", "Required in L0?", "Corresp. single-entry Argument", "Descr.")
colnames(req_cols) <- vars

# Argument names 
arguments <- c("FILENAME", "data.in", "sample.col", "lab.compound.col", "dtxsid.col", "date.col", 
                "compound.col", "area.col", "istd.col", "type.col", "direction.col", "membrane.area.col",
                "receiver.vol.col", "donor.vol.col", "test.conc.col", "cal.col", "dilution.col",
                "time.col", "istd.name.col", "istd.conc.col", "test.nominal.conc.col", "biological.replicates.col",
                "technical.replicates.col", "analysis.method.col", "analysis.instrument.col", "analysis.parameters.col",
                "note.col", "level0.file.col", "level0.sheet.col", "output.res", "save.bad.types", "sig.figs", "INPUT.DIR", "OUTPUT.DIR")
req_cols[, "Argument"] <- arguments

# Default arguments 
defaults <- c("MYDATA", NA, "Lab.Sample.Name", "Lab.Compound.Name",
              "DTXSID", "Date", "Compound.Name", "Area", 
              "ISTD.Area", "Type", "Direction", "Membrane.Area",
              "Vol.Receiver", "Vol.Donor", "Test.Compound.Conc", 
              "Cal", "Dilution.Factor", "Time", "ISTD.Name", 
              "ISTD.Conc", "Test.Target.Conc",
              "Biological.Replicates", "Technical.Replicates",
              "Analysis.Method", "Analysis.Instrument", 
              "Analysis.Parameters", "Note", "Level0.File",
              "Level0.Sheet", "FALSE", "FALSE", "5", "NULL", "NULL")
req_cols[, "Default"] <- defaults 

# Argument required in L0?
req_cols <- req_cols %>% 
  mutate("Required in L0?" = case_when(
    Argument %in% c("sample.col", "lab.compound.col", "dtxsid.col", "compound.col", "area.col", "istd.col", 
  "type.col", "direction.col", "receiver.vol.col", "donor.vol.col") ~ "Y",
  Argument %in% c("FILENAME", "data.in", "output.res", "save.bad.types", "sig.figs", "INPUT.DIR", "OUTPUT.DIR") ~ "N/A",
  .default = "N"
  ))

# Corresponding single-entry Argument 
req_cols <- req_cols %>% 
  mutate("Corresp. single-entry Argument" = ifelse(.data[[vars[[3]]]] == "N" & .data[[vars[[1]]]] != "note.col", 
                                                    gsub(".col" ,"", Argument), NA)) 

# Brief description 
description <- c("Output and input filename",
                 "Level 0 data frame", 
                 "Lab sample name", 
                 "Lab test compound name (abbr.)",
                 "EPA's DSSTox Structure ID",
                 "Lab measurement date",
                 "Formal test compound name",
                 "Target analyte peak area",
                 "Internal standard peak area",
                 "Sample type (Blank/D0/D2/R2)",
                 "Experiment direction",
                 "Membrane area",
                 "Receiver compartment volume",
                 "Donor compartment volume",
                 "Standard test chemical concentration",
                 "MS calibration",
                 "Number of times sample was diluted",
                 "Time before compartments were measured",
                 "Internal standard name",
                 "Internal standard concentration",
                 "Expected initial chemical concentration added to donor",
                 "Replicates with the same analyte",
                 "Repeated measurements from one sample",
                 "Analytical chemistry analysis method",
                 "Analytical chemistry analysis instrument",
                 "Analytical chemistry analysis parameters",
                 "Additional notes",
                 "Raw data filename",
                 "Raw data sheet name",
                 "Export results (TSV)?",
                 "Export bad data (TSV)?",
                 "Number of significant figures",
                 "Input directory of Level 0 file",
                 "Export directory to save Level 1 files")
req_cols[, "Descr."] <- description

## ----echo = FALSE-------------------------------------------------------------
req_cols %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list(height = 200)
      )
  ) %>% 
  set_caption("Table 2: Level 1 'format_caco2' function arguments", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----L1 processing------------------------------------------------------------
caco2_L1_curated <- format_caco2(FILENAME = "Caco2_vignette",
                                 data.in = caco2_L0,
                                 # columns present in L0 data 
                                 sample.col = "Sample",
                                 lab.compound.col = "Lab.Compound.ID",
                                 compound.col = "Compound",
                                 area.col = "Peak.Area",
                                 istd.col = "ISTD.Peak.Area",
                                 test.conc.col = "Compound.Conc",
                                 analysis.method.col = "Analysis.Params",
                                 # columns not present in L0 data
                                 biological.replicates = 1,
                                 technical.replicates = 1,
                                 membrane.area = 0.11,
                                 cal = 1,
                                 time = 2, 
                                 istd.conc = 1,
                                 test.nominal.conc = 10,
                                 analysis.instrument = "Agilent.GCMS",
                                 analysis.parameters = "Unknown",
                                 note.col = NULL,
                                 # don't export output TSV file
                                 output.res = FALSE
                                 )

## ----eval = TRUE--------------------------------------------------------------
all(caco2_L1_curated[caco2_L1_curated$ISTD.Area == 0, "Sample.Type"] == "Blank")

## -----------------------------------------------------------------------------
# Verify Blank samples with ISTD.Area = 0 also have Response = 0
resp <- caco2_L1_curated %>% 
  dplyr::filter(Sample.Type == "Blank" & ISTD.Area == 0) %>% 
  dplyr::select(Response) %>% 
  unlist()

all(resp == 0)

## ----echo = FALSE-------------------------------------------------------------
# Select non-Blank sample type to display Response col 
caco2_L1_curated %>% 
  filter(!Sample.Type %in% c("Blank", "CC")) %>% 
head(n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 3: Level 1 data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## ----L2 processing exclusion--------------------------------------------------
# Use verification data from loaded in `caco2_L2` data frame 
exclusion <- caco2_L2 %>% 
  filter(Verified != "Y") %>% 
  mutate("Variables" = "Lab.Sample.Name|DTXSID") %>% 
  mutate("Values" = paste(Lab.Sample.Name, DTXSID, sep = "|")) %>% 
  mutate("Message" = Verified) %>% 
  select(Variables, Values, Message)

## ----echo = FALSE-------------------------------------------------------------
exclusion %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 4: Exclusion data frame", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## -----------------------------------------------------------------------------
caco2_L2_curated <- sample_verification(FILENAME = "Caco2_vignette",
                                        data.in = caco2_L1_curated,
                                        assay = "Caco-2",
                                        # don't export output TSV file
                                        output.res = FALSE)

## ----echo = FALSE-------------------------------------------------------------
caco2_L2_curated %>% 
  head(n = 3) %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 5: Level 2 data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

## -----------------------------------------------------------------------------
caco2_L3_curated <- calc_caco2_point(FILENAME = "Caco2_vignette",
                                     data.in = caco2_L2_curated, 
                                     # don't export output TSV file 
                                     output.res = FALSE
                                     )

## ----echo = FALSE-------------------------------------------------------------
caco2_L3_curated %>% 
  flextable() %>% 
  bg(bg = "#DDDDDD", part = "header") %>% 
  autofit() %>% 
  set_table_properties(
    opts_html = list(
      scroll = list()
      )
  ) %>% 
  set_caption("Table 6: Level 3 data", align_with_table = FALSE) %>% 
  fontsize(size = 10, part = "body") %>% 
  theme_vanilla()

