% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ADMMsigma.R
\name{ADMMsigma}
\alias{ADMMsigma}
\title{Penalized precision matrix estimation via ADMM}
\usage{
ADMMsigma(X = NULL, S = NULL, lam = 10^seq(-5, 5, 0.5), alpha = seq(0,
  1, 0.1), diagonal = FALSE, rho = 2, mu = 10, tau1 = 2, tau2 = 2,
  crit = "ADMM", tol1 = 1e-04, tol2 = 1e-04, maxit = 1000, K = 5,
  cores = 1, quiet = TRUE)
}
\arguments{
\item{X}{option to provide a nxp data matrix. Each row corresponds to a single observation and each column contains n observations of a single feature/variable.}

\item{S}{option to provide a pxp sample covariance matrix (denominator n). If argument is \code{NULL} and \code{X} is provided instead then \code{S} will be computed automatically.}

\item{lam}{tuning parameter for elastic net penalty. Defaults to grid of values \code{10^seq(-5, 5, 0.5)}.}

\item{alpha}{elastic net mixing parameter contained in [0, 1]. \code{0 = ridge, 1 = lasso}. Defaults to grid of values \code{seq(-1, 1, 0.1)}.}

\item{diagonal}{option to penalize the diagonal elements of the estimated precision matrix (\eqn{\Omega}). Defaults to \code{FALSE}.}

\item{rho}{initial step size for ADMM algorithm.}

\item{mu}{factor for primal and residual norms in the ADMM algorithm. This will be used to adjust the step size \code{rho} after each iteration.}

\item{tau1}{factor in which to increase step size \code{rho}}

\item{tau2}{factor in which to decrease step size \code{rho}}

\item{crit}{criterion for convergence (\code{ADMM}, \code{grad}, or \code{loglik}). If \code{crit != ADMM} then \code{tol1} will be used as the convergence tolerance. Default is \code{ADMM}.}

\item{tol1}{absolute convergence tolerance. Defaults to 1e-4.}

\item{tol2}{relative convergence tolerance. Defaults to 1e-4.}

\item{maxit}{maximum number of iterations.}

\item{K}{specify the number of folds for cross validation.}

\item{cores}{option to run CV in parallel. Defaults to \code{cores = 1}.}

\item{quiet}{specify whether the function returns progress of CV or not.}
}
\value{
returns class object \code{ADMMsigma} which includes:
\item{Iterations}{number of iterations}
\item{Tuning}{optimal tuning parameters (lam and alpha).}
\item{Lambdas}{grid of lambda values for CV.}
\item{Alphas}{grid of alpha values for CV.}
\item{maxit}{maximum number of iterations.}
\item{Omega}{estimated penalized precision matrix.}
\item{Sigma}{estimated covariance matrix from the penalized precision matrix (inverse of Omega).}
\item{Gradient}{gradient of optimization function (penalized gaussian likelihood).}
\item{CV.error}{cross validation errors.}
}
\description{
Penalized precision matrix estimation using the ADMM algorithm.
Consider the case where \eqn{X_{1}, ..., X_{n}} are iid \eqn{N_{p}(\mu,
\Sigma)} and we are tasked with estimating the precision matrix,
denoted \eqn{\Omega \equiv \Sigma^{-1}}. This function solves the
following optimization problem:
\describe{
\item{Objective:}{
\eqn{\hat{\Omega}_{\lambda} = \arg\min_{\Omega \in S_{+}^{p}}
\left\{ Tr\left(S\Omega\right) - \log \det\left(\Omega \right) +
\lambda\left[\frac{1 - \alpha}{2}\left\| \Omega \right|_{F}^{2} +
\alpha\left\| \Omega \right\|_{1} \right] \right\}}}
}
where \eqn{0 \leq \alpha \leq 1}, \eqn{\lambda > 0},
\eqn{\left\|\cdot \right\|_{F}^{2}} is the Frobenius norm and we define
\eqn{\left\|A \right\|_{1} = \sum_{i, j} \left| A_{ij} \right|}.
This elastic net penalty is identical to the penalty used in the popular penalized
regression package \code{glmnet}. Clearly, when \eqn{\alpha = 0} the elastic-net
reduces to a ridge-type penalty and when \eqn{\alpha = 1} it reduces to a
lasso-type penalty.
}
\details{
For details on the implementation of 'ADMMsigma', see the vignette
\url{https://mgallow.github.io/ADMMsigma/}.
}
\examples{
# generate data from a dense matrix
# first compute covariance matrix
S = matrix(0.9, nrow = 5, ncol = 5)
diag(S) = 1

# generate 100 x 5 matrix with rows drawn from iid N_p(0, S)
Z = matrix(rnorm(100*5), nrow = 100, ncol = 5)
out = eigen(S, symmetric = TRUE)
S.sqrt = out$vectors \%*\% diag(out$values^0.5)
S.sqrt = S.sqrt \%*\% t(out$vectors)
X = Z \%*\% S.sqrt

# elastic-net type penalty (use CV for optimal lambda and alpha)
ADMMsigma(X)

# ridge penalty (use CV for optimal lambda)
ADMMsigma(X, alpha = 0)

# lasso penalty (lam = 0.1)
ADMMsigma(X, lam = 0.1, alpha = 1)

# produce CV heat map for ADMMsigma
plot(ADMMsigma(X))
}
\references{
\itemize{
\item 
For more information on the ADMM algorithm, see: \cr
Boyd, Stephen, Neal Parikh, Eric Chu, Borja Peleato, Jonathan Eckstein, and others. 2011. 'Distributed Optimization and Statistical Learning via the Alternating Direction Method of Multipliers.' \emph{Foundations and Trends in Machine Learning} 3 (1). Now Publishers, Inc.: 1-122.\cr
\url{https://web.stanford.edu/~boyd/papers/pdf/admm_distr_stats.pdf}
}
}
\seealso{
\code{\link{plot.ADMMsigma}}, \code{\link{RIDGEsigma}}
}
\author{
Matt Galloway \email{gall0441@umn.edu}
}
