% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bactid.R
\name{as.bactid}
\alias{as.bactid}
\alias{guess_bactid}
\alias{is.bactid}
\title{Transform to bacteria ID}
\source{
[1] Becker K \emph{et al.} \strong{Coagulase-Negative Staphylococci}. 2014. Clin Microbiol Rev. 27(4): 870–926. \cr
    \url{https://dx.doi.org/10.1128/CMR.00109-13} \cr
[2] Lancefield RC \strong{A serological differentiation of human and other groups of hemolytic streptococci}. 1933. J Exp Med. 57(4): 571–95. \cr
    \url{https://dx.doi.org/10.1084/jem.57.4.571}
}
\usage{
as.bactid(x, Becker = FALSE, Lancefield = FALSE)

guess_bactid(x, Becker = FALSE, Lancefield = FALSE)

is.bactid(x)
}
\arguments{
\item{x}{a character vector or a dataframe with one or two columns}

\item{Becker}{a logical to indicate whether \emph{Staphylococci} should be categorised into Coagulase Negative \emph{Staphylococci} ("CoNS") and Coagulase Positive \emph{Staphylococci} ("CoPS") instead of their own species, according to Karsten Becker \emph{et al.} [1]. This excludes \emph{Staphylococcus aureus} at default, use \code{Becker = "all"} to also categorise \emph{S. aureus} as "CoPS".}

\item{Lancefield}{a logical to indicate whether beta-haemolytic \emph{Streptococci} should be categorised into Lancefield groups instead of their own species, according to Rebecca C. Lancefield [2]. These \emph{Streptococci} will be categorised in their first group, i.e. \emph{Streptococcus dysgalactiae} will be group C, although officially it was also categorised into groups G and L. Groups D and E will be ignored, since they are \emph{Enterococci}.}
}
\value{
Character (vector) with class \code{"bactid"}. Unknown values will return \code{NA}.
}
\description{
Use this function to determine a valid ID based on a genus (and species). This input can be a full name (like \code{"Staphylococcus aureus"}), an abbreviated name (like \code{"S. aureus"}), or just a genus. You could also \code{\link{select}} a genus and species column, zie Examples.
}
\details{
\code{guess_bactid} is an alias of \code{as.bactid}.

Some exceptions have been built in to get more logical results, based on prevalence of human pathogens. These are:
\itemize{
  \item{\code{"E. coli"} will return the ID of \emph{Escherichia coli} and not \emph{Entamoeba coli}, although the latter would alphabetically come first}
  \item{\code{"H. influenzae"} will return the ID of \emph{Haemophilus influenzae} and not \emph{Haematobacter influenzae}}
  \item{Something like \code{"p aer"} will return the ID of \emph{Pseudomonas aeruginosa} and not \emph{Pasteurella aerogenes}}
  \item{Something like \code{"stau"} or \code{"staaur"} will return the ID of \emph{Staphylococcus aureus} and not \emph{Staphylococcus auricularis}}
}
Moreover, this function also supports ID's based on only Gram stain, when the species is not known. \cr
For example, \code{"Gram negative rods"} and \code{"GNR"} will both return the ID of a Gram negative rod: \code{GNR}.
}
\examples{
# These examples all return "STAAUR", the ID of S. aureus:
as.bactid("stau")
as.bactid("STAU")
as.bactid("staaur")
as.bactid("S. aureus")
as.bactid("S aureus")
as.bactid("Staphylococcus aureus")
as.bactid("MRSA") # Methicillin Resistant S. aureus
as.bactid("VISA") # Vancomycin Intermediate S. aureus
as.bactid("VRSA") # Vancomycin Resistant S. aureus

guess_bactid("S. epidermidis")                 # will remain species: STAEPI
guess_bactid("S. epidermidis", Becker = TRUE)  # will not remain species: STACNS

guess_bactid("S. pyogenes")                    # will remain species: STCAGA
guess_bactid("S. pyogenes", Lancefield = TRUE) # will not remain species: STCGRA

\dontrun{
df$bactid <- as.bactid(df$microorganism_name)

# the select function of tidyverse is also supported:
library(dplyr)
df$bactid <- df \%>\%
  select(microorganism_name) \%>\%
  guess_bactid()

# and can even contain 2 columns, which is convenient for genus/species combinations:
df$bactid <- df \%>\%
  select(genus, species) \%>\%
  guess_bactid()

# same result:
df <- df \%>\%
  mutate(bactid = guess_bactid(paste(genus, species)))
}
}
\seealso{
\code{\link{microorganisms}} for the dataframe that is being used to determine ID's.
}
\keyword{Becker}
\keyword{Lancefield}
\keyword{bactid}
\keyword{becker}
\keyword{guess}
\keyword{lancefield}
