% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multipliers.R
\name{multipliers}
\alias{multipliers}
\alias{multipliers.ardl}
\alias{multipliers.uecm}
\title{Multipliers estimation}
\usage{
multipliers(object, type = "lr", vcov_matrix = NULL)

\method{multipliers}{ardl}(object, type = "lr", vcov_matrix = NULL)

\method{multipliers}{uecm}(object, type = "lr", vcov_matrix = NULL)
}
\arguments{
\item{object}{An object of \code{\link[base]{class}} 'ardl' or 'uecm'.}

\item{type}{A character string describing the type of multipliers. Use "lr"
for long-run (total) multipliers (default), "sr" or 0 for short-run (impact)
multipliers or an integer between 1 and 200 for delay and interim multipliers.}

\item{vcov_matrix}{The estimated covariance matrix of the random variable
that the transformation function uses to estimate the standard errors (and
so the t-statistics and p-values) of the multipliers. The default is
\code{vcov(object)} (when \code{vcov_matrix = NULL}), but other estimations
of the covariance matrix of the regression's estimated coefficients can
also be used (e.g., using \code{\link[sandwich]{vcovHC}} or
\code{\link[sandwich]{vcovHAC}}).}
}
\value{
\code{multipliers} returns (for long and short run multipliers) a
  data.frame containing the independent variables (including possibly
  existing intercept or trend and excluding the fixed variables) and their
  corresponding standard errors, t-statistics and p-values. For delay and
  interim multipliers it returns a list with a data.frame for each variable,
  containing the delay and interim multipliers for each period.
}
\description{
\code{multipliers} is a generic function used to estimate short-run (impact),
delay, interim and long-run (total) multipliers. Long and short run
multipliers are accompanied by their corresponding standard errors,
t-statistics and p-values.
}
\details{
The function invokes two different \code{\link[utils]{methods}}, one for
objects of \code{\link[base]{class}} 'ardl' and one for objects of
\code{class} 'uecm'. This is because of the different (but equivalent)
transformation functions that are used for each class/model ('ardl' and
'uecm') to estimate the multipliers.

\code{type = 0} is equivalent to \code{type = "sr"}.

Note that the interim multipliers are the cumulative sum of the delays, and
that the sum of the interim multipliers (for long enough periods) and thus
a distant enough interim multiplier match the long-run multipliers.

The delay (interim) multiplier can be interpreted as the effect in period
t+s, of an instant (sustained) shock in period t.

The delta method is used for approximating the standard errors (and thus the
t-statistics and p-values) of the estimated long-run multipliers.
}
\section{Mathematical Formula}{


\strong{Short-Run Multipliers:}
\describe{
  \item{As derived from an ARDL:}{}
}
\deqn{\frac{\partial y_{t}}{\partial x_{j,t}} = b_{j,0} \;\;\;\;\; j \in \{1,\dots,k\}}

\describe{
  \item{As derived from an Unrestricted ECM:}{}
}
\deqn{\frac{\partial y_{t}}{\partial x_{j,t}} = \omega_{j} \;\;\;\;\; j \in \{1,\dots,k\}}

\describe{
  \item{Constant and Linear Trend:}{}
}
\deqn{c_{0}}
\deqn{c_{1}}

\strong{Delay & Interim Multipliers:}
\describe{
  \item{As derived from an ARDL:}{}
}
\deqn{Delay_{x_{j},s} = \frac{\partial y_{t+s}}{\partial x_{j,t}} = b_{j,s} + \sum_{i=1}^{min\{p,s\}} b_{y,i} \frac{\partial y_{t+(s-i)}}{\partial x_{j,t}} \;\;\;\;\; b_{j,s} = 0 \;\; \forall \;\; s > q}
\deqn{Interim_{x_{j},s} = \sum_{i=0}^{s} Delay_{x_{j},s}}

\describe{
  \item{Constant and Linear Trend:}{}
}
\deqn{Delay_{intercept,s} = c_{0} + \sum_{i=1}^{min\{p,s\}} b_{y,i} Delay_{intercept,s-i} \;\;\;\;\; c_{0} = 0 \;\; \forall \;\; s \neq 0}
\deqn{Interim_{intercept,s} = \sum_{i=0}^{s} Delay_{intercept,s}}
\deqn{Delay_{trend,s} = c_{1} + \sum_{i=1}^{min\{p,s\}} b_{y,i} Delay_{trend,s-i} \;\;\;\;\; c_{1} = 0 \;\; \forall \;\; s \neq 0}
\deqn{Interim_{trend,s} = \sum_{i=0}^{s} Delay_{trend,s}}

\strong{Long-Run Multipliers:}
\describe{
  \item{As derived from an ARDL:}{}
}
\deqn{\frac{\partial y_{t+\infty}}{\partial x_{j,t}} = \theta_{j} = \frac{\sum_{l=0}^{q_{j}}b_{j,l}}{1-\sum_{i=1}^{p}b_{y,i}} \;\;\;\;\; j \in \{1,\dots,k\}}
\describe{
  \item{Constant and Linear Trend:}{}
}
\deqn{\mu = \frac{c_{0}}{1-\sum_{i=1}^{p}b_{y,i}}}
\deqn{\delta = \frac{c_{1}}{1-\sum_{i=1}^{p}b_{y,i}}}

\describe{
  \item{As derived from an Unrestricted ECM:}{}
}
\deqn{\frac{\partial y_{t+\infty}}{\partial x_{j,t}} = \theta_{j} = \frac{\pi_{j}}{-\pi_{y}} \;\;\;\;\; j \in \{1,\dots,k\}}
\describe{
  \item{Constant and Linear Trend:}{}
}
\deqn{\mu = \frac{c_{0}}{-\pi_{y}}}
\deqn{\delta = \frac{c_{1}}{-\pi_{y}}}
}

\examples{
data(denmark)

## Estimate the long-run multipliers of an ARDL(3,1,3,2) model ---------

# From an ARDL model
ardl_3132 <- ardl(LRM ~ LRY + IBO + IDE, data = denmark, order = c(3,1,3,2))
mult_ardl <- multipliers(ardl_3132)
mult_ardl

# From an UECM
uecm_3132 <- uecm(ardl_3132)
mult_uecm <- multipliers(uecm_3132)
mult_uecm

all.equal(mult_ardl, mult_uecm)


## Estimate the short-run multipliers of an ARDL(3,1,3,2) model --------

mult_sr <- multipliers(uecm_3132, type = "sr")
mult_0 <- multipliers(uecm_3132, type = 0)
all.equal(mult_sr, mult_0)


## Estimate the delay & interim multipliers of an ARDL(3,1,3,2) model --

mult_lr <- multipliers(uecm_3132, type = "lr")
mult_inter80 <- multipliers(uecm_3132, type = 80)

mult_lr
sum(mult_inter80$`(Intercept)`$Delay)
mult_inter80$`(Intercept)`$Interim[nrow(mult_inter80$`(Intercept)`)]
sum(mult_inter80$LRY$Delay)
mult_inter80$LRY$Interim[nrow(mult_inter80$LRY)]
sum(mult_inter80$IBO$Delay)
mult_inter80$IBO$Interim[nrow(mult_inter80$IBO)]
sum(mult_inter80$IDE$Delay)
mult_inter80$IDE$Interim[nrow(mult_inter80$IDE)]
plot(mult_inter80$LRY$Delay, type='l')
plot(mult_inter80$LRY$Interim, type='l')
}
\seealso{
\code{\link{ardl}}, \code{\link{uecm}}
}
\author{
Kleanthis Natsiopoulos, \email{klnatsio@gmail.com}
}
\keyword{math}
