% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generic_support.r
\name{index.colours}
\alias{index.colours}
\title{Quantisation of colours and indexing a grDevices raster image}
\usage{
index.colours(x, length.out = 8, palette = NULL, background = "#FFFFFF",
  dither = c("none", "floyd-steinberg", "JJN", "stucki", "atkinson", "burkse",
  "sierra", "two-row-sierra", "sierra-lite"), colour.depth = c("12 bit",
  "24 bit"), ...)
}
\arguments{
\item{x}{A raster object (\code{\link[grDevices]{as.raster}}), or a \code{matrix}
containing \code{character} strings representing colours. \code{x} can also
be a \code{list} of such matrices or rasters. All elements of this list should
have identical dimensions. An overall palette will be generated for elements in the
list.}

\item{length.out}{A \code{numeric} value indicating the number of desired
colours in the indexed palette.}

\item{palette}{A vector of no more than \code{length.out} colours, to be used
for the bitmap image. When missing or set to \code{NULL}, a palette will be
generated based on the provided colours in raster \code{x}. In that case,
\code{\link[stats]{kmeans}} is used on the hue, saturation, brightness and
alpha values of the colours in \code{x} for clustering the colours. The cluster
centres will be used as palette colours.}

\item{background}{On the Amiga, indexed images could not be semi-transparent.
Only a single colour could be designated as being fully transparent. The
`\code{background}' argument should contain a background colour  with which
semi-transparent colours should be mixed, before colour quantisation. It is
white by default.}

\item{dither}{Dither the output image using the algorithm specified here.
See the usage section for possible options. By default no dithering ("\code{none}")
is applied. See \code{\link{dither}} for more details.}

\item{colour.depth}{A \code{character} string indicating the colour depth to be used.
Can be either "\code{12 bit}" (default, standard on an Amiga with original chipset),
or "\code{24 bit}".}

\item{...}{Arguments that are passed onto \code{\link[stats]{kmeans}} (see
\code{palette} argument).}
}
\value{
Returns a \code{matrix} with the same dimensions as \code{x} containing
\code{numeric} index values. The corresponding palette is returned as attribute,
as well as the index value for the fully transparent colour in the palette.
When \code{x} is a \code{list} a \code{list} of matrices is returned.
}
\description{
Converts an image represented by a grDevices \code{raster} object into a
matrix containing numeric indices of a quantised colour palette.
}
\details{
Determines the optimal limited palette by clustering colours in an image
with \code{\link[stats]{kmeans}}. The result of the optimisation routine
will depend on the randomly chosen cluster centres by this algorithm. This
means that the result may slightly differ for each call to this function. If
you want reproducible results, you may want to reset the random seed
(\code{\link{set.seed}}) before each call to this function.
}
\examples{
\dontrun{
## first: Let's make a raster out of the 'volcano' data, which we can use in the example:
volcano.raster <- as.raster(t(matrix(terrain.colors(1 + diff(range(volcano)))[volcano -
  min(volcano) + 1], nrow(volcano))))

## This will create an image of the original raster using an indexed palette:
volcano.index <- index.colours(volcano.raster)

## The index values can be converted back into colours, using the palette:
volcano.index <- as.raster(apply(volcano.index, 2,
                                 function(x) attributes(volcano.index)$palette[x]))

## Create an indexed image using dithering
volcano.dith <- index.colours(volcano.raster, dither = "floyd-steinberg")
volcano.dith <- as.raster(apply(volcano.dith, 2,
                                function(x) attributes(volcano.dith)$palette[x]))

## plot the images side by side for comparison
par(mfcol = c(1, 3))
plot(volcano.raster, interpolate = F)
plot(volcano.index, interpolate = F)
plot(volcano.dith, interpolate = F)
}
}
\seealso{
Other colour.quantisation.operations: \code{\link{dither}}

Other raster.operations: \code{\link{as.raster}},
  \code{\link{bitmapToRaster}}, \code{\link{dither}},
  \code{\link{rasterToBitmap}}, \code{\link{rasterToIFF}}
}
\author{
Pepijn de Vries
}
