% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bd.R
\name{bd.test}
\alias{bd.test}
\alias{bd.test.default}
\alias{bd.test.formula}
\title{Ball Divergence based Equality of Distributions Test}
\usage{
bd.test(x, ...)

\method{bd.test}{default}(x, y = NULL, num.permutations = 99,
  distance = FALSE, size = NULL, seed = 4, num.threads = 1,
  kbd.type = "sum", ...)

\method{bd.test}{formula}(formula, data, subset, na.action, ...)
}
\arguments{
\item{x}{a numeric vector, matrix, data.frame, \code{dist} object or list containing vector, matrix, or data.frame.}

\item{...}{further arguments to be passed to or from methods.}

\item{y}{a numeric vector, matrix or data.frame.}

\item{num.permutations}{the number of permutation replications, when \code{num.permutations} equals to 0, the function returns
the sample version of ball divergence. Default: \code{num.permutations = 99}}

\item{distance}{if \code{distance = TRUE}, \code{x} will be considered as a distance matrix. Default: \code{distance = FALSE}}

\item{size}{a vector record sample size of each group.}

\item{seed}{the random seed.}

\item{num.threads}{Number of threads. Default \code{num.threads = 1}.}

\item{kbd.type}{a character value controlling the output information. 
Setting \code{kdb.type = "sum"}, \code{kdb.type = "summax"}, or \code{kdb.type = "max"}, 
the corresponding statistics value and \eqn{p}-value of \eqn{K}-sample test procedure are demonstrated. 
Note that this arguments actually only influences the printed result in R console. Default: \code{kdb.type = "sum"}}

\item{formula}{a formula of the form \code{response ~ group} where \code{response} gives the data values and \code{group} a vector or factor of the corresponding groups.}

\item{data}{an optional matrix or data frame (or similar: see \code{model.frame}) containing the variables in the formula \code{formula}. By default the variables are taken from \code{environment(formula)}.}

\item{subset}{an optional vector specifying a subset of observations to be used.}

\item{na.action}{a function which indicates what should happen when the data contain \code{NA}s. Defaults to \code{getOption("na.action")}.}
}
\value{
bd.test returns a list with class "htest" containing the following components:
\item{\code{statistic}}{ball divergence statistic.}            
\item{\code{p.value}}{the p-value for the test.}
\item{\code{replicates}}{permutation replications of the test statistic.}
\item{\code{size}}{sample sizes.}
\item{\code{complete.info}}{a \code{list} containing multiple statistics value and their corresponding $p$ value.}
\item{\code{alternative}}{a character string describing the alternative hypothesis.}
\item{\code{method}}{a character string indicating what type of test was performed.}
\item{\code{data.name}}{description of data.}
}
\description{
Performs the nonparametric two-sample or \eqn{K}-sample ball divergence test for
equality of multivariate distributions
}
\details{
\code{bd.test} are ball divergence based multivariate nonparametric tests of two-sample or 
K-sample problem. If only \code{x} is given, the statistic is 
computed from the original pooled samples, stacked in 
matrix where each row is a multivariate observation, or from the distance matrix 
when \code{distance = TRUE}. The first \code{sizes[1]} rows of \code{x} are the first sample, the next 
\code{sizes[2]} rows of \code{x} are the second sample, etc.
If \code{x} is a \code{list}, its elements are taken as the samples to be compared, 
and hence have to be numeric data vectors, matrix or data.frame.

Based on sample version ball divergence (see \code{\link{bd}}), the test is implemented by 
permutation with \code{num.permutations} times. The function simply returns the test statistic 
when \code{num.permutations = 0}.
}
\examples{
################# Quick Start #################
x <- rnorm(50)
y <- rnorm(50, mean = 1)
# plot(density(x))
# lines(density(y), col = "red")
# ball divergence:
bd.test(x = x, y = y)

################# Quick Start #################
x <- matrix(rnorm(100), nrow = 50, ncol = 2)
y <- matrix(rnorm(100, mean = 3), nrow = 50, ncol = 2)
# Hypothesis test with Standard Ball Divergence:
bd.test(x = x, y = y)

################# Simlated Non-Hilbert data #################
data("bdvmf")
\dontrun{
library(scatterplot3d)
scatterplot3d(bdvmf[["x"]], color = bdvmf[["group"]], 
              xlab = "X1", ylab = "X2", zlab = "X3")
}
# calculate geodesic distance between sample:
Dmat <- nhdist(bdvmf[["x"]], method = "geodesic")
# hypothesis test with BD :
bd.test(x = Dmat, size = c(150, 150), num.permutations = 99, distance = TRUE)

################# Non-Hilbert Real Data #################
# load data:
data("macaques")
# number of femala and male Macaca fascicularis:
table(macaques[["group"]])
# calculate Riemannian shape distance matrix:
Dmat <- nhdist(macaques[["x"]], method = "riemann")
# hypothesis test with BD:
bd.test(x = Dmat, num.permutations = 99, size = c(9, 9), distance = TRUE)

################  K-sample Test  #################
n <- 150
bd.test(rnorm(n), size = c(40, 50, 60))
# alternative input method:
x <- lapply(c(40, 50, 60), rnorm)
bd.test(x)

################  Formula interface  ################
## Two-sample test
bd.test(extra ~ group, data = sleep)
## K-sample test
bd.test(Sepal.Width ~ Species, data = iris)
}
\references{
Pan, Wenliang; Tian, Yuan; Wang, Xueqin; Zhang, Heping. Ball Divergence: Nonparametric two sample test. Ann. Statist. 46 (2018), no. 3, 1109--1137. doi:10.1214/17-AOS1579. https://projecteuclid.org/euclid.aos/1525313077

Jin, Zhu, Wenliang Pan, Wei Zheng, and Xueqin Wang (2018). Ball: An R package for detecting distribution difference and association in metric spaces. arXiv preprint arXiv:1811.03750. URL http://arxiv.org/abs/1811.03750.
}
\seealso{
\code{\link{bd}}
}
\author{
Wenliang Pan, Yuan Tian, Xueqin Wang, Heping Zhang
}
