#' Consolidate results from partial files
#'
#' Consolidates results from a set of partial files (each generated by an
#' individual call to [calc_nreps2()]) into a single output structure, similar
#' (but not identical) to the output of [run_experiment()].
#'
#' @param folder folder where the partial files are located.
#'
#' @return a list object containing the following fields:
#' \itemize{
#'    \item \code{data.raw} - data frame containing all observations generated
#'    \item \code{data.summary} - data frame summarizing the experiment.
#'    \item \code{N} - number of instances sampled
#'    \item \code{instances.sampled} - names of the instances sampled
#' }
#'
#' @export
#'
consolidate.partial.results <- function(folder = "./nreps_files")
{

  # Error checking
  assertthat::assert_that(is.character(folder),
                          length(folder) == 1)

  # Get filenames from folder
  filenames <- dir(path = folder, pattern = ".rds")

  # Read results from files
  my.results <- vector(mode = "list", length = length(filenames))
  for (i in seq(length(filenames))){
    my.results[[i]] <- readRDS(file = paste0(folder, "/", filenames[i]))
  }

  # Consolidate raw results
  data.raw <- lapply(X   = my.results,
                     FUN = function(res_j){
                       inst  <- res_j$instance
                       nj    <- res_j$n1j + res_j$n2j
                       data.frame(Algorithm   = c(rep("algo1", res_j$n1j),
                                                  rep("algo2", res_j$n2j)),
                                  Instance    = rep(inst, nj),
                                  Observation = c(res_j$x1j, res_j$x2j),
                                  stringsAsFactors = FALSE)})

  data.raw <- do.call(rbind, data.raw)

  # Consolidate summary data
  data.summary <- lapply(X   = my.results,
                         FUN = function(res_j){
                           data.frame(Instance = res_j$instance,
                                      phi.j    = res_j$phi.est,
                                      std.err  = res_j$se,
                                      n1j      = res_j$n1j,
                                      n2j      = res_j$n2j,
                                      stringsAsFactors = FALSE)})

  data.summary <- do.call(rbind, data.summary)

  # Assemble output
  output <- list(data.raw          = data.raw,
                 data.summary      = data.summary,
                 N                 = length(filenames),
                 instances.sampled = unique(data.summary$Instance))

  # Return output
  return(output)
}
