% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/modelKriging.R
\name{combinatorialKriging}
\alias{combinatorialKriging}
\title{Combinatorial Kriging}
\usage{
combinatorialKriging(x, y, distanceFunction, control = list())
}
\arguments{
\item{x}{list of samples in input space}

\item{y}{column vector of observations for each sample}

\item{distanceFunction}{a suitable distance function of type f(x1,x2), returning a scalar distance value, preferably between 0 and 1.
Maximum distances larger 1 are no problem, but may yield scaling bias when different measures are compared.
   Should be non-negative and symmetric.  It can also be a list of several distance functions. In this case, Maximum Likelihood Estimation (MLE) is used
   to determine the most suited distance measure.}

\item{control}{(list), with the options for the model building procedure:\cr
\code{lower} lower boundary for theta, default is \code{1e-6}\cr
\code{upper} upper boundary for theta, default is \code{100}\cr
\code{corr} function to be used for correlation modelling, default is \code{fcorrGauss}\cr
\code{algTheta}  algorithm used to find theta (as well as p and lambda), default is \code{"L-BFGS-B"}. Else, any from the list of possible \code{method} values in \code{\link{optimInterface}} can be chosen.\cr
\code{budgetAlgTheta} budget for the above mentioned algorithm, default is \code{100}. The value will be multiplied with the length of the model parameter vector to be optimized.
\code{optimizeP} boolean that specifies whether the exponents (\code{p}) should be optimized. Else they will be set to two. \cr
\code{useLambda} whether or not to use the regularization constant lambda (nugget effect). Default is \code{FALSE}.
\code{lambdaLower} lower boundary for lambda, default is \code{-6}\cr
\code{lambdaUpper} upper boundary for lambda, default is \code{0}\cr
\code{distances} a distance matrix. If available, this matrix is used for model building, instead of calculating the distance matrix using the parameters \code{distanceFunction}. Default is \code{NULL}.}
}
\value{
an object of class \code{CKriging} containing the options and determined parameters for the model:\cr
\code{x} (see parameters)\cr
\code{y} (see parameters)\cr
\code{lower} (see parameters)\cr
\code{upper} (see parameters)\cr
\code{algTheta} (see parameters)\cr
\code{budgetAlgTheta} (see parameters)\cr
\code{optimizeP} (see parameters)\cr
\code{theta} activity or width parameter theta, a parameter of the correlation function determined with MLE\cr
\code{log10Theta} log10 \code{theta} (i.e. \code{log10(theta)})\cr
\code{lambda} regularization constant (nugget) lambda \cr
\code{log10Lambda} log10 of regularization constant (nugget) lambda (i.e. \code{log10(lambda)})\cr
\code{p} exponent p, parameter of the correlation function determined with MLE (if \code{optimizeP} is \code{TRUE})\cr
\code{yMu} vector of observations y, minus MLE of mu\cr
\code{SSQ} Maximum Likelihood Estimate (MLE) of model parameter sigma^2\cr
\code{mu} MLE of model parameter mu\cr
\code{Psi} correlation matrix Psi\cr
\code{Psinv} inverse of Psi\cr
\code{nevals} number of Likelihood evaluations during MLE of theta/lambda/p\cr
\code{distanceFunctionIndexMLE} If a list of several distance measures (\code{distanceFunction}) was given, this parameter contains the index value of the measure chosen with MLE.
}
\description{
Implementation of a Kriging model for mixed or combinatorial input spaces.
It is based on employing suitable distance measures for the samples in input space.
}
\details{
The basic Kriging implementation is based on the work of Forrester et al. (2008).
For adaptation of Kriging to mixed or combinatorial spaces, as well as
choosing distance measures with Maximum Likelihood Estimation, see the other two references (Zaefferer et al., 2014).
}
\examples{
#set random number generator seed
set.seed(1)
#simple test landscape
fn <- landscapeGeneratorUNI(1:5,distancePermutationHamming)
#generate data for training and test
x <- unique(replicate(40,sample(5),FALSE))
xtest <- x[-(1:15)]
x <- x[1:15]
#determin true objective function values
y <- sapply(x,fn)
ytest <- sapply(xtest,fn)
#build model
fit <- combinatorialKriging(x,y,distancePermutationHamming,
    control=list(algTheta="L-BFGS-B",useLambda=FALSE))
#predicted obj. function values
ypred <- predict(fit,xtest)$f
#uncertainty estimate
spred <- predict(fit,xtest,TRUE)$s
#plot
plot(ytest,ypred,xlab="true value",ylab="predicted value",
    pch=20,xlim=c(0.3,1),ylim=c(min(ypred)-0.1,max(ypred)+0.1))
segments(ytest, ypred-spred,ytest, ypred+spred)
epsilon = 0.02
segments(ytest-epsilon,ypred-spred,ytest+epsilon,ypred-spred)
segments(ytest-epsilon,ypred+spred,ytest+epsilon,ypred+spred)
abline(0,1,lty=2)
}
\references{
Forrester, Alexander I.J.; Sobester, Andras; Keane, Andy J. (2008). Engineering Design via Surrogate Modelling - A Practical Guide. John Wiley & Sons.

Zaefferer, Martin; Stork, Joerg; Friese, Martina; Fischbach, Andreas; Naujoks, Boris; Bartz-Beielstein, Thomas. (2014). Efficient global optimization for combinatorial problems. In Proceedings of the 2014 conference on Genetic and evolutionary computation (GECCO '14). ACM, New York, NY, USA, 871-878. DOI=10.1145/2576768.2598282 http://doi.acm.org/10.1145/2576768.2598282

Zaefferer, Martin; Stork, Joerg; Bartz-Beielstein, Thomas. (2014). Distance Measures for Permutations in Combinatorial Efficient Global Optimization. In Parallel Problem Solving from Nature - PPSN XIII (p. 373-383). Springer International Publishing.
}
\seealso{
\code{\link{predict.CKriging}}
}

