% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/impute.R
\name{Impute.data.frame}
\alias{Impute.data.frame}
\title{Impute a data frame}
\usage{
\method{Impute}{data.frame}(
  x,
  f_i = NULL,
  f_i_para = NULL,
  impute_by = "column",
  normalise_first = NULL,
  directions = NULL,
  ...
)
}
\arguments{
\item{x}{A data frame with only numeric columns.}

\item{f_i}{A function to use for imputation. By default, imputation is performed by simply substituting
the mean of non-\code{NA} values for each column at a time.}

\item{f_i_para}{Any additional parameters to pass to \code{f_i}, apart from \code{x}}

\item{impute_by}{Specifies how to impute: if \code{"column"}, passes each column separately as a numerical
vector to \code{f_i}; if \code{"row"}, passes each \emph{row} separately; and if \code{"df"} passes the entire data frame to
\code{f_i}. The function called by \code{f_i} should be compatible with the type of data passed to it.}

\item{normalise_first}{Logical: if \code{TRUE}, each column is normalised using a min-max operation before
imputation. By default this is \code{FALSE} unless \code{impute_by = "row"}. See details.}

\item{directions}{A vector of directions: either -1 or 1 to indicate the direction of each column
of \code{x} - this is only used if \code{normalise_first = TRUE}. See details.}

\item{...}{arguments passed to or from other methods.}
}
\value{
An imputed data frame
}
\description{
Impute a data frame using any function, either column-wise, row-wise or by the whole data frame in one
shot.
}
\details{
This function only accepts data frames with all numeric columns. It imputes any \code{NA}s in the data frame
by invoking the function \code{f_i} and any optional arguments \code{f_i_para} on each column at a time (if
\code{impute_by = "column"}), or on each row at a time (if \code{impute_by = "row"}), or by passing the entire
data frame to \code{f_i} if \code{impute_by = "df"}.

Clearly, the function \code{f_i} needs to be able to accept with the data class passed to it - if
\code{impute_by} is \code{"row"} or \code{"column"} this will be a numeric vector, or if \code{"df"} it will be a data
frame. Moreover, this function should return a vector or data frame identical to the vector/data frame passed to
it except for \code{NA} values, which can be replaced. The function \code{f_i} is not required to replace \emph{all} \code{NA}
values.

When imputing row-wise, prior normalisation of the data is recommended. This is because imputation
will use e.g. the mean of the unit values over all indicators (columns). If the indicators are on
very different scales, the result will likely make no sense. If the indicators are normalised first,
more sensible results can be obtained. There are two options to pre-normalise: first is by setting
\code{normalise_first = TRUE} - this is anyway the default if \code{impute_by = "row"}. In this case, you also
need to supply a vector of directions. The data will then be normalised using a min-max approach
before imputation, followed by the inverse operation to return the data to the original scales.

Another approach which gives more control is to simply run \code{\link[=Normalise]{Normalise()}} first, and work with the
normalised data from that point onwards. In that case it is better to set \code{normalise_first = FALSE},
since by default if \code{impute_by = "row"} it will be set to \code{TRUE}.

Checks are made on the format of the data returned by imputation functions, to ensure the
type and that non-\code{NA} values have not been inadvertently altered. This latter check is allowed
a degree of tolerance for numerical precision, controlled by the \code{sfigs} argument. This is because
if the data frame is normalised, and/or depending on the imputation function, there may be a very
small differences. By default \code{sfigs = 9}, meaning that the non-\code{NA} values pre and post-imputation
are compared to 9 significant figures.
}
\examples{
# a df of random numbers
X <- as.data.frame(matrix(runif(50), 10, 5))

# introduce NAs (2 in 3 of 5 cols)
X[sample(1:10, 2), 1] <- NA
X[sample(1:10, 2), 3] <- NA
X[sample(1:10, 2), 5] <- NA

# impute using column mean
Impute(X, f_i = "i_mean")

# impute using row median (no normalisation)
Impute(X, f_i = "i_median", impute_by = "row",
       normalise_first = FALSE)


}
