% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cste_surv.R
\name{cste_surv}
\alias{cste_surv}
\title{Estimate the CSTE curve for time to event outcome with right censoring.}
\usage{
cste_surv(
  x,
  y,
  z,
  status = NULL,
  beta_ini = NULL,
  lam = 0,
  nknots = 2,
  max.iter = 200,
  eps = 0.001
)
}
\arguments{
\item{x}{samples of covariates which is a \eqn{n*p} matrix.}

\item{y}{samples of time to event which is a \eqn{n*1} vector.}

\item{z}{samples of treatment indicator which is a \eqn{n*1} vector.}

\item{status}{samples of censoring indicator which is a \eqn{n*1} vector, default value is NULL, indicating no censoring.}

\item{beta_ini}{initial values for \eqn{(\beta_1', \beta_2')'}, default value is NULL.}

\item{lam}{value of the lasso penalty parameter \eqn{\lambda} for \eqn{\beta_1} and
\eqn{\beta_2}, default value is 0.}

\item{nknots}{number of knots for the B-spline for estimating \eqn{g_1} and \eqn{g_2}.}

\item{max.iter}{maximum iteration for the algorithm.}

\item{eps}{numeric scalar \eqn{\geq} 0, the tolerance for the estimation of \eqn{\beta_1} and \eqn{\beta_2}.}
}
\value{
A S3 class of cste, which includes:
\itemize{
   \item \code{beta1}: estimate of \eqn{\beta_1}.
   \item \code{beta2}: estimate of \eqn{\beta_2}.
   \item \code{B1}: the B-spline basis for estimating \eqn{g_1}.
   \item \code{B2}: the B-spline basis for estimating \eqn{g_2}. 
   \item \code{delta1}: the coefficient of B-spline for estimating \eqn{g_1}.
   \item \code{delta2}: the coefficient for B-spline for estimating \eqn{g_2}.
   \item \code{iter}: number of iteration.
   \item \code{g1}: the estimate for \eqn{g_1(X\beta_1)}.
   \item \code{g2}: the estimate for \eqn{g_2(X\beta_2)}. 
}
}
\description{
Estimate the CSTE curve for time to event outcome with right censoring. 
The working model 
is \deqn{\lambda(t| X, Z) = \lambda_0(t) \exp(g_1(X\beta_1)Z + g_2(X\beta_2)),}
which implies that \eqn{CSTE(x) = g_1(x\beta_1)}.
}
\examples{
## Quick example for the cste 

## generate data 
set.seed(313)
X1 <- runif(200,0,20)
X2 <- runif(200,0,20)
X <- cbind(X1, X2)
Z <- c(rep(1,100),rep(0,100))

beta <- c(0.3,0.4)
lambda <- 0.1 * exp(log(X\%*\%beta/10+0.6)*Z+(X\%*\%beta)/20)
Time <- -log(runif(200,0,1))/lambda
C <- runif(200,12,15)
S <- as.numeric(Time <= C)
Time <- Time*S + C*(1-S)
## estimate the CSTE curve
fit <- cste_surv(X, Time, Z, S, nknots=3, max.iter=200)
## pointwise confidence interval (PCI)
res <- cste_surv_PCI(fit)

## plot
plot(res$or_x, res$fit_x, col = 'red', type = "l", 
     xlim=c(5,12), ylim=c(-1,2), lwd = 2, 
     ylab = "CSTE", xlab = "X * beta1",
     main ="Pointwise Confidence interval")
lines(res$or_x, res$lower_bound, lwd = 3, col = 'purple', lty = 2)
lines(res$or_x, res$upper_bound, lwd = 3, col = 'purple', lty = 2)
abline(h = 0, lty = 2, cex = 0.2)
legend("topleft", legend = c("Estimates", "PCI"), 
       lwd = c(2,3), lty = c(1, 2), col = c('red','purple'))
       
## True CSTE curve       
xb <- sort(as.numeric(X\%*\%beta))
lines(xb, log(xb/10+0.6), type='l', col='blue')

}
\references{
Ma Y. and Zhou X. (2017). 
Treatment selection in a randomized clinical trial via covariate-specific 
treatment effect curves, \emph{Statistical Methods in Medical Research}, 26(1), 124-141.
}
\seealso{
\code{\link{cste_surv_PCI}, \link{predict_cste_surv}, \link{select_cste_surv}}
}
