% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/query_model.R
\name{query_model}
\alias{query_model}
\title{Generate data frame for batches of causal queries}
\usage{
query_model(
  model,
  queries = NULL,
  given = NULL,
  using = list("parameters"),
  parameters = NULL,
  stats = NULL,
  n_draws = 4000,
  expand_grid = NULL,
  case_level = FALSE,
  query = NULL,
  cred = 95,
  labels = NULL
)
}
\arguments{
\item{model}{A \code{causal_model}. A model object generated by
\code{\link{make_model}}.}

\item{queries}{A vector of strings or list of strings specifying queries
on potential outcomes such as "Y[X=1] - Y[X=0]".
Queries can also indicate conditioning sets by placing second queries after a colon:
"Y[X=1] - Y[X=0] :|: X == 1 & Y == 1". Note a colon, ':|:' is used rather than the traditional
conditioning marker '|' to avoid confusion with logical operators.}

\item{given}{A character vector specifying given conditions for each query.
A 'given' is a quoted expression that evaluates to logical statement.
\code{given} allows the query to be conditioned on either observed
or counterfactural distributions. A value of TRUE is interpreted as no conditioning.
A given statement can alternatively be provided after a colon in the query statement.}

\item{using}{A vector or list of strings. Whether to use priors,
posteriors or parameters.}

\item{parameters}{A vector of real numbers in [0,1]. Values of parameters to
specify (optional). By default, parameters is drawn from the parameters dataframe.
See \code{inspect(model, "parameters_df")}.}

\item{stats}{Functions to be applied to the query distribution.
If NULL, defaults to mean, standard deviation,
and 95\% confidence interval. Functions should return a single numeric
value.}

\item{n_draws}{An integer. Number of draws.}

\item{expand_grid}{Logical. If \code{TRUE} then all combinations of
provided lists are examined. If not then each list is cycled through
separately. Defaults to FALSE.}

\item{case_level}{Logical. If TRUE estimates the probability of the
query for a case.}

\item{query}{alias for queries}

\item{cred}{size of the credible interval ranging between 0 and 100}

\item{labels}{labels for queries: if provided labels should have the length of the combinations of requests}
}
\value{
A data frame with columns Model, Query, Given and Using
  defined by corresponding input values. Further columns are generated
  as specified in \code{stats}.
}
\description{
Calculated from a parameter vector, from a prior or
from a posterior distribution.
}
\details{
Queries can condition on observed or counterfactual quantities.
Nested or "complex" counterfactual queries of the form
\code{Y[X=1, M[X=0]]} are allowed.
}
\examples{
model <- make_model("X -> Y")
query_model(model, "Y[X=1] - Y[X = 0]", using = "priors")
query_model(model, "Y[X=1] - Y[X = 0] :|: X==1 & Y==1", using = "priors")
query_model(model,
  list("Y[X=1] - Y[X = 0]",
       "Y[X=1] - Y[X = 0] :|: X==1 & Y==1"),
  using = "priors")
query_model(model, "Y[X=1] > Y[X = 0]", using = "parameters")
query_model(model, "Y[X=1] > Y[X = 0]", using = c("priors", "parameters"))
\donttest{

# `expand_grid= TRUE` requests the Cartesian product of arguments

models <- list(
 M1 = make_model("X -> Y"),
 M2 = make_model("X -> Y") |>
   set_restrictions("Y[X=1] < Y[X=0]")
 )

# No expansion: lists should be equal length
query_model(
  models,
  query = list(ATE = "Y[X=1] - Y[X=0]",
               Share_positive = "Y[X=1] > Y[X=0]"),
  given = c(TRUE,  "Y==1 & X==1"),
  using = c("parameters", "priors"),
  expand_grid = FALSE)

# Expansion when query and given arguments coupled
query_model(
  models,
  query = list(ATE = "Y[X=1] - Y[X=0]",
               Share_positive = "Y[X=1] > Y[X=0] :|: Y==1 & X==1"),
  using = c("parameters", "priors"),
  expand_grid = TRUE)

# Expands over query and given argument when these are not coupled
query_model(
  models,
  query = list(ATE = "Y[X=1] - Y[X=0]",
               Share_positive = "Y[X=1] > Y[X=0]"),
  given = c(TRUE,  "Y==1 & X==1"),
  using = c("parameters", "priors"),
  expand_grid = TRUE)

# An example of a custom statistic: uncertainty of token causation
f <- function(x) mean(x)*(1-mean(x))

query_model(
  model,
  using = list( "parameters", "priors"),
  query = "Y[X=1] > Y[X=0]",
  stats = c(mean = mean, sd = sd, token_variance = f))
}
}
