% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mh_rep.R
\name{mh_rep}
\alias{mh_rep}
\title{Multivariate Climate Representativeness Analysis}
\usage{
mh_rep(
  polygon,
  col_name,
  climate_variables,
  th = 0.95,
  dir_output = file.path(tempdir(), "ClimaRep"),
  save_raw = FALSE
)
}
\arguments{
\item{polygon}{An \code{sf} object containing the defined areas. \strong{Must have the same CRS as} \code{climate_variables}.}

\item{col_name}{\code{character}. Name of the column in the \code{polygon} object that contains unique identifiers for each polygon.}

\item{climate_variables}{A \code{SpatRaster} stack of climate variables representing the conditions. Its CRS will be used as the reference system.}

\item{th}{\code{numeric} (0-1). Percentile threshold used to define representativeness. Cells with a Mahalanobis distance below or equal to the \code{th} are classified as representative (default: 0.95).}

\item{dir_output}{\code{character}. Path to the directory where output files will be saved. The function will create subdirectories within this path.}

\item{save_raw}{\code{logical}. If \code{TRUE}, saves the intermediate continuous Mahalanobis distance rasters calculated for each polygon before binary classification. The final binary classification rasters are always saved (default: \code{FALSE}).}
}
\value{
Writes the following outputs to disk within subdirectories of \code{dir_output}:
\itemize{
\item Classification (\code{.tif} ) rasters: Binary rasters (\code{0} for \strong{Non-representative} and\code{1} for \strong{Representative}) for each input polygon are saved in the \verb{Representativeness/} subdirectory.
\item Visualization (\code{.jpeg}) maps: Image files visualizing the classification results for each \code{polygon} are saved in the \verb{Charts/} subdirectory.
\item Raw Mahalanobis distance rasters: Optionally saved as \code{.tif} files in the \verb{Mh_Raw/} subdirectory if \code{save_raw = TRUE}.
}
}
\description{
This function calculates Mahalanobis-based Climate Representativeness for input polygon within a defined area.

Representativeness is assessed by comparing the multivariate climate conditions of each cell, of the reference climate space (\code{climate_variables}), with the climate conditions within each specific input \code{polygon}.
}
\details{
This function performs a multivariate analysis using Mahalanobis distance to assess the Climate Representativeness of input \code{polygons} for a single time period.

Crucially, this function assumes that all spatial inputs (\code{polygon}, \code{climate_variables}) are already correctly aligned and share the same Coordinate Reference System (CRS). If inputs do not meet these criteria, the function will stop with an informative error.

Here are the key steps:
\enumerate{
\item Checking of spatial inputs: Ensures that \code{polygon} and \code{climate_variables} have matching CRSs.
\item Calculate the multivariate covariance matrix using climate data from all cells.
\item For each polygon in the \code{polygon} object:
\itemize{
\item Crop and mask the climate variables raster (\code{climate_variables}) to the boundary of the current polygon.
\item Calculate the multivariate mean using the climate data from the previous step. This defines the climate centroid for the current polygon.
\item Calculate the Mahalanobis distance for each cell relative to the centroid and covariance matrix.
\item Apply the specified threshold (\code{th}) to Mahalanobis distances to determine which cells are considered representative. This threshold is a percentile of the Mahalanobis distances within the current polygon.
\item Classify each cell as Representative = \code{1} (Mahalanobis distance \eqn{\le} \code{th}) or Non-Representative = \code{0} (Mahalanobis distance $>$ \code{th}).
}
\item Saves the binary classification raster (\code{.tif}) and generates a corresponding visualization map (\code{.jpeg}) for each polygon. These are saved within the specified output directory (\code{dir_output}).
}

It is important to note that Mahalanobis distance is sensitive to collinearity among variables.
While the covariance matrix accounts for correlations, it is strongly recommended that the \code{climate_variables} are not strongly correlated.
Consider performing a collinearity analysis beforehand, perhaps using the \code{vif_filter} function from this package.
}
\examples{
library(terra)
library(sf)
set.seed(2458)
n_cells <- 100 * 100
r_clim_present <- terra::rast(ncols = 100, nrows = 100, nlyrs = 7)
values(r_clim_present) <- c(
   (terra::rowFromCell(r_clim_present, 1:n_cells) * 0.2 + rnorm(n_cells, 0, 3)),
   (terra::rowFromCell(r_clim_present, 1:n_cells) * 0.9 + rnorm(n_cells, 0, 0.2)),
   (terra::colFromCell(r_clim_present, 1:n_cells) * 0.15 + rnorm(n_cells, 0, 2.5)),
   (terra::colFromCell(r_clim_present, 1:n_cells) +
     (terra::rowFromCell(r_clim_present, 1:n_cells)) * 0.1 + rnorm(n_cells, 0, 4)),
   (terra::colFromCell(r_clim_present, 1:n_cells) /
     (terra::rowFromCell(r_clim_present, 1:n_cells)) * 0.1 + rnorm(n_cells, 0, 4)),
   (terra::colFromCell(r_clim_present, 1:n_cells) *
     (terra::rowFromCell(r_clim_present, 1:n_cells) + 0.1 + rnorm(n_cells, 0, 4))),
   (terra::colFromCell(r_clim_present, 1:n_cells) *
     (terra::colFromCell(r_clim_present, 1:n_cells) + 0.1 + rnorm(n_cells, 0, 4)))
)
names(r_clim_present) <- c("varA", "varB", "varC", "varD", "varE", "varF", "varG")
terra::crs(r_clim_present) <- "EPSG:4326"

vif_result <- ClimaRep::vif_filter(r_clim_present, th = 5)
print(vif_result$summary)
r_clim_present_filtered <- vif_result$filtered_raster
hex_grid <- sf::st_sf(
   sf::st_make_grid(
     sf::st_as_sf(
       terra::as.polygons(
         terra::ext(r_clim_present_filtered))),
     square = FALSE)
     )
sf::st_crs(hex_grid) <- "EPSG:4326"
polygons <- hex_grid[sample(nrow(hex_grid), 2), ]
polygons$name <- c("Pol_A", "Pol_B")
terra::plot(r_clim_present_filtered[[1]])
terra::plot(polygons, add = TRUE, color = "transparent", lwd = 3)

ClimaRep::mh_rep(
   polygon = polygons,
   col_name = "name",
   climate_variables = r_clim_present_filtered,
   th = 0.95,
   dir_output = file.path(tempdir(), "ClimaRep"),
   save_raw = TRUE
   )
}
