% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lyapunov.R
\name{lyapunov}
\alias{lyapunov}
\title{Estimation of the Lyapunov exponent through several methods}
\usage{
lyapunov(x, lag = 1, timelapse = "FIXED", M0 = 3, M1 = 10,
  H0 = 2, H1 = 10, I = 100, lyapmethod = c("LLE", "SLE", "ALL"),
  blocking = c("FULL", "NOVER", "EQS", "BOOT", "ALL"),
  pre.white = TRUE, B = 100, netplot = TRUE, doplot = TRUE)
}
\arguments{
\item{x}{a numeric vector, time serie, data frame or matrix depending on the method selected in \code{timelapse}.}

\item{lag}{a non-negative integer denoting the reconstruction delay (Default 1).}

\item{timelapse}{a character denoting if you consider that the observations are sampled at uniform time intervals \code{FIXED} or with a variable time-lapse between each observation \code{VARIABLE} (Default \code{FIXED}).}

\item{M0}{a non-negative integer denoting a lower bound for the embedding dimension (Default 3).}

\item{M1}{a non-negative integer denoting an upper bound for the embedding dimension (Default 10).}

\item{H0}{a non-negative integer denoting a lower bound for the number of neurones in the hidden layer (Default 2).}

\item{H1}{a non-negative integer denoting an upper bound for the number of neurones in the hidden layer (Default 10).}

\item{I}{a non-negative integer denoting a number of neural networks iterations (Default 100).}

\item{lyapmethod}{a character denoting if you want to estimate the largest Lyapunov exponent \code{LLE}, the Lyapunov exponent spectrum \code{SLE} or both \code{ALL} (Default \code{LLE}).}

\item{blocking}{a character denoting if you consider the full sample \code{FULL}, the non-overlapping sample \code{NOVER}, the equally spaced sample \code{EQS}, the bootstrap sample \code{BOOT} or all of them \code{ALL} (Default \code{FULL}).}

\item{pre.white}{a character denoting if you want to use as points to evaluate the partial derivatives the delayed vectors filtered by the neural network \code{TRUE} or not \code{FALSE} (Default \code{TRUE}).}

\item{B}{a non-negative integer denoting the number of bootstrap iterations (Default 100).}

\item{netplot}{a logical value denoting if you want to draw a plot \code{TRUE} or not \code{FALSE}. If it is \code{TRUE} shows as many graphs as networks have been considered. Each of them represents the network structure by drawing the weights with positive values in black and the weights with negative values in grey. The thickness of the lines represents a greater or lesser value (Default \code{TRUE}).}

\item{doplot}{a logical value denoting if you want to draw a plot \code{TRUE} or not \code{FALSE}. If it is \code{TRUE} the evolution of the Lyapunov exponent values are represented for the whole period considering the lyapunov methods and the blocking methods selected by the user (Default \code{TRUE}).}
}
\value{
A list containing the largest Lyapunov exponent, the Lyapunov exponent spectrum or both for each neural network structure considered by keeping to \code{Lyapunov.net}. The dataset saved by each blocking method are the estimated Lyapunov exponent value, the standard error, the z-test value and the p-value. If the user chooses the non-overlapping sample, the equally spaced sample or the bootstrap sample the mean and median values of the Lyapunov exponent are showed. Also some details about the embedding dimension, the sample size, the block length and the block number are recorded.
}
\description{
This function estimates both the largest Lyapunov exponent through the Norma-2 method and the Lyapunov exponent spectrum through the QR decomposition method taking into account the full sample and three different methods of subsampling by blocks considering the argument set selected by the user.
}
\details{
If \code{FIXED} has been selected \code{x} must be a numeric vector or time serie. Otherwise \code{VARIABLE} has to be specified. In this case \code{x} must be a data frame or matrix with two columns. First, the date with the following format \code{YMD H:M:OS3} considering milliseconds e.g., 20190407 00:00:03.347. If you don't consider milliseconds you must put .000 after the seconds. It should be an object of class \code{Factor}. Second, the univariate time serie as a sequence of numerical values.
}
\examples{
## We show below an example considering time series from the
## logistic equation. We have estimated the Lyapunov exponent
## spectrum by each blocking method for an embedding dimension (m=4).
data<-logistic.ts(u.min=4,u.max=4,B=100,doplot=FALSE)
ts<-data$`Logistic 100`$time.serie
lyapu<-lyapunov(ts,lag=1,timelapse="FIXED",M0=4,M1=4,H0=2,H1=7,I=10,
       lyapmethod="SLE",blocking="ALL",pre.white=TRUE,B=30,netplot=FALSE,
       doplot=FALSE)
show(lyapu$Lyapunov.net$Spectrum.full$Exponent)
show(lyapu$Lyapunov.net$Spectrum.nonoverlap$Exponent)
show(lyapu$Lyapunov.net$Spectrum.equally$Exponent)
show(lyapu$Lyapunov.net$Spectrum.bootstrap$Exponent)
}
\references{
Ellner, S., Gallant, A., McCaffrey, D., Nychka, D. 1991 Convergence rates and data requirements for jacobian-based estimates of lyapunov exponents from data. Physics Letters A 153(6):357-363.

McCaffrey, D.F., Ellner, S., Gallant, A.R., Nychka, D.W. 1992 Estimating the lyapunov exponent of a chaotic system with nonparametric regression. Journal of the American Statistical Association 87(419):682-695.

Nychka, D., Ellner, S., Gallant, A.R., McCaffrey, D. 1992 Finding chaos in noisy systems. Journal of the Royal Statistical Society 54(2):399-426.

Whang, Y.J., Linton, O. 1999 The asymptotic distribution of nonparametric estimates of the lyapunov exponent for stochastic time series. Journal of Econometrics 91(1):1-42.

Shintani, M., Linton, O. 2004 Nonparametric neural network estimation of Lyapunov exponents and a direct test for chaos. Journal of Econometrics 120(1):1-33.
}
\seealso{
\code{\link{embedding}}, \code{\link{jacobi}}, \code{\link{lyapunov.max}}, \code{\link{lyapunov.spec}}
}
\author{
Julio E. Sandubete, Lorenzo Escot
}
