% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_truncation.R
\name{estimate_truncation}
\alias{estimate_truncation}
\title{Estimate Truncation of Observed Data}
\usage{
estimate_truncation(
  data,
  truncation = trunc_opts(LogNormal(meanlog = Normal(0, 1), sdlog = Normal(1, 1), max =
    10)),
  model = NULL,
  stan = stan_opts(),
  CrIs = c(0.2, 0.5, 0.9),
  filter_leading_zeros = FALSE,
  zero_threshold = Inf,
  weigh_delay_priors = FALSE,
  verbose = TRUE,
  ...,
  obs
)
}
\arguments{
\item{data}{A list of \verb{<data.frame>}s each containing a date variable
and a confirm (numeric) variable. Each data set should be a snapshot
of the reported data over time. All data sets must contain a complete vector
of dates.}

\item{truncation}{A call to \code{\link[=trunc_opts]{trunc_opts()}} defining the truncation of
the observed data. Defaults to \code{\link[=trunc_opts]{trunc_opts()}}, i.e. no truncation.  See the
\code{\link[=estimate_truncation]{estimate_truncation()}} help file for an approach to estimating this from
data where the \code{dist} list element returned by \code{\link[=estimate_truncation]{estimate_truncation()}} is
used as the \code{truncation} argument here, thereby propagating the uncertainty
in the estimate.}

\item{model}{A compiled stan model to override the default model. May be
useful for package developers or those developing extensions.}

\item{stan}{A list of stan options as generated by \code{\link[=stan_opts]{stan_opts()}}. Defaults
to \code{\link[=stan_opts]{stan_opts()}}. Can be used to override \code{data}, \code{init}, and \code{verbose}
settings if desired.}

\item{CrIs}{Numeric vector of credible intervals to calculate.}

\item{filter_leading_zeros}{Logical, defaults to TRUE. Should zeros at the
start of the time series be filtered out.}

\item{zero_threshold}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}} Numeric defaults
to Inf. Indicates if detected zero cases are meaningful by using a threshold
number of cases based on the 7-day average. If the average is above this
threshold then the zero is replaced using \code{fill}.}

\item{weigh_delay_priors}{Deprecated; use the \code{weight_prior} option in
\code{\link[=trunc_opts]{trunc_opts()}} instead.}

\item{verbose}{Logical, should model fitting progress be returned.}

\item{...}{Additional parameters to pass to \code{\link[rstan:stanmodel-method-sampling]{rstan::sampling()}}.}

\item{obs}{Deprecated; use \code{data} instead.}
}
\value{
A list containing: the summary parameters of the truncation
distribution (\code{dist}), which could be passed to the \code{truncation} argument
of \code{\link[=epinow]{epinow()}}, \code{\link[=regional_epinow]{regional_epinow()}}, and \code{\link[=estimate_infections]{estimate_infections()}}, the
estimated CMF of the truncation distribution (\code{cmf}, can be used to
adjusted new data), a \verb{<data.frame>} containing the observed truncated
data, latest observed data and the adjusted for
truncation observations (\code{obs}), a \verb{<data.frame>} containing the last
observed data (\code{last_obs}, useful for plotting and validation), the data
used for fitting (\code{data}) and the fit object (\code{fit}).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}
Estimates a truncation distribution from multiple snapshots of the same
data source over time. This distribution can then be used passed to the
\code{truncation} argument in \code{\link[=regional_epinow]{regional_epinow()}}, \code{\link[=epinow]{epinow()}}, and
\code{\link[=estimate_infections]{estimate_infections()}} to adjust for truncated data and propagate the
uncertainty associated with data truncation into the estimates.

See \href{https://gist.github.com/seabbs/176b0c7f83eab1a7192a25b28bbd116a}{here}
for an example of using this approach on Covid-19 data in England. The
functionality offered by this function is now available in a more principled
manner in the \href{https://package.epinowcast.org/}{\code{epinowcast} R package}.

The model of truncation is as follows:
\enumerate{
\item The truncation distribution is assumed to be discretised log normal wit
a mean and standard deviation that is informed by the data.
\item The data set with the latest observations is adjusted for truncation using
the truncation distribution.
\item Earlier data sets are recreated by applying the truncation distribution to
the adjusted latest observations in the time period of the earlier data set.
These data sets are then compared to the earlier observations assuming a
negative binomial observation model with an additive noise term to deal with
zero observations.
}

This model is then fit using \code{stan} with standard normal, or half normal,
prior for the mean, standard deviation, 1 over the square root of the
overdispersion and additive noise term.

This approach assumes that:
\itemize{
\item Current truncation is related to past truncation.
\item Truncation is a multiplicative scaling of underlying reported cases.
\item Truncation is log normally distributed.
}
}
\examples{
\donttest{
# set number of cores to use
old_opts <- options()
options(mc.cores = ifelse(interactive(), 4, 1))

# fit model to example data
# See [example_truncated] for more details
est <- estimate_truncation(example_truncated,
  verbose = interactive(),
  chains = 2, iter = 2000
)

# summary of the distribution
est$dist
# summary of the estimated truncation cmf (can be applied to new data)
print(est$cmf)
# observations linked to truncation adjusted estimates
print(est$obs)
# validation plot of observations vs estimates
plot(est)

# Pass the truncation distribution to `epinow()`.
# Note, we're using the last snapshot as the observed data as it contains
# all the previous snapshots. Also, we're using the default options for
# illustrative purposes only.
out <- epinow(
  generation_time = generation_time_opts(example_generation_time),
  example_truncated[[5]],
  truncation = trunc_opts(est$dist)
)
plot(out)
options(old_opts)
}
}
