% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gPoMo.R
\name{gPoMo}
\alias{gPoMo}
\title{gPoMo: Generalized Polynomial Modeling}
\usage{
gPoMo(data, tin = NULL, dtFixe = NULL, dMax = 2, nS = c(3), winL = 9,
  weight = NULL, show = 1, verbose = 1, underSamp = NULL, EqS = NULL,
  IstepMin = 2, IstepMax = 2000, nPmin = 1, nPmax = 14,
  method = "lsoda")
}
\arguments{
\item{data}{Input Time series: Each column corresponds to one input variable.}

\item{tin}{Input date vector which length should correspond to the variables of
the input data (same number of lines).}

\item{dtFixe}{Time step used for the analysis. In principle, it should correspond
to the time step of the input data. Modification of this time step may be used to
stabilize the numerical computation. Such modification should be performed in full
consciouness that it will lead to change the time scale accordingly.}

\item{dMax}{Maximum degree of the polynomial functions allowed
in the model (see \code{poLabs}).}

\item{nS}{A vector providing the number of dimensions that will be used for each
input variables (see Examples 1 and 2). The dimension of the resulting model will
be nVar = sum(nS).}

\item{winL}{Total number of points used for computing the derivatives of the input
time series (data). This parameter will be used as an input in function \code{drvSucc}.}

\item{weight}{Weighting function of input data series. By default uniform
weight is applied. This weighting function can also be used to apply
the analysis piecewise using zeros and ones.}

\item{show}{Indicates (2) or not (0-1) the algorithm progress}

\item{verbose}{Gives information (if set to 1) about the algorithm progress
and keeps silent if set to 0}

\item{underSamp}{Points used for undersampling the data. For undersamp = 1
the complete time series is used. For undersamp = 2, only one data out of
two is kept, etc.}

\item{EqS}{Model template including all allowed regressors.
Each column corresponds to one equation. Each line corresponds to one
regressor following the convention given by \code{poLabs(nVar,dMax)}.}

\item{IstepMin}{Minimum step of integration at the beginning
of the analysis (by default IstepMin=10).}

\item{IstepMax}{Maximum step of integration before stopping the
analysis, with the interface this value can be changed during
the analysis}

\item{nPmin}{Corresponds to the minimum number of parameters (and thus
of regressor) of the model}

\item{nPmax}{Corresponds to the maximum number of parameters (and thus
of regressor) of the model}

\item{method}{The integration technique used for the numerical
integration. Default is 'lsoda'. Others such as 'rk4' or 'ode45'
may also be used. See package deSolve for details.}
}
\value{
A list containing:

\code{$tin}        The time vector of the input time series

\code{$inputdata}  The input time series

\code{$tfiltdata}  The time vector of the filtered time series (boudary removed)

\code{$filtdata}   A matrix of the filtered time series with its derivatives

\code{$okMod}      A vector classifying the models: diverging models (0), periodic
models of period-1 (-1), unclassified models (1).

\code{$coeff}      A matrix with the coefficients of one selected model

\code{$models}     A list of all the models to be tested \code{$mToTest1},
\code{$mToTest2}, etc. and all selected models \code{$model1}, \code{$model2}, etc.

\code{$tout}       The time vector of the output time series (vector length
corresponding to the longest numerical integration duration)

\code{$stockoutreg} A list of matrices with the integrated trajectories
(variable \code{X1} in column 1, \code{X2} in 2, etc.) of all the models \code{$model1}, \code{$model2}, etc.
}
\description{
Algorithm for a Generalized Polynomial formulation
of multivariate Global Modeling.
Global modeling aims to obtain multidimensional models
from single time series [1-2].
In the generalized (polynomial) formulation provided in this
function, it can also be applied to multivariate time series [3-4].

Example:\cr
Note that nS provides the number of dimensions used from each variable

case I \cr
For nS=c(2,3) means that 2 dimensions are reconstructed from variable 1:
the original variable X1 and its first derivative X2), and
3 dimensions are reconstructed from variable 2: the original
variable X3 and its first and second derivatives X4 and X5.
The generalized model will thus be such as: \cr
\eqn{dX1/dt = X2}\cr
\eqn{dX2/dt = P1(X1,X2,X3,X4,X5)}\cr
\eqn{dX3/dt = X4}\cr
\eqn{dX4/dt = X5}\cr
\eqn{dX5/dt = P2(X1,X2,X3,X4,X5).}\cr

case II\cr
For nS=c(1,1,1,1) means that only the original variables
X1, X2, X3 and X4 will be used.
The generalized model will thus be such as:\cr
\eqn{dX1/dt = P1(X1,X2,X3,X4)}\cr
\eqn{dX2/dt = P2(X1,X2,X3,X4)}\cr
\eqn{dX3/dt = P3(X1,X2,X3,X4)}\cr
\eqn{dX4/dt = P4(X1,X2,X3,X4).}\cr
}
\examples{
#Example 1
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,3]
dev.new()
out1 <- gPoMo(data, tin=tin, dMax = 2, nS=c(3), show = 1,
              IstepMax = 1000, nPmin = 9, nPmax = 11)
visuEq(3, 2, out1$models$model1, approx = 4)

\dontrun{
#Example 2
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,2:4]
dev.new()
out2 <- gPoMo(data, tin=tin, dMax = 2, nS=c(1,1,1), show = 1,
              IstepMin = 10, IstepMax = 3000, nPmin = 7, nPmax = 8)
# the simplest model able to reproduce the observed dynamics is model #5
visuEq(3, 2, out2$models$model5, approx = 4) # the original Rossler system is thus retrieved
}

\dontrun{
#Example 3
data("Ross76")
tin <- Ross76[,1]
data <- Ross76[,2:3]
# model template:
EqS <- matrix(1, ncol = 3, nrow = 10)
EqS[,1] <- c(0,0,0,1,0,0,0,0,0,0)
EqS[,2] <- c(1,1,0,1,0,1,1,1,1,1)
EqS[,3] <- c(0,1,0,0,0,0,1,1,0,0)
visuEq(3, 2, EqS, substit = c('X','Y','Z'))
dev.new()
out3 <- gPoMo(data, tin=tin, dMax = 2, nS=c(2,1), show = 1,
      EqS = EqS, IstepMin = 10, IstepMax = 2000,
      nPmin = 9, nPmax = 11)
}

\dontrun{
#Example 4
data(sprottK)
data(rossler)
data <- cbind(rossler,sprottK)[1:400,]
dev.new()
out4 <- gPoMo(data, dt=1/20, dMax = 2, nS=c(1,1,1,1,1,1),
             show = 1, method = 'rk4',
             IstepMin = 2, IstepMax = 3, nPmin = 13, nPmax = 13)
# the original Rossler (variables x, y and z) and Sprott (variables X, Y and Z)
# systems are retrieved:
visuEq(6, 2, out4$models$model347, approx = 4,
           substit = c('x', 'y', 'z', 'X', 'Y', 'Z'))
   # to check the robustness of the model, the integration duration
   # should be chosen longer (at least IstepMax = 4000)
}

}
\author{
Sylvain Mangiarotti, Flavie Le Jean, Mireille Huc
}
\references{
[1] Gouesbet G. & Letellier C., 1994. Global vector-field reconstruction by using a
multivariate polynomial L2 approximation on nets, Physical Review E, 49 (6),
4955-4972. \cr
[2] Mangiarotti S., Coudret R., Drapeau L. & Jarlan L., Polynomial search and
Global modelling: two algorithms for modeling chaos. Physical Review E, 86(4),
046205. \cr
[3] Mangiarotti S., Le Jean F., Huc M. & Letellier C., Global Modeling of aggregated
and associated chaotic dynamics. Chaos, Solitons and Fractals, 83, 82-96. \cr
[4] S. Mangiarotti, M. Peyre & M. Huc, 2016.
A chaotic model for the epidemic of Ebola virus disease
in West Africa (2013-2016). Chaos, 26, 113112. \cr
}
\seealso{
\code{\link{gloMoId}}, \code{\link{autoGPoMoSearch}},
         \code{\link{autoGPoMoTest}}
}

