
#' gsnImportGSEA
#'
#' @description Add GSEA search data to a GSNData object, as generated by the the GSEA package. The data set can be
#' either in the form of a data.frame or specified as import from a delimited text file. (See Details below)
#'
#' @param object A GSNData object.
#' @param pathways_data An (optional) data.frame containing the results of GSEA analysis. (Either this or the
#' \code{filename} argument must be set.
#' @param filename An (optional) filename for data sets read from a text file containing GSEA results. This is ignored
#' if the \code{pathways_data} argument is set.
#' @param id_col (optional) A character vector of length 1 indicating the name of the column used as a key for gene
#' sets or modules. This is normally the \code{NAME} field of GSEA data, which must be the same as the names of gene
#' sets specified in the tmod object or in the list of gene set vectors specified with the \code{geneSetCollection}
#' argument used when building the gene set network. By default this value is \code{'NAME'}. The IDs must correspond
#' to the names of the gene sets provided, or an error will be thrown. **NOTE:** In the tmod::tmodImportMSigDB function
#' provided by the tmod package, the default ID is an MSigDB accession, but GSEA data sets do not use this accession.
#' The \code{NAME} column used in GSEA results set corresponds instead to the \code{STANDARD_NAME} field in the MSigDB
#' XML database file. This \code{STANDARD_NAME} field is not preserved by the standard \code{tmod::tmodImportMSigDB}
#' utility function, but instead reformatted converting underscores to spaces and non-initial letters to lower case.
#' Therefore, when using GSEA data sets with an MSigDB gene set collection imported using \code{tmod::tmodImportMSigDB}
#' the \code{NAME} fields need to be mapped to the \code{ID} or vice versa.
#' @param stat_col (optional) A character vector of length 1 indicating the name of the column used as a statistic
#' to evaluate the quality of pathways results. The function scans through possible \code{stat_col} values
#' (\code{"FDR q-val"}, \code{"FDR.q.val"}, \code{"FWER p-val"}, \code{"FWER.p.val"}, \code{"NOM p-val"},
#' \code{"NOM.p.val"}), and uses the first one it finds. (The presence of spaces and hyphens in the column names
#' necessitates flexibility here. Depending on how GSEA results sets are read in, spaces and hyphens may be substituted
#' with periods.)
#' @param sig_order (optional) Either \code{'loToHi'} (default) or \code{'hiToLo'} depending on the statistic used to
#' evaluate pathways results.
#' @param n_col (optional) Specifies the column containing the number of genes in the gene set. Generally, this is the number
#' of genes in the gene set that are attested in an expression data set (Defaults to 'SIZE').
#' @param sep A separator for text file import, defaults to "\\t". Ignored if \code{filename} is not specified.
#'
#' @return This returns a GSNData object containing imported pathways data.
#'
#' @details GSEA results directories generally contain files with names beginning with \code{gsea_report_for_} and with
#' the \code{.xls} suffix. This method is designed to handle such data sets.
#'
#' Note: An error is thrown if all gene set IDs in the genePresenceAbsense are not present in the GSEA \code{NAME} column.
#' However, if there are gene set IDs present in the pathways data that are absent from the \code{$genePresenceAbsence}
#' matrix, then this method emits a warning. It also checks for the standard GSEA data set column names, and if some are
#' missing, it will emit a warning.
#'
#' @export
#'
#' @examples
#'
#' library(GSNA)
#'
#' # In this example, we generate a gene set network from GSEA example
#' # data. We begin by subsetting the GSEA data for significant results:
#' sig_pathways.gsea <- subset( Bai_CiHep_dorothea_DN.Gsea, `FDR q-val` <= 0.05 )
#'
#' # Now create a gene set collection containing just the gene sets
#' # with significant GSEA results, by subsetting Bai_gsc.tmod using
#' # the gene set NAME as keys:
#' sig_pathways.tmod <- Bai_gsc.tmod[sig_pathways.gsea$NAME]
#'
#' # And obtain a background gene set from the expression data used
#' # to generate the gsea results:
#' background_genes <- toupper( rownames( Bai_empty_expr_mat ) )
#'
#' # Build a gene set network:
#' sig_pathways.GSN <-
#'    buildGeneSetNetworkSTLF(geneSetCollection = sig_pathways.tmod,
#'                            ref.background = background_genes )
#'
#' # Now import the GSEA data.
#' sig_pathways.GSN <- gsnImportGSEA( sig_pathways.GSN,
#'                                    pathways_data = sig_pathways.gsea )
#'
#' @seealso
#'  \code{\link{gsnAddPathwaysData}}
#'  \code{\link{gsnImportCERNO}}
#'  \code{\link{gsnImportGSNORA}}
#'  \code{\link{gsnImportGenericPathways}}
#'
#' @importFrom utils read.table
#'
gsnImportGSEA <- function( object, pathways_data = NULL, filename = NULL, id_col = NULL, stat_col = NULL, sig_order = NULL, n_col = NULL, sep = "\t" ){
  stopifnot( "GSNData" %in% class( object ) )

  # Not searching for *all* the fields, just the critical ones. (Some are repeats, with weird names.)
  if( is.null( pathways_data ) && is.null( filename ) ) stop( "The 'pathways_data' and 'filename' arguments cannot both be NULL." )
  if( is.null( pathways_data ) ){
    pathways_data <- utils::read.table( file = filename, header = TRUE, sep = sep, stringsAsFactors = FALSE, check.names = FALSE )
  }
  if( !is.null(sig_order) && ! sig_order %in% c( "loToHi", "hiToLo" ) )
    stop( "Invalid sig_order: ", as.character( sig_order ) )
  if( ! is.null(stat_col) && ! stat_col %in% colnames( pathways_data ) )
    stop( "stat_col '", stat_col, "' not found in pathways data."  )

  field_names <- colnames( pathways_data )

  # "NAME", "GS<br> follow link to MSigDB", "GS DETAILS", "SIZE", "ES", "NES", "NOM p-val", "FDR q-val",
  # "FWER p-val", "RANK AT MAX", "LEADING EDGE"
  gsea_fieldnames <- c( "NAME", "GS DETAILS", "SIZE", "ES", "NES", "NOM p-val", "FDR q-val", "FWER p-val",
                        "RANK AT MAX", "LEADING EDGE" )

  if( length( missing_fieldnames <- gsea_fieldnames[! make.names(gsea_fieldnames) %in% make.names(field_names)] ) > 0 ){
    warning( "Pathways data are missing the following GSEA fields:", paste0( missing_fieldnames,  collapse = ", " ), "\n" )
  }

  pathways <- list( data = pathways_data, type = "gsea" )
  if( "NAME" %in% field_names ) pathways$id_col <- "NAME"

  stat_cols <- c( "FDR q-val", "FDR.q.val", "FWER p-val", "FWER.p.val", "NOM p-val", "NOM.p.val" )
  if( length( present_stat_cols <- field_names[field_names %in% stat_cols] ) > 0 ){
    pathways$stat_col <- present_stat_cols[1]
    pathways$sig_order <- "loToHi"
  }

  n_cols <- c( "SIZE" )
  if( length( present_n_cols <- field_names[field_names %in% n_cols] ) > 0 ){
    pathways$n_col <- present_n_cols[1]
  }

  # Add a Title column to gsea data for later:
  if( "NAME" %in% field_names && ! "Title" %in% field_names )
    pathways$data$Title <- stringr::str_to_title( gsub( pattern = "_", replacement = " ", x = pathways$data$NAME ) )

  if( !is.null(id_col) ) pathways$id_col <- id_col
  if( !is.null(stat_col) ) pathways$stat_col <- stat_col
  if( !is.null(sig_order) ) pathways$sig_order <- sig_order
  if( !is.null(n_col) ) pathways$n_col <- n_col

  if( is.null( pathways$id_col ) ) stop( "id_col (ID Column specification) required." )

  if( ! all( colnames( object$genePresenceAbsence ) %in% pathways$data[[pathways$id_col]] ) )
    stop("Error: Pathways data do not match gene set collection. They are missing gene sets from gene set collection.")
  if( ! all( pathways$data[[pathways$id_col]] %in% colnames( object$genePresenceAbsence ) ) )
    warning("Warning: Pathways data do not match gene set collection. They contain gene sets not present in gene set collection.\n")

  object$pathways <- pathways
  object
}

