% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MHalgoGen.R
\name{MHalgoGen}
\alias{MHalgoGen}
\title{Monte-Carlo Markov-chain with Metropolis-Hastings algorithm}
\usage{
MHalgoGen(likelihood = stop("A likelihood function is mandatory"),
  parameters = stop("Priors are mandatory"), ..., n.iter = 10000,
  n.chains = 1, n.adapt = 100, thin = 30, trace = FALSE,
  intermediate = NULL, filename = "intermediate.Rdata", previous = NULL)
}
\arguments{
\item{likelihood}{The function that returns -ln likelihood using data and parameters}

\item{parameters}{A data.frame with priors; see description and examples}

\item{...}{Parameters to be transmitted to likelihood function}

\item{n.iter}{Number of iterations for each chain}

\item{n.chains}{Number of chains}

\item{n.adapt}{Number of iteration to stabilize likelihood}

\item{thin}{Interval for thinning likelihoods}

\item{trace}{Or FALSE or period to show progress}

\item{intermediate}{Or NULL of period to save intermediate result}

\item{filename}{Name of file in which intermediate results are saved}

\item{previous}{The content of the file in which intermediate results are saved}
}
\value{
A mcmcComposite object with all characteristics of the model and mcmc run
}
\description{
The parameters must be stored in a data.frame with named rows for each parameter with the following columns:\cr
\itemize{
  \item Density. The density function name, example \code{dnorm}, \code{dlnorm}, \code{dunif}
  \item Prior1. The first parameter to send to the \code{Density} function
  \item Prior2. The second parameter to send to the \code{Density} function
  \item SDProp. The standard error from new proposition value of this parameter
  \item Min. The minimum value for this parameter
  \item Max. The maximum value for this parameter
  \item Init. The initial value for this parameter
}
This script has been deeply modified from a MCMC script provided by Olivier Martin (INRA, Paris-Grignon).
}
\details{
MHalgoGen is a function to use mcmc with Metropolis-Hastings algorithm
}
\examples{
\dontrun{
library(HelpersMG)
require(coda)
x <- rnorm(30, 10, 2)
dnormx <- function(x, par) return(-sum(dnorm(x, mean=par['mean'], sd=par['sd'], log=TRUE)))
parameters_mcmc <- data.frame(Density=c('dnorm', 'dlnorm'), 
Prior1=c(10, 0.5), Prior2=c(2, 0.5), SDProp=c(0.35, 0.2), 
Min=c(-3, 0), Max=c(100, 10), Init=c(10, 2), stringsAsFactors = FALSE, 
row.names=c('mean', 'sd'))
mcmc_run <- MHalgoGen(n.iter=10000, parameters=parameters_mcmc, x=x, 
likelihood=dnormx, n.chains=1, n.adapt=100, thin=1, trace=1)
plot(mcmc_run, xlim=c(0, 20))
plot(mcmc_run, xlim=c(0, 10), parameters="sd")
mcmcforcoda <- as.mcmc(mcmc_run)
#' heidel.diag(mcmcforcoda)
raftery.diag(mcmcforcoda)
autocorr.diag(mcmcforcoda)
acf(mcmcforcoda[[1]][,"mean"], lag.max=20, bty="n", las=1)
acf(mcmcforcoda[[1]][,"sd"], lag.max=20, bty="n", las=1)
batchSE(mcmcforcoda, batchSize=100)
# The batch standard error procedure is usually thought to 
# be not as accurate as the time series methods used in summary
summary(mcmcforcoda)$statistics[,"Time-series SE"]
summary(mcmc_run)
as.parameters(mcmc_run)
lastp <- as.parameters(mcmc_run, index="last")
parameters_mcmc[,"Init"] <- lastp
# The n.adapt set to 1 is used to not record the first set of parameters
# then it is not duplicated (as it is also the last one for 
# the object mcmc_run)
mcmc_run2 <- MHalgoGen(n.iter=1000, parameters=parameters_mcmc, x=x, 
likelihood=dnormx, n.chains=1, n.adapt=1, thin=1, trace=1)
mcmc_run3 <- merge(mcmc_run, mcmc_run2)
####### no adaptation, n.adapt must be 0
parameters_mcmc[,"Init"] <- c(mean(x), sd(x))
mcmc_run3 <- MHalgoGen(n.iter=1000, parameters=parameters_mcmc, x=x, 
likelihood=dnormx, n.chains=1, n.adapt=0, thin=1, trace=1)
}
}
\author{
Marc Girondot
}
\seealso{
Other mcmcComposite functions: \code{\link{as.mcmc.mcmcComposite}},
  \code{\link{as.parameters}},
  \code{\link{merge.mcmcComposite}},
  \code{\link{plot.mcmcComposite}},
  \code{\link{summary.mcmcComposite}}
}

