#' Particle Swarm Optimization for LHD
#'
#' \code{LaPSO} returns an LHD matrix generated by particle swarm optimization algorithm (PSO)
#'
#' @param n A positive integer.
#' @param k A positive integer.
#' @param m A positive integer.
#' @param N A positive integer.
#' @param SameNumP A non-negative integer.
#' @param SameNumG A non-negative integer.
#' @param p0 A probability.
#' @param OC An optimality criterion.
#' @param p A positive integer.
#' @param q The default is set to be 1, and it could be either 1 or 2.
#'
#' @return If all inputs are logical, then the output will be a \code{n} by \code{k} LHD.
#' @details \itemize{
#' \item \code{n} stands for the number of rows (or run size).
#' \item \code{k} stands for the number of columns (or the number of factors).
#' \item \code{m} stands for the number of particles. The default is set to be 10.
#' \item \code{N} stands for the number of iterations. The default is set to be 10.
#' \item \code{SameNumP} stands for how many elements in current column of current particle LHD should be the same as corresponding Personal Best. SameNumP=0, 1, 2, ..., n, and 0 means to skip the "exchange".  The default is set to be 0.
#' \item \code{SameNumG} stands for how many elements in current column of current particle LHD should be the same as corresponding Global Best. SameNumP=0, 1, 2, ..., n, and 0 means to skip the "exchange".  The default is set to be \code{n}/4.
#' \item \code{SameNumP} and \code{SameNumG} cannot be 0 at the same time.
#' \item \code{p0} stands the probability of exchange two randomly selected elements in current column of current particle LHD. The default is set to be 1/(\code{k} - 1).
#' \item \code{OC} stands for the optimality criterion, the default setting is "phi_p", and it could be one of the following: "phi_p", "AvgAbsCor", "MaxAbsCor", "MaxProCriterion".
#' \item \code{p} is the parameter in the phi_p formula, and \code{p} is prefered to be large. The default is set to be 15.
#' \item If \code{q} is 1 (the default setting), \code{dij} is the rectangular distance. If \code{q} is 2, \code{dij} is the Euclidean distance.
#' }
#'
#' @note Here are some general suggestions about the parameters: \itemize{
#' \item \code{SameNumP} is approximately \code{n}/2 when \code{SameNumG} is 0.
#' \item \code{SameNumG} is approximately \code{n}/4 when \code{SameNumP} is 0.
#' \item \code{p0} * (\code{k} - 1) = 1 or 2 is often sufficient. So \code{p0} = 1/(\code{k} - 1) or 2/(\code{k} - 1).
#' }
#'
#' @references Chen, R.-B., Hsieh, D.-N., Hung, Y., and Wang, W. (2013) Optimizing Latin hypercube designs by particle swarm. \emph{Stat. Comput.}, \strong{23}, 663-676.
#'
#' @examples
#' #generate a 5 by 3 maximin distance LHD with the default setting
#' try=LaPSO(n=5,k=3)
#' try
#' phi_p(try)   #calculate the phi_p of "try".
#'
#' #Another example
#' #generate a 8 by 4 nearly orthogonal LHD
#' try2=LaPSO(n=8,k=4,OC="AvgAbsCor")
#' try2
#' AvgAbsCor(try2)  #calculate the average absolute correlation.
#' @export


LaPSO=function(n,k,m=10,N=10,SameNumP=0,SameNumG=n/4,p0=1/(k-1),OC="phi_p",p=15,q=1){
  #n and k are the rs and fa.
  #m: the number of particles
  #N: maximum number of iterations.

  #SameNumP: choose how many elements in column j of current LHD should be the same as
  #corresponding Personal Best. SameNumP=0, 1, 2, ..., n, and 0 means not run.
  #SameNumG: choose how many elements in column j of current LHD should be the same as
  #corresponding Global Best. SameNumG=0, 1, 2, ..., n, and 0 means not run.
  #p0: the probability of exchange two randomly selected elements in j of current LHD
  #OC: optimality criterion, the default is "phi_p", along with default p and q

  C=1  #Initialize counter index

  #step 2 starts, each X[,,i] is the L_i^{0}, i=1, ..., m
  X=rep(0,n*k*m)

  dim(X)=c(n,k,m)

  for (i in 1:m) {
    X[,,i]=rLHD(n=n,k=k)
  }
  #step 2 ends

  if(OC=="phi_p"){
    #step 3: Initialize personal best for each particle.
    pbest=X

    #step 4 starts: Initialize the global best
    gbest=X[,,1]

    for (i in 2:m) {
      if (phi_p(X[,,i],p=p,q=q)<phi_p(gbest,p=p,q=q)){gbest=X[,,i]}
    }

    #step 4 ends

    Xnew=X     #step 5


    while(C<=N){

      for (i in 1:m) {     #step 6
        for (j in 1:k) {   #step 7

          #step 8 starts

          if (SameNumP>0){
            for (a in 1:SameNumP) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_p=pbest[rrow,j,i]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_p){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_p]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_p
              }

            }
          }
          #step 8 ends

          #step 9 starts

          if (SameNumG>0){
            for (b in 1:SameNumG) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_g=gbest[rrow,j]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_g){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_g]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_g
              }

            }
          }
          #step 9 ends


          z=stats::runif(1,0,1)             #step 10

          if (z<p0){

            Xnew[,,i]=exchange(X=X[,,i],j=j)   #step 11

          }          #step 12: end if


        }   #step 13: end for

      }    #step 14: end for


      for (i in 1:m) {     #step 15

        if (phi_p(Xnew[,,i],p=p,q=q)<phi_p(pbest[,,i],p=p,q=q)){pbest[,,i]=Xnew[,,i]}   #step 16-19
        if (phi_p(Xnew[,,i],p=p,q=q)<phi_p(gbest,p=p,q=q)){gbest=Xnew[,,i]}             #step 20-22
      }      #step 23: end for

      C=C+1
    }

  }

  if(OC=="AvgAbsCor"){
    #step 3: Initialize personal best for each particle.
    pbest=X

    #step 4 starts: Initialize the global best
    gbest=X[,,1]

    for (i in 2:m) {
      if (AvgAbsCor(X[,,i])<AvgAbsCor(gbest)){gbest=X[,,i]}
    }

    #step 4 ends

    Xnew=X     #step 5


    while(C<=N){

      for (i in 1:m) {     #step 6
        for (j in 1:k) {   #step 7

          #step 8 starts

          if (SameNumP>0){
            for (a in 1:SameNumP) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_p=pbest[rrow,j,i]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_p){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_p]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_p
              }

            }
          }
          #step 8 ends

          #step 9 starts

          if (SameNumG>0){
            for (b in 1:SameNumG) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_g=gbest[rrow,j]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_g){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_g]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_g
              }

            }
          }
          #step 9 ends


          z=stats::runif(1,0,1)             #step 10

          if (z<p0){

            Xnew[,,i]=exchange(X=X[,,i],j=j)   #step 11

          }          #step 12: end if


        }   #step 13: end for

      }    #step 14: end for


      for (i in 1:m) {     #step 15

        if (AvgAbsCor(Xnew[,,i])<AvgAbsCor(pbest[,,i])){pbest[,,i]=Xnew[,,i]}   #step 16-19
        if (AvgAbsCor(Xnew[,,i])<AvgAbsCor(gbest)){gbest=Xnew[,,i]}             #step 20-22
      }      #step 23: end for

      C=C+1
    }

  }

  if(OC=="MaxAbsCor"){
    #step 3: Initialize personal best for each particle.
    pbest=X

    #step 4 starts: Initialize the global best
    gbest=X[,,1]

    for (i in 2:m) {
      if (MaxAbsCor(X[,,i])<MaxAbsCor(gbest)){gbest=X[,,i]}
    }

    #step 4 ends

    Xnew=X     #step 5


    while(C<=N){

      for (i in 1:m) {     #step 6
        for (j in 1:k) {   #step 7

          #step 8 starts

          if (SameNumP>0){
            for (a in 1:SameNumP) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_p=pbest[rrow,j,i]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_p){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_p]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_p
              }

            }
          }
          #step 8 ends

          #step 9 starts

          if (SameNumG>0){
            for (b in 1:SameNumG) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_g=gbest[rrow,j]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_g){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_g]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_g
              }

            }
          }
          #step 9 ends


          z=stats::runif(1,0,1)             #step 10

          if (z<p0){

            Xnew[,,i]=exchange(X=X[,,i],j=j)   #step 11

          }          #step 12: end if


        }   #step 13: end for

      }    #step 14: end for


      for (i in 1:m) {     #step 15

        if (MaxAbsCor(Xnew[,,i])<MaxAbsCor(pbest[,,i])){pbest[,,i]=Xnew[,,i]}   #step 16-19
        if (MaxAbsCor(Xnew[,,i])<MaxAbsCor(gbest)){gbest=Xnew[,,i]}             #step 20-22
      }      #step 23: end for

      C=C+1
    }

  }

  if(OC=="MaxProCriterion"){
    #step 3: Initialize personal best for each particle.
    pbest=X

    #step 4 starts: Initialize the global best
    gbest=X[,,1]

    for (i in 2:m) {
      if (MaxProCriterion(X[,,i])<MaxProCriterion(gbest)){gbest=X[,,i]}
    }

    #step 4 ends

    Xnew=X     #step 5


    while(C<=N){

      for (i in 1:m) {     #step 6
        for (j in 1:k) {   #step 7

          #step 8 starts

          if (SameNumP>0){
            for (a in 1:SameNumP) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_p=pbest[rrow,j,i]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_p){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_p]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_p
              }

            }
          }
          #step 8 ends

          #step 9 starts

          if (SameNumG>0){
            for (b in 1:SameNumG) {
              rrow=sample(1:n,1)
              e_r=Xnew[rrow,j,i]        #Randomly choose an element in column j, denoted by e_r

              e_g=gbest[rrow,j]       #e_p: the element in pbest whose location is same as e_r

              if (e_r!=e_g){

                #locate the row number of e_p in column j of current LHD, L_i^{new}.
                location=cbind(X[,j,i],1:n)[,2][cbind(X[,j,i],1:n)[,1]==e_g]

                Xnew[location,j,i]=e_r

                Xnew[rrow,j,i]=e_g
              }

            }
          }
          #step 9 ends


          z=stats::runif(1,0,1)             #step 10

          if (z<p0){

            Xnew[,,i]=exchange(X=X[,,i],j=j)   #step 11

          }          #step 12: end if


        }   #step 13: end for

      }    #step 14: end for


      for (i in 1:m) {     #step 15

        if (MaxProCriterion(Xnew[,,i])<MaxProCriterion(pbest[,,i])){pbest[,,i]=Xnew[,,i]}   #step 16-19
        if (MaxProCriterion(Xnew[,,i])<MaxProCriterion(gbest)){gbest=Xnew[,,i]}             #step 20-22
      }      #step 23: end for

      C=C+1
    }

  }

  gbest
}
