% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/InsuranceContract.R
\name{InsuranceContract}
\alias{InsuranceContract}
\title{Base Class for Insurance Contracts}
\description{
Base Class for Insurance Contracts

Base Class for Insurance Contracts
}
\details{
R6 class that models a complete, general insurance contract.
The corresponding tariff and the profit participation scheme, as well as
all other relevant contract parameters (if not defined by the tariff or
explicitly overridden by the contract) can be given in the constructor.
}
\section{Usage}{
The typical usage of this class is to simply call
\ifelse{html}{\href{#method-new}{\code{InsuranceContract$new()}}}{\code{InsuranceContract$new()()}}.

All parameters from the \link{InsuranceContract.ParameterDefaults} can be passed
to the constructor of the class (i.e. the \ifelse{html}{\href{#method-new}{\code{InsuranceContract$new()}}}{\code{InsuranceContract$new()()}}-call).
Parameters not explicitly given, will be taken from the tariff or as a fall-back
mechanism from the \link{InsuranceContract.ParameterDefaults} defaults.

Immediately upon construction, all premiums, reserves and cash flows for the
whole contract period are calculated and can be accessed via the \code{Values}
field of the object.
}

\section{Calculation approach: Valuation}{
The calculation of all contract values is controlled by the function
\ifelse{html}{\href{#method-calculateContract}{\code{InsuranceContract$calculateContract()}}}{\code{InsuranceContract$calculateContract()()}} (using methods of the \link{InsuranceTarif}
object) and follows the following logic:
\enumerate{
\item First the \strong{contingent (unit) cash flows} and the \strong{transition probbilities}
are determined.
\item The \strong{actuarial equivalence principle} states that at time of inception, the
(net and gross) premium must be determined in a way that the present value
of the future benefits and costs minus the present value of the future premiums
must be equal, i.e. in expectation the future premiums ove the whole lifetime
of the contract will exactly cover the benefits and costs. Similarly, at all
later time steps, the difference between these two present values needs to be
reserved (i.e. has already been paid by the customer by previous premiums).
\item This allows the premiums to be calculated by first calculating the \strong{present
values} for all of the \strong{benefit and costs cash flow} vectors.
\item The formulas
to calculate the gross, Zillmer and net \strong{premiums} involve simple linear
combinations of these present values, so the \strong{coefficients of these formulas}
are determined next.
\item With the coefficients of the premium formulas calculated, all \strong{premiums
can be calculated} (first the gross premium, because due to potential gross
premium refunds in case of death, the formula for the net premium requires
the gross premium, which the formula for the gross premium involves no other
type of premuim).
\item With premiums determined, all unit cash flows and unit present values can
now be expressed in monetary terms / as \strong{absolute cash flows} (i.e. the actual Euro-amount that flows
rather than a percentage).
\item As described above, the difference between the present values of premiums
and present values of benefits and costs is defined as the required amount
of reserves, so the \strong{reserves (net, gross, administration cost, balance sheet)}
and all values derived from them (i.e. surrender value, sum insured in case of
premium waiver, etc.) are calculated.
\item The \strong{decomposition of the premium} into parts dedicated to specific purposes
(tax, rebates, net premium, gross premium, Zillmer premium, cost components,
risk premium, savings premium, etc.) can be done once the reserves are
ready (since e.g. the savings premium is defined as the difference of
discounted reserves at times $t$ and $t+1$).
\item If the contract has \strong{(discretionary or obligatory) profit sharing}B mechanisms
included, the corresponding \link{ProfitParticipation} object can calculate that
profit sharing amounts, once all guaranteed values are calculated. This can
also be triggered manually (with custom profit sharing rates) by calling
the methods \ifelse{html}{\href{#method-profitScenario}{\code{InsuranceContract$profitScenario()}}}{\code{InsuranceContract$profitScenario()()}}]
or \ifelse{html}{\href{#method-addProfitScenario}{\code{InsuranceContract$addProfitScenario()}}}{\code{InsuranceContract$addProfitScenario()()}}.
}
}

\section{Calculation approach: Cash Flows}{
An insurance contract is basically defined by the (unit) cash flows it produces:
\itemize{
\item \strong{Premium payments} (in advance or in arrears) at each timestep
\item \strong{Survival payments} at each timestep
\item \strong{Guaranteed payments} at each timestep
\item \strong{Death benefits} at each timestep
\item \strong{Disease benefits} at each timestep
}
Together with the transition probabilities (mortalityTable parameter)
the present values can be calculated, from which the premiums follow and
finally the reserves and a potential profit sharing.

For example, a \emph{\strong{term life insurance with regular premiums}} would have the following
cash flows:
\itemize{
\item premium cash flows: 1, 1, 1, 1, 1, ...
\item survival cash flows: 0, 0, 0, 0, 0, ...
\item guaranteed cash flows: 0, 0, 0, 0, 0, ...
\item death benefit cash flows: 1, 1, 1, 1, 1, ...
}

A \emph{\strong{single-premium term life insurance}} would look similar, except for the premiums:
\itemize{
\item premium cash flows: 1, 0, 0, 0, 0, ...
}

A \emph{\strong{pure endowment}} has no death benefits, but a survival benefit of 1 at the
maturity of the contract:
\itemize{
\item premium cash flows: 1, 1, 1, 1, 1, ...
\item survival cash flows: 0, 0, ..., 0, 1
\item guaranteed cash flows: 0, 0, 0, 0, 0, ...
\item death benefit cash flows: 0, 0, 0, 0, 0, ...
}

An \emph{\strong{endowment}} has also death benefits during the contract duration:
\itemize{
\item premium cash flows: 1, 1, 1, 1, 1, ...
\item survival cash flows: 0, 0, ..., 0, 1
\item guaranteed cash flows: 0, 0, 0, 0, 0, ...
\item death benefit cash flows: 1, 1, 1, 1, 1, ...
}

A \emph{\strong{(deferred) annuity}B} has premium cash flows only during the deferral peroid
and only survival cash flows during the annuity payment phase. Often, in case
of death during the deferral period, all premiums paid are refunded as a death
benefit.:
\itemize{
\item premium cash flows: 1, 1, ...,  1, 0, 0, 0, ...
\item survival cash flows: 0, 0, ..., 0, 1, 1, 1,...
\item guaranteed cash flows: 0, 0, 0, 0, 0, ...
\item death benefit cash flows: 1, 2, 3, 4, 5, ..., 0, 0, ...
}

A \emph{\strong{terme-fix insurance}} has a guaranteed payment at maturity, even if the insured
has already died. The premiums, however, are only paid until death (which is
not reflected in the contingent cash flows, but rather in the transition
probabilities):
\itemize{
\item premium cash flows: 1, 1, 1, 1, ...,  1
\item survival cash flows: 0, 0, 0, 0, ..., 0
\item guaranteed cash flows: 0, 0, 0, ..., 0, 1
\item death benefit cash flows: 0, 0, 0, 0, ..., 0
}

The \code{InsuranceContract$new()} function creates a new
insurance contract for the given tariff, using the parameters passed
to the function (and the defaults specified in the tariff).

As soon as this function is called, the contract object calculates
all time series (cash flows, premiums, reserves, profit participation)
for the whole contract duration.

The most important parameters that are typically passed to the
constructor are:
\itemize{
\item \code{age} ... Age of the insured person (used to derive mortalities / transition probabilities)
\item \code{policyPeriod} ... Maturity of the policy (in years)
\item \code{premiumPeriod} ... How long premiums are paid (\code{premiumPeriod = 1}
for single-premium contracts, \code{premiumPeriod} equals
\code{policyPeriod} for regular premium payments for the whole
contract period, while other premium payment durations indicate
premium payments only for shorter periods than the whole contract
duration). Default is equal to \code{policyPeriod}
\item \code{sumInsured} ... The sum insured (i.e. survival benefit for
endowments, death benefit for whole/term life insurances,
annuity payments for annuities)
\item \code{contractClosing} ... Date of the contract beginning (typically
created using something like \code{as.Date("2020-08-01")})
\item \code{YOB} ... Year of birth of the insured (for cohort mortality
tables). If not given, YOB is derived from \code{age} and
\code{contractClosing}.
\item \code{deferralPeriod} ... Deferral period for deferred annuities
(i.e. when annuity payments start at a future point in time).
Default is 0.
\item \code{premiumFrequency} ... How many premium payments per year are
made (e.g. 1 for yearly premiums, 4 for quarterly premiumd,
12 for monthly premium payments). Default is 1 (yearly premiums).
}

While these are the most common and most important parameters, all
parameters can be overwritten on a per-contract basis, even those
that are defined by the tariff. For a full list and explanation of all
parameters, see \link{InsuranceContract.ParameterDefaults}.

The \code{InsuranceContract$addHistorySnapshot()} function
adds the current (or the explicitly given) state of the contract
(parameters, calculated values, tarif, list of all contract blocks)
to the history list (available in the \code{history} field of the
contract, i.e. \code{InsuranceContract$history}).

Contracts with multiple contract blocks (typically either
contracts with dynamic increases, sum increases or protection riders)
are constructed by instantiating the child block (e.g. a single
dynamic increase or the rider) independently with its own (shorter)
duration and then inserting it into the parent contract with this
function at the given time.

If no \link{InsuranceContract} object is passed as \code{block}, a copy
of the parent is created with overriding parameters given in \code{...}.

This method adds a new contract block describing a dynamic
or sum increase (increasing the sum insured at a later time
$t$ than contract inception). This increase is modelled by a
separate \link{InsuranceContract} object with the sum difference
as its own \code{sumInsured}.

\if{html}{\out{<div class="sourceCode">}}\preformatted{     By default, all parameters are taken from the main contract,
     with the maturity adjusted to match the original contract's
     maturity.

     The main contract holds all child blocks, controls their
     valueation and aggregates all children's values to the
     total values of the overall contract.
}\if{html}{\out{</div>}}

This method calculates all contract values (potentially
starting from and preserving all values before a later time
\code{valuesFrom}). This function is not meant to be called
directly, but internally, whenever a contract is created or
modified.

\if{html}{\out{<div class="sourceCode">}}\preformatted{     There is, hoever, a legitimate case to call this function
     when a contract was initially created with a value of
     \code{calculate} other than "all", so not all values of the
     contract were calculated. When one later needs more values
     than were initially calculated, this function can be called.
     However, any contract changes might need to be rolled back
     and reapplied again afterwards. So even in this case it is
     probably easier to create the contract object from scratch
     again.
}\if{html}{\out{</div>}}

This function is an internal function for contracts with
multiple child blocks (dynamic increases, sum increases, riders).
It takes the values from all child blocks and calculates the
overall values from all child blocks aggregated.

\if{html}{\out{<div class="sourceCode">}}\preformatted{   This function should not be called manually.
}\if{html}{\out{</div>}}

This function modifies the contract at time $t$ so that
no further premiums are paid (i.e. a paid-up contract) and the
\code{sumInsured} is adjusted according to the existing reserves.

This function calculates one profit scenario with the
provided profit participation parameters (all parameters
not given in the call are taken from their values of the
contract, profit participation scheme or tariff).

This function calculates one profit scenario with the
provided profit participation parameters (all parameters
not given in the call are taken from their values of the
contract, profit participation scheme or tariff). The results
are stored in a list of profit scenarios inside the contract.

\if{html}{\out{<div class="sourceCode">}}\preformatted{    This function can be chained to calculate and add multiple
    profit scenarios.
}\if{html}{\out{</div>}}
}

\examples{
# TODO


## ------------------------------------------------
## Method `InsuranceContract$addHistorySnapshot`
## ------------------------------------------------

# TODO

## ------------------------------------------------
## Method `InsuranceContract$addBlock`
## ------------------------------------------------

# TODO

## ------------------------------------------------
## Method `InsuranceContract$addDynamics`
## ------------------------------------------------

# TODO

## ------------------------------------------------
## Method `InsuranceContract$premiumWaiver`
## ------------------------------------------------

# TODO

## ------------------------------------------------
## Method `InsuranceContract$profitScenario`
## ------------------------------------------------

# TODO

## ------------------------------------------------
## Method `InsuranceContract$addProfitScenario`
## ------------------------------------------------

# TODO
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{tarif}}{The \link{InsuranceTarif} underlying this contract. The tarif is the abstract
product description (i.e. defining the type of insurance, fixing tpyes
of benefits, specifying costs, guaranteed interest rate, mortality tables,
potential profit sharing mechanisms, etc.), while the contract holds
the individual parts like age, sum insured, contract duration, premium
payment frequency, etc.}

\item{\code{parent}}{A pointer to the parent contract. Some contracts consist of multiple
parts (e.g. a main savings contract with a dread-disease rider, or
a contract with multiple dynamic increases). These are internally
represented by one \link{InsuranceContract} object per contract part, plus
one contract object combining them and deriving combined premiums,
reserves and profit participation. The child contracts (i.e. the
objects representing the individual parts) have a pointer to their
parent, while the overall contract holds a list of all its child contract parts.}

\item{\code{ContractParameters}}{Insurance contract parameters explicitly specified in the contract
(i.e. parameters that are NOT taken from the tariff of the defaults).}

\item{\code{Parameters}}{Full set of insurance contract parameters applying to this contract.
The set of parameters is a combination of explicitly given (contract-specific)
values, parameters determined by the tariff and default values.}

\item{\code{Values}}{List of all contract values (cash flows, present values, premiums,
reserves, premium decomposition, profit participation, etc.). These
values will be calculated and filled when the contract is created
and updated whenever the contract is changed.}

\item{\code{blocks}}{For contracts with multiple contract parts: List of all tariff blocks
(independently calculated \link{InsuranceContract} objects, that are combined
to one contract, e.g. dynamic/sum increases). If this field is empty,
this object describes a contract block (calculated as a stand-alone
tariff), otherwise it will simply be the sum of its blocks (adjusted
to span the same time periods)}

\item{\code{history}}{A list keeping track of all contract changes (including the whole
contract state and its values before the change).}

\item{\code{dummy.public}}{dummy field to allow a trailing comma after the previous field/method}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-InsuranceContract-new}{\code{InsuranceContract$new()}}
\item \href{#method-InsuranceContract-addHistorySnapshot}{\code{InsuranceContract$addHistorySnapshot()}}
\item \href{#method-InsuranceContract-addBlock}{\code{InsuranceContract$addBlock()}}
\item \href{#method-InsuranceContract-addDynamics}{\code{InsuranceContract$addDynamics()}}
\item \href{#method-InsuranceContract-calculateContract}{\code{InsuranceContract$calculateContract()}}
\item \href{#method-InsuranceContract-consolidateBlocks}{\code{InsuranceContract$consolidateBlocks()}}
\item \href{#method-InsuranceContract-premiumWaiver}{\code{InsuranceContract$premiumWaiver()}}
\item \href{#method-InsuranceContract-profitScenario}{\code{InsuranceContract$profitScenario()}}
\item \href{#method-InsuranceContract-addProfitScenario}{\code{InsuranceContract$addProfitScenario()}}
\item \href{#method-InsuranceContract-clone}{\code{InsuranceContract$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-new"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-new}{}}}
\subsection{Method \code{new()}}{
Create a new insurance contract (for the given tariff/product) and calculate all time series
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$new(
  tarif,
  parent = NULL,
  calculate = "all",
  profitid = "default",
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{tarif}}{The \link{InsuranceTarif} object describing the Tariff/Product
and providing defaults for the parameters.}

\item{\code{parent}}{For contracts with multiple contract blocks (dynamic
increases, sum increases, riders), each child is created with
a pointer to its parent. NULL for single-block contracts or
for the overall-contract of a multi-block contract. This
parameter is used internally, but should not be used in
user-written code.}

\item{\code{calculate}}{how much of the contract's time series need to be
calculated. See \link{CalculationEnum} for all possible values. This
is usefull to prevent calculation of e.g. reserves and profit
participation, when one only wants to create a grid of premiums.}

\item{\code{profitid}}{The ID of the default profit participation scenario.
The default profit participation scenario uses the default
values passed, while further scenarios can be added by
\ifelse{html}{\href{#method-addProfitScenario}{\code{InsuranceContract$addProfitScenario()}}}{\code{InsuranceContract$addProfitScenario()()}}.}

\item{\code{...}}{Further parameters (age, sum insured, contract closing /
begin, premium payment details, etc.) of the contract, which
can also override parameters defined at the tariff-level.
Possible values are all sub-fields of the
\link{InsuranceContract.ParameterDefaults} data structure.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-addHistorySnapshot"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-addHistorySnapshot}{}}}
\subsection{Method \code{addHistorySnapshot()}}{
Add the current state of the contract to the history list
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$addHistorySnapshot(
  time = 0,
  comment = "Initial contract values",
  type = "Contract",
  params = self$Parameters,
  values = self$Values,
  tarif = self$tarif,
  blocks = self$blocks
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{time}}{the time described by the snapshot}

\item{\code{comment}}{a comment to store together with the contract state}

\item{\code{type}}{The type of action that caused a history snapshot to
be stored. Typical values are "Contract" to describe the initial
contract, "Premium Waiver" or "Dynamic Increase".}

\item{\code{params}}{The set of params to be stored in the history snapshot
(default is \code{self$Parameters}, if not explicitly given)}

\item{\code{values}}{The calculated time series of all contract values
calculated so far. Default is \code{self$Values}, if not
explicitly given}

\item{\code{tarif}}{The underlying \link{InsuranceTarif} object describing the
Product/Tariff. Default is \code{self$tarif}, if not explicitly given.}

\item{\code{blocks}}{The list of all contract children for contracts with
multiple insurance blocks (e.g. dynamic increases, riders, etc.)}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-addBlock"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-addBlock}{}}}
\subsection{Method \code{addBlock()}}{
Add a child contract block (e.g. a dynamic increase or a rider) to an insurance contract
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$addBlock(
  id = NULL,
  block = NULL,
  t = block$Values$int$blockStart,
  comment = paste0("Additional block at time t=", t),
  ...
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{The identifier of the child block to be inserted}

\item{\code{block}}{The \link{InsuranceContract} object describing the child block.
If NULL (or not given at all), a copy of the parent will be
created.}

\item{\code{t}}{Then the child block starts, relative to the parent block.
The child block is calculated independently (with time 0
describing its own start), so when aggregating all values from
the individual blocks to overall values for the whole contract,
the child's values need to be translated to the parent contracts's
time frame using this parameter}

\item{\code{comment}}{The comment to use in the history snapshot.}

\item{\code{...}}{parameters to be passed to \ifelse{html}{\href{#method-new}{\code{InsuranceContract$new()}}}{\code{InsuranceContract$new()()}} when
\code{block} is not given and a copy of the parent should be
created with overrides.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-addDynamics"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-addDynamics}{}}}
\subsection{Method \code{addDynamics()}}{
Add a dynamic increase with the same parameters as the main contract part
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$addDynamics(t, NewSumInsured, SumInsuredDelta, id, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{The time within the main contract when the sum increase happens.
The \link{InsuranceContract} object describing the dynamic increase
will still internally start at its own time 0, but the
aggregation by the main contract will correctly offset to
the time $t$ within the main contract.}

\item{\code{NewSumInsured}}{The over-all new sum insured (sum of original
contract and all dynamica increaeses). The \code{sumInsured}
of the new dynamic increase block will be determined as the
difference of the old and new overall sum insured. Alternatively,
it can directly be given as the \code{SumInsuredDelta}
argument instead.}

\item{\code{SumInsuredDelta}}{The sum insured of only the dynamic increase,
i.e. the sumInsured of the dynamic contract block only. The
overall sum insured will increase by this amount. Only one of
\code{NewSumInsured} and \code{SumInsuredDelta} is needed,
the other one will be calculated accordingly. If both are
given, the \code{SumInsuredDelta} will take precedence.}

\item{\code{id}}{The identifier of the contract block describing the dynamic
increase. This is a free-form string that should be unique
within the list of child blocks. It will be displayed in the
Excel export feature and in the history snapshot list.}

\item{\code{...}}{Paramters to override in the dynamic block. By default,
all parameters of the main contract block will be used, but
they can be overridden per dynamic increase block.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-calculateContract"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-calculateContract}{}}}
\subsection{Method \code{calculateContract()}}{
Calculate all time series of the contract from the parameters
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$calculateContract(
  calculate = "all",
  valuesFrom = 0,
  premiumCalculationTime = 0,
  preservePastPV = TRUE,
  additionalCapital = 0,
  recalculatePremiums = TRUE,
  recalculatePremiumSum = TRUE,
  history_comment = NULL,
  history_type = "Contract"
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{calculate}}{Which values to calculate. See \link{CalculationEnum}}

\item{\code{valuesFrom}}{Calculate only values starting from this time step
on (all values before that time will be preserved). This is
required when a contract is changed significantly (potentially
even switching to a new tariff), so that the calculation bases
for previous periods are no longer available.}

\item{\code{premiumCalculationTime}}{The time point when the premium should
be re-calculated (including existing reserves) based on the
actuarial equivalence principle. All reserves will be based on
these new premiums.}

\item{\code{preservePastPV}}{Whether present value before the recalculation
time \code{valuesFrom} should be preserved or recalculated.
When they are recalculated, the present values are consistent
to the new cash flows over the whole contract period, but
they no longer represent the actual contract state at these
times. If values are not recalculated, the reserves at each
time step represent the proper state at that point in time.}

\item{\code{additionalCapital}}{The capital that is added to the contract
(e.g. capital carried over from a previous contract) at the
premium calculation time.}

\item{\code{recalculatePremiums}}{Whether the premiums should be recalculated
at time \code{premiumCalculationTime} at all.}

\item{\code{recalculatePremiumSum}}{Whether to recalculate the overall premium
sum when the premium is recalculated.}

\item{\code{history_comment}}{The comment for the history snapshot entyr}

\item{\code{history_type}}{The type (free-form string) to record in the history snapshot}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-consolidateBlocks"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-consolidateBlocks}{}}}
\subsection{Method \code{consolidateBlocks()}}{
Aggregate values from all child contract blocks (if any)
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$consolidateBlocks(valuesFrom = 0)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{valuesFrom}}{The time from when to aggragate values. Values before
that time will be left unchanged.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-premiumWaiver"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-premiumWaiver}{}}}
\subsection{Method \code{premiumWaiver()}}{
Stop premium payments and re-calculate sumInsured of the paid-up contract
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$premiumWaiver(t, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{t}}{Time of the premium waiver.}

\item{\code{...}}{Further parameters (currently unused)}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-profitScenario"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-profitScenario}{}}}
\subsection{Method \code{profitScenario()}}{
Calculate one profit scenario and return all values
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$profitScenario(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{Scenario-specific profit sharing parameters, overriding
the default values. Typically, adjusted profit rates are required
in a profitScenario.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a data.frame holding all profit participation values (rates,
bases for the different profit types, profit allocations,
terminal bonus funds, profit in case of death/surrender/premium waiver)
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-addProfitScenario"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-addProfitScenario}{}}}
\subsection{Method \code{addProfitScenario()}}{
Calculate one profit scenario and store it in the contract
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$addProfitScenario(id, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{The unique ID of the profit scenario. Will be used as key
in the list of profit scenarios and printed out in the Excel
export.}

\item{\code{...}}{Scenario-specific profit sharing parameters, overriding
the default values. Typically, adjusted profit rates are required
in a profitScenario.}
}
\if{html}{\out{</div>}}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# TODO
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-InsuranceContract-clone"></a>}}
\if{latex}{\out{\hypertarget{method-InsuranceContract-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{InsuranceContract$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
