% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.functions.R
\name{mb.quadratic}
\alias{mb.quadratic}
\title{Run MBNMA model with a quadratic time-course function}
\usage{
mb.quadratic(network, beta.1 = list(pool = "rel", method = "common"),
  beta.2 = list(pool = "rel", method = "common"), alpha = "study",
  positive.scale = FALSE, intercept = TRUE, rho = NULL,
  covar = NULL, var.scale = NULL, class.effect = list(),
  UME = FALSE, pd = "pv", parallel = TRUE, priors = NULL, ...)
}
\arguments{
\item{network}{An object of class \code{mb.network}.}

\item{beta.1}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{beta.2}{A list with named elements \code{pool} and \code{method} that refers to
time-course parameter(s) specified within the time-course function (see details).}

\item{alpha}{Refers to the baseline mean response and is a character object
that can take either:
\itemize{
\item \code{"study"} to constrain baseline to be equal for all
arms within a study (\code{i} index is added))
\item \code{"arm"} to allow baseline to
vary between arms within a study (\code{i,k} index is added)).
}}

\item{positive.scale}{A boolean object that indicates whether all continuous
mean responses (y) are positive and therefore whether the baseline response
should be given a prior that constrains it to be positive.}

\item{intercept}{A boolean object that indicates whether an intercept is to
be included in the model. Can be used to imply whether mean responses in
data are change from baseline (\code{FALSE}) or not (setting it to \code{FALSE}
removes the intercept, \code{alpha}, from the model).}

\item{rho}{The correlation coefficient when modelling correlation between time points. If left
as \code{NULL} (the default) then this implies modelling no correlation between time points.
Can either be assigned the string \code{"estimate"} to indicate that rho should be estimated
from the data, or assigned a numeric value, which fixes \code{rho} in the model to the assigned
value, either for when \code{rho} is calculated externally or for use in deterministic sensitivity
analyses.}

\item{covar}{A character specifying the covariance structure to use for the
multivariate normal likelihood. Can currently take either \code{"CS"} (compound
symmetry) or \code{"AR1"} (autoregressive AR1).}

\item{var.scale}{A numeric vector indicating the relative scale of variances between
correlated time-course parameters when relative effects are modelled on more than
one time-course parameter(see Details LINK). Each element of
the vector refers to the relative scale of each of the time-course parameters that is
modelled using relative effects.}

\item{class.effect}{A list of named strings that determines which time-course
parameters to model with a class effect and what that effect should be
(\code{"common"} or \code{"random"}). For example: \code{list("beta.2"="common", "beta.3"="random")}.}

\item{UME}{Can take either \code{TRUE} or \code{FALSE} (for an unrelated mean effects
model on all or no time-course parameters respectively) or can be a vector
of parameter name strings to model as UME. For example: \code{c("beta.1", "beta.2")}.}

\item{pd}{Can take either:
\itemize{
\item \code{pv} only pV will be reported (as automatically outputted by R2jags).
\item \code{plugin} calculates pD by the plug-in
method \insertCite{spiegelhalter2002}{MBNMAtime}. It is faster, but may output negative
non-sensical values, due to skewed deviances that can arise with non-linear models.
\item \code{pd.kl} calculates pD by the Kullback–Leibler divergence \insertCite{plummer2008}{MBNMAtime}. This
will require running the model for additional iterations but
will always produce a positive result.
\item \code{popt} calculates pD using an optimism adjustment which allows for calculation
of the penalized expected deviance \insertCite{plummer2008}{MBNMAtime}
}}

\item{parallel}{A boolean value that indicates whether JAGS should be run in
parallel (\code{TRUE}) or not (\code{FALSE}). If \code{TRUE} then the number of cores to
use is automatically calculated.}

\item{priors}{A named list of parameter values (without indices) and
replacement prior distribution values given as strings
\strong{using distributions as specified in JAGS syntax}.}

\item{...}{Arguments to be sent to \code{mb.run()}}
}
\value{
An object of S3 class \code{c("mbnma", "rjags")`` containing parameter results from the model. Can be summarized by }print()\code{and can check traceplots using}R2jags::traceplot()\code{or various functions from the package}mcmcplots`.

Nodes that are automatically monitored (if present in the model) have the
following interpretation. They will have an additional suffix that relates
to the name/number of the time-course parameter to which they correspond
(e.g. \code{d.et50} or \code{d.1}):
\itemize{
\item \code{d} The pooled relative effect for a given
treatment compared to the network reference treatment for a particular
time-course parameter, reported if \code{pool="rel"}
\item \code{sd.d} The between-study SD (heterogeneity) for relative effects,
reported if \code{pool="rel"} and \code{method="random"}
\item \code{D} The class effect for a given class compared to the
network reference class for a particular time-course parameter
\item \code{sd.D} The standard deviation for the pooled relative effects of treatments within a
given class from a model with a random class effect.
\item \code{beta} If \code{pool="const"} then only a single node will be present in the
output, which corresponds to the absolute value of a particular time-course
parameter across the network, If \code{pool="arm"}
then for the relevant time-course parameter there will be one node for
each treatment, which represents the absolute value of the time-course
parameter for each treatment
\item \code{sd.beta} Reported if \code{method="random"} and \code{pool} is either \code{"const"} or \code{"arm"}.
If \code{pool="const"} this represents the between-study SD for the absolute value of a particular
time-course parameter exchangeable across the network. If \code{pool="arm"}
this represents the between-study SD for the absolute value of a
particular time-course parameter exchangeable by treatment
\item \code{rho} The correlation coefficient for correlation between time-points. Its
interpretation will differ depending on the covariance structure used
\item \code{totresdev} The residual deviance of the model
\item \code{deviance} The deviance of the model
}

If there are errors in the JAGS model code then the object will be a list
consisting of two elements - an error message from JAGS that can help with
debugging and \code{model.arg}, a list of arguments provided to \code{mb.run()}
which includes \code{jagscode}, the JAGS code for the model that can help
users identify the source of the error.
}
\description{
Fits a Bayesian model-based network meta-analysis (MBNMA) with a defined
time-course function. This function accounts for repeated measures over time
within studies by applying a quadratic time-course function. Follows the
methods of \insertCite{pedder2019;textual}{MBNMAtime}. This function acts as a wrapper for \code{mb.run()}
that allows for more clearly defined parameter names.
}
\section{Time-course parameters}{

Time-course parameters in the model must be provided as a list with named elements
\code{pool} and \code{method}.

\code{pool} is used to define the approach used for pooling of a given time-course parameter and
can take any of the following values:
\itemize{
\item \code{"rel"} indicates that relative effects should be pooled for this time-course parameter.
This preserves randomisation
within included studies, are likely to vary less between studies (only due to effect modification),
and allow for testing of consistency between direct and indirect evidence. Pooling follows the
general approach for Network Meta-Analysis proposed by \insertCite{lu2004;textual}{MBNMAtime}.
\item \code{"arm"} indicates that study arms should be pooled within each treatment in the network
for this time-course parameter.
This allows estimation of absolute time-course parameter values, but makes stronger assumptions
regarding similarity of studies.
\item \code{"const"} indicates that study arms should be pooled across the whole network for this
time-course parameter  \emph{independently of assigned treatment}.
This implies using a single value across the network for this time-course parameter,
and may therefore be making very strong assumptions of similarity.
}

\code{method} is used to define the model used for meta-analysis for a given time-course parameter
and can take any of the following values:
\itemize{
\item \code{"common"} implies that all studies estimate the same true effect
(akin to a "fixed effects" meta-analysis)
\item \code{"random"} implies that all studies estimate a separate true effect, but that each
of these true effects vary randomly around a true mean effect. This approach allows
for modelling of between-study heterogeneity.
\item \code{numeric()} Assigned a numeric value - this can only be used if \code{pool="const"}. It indicates that
this time-course parameter should not be estimated from the data but should be assigned
the numeric value determined by the user. This can be useful for fixing specific time-course
parameters (e.g. Hill parameters in Emax functions or knot location in piecewise functions).
}

When relative effects are modelled on more than one time-course parameter,
correlation between the time-course parameters is automatically
estimated using a vague Wishart prior. This prior can be made slightly more informative
by specifying the relative scale of variances between the time-course parameters using
\code{var.scale}.
}

\section{Correlation between observations}{

When modelling correlation between observations using \code{rho}, values for \code{rho} must imply a
positive semidefinite covariance matrix. If estimating \code{rho} from the data (by assigning it
\code{"estimate"}), the default prior distribution (\code{dunif(-1,1)}) may include values that exclude
a positive semidefinite covariance matrix. This prior can be restricted (e.g. to \code{dunif(0,1)})
using the \code{priors} argument (see \code{\link{get.prior}})
}

\examples{
\donttest{
# Create mb.network object
network <- mb.network(osteopain)

# Fit quadratic time-course with fixed consistency treatment effects on beta.1 and
#random consistency treatment effects on beta.2
result <- mb.quadratic(network,
  beta.1=list(pool="rel", method="common"),
  beta.2=list(pool="rel", method="random"))


####### Examine MCMC diagnostics (using mcmcplots package) #######

# Density plots
mcmcplots::denplot(result, parms=c("sd.2", "d.1[3]", "d.2[3]", "totresdev"))

# Traceplots
mcmcplots::traplot(result)

# Caterpillar plots
mcmcplots::caterplot(result, "d.2")


########## Output ###########

# Print R2jags output and summary
print(result)
summary(result)

# Plot forest plot of results
plot(result)


###### Additional model arguments ######

# Fit model with unrelated mean effects on beta.1
mb.quadratic(network, alpha="study",
  beta.1=list(pool="rel", method="random"),
  beta.2=list(pool="const", method="common"),
  UME="beta.1"
  )
}
}
\references{
\insertAllCited
}
