% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/moead.R
\name{moead}
\alias{moead}
\title{MOEA/D}
\usage{
moead(preset = NULL, problem = NULL, decomp = NULL, aggfun = NULL,
  neighbors = NULL, variation = NULL, update = NULL, constraint = NULL,
  scaling = NULL, stopcrit = NULL, showpars = NULL, seed = NULL, ...)
}
\arguments{
\item{preset}{List object containing preset values for one or more
of the other parameters of the \code{moead} function. Values provided in
the \code{preset} list will override any other value provided. Presets should be
generated by the \code{\link[=preset_moead]{preset_moead()}} function.}

\item{problem}{List containing the problem parameters.
See \code{Problem Description} for details.}

\item{decomp}{List containing the decomposition method parameters
See \code{Decomposition methods} for details.}

\item{aggfun}{List containing the aggregation function parameters
See \code{Scalarization methods} for details.}

\item{neighbors}{List containing the decomposition method parameters
See \code{Neighborhood strategies} for details.}

\item{variation}{List containing the variation operator parameters
See \code{Variation operators} for details.}

\item{update}{List containing the population update parameters
See \code{Update strategies} for details.}

\item{constraint}{List containing the constraint handing parameters
See \code{Constraint operators} for details.}

\item{scaling}{List containing the objective scaling parameters
See \code{Objective scaling} for details.}

\item{stopcrit}{list containing the stop criteria parameters.
See \code{Stop criteria} for details.}

\item{showpars}{list containing the echoing behavior parameters.
See \code{\link[=print_progress]{print_progress()}} for details.}

\item{seed}{seed for the pseudorandom number generator. Defaults to NULL,
in which case \code{as.integer(Sys.time())} is used for the definition.}

\item{...}{Other parameters (useful for development and debugging, not
necessary in regular use)}
}
\value{
List object of class \emph{moead} containing:
\itemize{
\item information on the final population (\code{X}), its objective values (\code{Y}) and
constraint information list (\code{V}) (see \code{\link[=evaluate_population]{evaluate_population()}} for details);
\item Archive population list containing its corresponding \code{X}, \code{Y} and \code{V}
fields (only if \code{update$UseArchive = TRUE}).
\item Estimates of the \emph{ideal} and \emph{nadir} points, calculated for the final
population;
\item Number of function evaluations, iterations, and total execution time;
\item Random seed employed in the run, for reproducibility
}
}
\description{
MOEA/D implementation in R
}
\details{
Component-wise implementation of the Multiobjective Evolutionary Algorithm
based on decomposition - MOEA/D.
}
\section{Problem Description}{

The \code{problem} parameter consists of a list with all necessary
definitions for the multiobjective optimization problem to be solved.
\code{problem} must contain at least the following fields:
\itemize{
\item \code{problem$name}: name of the problem instance function, that is, a
routine that calculates \strong{Y} = \strong{f}(\strong{X});
\item \code{problem$xmin}: vector of lower bounds of each variable
\item \code{problem$xmax}: vector of upper bounds of each variable
\item \code{problem$m}:  integer indicating the number of objectives
}

Besides these fields, \code{problem} should contain any other relevant inputs
for the routine listed in \code{$name}. \code{problem} may also contain the
(optional) field \code{problem$constraints}, which is a list object
containing information about the problem constraints. If present, this list
must have the following fields:
\itemize{
\item \code{problem$constraints$name} - (required) name of the function that
calculates the constraint values (see below for details)
\item \code{problem$constraints$epsilon} - (optional) a small non-negative value
indicating the tolerance to be considered for equality constraints.
Defaults to zero.
}

Besides these fields, \code{problem$constraint} should contain any other
relevant inputs for the routine listed in \code{problem$constraint$name}.

Detailed instructions for defining the routines for calculating the
objective and constraint functions are provided in the vignette
\emph{Defining Problems in the MOEADr Package}. Check that documentation for
details.
}

\section{Decomposition Methods}{

The \code{decomp} parameter is a list that defines the method to be used for the
generation of the weight vectors. \code{decomp} must have
at least the \code{$name} parameter. Currently available methods can be
verified using \code{\link[=get_decomposition_methods]{get_decomposition_methods()}}. Check
\code{\link[=generate_weights]{generate_weights()}} and the information provided by
\code{\link[=get_decomposition_methods]{get_decomposition_methods()}} for more details.
}

\section{Neighborhood Strategies}{

The \code{neighbors} parameter is a list that defines the method for defining the
neighborhood relations among subproblems. \code{neighbors} must have
at least three parameters:
\itemize{
\item \code{neighbors$name}, name of the strategy used to define the neighborhoods.
Currently available methods are:
- \code{$name = "lambda"}: uses the distances between weight vectors.
The calculation is performed only once for the entire run,
since the weight vectors are assumed static.
- \code{$name = "x"}: uses the distances between the incumbent solutions
associated with each subproblem. In this case the calculation is
performed at each iteration, since incumbent solutions may change.
\item \code{neighbors$T}: defines the neighborhood size. This parameter must receive
a value smaller than the number of subproblems defined for the MOEA/D.
\item \code{neighbors$delta.p}: parameter that defines the probability of sampling
from the neighborhood when performing variation.
}

Check \code{\link[=define_neighborhood]{define_neighborhood()}} for more details.
}

\section{Variation Operators}{

The \code{variation} parameter consists of a list vector, in which each
sublist defines a variation operator to be used as part of the variation
block. Each sublist must have at least a field \code{$name}, containing the name
of the \code{i}-th variation operator to be applied. Use
\code{\link[=get_variation_operators]{get_variation_operators()}} to generate a list of available operators, and
consult the vignette \code{Variation Stack in the MOEADr Package} for more
details.
}

\section{Scalar Aggregation Functions}{

The \code{aggfun} parameter is a list that defines the scalar aggregation function
to be used. \code{aggfun} must have at least the \code{$name} parameter. Currently
available methods can be verified using \code{\link[=get_scalarization_methods]{get_scalarization_methods()}}. Check
\code{\link[=scalarize_values]{scalarize_values()}} and the information provided by
\code{\link[=get_scalarization_methods]{get_scalarization_methods()}} for more details.
}

\section{Update Methods}{

The \code{update} parameter is a list that defines the population update strategy
to be used. \code{update} must have at least the \code{$name} parameter. Currently
available methods can be verified using \code{\link[=get_update_methods]{get_update_methods()}}. Check
\code{\link[=update_population]{update_population()}} and the information provided by
\code{\link[=get_update_methods]{get_update_methods()}} for more details.

Another (optional) field of the \code{update} parameter is \code{update$UseArchive},
which is a binary flag defining whether the algorithm should keep an
external solution archive (\code{TRUE}) or not (\code{FALSE}). Since it adds to the
computational burden and memory requirements of the algorithm, the use of an
archive population is recommended only in the case of constrained problems
with constraint handling method that can occasionally accept unfeasible
solutions, leading to the potential loss of feasible efficient solutions for
certain subproblems (e.g., \code{\link[=constraint_vbr]{constraint_vbr()}} with \code{type} = "sr" or "vt").
}

\section{Constraint Handling Methods}{

The \code{constraint} parameter is a list that defines the constraint-handling
technique to be used. \code{constraint} must have at least the \code{$name} parameter.
Currently available methods can be verified using \code{\link[=get_constraint_methods]{get_constraint_methods()}}.
Check \code{\link[=update_population]{update_population()}} and the information provided by
\code{\link[=get_constraint_methods]{get_constraint_methods()}} for more details.
}

\section{Objective Scaling}{

Objective scaling refers to the re-scaling of the objective values at each
iteration, which is generally considered to prevent problems arising from
differently-scaled objective functions. \code{scaling} is a list that must have
at least the \code{$name} parameter. Currently available options are
\code{$name = "none"}, which does not perform any scaling, and \code{$name = "simple"},
which performs a simple linear scaling of the objectives to the interval
\code{[0, 1]}.
}

\section{Stop Criteria}{

The \code{stopcrit} parameter consists of a list vector, in which each
sublist defines a termination criterion to be used for the MOEA/D. Each
sublist must have at least a field \code{$name}, containing the name of the
\code{i}-th criterion to be verified. The iterative cycle of the MOEA/D is
terminated whenever any criterion is met. Use \code{\link[=get_stop_criteria]{get_stop_criteria()}} to
generate a list of available criteria, and check the information provided by
that function for more details.
}

\section{Echoing Options}{

The \code{showpars} parameter is a list that defines the echoing options of the
MOEA/D. \code{showpars} must contain two fields:
\itemize{
\item \code{showpars$show.iters}, defining the type of echoing output. \code{$show.iters}
can be set as \code{"none"}, \code{"numbers"}, or \code{"dots"}.
\item \code{showpars$showevery}, defining the period of echoing (in iterations).
\code{$showevery} must be a positive integer.
}
}

\section{References}{

F. Campelo, L.S. Batista, C. Aranha:
"The MOEADr Package - A Component-Based Framework for Multiobjective
Evolutionary Algorithms Based on Decomposition". In preparation, 2017.
}

\examples{
## Prepare a test problem composed of minimization of the (shifted)
## sphere and Rastrigin functions
sphere     <- function(x){sum((x + seq_along(x) * 0.1) ^ 2)}
rastringin <- function(x){
                x.shift <- x - seq_along(x) * 0.1
                sum((x.shift) ^ 2 - 10 * cos(2 * pi * x.shift) + 10)}
problem.sr <- function(X){
                t(apply(X, MARGIN = 1,
                FUN = function(X){c(sphere(X), rastringin(X))}))}


## Set the input parameters for the moead() routine
## This reproduces the Original MOEA/D of Zhang and Li (2007)
## (with a few changes in the computational budget, to make it run faster)
problem   <- list(name       = "problem.sr",
                  xmin       = rep(-1, 30),
                  xmax       = rep(1, 30),
                  m          = 2)
decomp    <- list(name       = "SLD", H = 49) # <-- H = 99 in the original
neighbors <- list(name       = "lambda",
                  T          = 20,
                  delta.p    = 1)
aggfun    <- list(name       = "wt")
variation <- list(list(name  = "sbx",
                       etax  = 20, pc = 1),
                  list(name  = "polymut",
                       etam  = 20, pm = 0.1),
                  list(name  = "truncate"))
update    <- list(name       = "standard", UseArchive = FALSE)
scaling   <- list(name       = "none")
constraint<- list(name       = "none")
stopcrit  <- list(list(name  = "maxiter",
                    maxiter  = 50))      # <-- maxiter = 200 in the original
showpars  <- list(show.iters = "dots",
                  showevery  = 10)
seed      <- 42

## Run MOEA/D
out1 <- moead(preset = NULL,
              problem, decomp, aggfun, neighbors, variation, update,
              constraint, scaling, stopcrit, showpars, seed)

## Examine the output:
summary(out1)

## Alternatively, the standard MOEA/D could also be set up using the
## preset_moead() function. The code below runs the original MOEA/D with
## exactly the same configurations as in Zhang and Li (2007).
\dontrun{
  out2 <- moead(preset   = preset_moead("original"),
                problem  = problem,
                showpars = showpars,
                seed     = 42)

  ## Examine the output:
  summary(out2)
  plot(out2, suppress.pause = TRUE)
}

# Rerun with MOEA/D-DE configuration and AWT scalarization
out3 <- moead(preset   = preset_moead("moead.de"),
              problem  = problem,
              aggfun   = list(name = "awt"),
              stopcrit = list(list(name    = "maxiter",
                                   maxiter = 50)),
              seed    = seed)
plot(out3, suppress.pause = TRUE)
}
