# Basic function to take in the one-step generation function of the 
# Markov chain.
#
# Parallel Version 3.2 - last change Jun 16 2011

runMarkovChainRegenP <- function(genNextState, genNextStateArgList=NULL, 
                       initialPoint, 
                       regenProbs, regenProbsArgList=NULL,
                       transDens, transDensArgList=NULL,
                       smallMeasure, smallMeasureArgList=NULL,
                       smallFn, smallFnArgList=NULL, numericParams,
                       nsim, nTours, prefix, pathToLog, 
                       outfile="output.rdt") {

################# Set up log-files ###############################
  commSize <- mpi.comm.size(0)
  if(commSize == 0)
    stop("No MPI Communicator present. Use 'runMarkovChainRegenS()'.\n")
  myRank <- mpi.comm.rank(0)
  numSlaves <- commSize - 1
  
  output.fname <- paste(pathToLog, prefix, sprintf("%02d",myRank), ".log", 
                        sep="")
  output.con <- file(output.fname, open="wt")
  cat("Log file set up. Starting checks .. \n", file=output.con)
################# Log-files set up ###############################

  # Check that exactly one of nsim or nTours is supplied
  nsimMissing <- missing(nsim)
  nToursMissing <- missing(nTours)
  if(nsimMissing & nToursMissing)
    cat("Either nsim or nTours has to be supplied.\n", file=output.con) else
  if(!nsimMissing & !nToursMissing)
    cat("Either nsim or nTours has to be supplied, but not both.\n", 
        file=output.con)

  # Check that either regenProbs, or all of smallFn, smallMeasure and 
  # transDens are supplied.
  regenProbsMissing <- missing(regenProbs)
  if(regenProbsMissing){
    transDensMissing <- missing(transDens)
    smallFnMissing <- missing(smallFn)
    smallMeasureMissing <- missing(smallMeasure)
    if(transDensMissing || smallFnMissing || smallMeasureMissing)
      cat("Either regenProbs(), or ALL of transDens(), smallFn() and 
            smallMeasure() have to be supplied.\n", file=output.con)
  }

  # check that the number of numeric parameters is known
  if(missing(numericParams))
    numericParams <- 1:length(initialPoint)

# CHECKS ON genNextState AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(genNextState)!="function")
    cat("genNextState() has to be of 'function' class.\n", file=output.con)

  # check that genNextStateArgList is not missing and is a list.
  if(!is.null(genNextStateArgList)){
    if(class(genNextStateArgList)!="list") 
      cat("genNextStateArgList has to be of 'list' class.\n", file=output.con)

    # After this, we should only have to call genNextState(x=...)
    formals(genNextState) <- c(genNextStateArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if one-step generation function works .. ", 
      file=output.con)
  firstPointGenerated <- genNextState(x=initialPoint)
  if(sum(is.na(firstPointGenerated)) > 0) 
    cat("NA's generated in genNextState().\n", file=output.con)

  # check length of output argument
  if(length(firstPointGenerated)!=length(initialPoint))
    cat("Input/output states for genNextState() do not match in length.\n", 
        file=output.con)
# END OF CHECKS ON genNextState AND ITS ARGUMENTS. ###############
  cat("genNextState() OK.\n", file=output.con)

  if(regenProbsMissing) {
# CHECKS ON transDens AND ITS ARGUMENTS. ######################
  # check that the genNextState argument is a function.
  if(class(transDens)!="function")
    cat("transDens() has to be of 'function' class.\n", file=output.con)

  # check that transDensArgList is not missing and is a list.
  if(!is.null(transDensArgList)){
    if(class(transDensArgList)!="list")
      cat("transDensArgList has to be of 'list' class.\n", file=output.con)

    # After this, we should only have to call genNextState(x=...)
    formals(transDens) <- c(transDensArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if transition density function works .. ", 
      file=output.con)
  pxy <- transDens(x=initialPoint, y=firstPointGenerated)
  if(is.na(pxy))
    cat("NA's generated by transition density function.\n", file=output.con)

  # check length of output argument
  if(length(pxy)!=1)
    cat("Transition density function should return a real scalar.\n", 
        file=output.con)
# END OF CHECKS ON transDens AND ITS ARGUMENTS. ###############
  cat("transDens() OK.\n", file=output.con)

# CHECKS ON smallMeasure AND ITS ARGUMENTS. ######################
  # check that the smallMeasure argument is a function.
  if(class(smallMeasure)!="function")
    cat("smallMeasure() has to be of 'function' class.\n", file=output.con)

  # check that smallMeasureArgList is not missing and is a list.
  if(!is.null(smallMeasureArgList)){
    if(class(smallMeasureArgList)!="list")
      cat("smallMeasureArgList has to be of 'list' class.\n", file=output.con)

    # After this, we should only have to call genNextState(x=...)
    formals(smallMeasure) <- c(smallMeasureArgList, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small measure function works .. ", file=output.con)
  nuY <- smallMeasure(y=firstPointGenerated)
  if(is.na(nuY))
    cat("NA's generated by small measure function.\n", file=output.con)

  # check length of output argument
  if(length(nuY)!=1)
    cat("Small measure function should return a real scalar.\n", 
        file=output.con)
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallMeasure() OK.\n", file=output.con)

# CHECKS ON smallFn AND ITS ARGUMENTS. ######################
  # check that the smallFn argument is a function.
  if(class(smallFn)!="function")
    cat("smallFn() has to be of 'function' class.\n", file=output.con)

  # check that smallFnArgList is not missing and is a list.
  if(!is.null(smallFnArgList)){
    if(class(smallFnArgList)!="list")
      cat("smallFnArgList has to be of 'list' class.\n", file=output.con)

    # After this, we should only have to call genNextState(x=...)
    formals(smallFn) <- c(smallFnArgList, x=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if small function works .. ", file=output.con)
  sX <- smallFn(x=firstPointGenerated)
  if(is.na(sX))
    cat("NA's generated by small function.\n", file=output.con)

  # check length of output argument
  if(length(sX)!=1)
    cat("Small function should return a real scalar.\n", file=output.con)
# END OF CHECKS ON smallMeasure AND ITS ARGUMENTS. ###############
  cat("smallFn () OK.\n", file=output.con)
  }
  else {
# CHECKS ON regenProbs AND ITS ARGUMENTS. ######################
  # check that the regenProbs argument is a function.
  if(class(regenProbs)!="function")
    cat("regenProbs() has to be of 'function' class.\n", file=output.con)

  # When regenProbsArgList is not NULL, check that it is a list.
  if(!is.null(regenProbsArgList)){
    if(class(regenProbsArgList)!="list")
      cat("regenProbsArgList has to be of 'list' class.\n", file=output.con)

    # After this, we should only have to call genNextState(x=...)
    formals(regenProbs) <- c(regenProbsArgList, x=1, y=1)
  }

  # try running one step and stop if there are errors.
  cat("Checking to see if regenProbs function works .. ", file=output.con)
  firstRegenProb <- regenProbs(x=initialPoint, y=firstPointGenerated)
  if(is.na(firstRegenProb))
    cat("NA's generated by regenProbs function with x=", initialPoint, 
        "and y=", firstPointGenerated, ".\n", sep=" ", file=output.con)

  # check length of output argument
  if(length(firstRegenProb)!=1)
    cat("regenProbs function should return a real scalar.\n", file=output.con)
# END OF CHECKS ON regenProbs AND ITS ARGUMENTS. ###############
  cat("regenProbs() OK.\n", file=output.con)
  }
  cat("All checks done.\n\n", file=output.con)

############# genOneTour function definition ##################
genOneTour <- function(startPt, numericParams) {
  steps <- 1
  Si <- startPt
  complete <- 0

   repeat{
      potentialNewY <- genNextState(x=startPt)
      if(regenProbsMissing) {
        regenProb <- smallFn(x=startPt) *
                     smallMeasure(y=potentialNewY) /
                     transDens(x=startPt, y=potentialNewY)
      }
      else {
        regenProb <- regenProbs(x=startPt, y=potentialNewY)
      }
      complete <- ifelse(runif(1) < regenProb, 1, 0)
      if (complete==1)
        break
      else {
        Si <- Si + potentialNewY
        startPt <- potentialNewY
        steps <- steps + 1
      }
    }

  return(list(out=c(Si[numericParams], steps), newStart=potentialNewY))
}
###############################################################
#  # if nTours is missing, run the chain for nsim iterations and split that up.
  if(nToursMissing) {
    ## Run the Markov chain for nsim steps
    cat("Running Markov chain .. ", file=output.con)
    outMatrix <- matrix(0, nrow=nsim, ncol=length(initialPoint))
    regenProbsVec <- rep(0, times=nsim)
    bellVars <- rep(0, times=nsim)

    outMatrix[1,] <- firstPointGenerated
    for(i in 2:nsim) {
      outMatrix[i,] <- genNextState(x=outMatrix[i-1,])
      if(regenProbsMissing) {
        regenProbsVec[i-1] <- smallFn(x=outMatrix[i-1,]) *
                              smallMeasure(y=outMatrix[i,]) /
                              transDens(x=outMatrix[i-1,], y=outMatrix[i,])
      }
      else {
        regenProbsVec[i-1] <- regenProbs(x=outMatrix[i-1,], y=outMatrix[i,])
      }
    }

    # generate regeneration points
    bellVars[-nsim] <- ifelse(runif(nsim-1) < regenProbsVec[-nsim], 1, 0)
    cat("Done.\n", file=output.con)
    tourMatrix <- splitIntoTours(outMatrix, bellVars, numericParams)

  # begin passing tourMatrix to master
    if(myRank > 0) {
      if(is.na(tourMatrix))
        cat("No tours found! Transferring NA .. ", file=output.con) else {
        cat("Transferring", nrow(tourMatrix), "tours to master .. ", 
            sep=" ", file=output.con) }
      mpi.send.Robj(tourMatrix, dest=0, tag=2, comm=0)
      cat("Done.\n", file=output.con)
    }
    if(myRank == 0) {
      if(is.na(tourMatrix)) {
        cat("No tours found here!\n", file=output.con) 
        tourMatrix <- NULL
      }
      else {
        cat(nrow(tourMatrix), "tours completed here.\n", sep=" ", 
            file=output.con) 
      }
      recvdFrom <- 0
      for(i in 1:numSlaves) {
        tmp <- mpi.recv.Robj(source=mpi.any.source(), tag=2, comm=0)
        cat("Receiving tour matrix from", mpi.get.sourcetag(0)[1], ".. ", 
            sep=" ", file=output.con)
        if(is.na(tmp)) tmp <- NULL
        tourMatrix <- rbind(tourMatrix, tmp)
        cat("Done.\n\n", file=output.con)
      }

      if(is.null(tourMatrix)) {
        cat("No tours generated at all.\n", file=output.con)
        cat("Saving outMatrix from rank 0 node along with regenProbs.\n", 
            file=output.con)
        save(outMatrix, regenProbsVec, file=outfile)
      }
      else {
        cat("Total number of tours generated from", commSize, "processors :",
            nrow(tourMatrix), "\n", sep=" ", file=output.con)
        save(tourMatrix, file=outfile)
      }
    }
  }
  else { # Run for a specified number of tours
  # Look for first regeneration point 
    cat("Looking for first regeneration point .. ", file=output.con)
    findFirstPt <- 0
    while(findFirstPt == 0) {
      potentialNewY <- genNextState(x=initialPoint)
      if(regenProbsMissing) {
        regenProb <- smallFn(x=initialPoint) *
                     smallMeasure(y=potentialNewY) /
                     transDens(x=initialPoint, y=potentialNewY)
      }
      else {
        regenProb <- regenProbs(x=initialPoint, y=potentialNewY)
      }
      complete <- ifelse(runif(1) < regenProb, 1, 0)
      if(complete == 1) {
        findFirstPt <- 1
        newStart <- potentialNewY
      }
      else {
        initialPoint <- potentialNewY
      }
    }
    cat("Done.\n", file=output.con)

  # Run the Markov chain for nTours tours
    if(myRank == 0)
      timeStart <- Sys.time()
    cat("Generating", nTours, "tours .. ", sep=" ", file=output.con)
    tourMatrix <- matrix(0, nrow=nTours, ncol=length(numericParams)+1)
    for(i in 1:nTours) {
      tmp <- genOneTour(newStart, numericParams)
      tourMatrix[i,] <- tmp$out
      newStart <- tmp$newStart
    }
    cat("Done.\n", file=output.con)
    colnames(tourMatrix) <- c(paste("S", numericParams, sep=""), "N")

  # begin passing tourMatrix to master
    if(myRank > 0) {
      cat("Transferring", nrow(tourMatrix), "tours to master .. ", sep=" ", 
          file=output.con)
      mpi.send.Robj(tourMatrix, dest=0, tag=2, comm=0)
      cat("Done.\n", file=output.con)
    }
    if(myRank == 0) {
      recvdFrom <- 0
       for(i in 1:numSlaves) {
        tmp <- mpi.recv.Robj(source=mpi.any.source(), tag=2, comm=0)
        cat("Receiving tour matrix from", mpi.get.sourcetag(0)[1], ".. ", 
            sep=" ", file=output.con)
        tourMatrix <- rbind(tourMatrix, tmp)
        cat("Done.\n", file=output.con)
      }
      timeEnd <- Sys.time()
      timeTaken <- as.numeric(timeEnd - timeStart)
      cat("Total time taken was", timeTaken, "seconds\n", sep=" ", 
          file=output.con)
 
      cat("Total number of tours generated from", commSize, "processors :",
          nrow(tourMatrix), "\n", sep=" ", file=output.con)
      save(tourMatrix, file=outfile)
    }
  }

  close(output.con)
}
