% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NMAoutlier.R
\name{NMAoutlier}
\alias{NMAoutlier}
\title{Forward Search algorithm in network meta-analysis}
\usage{
NMAoutlier(
  TE,
  seTE,
  treat1,
  treat2,
  studlab,
  data = NULL,
  crit1 = "R",
  crit2 = "R",
  studies = NULL,
  P = 100,
  sm,
  Isub = NULL,
  reference = "",
  small.values = "good",
  n_cores = NULL,
  ...
)
}
\arguments{
\item{TE}{Estimate of treatment effect, i.e. difference between
first and second treatment (e.g. log odds ratio, mean difference,
or log hazard ratio). This can also be a pairwise object
(i.e. the result of pairwise function of netmeta package).  In
this case, the pairwise object should include the following: TE,
seTE, treat1, treat2, studlab}

\item{seTE}{Standard error of treatment estimate.}

\item{treat1}{Label/Number for first treatment.}

\item{treat2}{Label/Number for second treatment.}

\item{studlab}{Study labels (important when multi arm studies are
included).}

\item{data}{A data frame containing the study information.}

\item{crit1}{A character string indicating the criterion to be used
for selecting the initial subset, this criterion may be the
minimum of median absolute residuals ("R") or the maximum of
median absolute likelihood contributions ("L"). Default value is
"R".}

\item{crit2}{A character string indicating the criterion to be used
for selecting the study entered from non-basic set to basic set,
this criterion may be the minimum of absolute residuals ("R") or
the maximum of absolute likelihood contributions ("L"). Default
value is "R".}

\item{studies}{An optional vector specifying the number of the
initial subset of studies. The default value is the maximum of
the number of treatments and the 20 percent of the total number
of studies.}

\item{P}{An optional vector specifying the number of candidate
sample of studies (with length equal to \code{studies}) for the
choice of the initial subset. Default value is 100.}

\item{sm}{A character string indicating underlying summary measure,
e.g., \code{"RD"}, \code{"RR"}, \code{"OR"}, \code{"ASD"},
\code{"HR"}, \code{"MD"}, \code{"SMD"}, or \code{"ROM"}.}

\item{Isub}{A vector for the studies to be included in the initial
subset (default: NULL, the initial subset not specified by the
user).}

\item{reference}{Reference treatment group.}

\item{small.values}{A character string indicating if small values
are considered beneficial (option:"good") or harmfull
(option:"bad") on outcome.  This is requirement for p-scores
computation. The default value is considered benefial outcome
("good").}

\item{n_cores}{The number of cores that the process is running
using the parallel (default: NULL, the process is running using
all the available cores)}

\item{\dots}{Additional arguments passed on to
\code{\link[netmeta]{netmeta}}.}
}
\value{
An object of class \code{NMAoutlier}; a list containing the
following components:
   \item{dat}{Matrix containing the data \code{"TE"}, \code{"seTE"}, \code{"studlab"}, \code{"treat1"}, \code{"treat2"} as defined above.}
   \item{length.initial}{The number of studies that constitute the initial (outlying-clean) subset of studies.}
   \item{index}{The number of iterations of forward search algorithm.}
   \item{basic}{Studies entered into the basic set in each iteration of the search.
   At the first iteration, basic set constitute the studies that are included in the basic-initial subset.
   The number of studies in the first iteration is equal to length.initial.}
   \item{taub}{Heterogeneity estimator variance for basic set in each iteration of forward search algorithm.}
   \item{Qb}{Overall heterogeneity - inconsistency Q statistic (\code{Q}) for the basic set in each iteration of forward search algorithm.}
   \item{Qhb}{Overall heterogeneity Q statistic (\code{Q}) for the basic set in each iteration of forward search algorithm.}
   \item{Qib}{Overall inconsistency Q statistic (\code{Q}) from design-by-treatment interaction model for the basic set in each iteration of forward search algorithm.}
   \item{estb}{Summary estimates for each treatment for the basic set in each iteration of forward search algorithm.}
   \item{lb}{Lower 95\% confidence interval of summary estimates for the basic set in each iteration of forward search algorithm.}
   \item{ub}{Upper 95\% confidence interval of summary estimates for the basic set in each iteration of forward search algorithm.}
   \item{Ratio}{Ratio of determinants (\code{COVRATIOj}) of variance-covariance matrix of treatment estimates at iteration j to that iteration at (j-1).}
   \item{cook_d}{Cook's statistic (\code{Cj}) at iteration j of forward search algorithm.}
   \item{p.score}{P-score for ranking each treatment for the basic set in each iteration of forward search algorithm.}
   \item{dif}{Z-values for comparison between direct and indirect evidence for each iteration of forward search algorithm.
    Based on back-calculation method to derive indirect estimates from direct pairwise comparisons and network estimates.}
   \item{estand}{Standardized residuals for each study for the basic set in each iteration of forward search algorithm.}
   \item{call}{Function call}
}
\description{
This function employs the Forward Search algorithm to detect
outliers and influential studies fitted in network meta-analysis
model from graph-theory.  This is an outlying diagnostic tool to
detect outliers and studies that are potential sources for
heterogeneity and inconsistency in network meta-analysis.

Monitoring measures during the search are:
\itemize{
\item outlier detection measures (standardized residuals, Cook's
  distance, ratio of variance);
\item ranking measures (P-scores);
\item heterogeneity and inconsistency measures (Q statistics for
  overall heterogeneity / inconsistency, inconsistency by
  design-by-treatment interaction model, z-values for comparison
  between direct and indirect evidence by back-calculation method).
}

A description of the outlier detection methodology can be found in
Petropoulou et al. (2021).
}
\details{
FS algorithm for network meta-analysis model from graph theory is
described in Petropoulou et al. (2021).

Let \emph{n} be the number of treatments and let \emph{m} be the
number of pairwise treatment comparisons.  If there are only
two-arm studies, \emph{m} is equal to the number of studies. Let TE
and seTE be the vectors of observed effects and their standard
errors.  Comparisons belonging to multi-arm studies are identified
by identical study labels (argument \code{studlab}).

The FS algorithm is an outlier diagnostic iterative procedure. FS
algorithm apart from three steps. It starts with a subset of
studies and it gradually adds studies until all studies entered.
After the search, statistical measures are monitored for sharp
changes.

In more detail, the FS algorithm starts with an initial subset of
the dataset with size \emph{l}. Let (argument \code{P})
(eg. \emph{P} = 100) a large number of candidate subset of studies
with size \emph{l}. The candidate subset that optimize the
criterion (argument \code{crit1}) is taken as the initial subset
(considered ideally to be outlying-free).  Criterion (\code{crit1})
to be used for selecting the initial subset, can be the minimum of
median absolute residuals \code{"R"} or the maximum of median
absolute likelihood contributions \code{"L"}. It is conventionally
refer this subset as basic set, whereas the remaining studies
constitute the non-basic set.

The FS algorithm gradually adds studies from the non-basic to the
basic subset based on how close the former studies are to the
hypothesized model fit in the basic set.  A study from non-basic
set entered into the basic set if optimize the criterion (argument
\code{crit2}).  Criterion (\code{crit2}) for selecting the study
from non-basic to basic set may be the minimum of median absolute
residuals \code{"R"} or the maximum of median absolute likelihood
contributions \code{"L"}.  The algorithm order the studies
according to their closeness to the basic set by adding the study
that optimize the criterion from non-basic set to basic set.

The process is repeated until all studies are entered into the
basic set.  The number of iterations of algorithm \emph{index} is
equal to the total number of studies minus the number of studies
entered into the initial subset. Through the FS procedure,
parameter estimates (summary estmates, heterogeneity estimator) and
other statistics of interest (outlying measures, heterogeneity and
inconsistency measures, ranking measures) are monitored. In each
iteration, network meta-analysis model from graph theory (Rücker,
2012) is fitted (\code{netmeta} function) with R package
\bold{netmeta}.

Monitoring statistical measures for each FS iteration can be:

\bold{Outlying detection measures:}
Standardized residuals (arithmetic mean in case of multi-arm
studies); Cook's statistic; Ratio of determinants of
variance-covariance matrix

\bold{Ranking measures:}
P-scores for ranking of treatments (Rücker G & Schwarzer G, 2015)
for each basic set with implementation of (\code{netrank} function)
from R package \bold{netmeta}.

\bold{Heterogeneity and inconsistency measures:}
Overall heterogeneity / inconsistency Q statistic (\code{Q}) This
is the design-based decomposition of Cochran Q as provided by Krahn
et al. (2013); Overall heterogeneity Q statistic (\code{Q});
Between-designs Q statistic (\code{Q}), based on a random effects
model with square-root of between-study variance estimated embedded
in a full design-by-treatment interaction model.  Implementation
with (\code{decomp.design} function) from R package \bold{netmeta};
Z-values (Dias et al., 2010; König et al., 2013) for comparison
between direct and indirect evidence in
each iteration of forward search algorithm.  By monitoring
difference of direct and indirect evidence, potential sources of
consistency can be detected with the implementation of
(\code{netsplit} function) from R package \bold{netmeta} for each
iteration of the search.
}
\examples{
\dontrun{
library("netmeta")
data(smokingcessation)
smokingcessation$id <- 1:nrow(smokingcessation)

study912 <- subset(smokingcessation, id \%in\% 9:12)
p1 <- pairwise(list(treat1, treat2, treat3),
                        list(event1, event2, event3),
                        list(n1, n2, n3),
                        data = study912,
                        sm = "OR")

# Forward search algorithm
#
FSresult <- NMAoutlier(p1, P = 1, small.values = "bad", n_cores = 2)
FSresult

data(smokingcessation)

# Transform data from arm-based to contrast-based format
# We use 'sm' argument for odds ratios.
# We use function pairwise from netmeta package
#
p1 <- pairwise(list(treat1, treat2, treat3),
                        list(event1, event2, event3),
                        list(n1, n2, n3),
                        data = smokingcessation,
                        sm = "OR")

# Forward search algorithm
#
FSresult1 <- NMAoutlier(p1, small.values = "bad")

# Basic set for each iteration of forward search algorithm
#
FSresult1$basic

# Forward search algorithm using the criteria (crit1, crit2)
# with the maximum of absolute likelihood contributions ("L")
#
FSresult2 <- NMAoutlier(p1, crit1 = "L", crit2 = "L",
                        small.values = "bad")
FSresult2
}

}
\references{
Dias S, Welton NJ, Caldwell DM, Ades AE (2010):
Checking consistency in mixed treatment comparison meta-analysis.
\emph{Statistics in Medicine},
\bold{29}, 932--44

König J, Krahn U, Binder H (2013):
Visualizing the flow of evidence in network meta-analysis and
characterizing mixed treatment comparisons.
\emph{Statistics in Medicine},
\bold{32}, 5414--29

Krahn U, Binder H, König J (2013):
A graphical tool for locating inconsistency in network meta-analyses.
\emph{BMC Medical Research Methodology},
\bold{13}, 35

Petropoulou M, Salanti G, Rücker G, Schwarzer G, Moustaki I,
Mavridis D (2021):
A forward search algorithm for detecting extreme study effects in
network meta-analysis.
\emph{Statistics in Medicine}

Rücker G (2012):
Network meta-analysis, electrical networks and graph theory.
\emph{Research Synthesis Methods},
\bold{3}, 312--24

Rücker G, Schwarzer G (2015):
Ranking treatments in frequentist network meta-analysis works
without resampling methods.
\emph{BMC Medical Research Methodology},
\bold{15}, 58
}
\author{
Maria Petropoulou <maria.petropoulou@uniklinik-freiburg.de>
}
