% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msmm.R
\name{msmm}
\alias{msmm}
\title{Multiplicative structural mean model}
\usage{
msmm(
  formula,
  instruments,
  data,
  subset,
  na.action,
  contrasts = NULL,
  estmethod = c("gmm", "gmmalt", "tsls", "tslsalt"),
  t0 = NULL,
  ...
)
}
\arguments{
\item{formula, instruments}{formula specification(s) of the regression
relationship and the instruments. Either \code{instruments} is missing and
\code{formula} has three parts as in \code{y ~ x1 + x2 | z1 + z2 + z3}
(recommended) or \code{formula} is \code{y ~ x1 + x2} and \code{instruments}
is a one-sided formula \code{~ z1 + z2 + z3} (only for backward
compatibility).}

\item{data}{an optional data frame containing the variables in the model.
By default the variables are taken from the environment of the
\code{formula}.}

\item{subset}{an optional vector specifying a subset of observations to be
used in fitting the model.}

\item{na.action}{a function that indicates what should happen when the data
contain \code{NA}s. The default is set by the \code{na.action} option.}

\item{contrasts}{an optional list. See the \code{contrasts.arg} of
\code{\link[stats:model.matrix]{model.matrix.default}}.}

\item{estmethod}{Estimation method, please use one of
\itemize{
\item \code{"gmm"} GMM estimation of the MSMM (the default).
\item \code{"gmmalt"} GMM estimation of the alternative moment conditions
for the MSMM as per Clarke et al. (2015). These are the same moment
conditions fit by the user-written Stata command \code{ivpois} (Nichols, 2007)
and by the official Stata command \verb{ivpoisson gmm ..., multiplicative}
(StataCorp., 2013).
\item \code{"tsls"} the TSLS method of fitting the MSMM of Clarke et al. (2015).
For binary \eqn{Y} and \eqn{X} this uses \eqn{Y*(1-X)} as the outcome and
\eqn{Y*X} as the exposure.
\item \code{"tslsalt"} the alternative TSLS method of fitting the MSMM of
Clarke et al. (2015). For binary \eqn{Y} and \eqn{X} this uses \eqn{Y*X}
as the outcome and \eqn{Y*(1-X)} as the exposure.
}}

\item{t0}{A vector of starting values for the gmm optimizer. This should
have length equal to the number of exposures plus 1.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
An object of class \code{"msmm"}. A list with the following items:

\item{fit}{The object from either a \code{\link[gmm:gmm]{gmm::gmm()}} or \code{\link[ivreg:ivreg]{ivreg::ivreg()}} fit.}
\item{crrci}{The causal risk ratio/s and it corresponding 95\% confidence
interval limits.}
\item{estmethod}{The specified \code{estmethod}.}

If \code{estmethod} is \code{"tsls"}, \code{"gmm"}, or \code{"gmmalt"}:

\item{ey0ci}{The estimate of the treatment/exposure free potential outcome and
its 95\% confidence interval limits.}

If \code{estmethod} is \code{"tsls"} or \code{"tslsalt"}:

\item{stage1}{An object containing the first stage regression from an
\code{\link[stats:lm]{stats::lm()}} fit.}
}
\description{
Function providing several methods to estimate the multiplicative
structural mean model (MSMM) of Robins (1989).
}
\details{
Function providing several methods to estimate the multiplicative
structural mean model (MSMM) of Robins (1989). These are the methods
described in Clarke et al. (2015), most notably
generalised method of moments (GMM) estimation of the MSMM.

An equivalent estimator to the MSMM was proposed in Econometrics by Mullahy (1997) and
then discussed in several articles by Windmeijer (1997, 2002) and Cameron
and Trivedi (2013). This was implemented in the user-written Stata command \code{ivpois}
(Nichols, 2007) and then implemented in official Stata in the \code{ivpoisson}
command (StataCorp., 2013).
}
\examples{
# Single instrument example
# Data generation from the example in the ivtools ivglm() helpfile
set.seed(9)
n <- 1000
psi0 <- 0.5
Z <- rbinom(n, 1, 0.5)
X <- rbinom(n, 1, 0.7*Z + 0.2*(1 - Z))
m0 <- plogis(1 + 0.8*X - 0.39*Z)
Y <- rbinom(n, 1, plogis(psi0*X + log(m0/(1 - m0))))
dat <- data.frame(Z, X, Y)
fit <- msmm(Y ~ X | Z, data = dat)
summary(fit)

# Multiple instrument example
set.seed(123456)
n <- 1000
psi0 <- 0.5
G1 <- rbinom(n, 2, 0.5)
G2 <- rbinom(n, 2, 0.3)
G3 <- rbinom(n, 2, 0.4)
U <- runif(n)
pX <- plogis(0.7*G1 + G2 - G3 + U)
X <- rbinom(n, 1, pX)
pY <- plogis(-2 + psi0*X + U)
Y <- rbinom(n, 1, pY)
dat2 <- data.frame(G1, G2, G3, X, Y)
fit2 <- msmm(Y ~ X | G1 + G2 + G3, data = dat2)
summary(fit2)
}
\references{
Cameron AC, Trivedi PK. Regression analysis of count data. 2nd ed. 2013.
New York, Cambridge University Press. ISBN:1107667275

Clarke PS, Palmer TM, Windmeijer F. Estimating structural
mean models with multiple instrumental variables using the
Generalised Method of Moments. Statistical Science, 2015, 30, 1,
96-117. \doi{10.1214/14-STS503}

Hernán and Robins. Instruments for causal inference: An
Epidemiologist's dream? Epidemiology, 2006, 17, 360-372.
\doi{10.1097/01.ede.0000222409.00878.37}

Mullahy J. Instrumental-variable estimation of count data models:
applications to models of cigarette smoking and behavior. The Review of
Economics and Statistics. 1997, 79, 4, 586-593.
\doi{10.1162/003465397557169}

Nichols A. ivpois: Stata module for IV/GMM Poisson regression. 2007.
\href{https://ideas.repec.org/c/boc/bocode/s456890.html}{url}

Palmer TM, Sterne JAC, Harbord RM, Lawlor DA, Sheehan NA, Meng S,
Granell R, Davey Smith G, Didelez V.
Instrumental variable estimation of causal risk ratios and causal odds ratios
in Mendelian randomization analyses.
American Journal of Epidemiology, 2011, 173, 12, 1392-1403.
\doi{10.1093/aje/kwr026}

Robins JM. The analysis of randomised and
nonrandomised AIDS treatment trials using a new approach to
causal inference in longitudinal studies.
In Health Service Research Methodology: A Focus on AIDS
(L. Sechrest, H. Freeman and A. Mulley, eds.). 1989. 113–159.
US Public Health Service, National Center for Health Services Research,
Washington, DC.

StataCorp. Stata Base Reference Manual. Release 13.
ivpoisson - Poisson model with continuous endogenous covariates. 2013.
\href{https://www.stata.com/manuals13/rivpoisson.pdf}{url}

Windmeijer FAG, Santos Silva JMC. Endogeneity in Count Data Models:
An Application to Demand for Health Care. Journal of Applied Econometrics.
1997, 12, 3, 281-294.
\doi{10/fdkh4n}

Windmeijer, F. ExpEnd, A Gauss programme for non-linear GMM estimation of
EXPonential models with ENDogenous regressors for cross section and panel
data. CEMMAP working paper CWP14/02. 2002.
\href{https://www.cemmap.ac.uk/wp-content/uploads/2020/08/CWP1402.pdf}{url}
}
