#' @title Construct OMLFlow.
#'
#' @description More details about the elements of a \code{OMLFlow} can be found in the
#' \href{https://www.openml.org/api/v1/xsd/openml.implementation.upload}{XSD scheme}.
#'
#' @param flow.id [\code{integer(1)}]\cr
#'   ID of the flow. Generated by the server, based on name and version of the flow.
#'   Ignored when uploaded manually.
#' @param uploader [\code{integer(1)}]\cr
#'   The user that uploaded the flow. Added by the server. Ignored when uploaded manually.
#' @param name [\code{character(1)}]\cr
#'   The name of the flow. Name-version combinations should be unique.\cr
#'   Allowed characters: () [] a-z A-Z 0-9 . _ - +
#' @param version [\code{character(1)}]\cr
#'   The version of the flow. Default is 1.0. Ignored at upload time.
#' @param external.version [\code{character(1)}]\cr
#'   An external version, defined by the user. In combination with the name, it must be unique.
#' @param description [\code{character(1)}]\cr
#'   A user description of the flow.
#' @param creator [\code{character}]\cr
#'   Optional. The persons/institutions that created the flow.
#' @param contributor [\code{character}]\cr
#'   Optional. (Minor) contributors to the workflow
#' @param upload.date [\code{character(1)}]\cr
#'   The date on which the flow was uploaded.\cr
#'   Format YYYY-mm-ddThh:MM:SS.
#'   Added by the server. Ignored when uploaded manually.
#' @param licence [\code{character(1)}]\cr
#'   Optional. Default is none, meaning Public Domain or "don't know/care".
#' @param language [\code{character(1)}]\cr
#'   Optional. Starts with one upper case letter, rest is lower case. Default is English.
#' @param full.description [\code{character(1)}]\cr
#'   Optional. Full description of the workflow, e.g, man pages filled in by tool.
#'   This is a much more elaborate description than given in the 'description field'. It may include
#'   information about all components of the workflow.
#' @param installation.notes [\code{character(1)}]\cr
#'   Optional. Additional hints on how to run the flow.
#' @param dependencies [\code{character(1)}]\cr
#'   Optional. The dependencies of the flow.
#' @param bibliographical.reference [\code{list}]\cr
#'   An optional list containing information on bibliographical references in form of
#'   \code{OMLBibRef}.
#' @param implements [\code{character(1)}]\cr
#'   Ontological reference.
#' @param parameters [\code{list}]\cr
#'   The parameters of the flow. A list containing
#'   \code{\link{OMLFlowParameter}s}.
#' @param components [\code{list}]\cr
#'   A list containing \code{\link{OMLFlow}s}. Typically components of a workflow or
#'   subfunctions of an algorithm (e.g. kernels). Components can have their own parameters.
#' @param qualities [\code{list}]\cr
#'   Qualities of the algorithm. Each member of the list is an \code{OMLFlowQuality}.
#' @param tags [\code{character}]\cr
#'   Tags describing the algorithm.
#' @param source.url [\code{character(1)}]\cr
#'   URL from which the source code can be downloaded. Added by the server. Ignored when uploaded manually.
#' @param binary.url [\code{character(1)}]\cr
#'   URL from which the binary can be downloaded. Added by the server. Ignored when uploaded manually.
#' @param source.format [\code{character(1)}]\cr
#'   Format of the source file.
#' @param binary.format [\code{character(1)}]\cr
#'   Format of the binary file.
#' @param source.md5 [\code{character(1)}]\cr
#'   MD5 checksum to check if the source code was uploaded correctly.
#' @param binary.md5 [\code{character(1)}]\cr
#'   MD5 checksum to check if the binary code was uploaded correctly.
#' @param source.path [\code{character(1)}]\cr
#'   The path to the cached source file, once \code{\link{getOMLFlow}} was run.
#' @param binary.path [\code{character(1)}]\cr
#'   The path to the cached binary file, once \code{\link{getOMLFlow}} was run.
#' @param object [\code{any}]\cr
#'   (optional) Any R object referring to the flow.
#' @export
#' @family flow-related functions
#' @aliases OMLFlow
# @param implements [\code{character}]\cr
#   Ontological reference.
makeOMLFlow = function(
  flow.id = NA_integer_,
  uploader = NA_integer_,
  name,
  version = NA_character_,
  external.version = NA_character_,
  description,
  creator = NA_character_,
  contributor = NA_character_,
  upload.date = NA_character_,
  licence = NA_character_,
  language = "English",
  full.description = NA_character_,
  installation.notes = NA_character_,
  dependencies = NA_character_,
  bibliographical.reference = NULL,
  implements = NA_character_,
  parameters = NULL,
  components = NULL,
  qualities = NULL,
  tags = NA_character_,
  source.url = NA_character_,
  binary.url = NA_character_,
  source.format = NA_character_,
  binary.format = NA_character_,
  source.md5 = NA_character_,
  binary.md5 = NA_character_,
  source.path = NA_character_,
  binary.path = NA_character_,
  object = NULL
) {

  assertInt(flow.id, na.ok = TRUE)
  assertInt(uploader, na.ok = TRUE)
  assertString(name)
  assertString(version, na.ok = TRUE)
  assertString(external.version, na.ok = TRUE)
  assertString(description)
  assertCharacter(creator)
  assertCharacter(contributor)
  assertString(upload.date, na.ok = TRUE)
  assertString(licence, na.ok = TRUE)
  assertString(language)
  assertString(full.description, na.ok = TRUE)
  assertString(installation.notes, na.ok = TRUE)
  assertString(dependencies, na.ok = TRUE)
  if (!is.null(bibliographical.reference))
    assertList(bibliographical.reference)
  assertString(implements, na.ok = TRUE)
  if (!is.null(parameters))
    assertList(parameters)
  if (!is.null(components))
    assertList(components)
  if (!is.null(qualities))
    assertList(qualities)
  assertCharacter(tags)
  assertString(source.url, na.ok = TRUE)
  assertString(binary.url, na.ok = TRUE)
  assertString(source.format, na.ok = TRUE)
  assertString(binary.format, na.ok = TRUE)
  assertString(source.md5, na.ok = TRUE)
  assertString(binary.md5, na.ok = TRUE)
  assertString(source.path, na.ok = TRUE)
  assertString(binary.path, na.ok = TRUE)

  makeS3Obj("OMLFlow",
    flow.id = flow.id,
    uploader = uploader,
    name = name,
    version = version,
    external.version = external.version,
    description = description,
    creator = creator,
    contributor = contributor,
    upload.date = upload.date,
    licence = licence,
    language = language,
    full.description = full.description,
    installation.notes = installation.notes,
    dependencies = dependencies,
    bibliographical.reference = bibliographical.reference,
    implements = implements,
    parameters = parameters,
    components = components,
    qualities = qualities,
    tags = tags,
    source.url = source.url,
    binary.url = binary.url,
    source.format = source.format,
    binary.format = binary.format,
    source.md5 = source.md5,
    binary.md5 = binary.md5,
    source.path = source.path,
    binary.path = binary.path,
    object = object
  )
}

# comp.length = function(x) {
#   len = 0
#   while(length(x$components) != 0) {
#     len = len + length(x$components)
#     if(length(x$components) == 1) x = x$components[[1]]
#   }
#   return(len)
# }

# show
#' @export
print.OMLFlow = function(x, ...)  {
  ## General flow info
  catf("\nFlow '%s' :: (Version = %s, Flow ID = %i)", x$name, x$version, x$flow.id)
  catfNotNA("\tExternal Version         : %s", x$external.version)
  catfNotNA("\tCreator(s)               : %s", x$creator)
  catfNotNA("\tLicence                  : %s", x$licence)
  catfNotNA("\tInstallation Notes       : %s", x$installation.notes)
  catfNotNA("\tDependencies             : %s", x$dependencies)
  catf("\tNumber of Flow Parameters: %i", length(x$parameters))
  catf("\tNumber of Flow Components: %i", length(x$components))
}

#' Construct OMLFlowParameter.
#'
#' @param name [\code{character(1)}]\cr
#'   The name of the parameter.
#' @param data.type [\code{character(1)}]\cr
#'   The data type of the parameter. Should be either integer, numeric, string, vector, matrix or object.
#' @param default.value [\code{character(1)}]\cr
#'   The default value of the parameter.
#' @param description [\code{character(1)}]\cr
#'   A description of what this parameter does.
#' @param recommended.range [\code{character(1)}]\cr
#'   Minimal/maximal value and/or a recommended range of values.
#' @export
#' @keywords internal
#' @family flow-related functions
#' @aliases OMLFlowParameter
makeOMLFlowParameter = function(
  name,
  data.type = NA_character_,
  default.value = NA_character_,
  description = NA_character_,
  recommended.range = NA_character_
) {

  assertString(name)
  assertString(default.value, na.ok = TRUE)
  assertString(description, na.ok = TRUE)
  assertString(recommended.range, na.ok = TRUE)

  makeS3Obj("OMLFlowParameter",
    name = name,
    data.type = data.type,
    default.value = default.value,
    description = description,
    recommended.range = recommended.range
  )
}

# show
#' @export
print.OMLFlowParameter = function(x, ...) {
  catf("Parameter %s:", x$name)
  catfNotNA("  type             : %s", x$data.type)
  catfNotNA("  default          : %s", x$default.value)
  catfNotNA("  recommended range: %s", x$recommended.range)
  catfNotNA("\n%s", x$description)
}

# @title Construct OMLBibRef.
#
# @param citation [\code{character(1)}]\cr
#    Free form reference for this implementation.
# @param url [\code{character(1)}]\cr
#   URL to an online version of the paper, e.g. PDF.
# @keywords internal
# @aliases OMLBibRef
makeOMLBibRef = function(citation, url) {
  assertString(citation)
  assertString(url)
  makeS3Obj("OMLBibRef",
    citation = citation,
    url = url
  )
}

# show
# FIXME: how should missing values be represented? here, character(0) AND "" are possible.
print.OMLBibRef = function(x, ...) {
  catf("  %s", x$citation)
  if (length(x$url) > 0L && nzchar(x$url))
    catf("  url :: %s\n", x$url)
}

# @title Construct OMLFlowQuality.
#
# @param name [\code{character(1)}]\cr
#   Name of the quality.\cr
#   Suggested: LearnerType (classification, regression, cost sensitive classification, survival analysis, clustering),
#   HandlesNumericFeatures, HandlesNominalFeatures, HandlesMissingValues, HandlesInstanceWeights, HandlesMultiClass,
#   HandlesBinaryClass, HandlesSingleClass, HandlesOrderedFeatures, HandlesCostMatrices, HandlesClassWeights,
#   CanOutputProbabilities (classification+clustering), CanOutputStandardError (regression).
# @param value [\code{character(1)}]\cr
#   The quality's value. E.g., 'true' or 'false'.
# @export
# @family flow-related functions
# @aliases OMLFlowQuality
makeOMLFlowQuality = function(name, value) {
  assertString(name)
  assertString(value)
  makeS3Obj("OMLFlowQuality",
    name = name,
    value = value
  )
}

# @export
print.OMLFlowQuality = function(x, ...) {
  sprintf("%s: %s", x$name, x$value)
}
