% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brdate.R
\name{brdate}
\alias{brdate}
\title{Estimating Break Dates in a Quantile Regression}
\usage{
brdate(y, x, n.size = 1, m, trim.size, vec.long)
}
\arguments{
\item{y}{A numeric vector of dependent variables (\eqn{NT \times 1}).}

\item{x}{A numeric matrix of regressors (\eqn{NT \times p}); a column of ones will be automatically added to \eqn{x}.}

\item{n.size}{An integer specifying the number of cross-sectional units (\eqn{N}), equal to 1 for time series data.}

\item{m}{An integer specifying the maximum number of breaks allowed.}

\item{trim.size}{An integer representing the minimum length of a regime, which ensures break dates are not too short or too close to the sample's boundaries.}

\item{vec.long}{A numeric vector/matrix used in dynamic programming, storing pre-computed objective function values for all possible segments of the sample for optimization.
Produced by the function \code{gen.long()}.}
}
\value{
An upper triangular matrix (\verb{m × m}) containing estimated break dates. The row index represents break dates, and the column index corresponds to the permitted number of breaks before the ending date.
}
\description{
This function estimates break dates in a quantile regression model,
allowing for up to \code{m} breaks.
When \code{m = 1}, this function finds the optimal single-break partition within
a segment by evaluating the objective function at each possible break point
to determine the break date.
When \code{m > 1}, a dynamic programming algorithm is used to efficiently determine
the break dates.
}
\details{
The function first determines the optimal one-break partition.
If multiple breaks are allowed (\code{m > 1}), it applies a dynamic programming
algorithm as in Bai and Perron (2003) to minimize the objective function.
The algorithm finds break dates by iterating over all possible partitions,
returning optimal break locations and associated objective function values.
}
\examples{
\donttest{
  # data
  data(gdp)
  y = gdp$gdp
  x = gdp[,c("lag1", "lag2")]
  n.size = 1
  T.size = length(y) # number of time periods

  # setting
  vec.tau   = seq(0.20, 0.80, by = 0.150)
  trim.e    = 0.2
  trim.size = round(T.size * trim.e)  #minimum length of a regime
  m.max     = 3

  # compute the objective function under all possible partitions
  out.long   = gen.long(y, x, vec.tau, n.size, trim.size)
  vec.long.m = out.long$vec.long  ## for break estimation using multiple quantiles

  # break date
  mat.date = brdate(y, x, n.size, m.max, trim.size, vec.long.m)
  print(mat.date)
}
}
\references{
Bai, J. and P. Perron (2003).
Computation and analysis of multiple structural change models.
\emph{Journal of Applied Econometrics}, 18(1), 1-22.

Oka, T. and Z. Qu (2011). Estimating structural changes in regression quantiles,
\emph{Journal of Econometrics}, 162(2), 248-267.
}
