\name{openbugs}
\alias{openbugs}
\title{Wrapper to run OpenBUGS}
\description{The \code{openbugs} function takes data and starting values
  as input.  It automatically calls the package \pkg{BRugs} and runs
  something similar to \code{\link[BRugs]{BRugsFit}}.  Not available in
  S-PLUS.}

\usage{
openbugs(data, inits, parameters.to.save,
    model.file = "model.txt", n.chains = 3, n.iter = 2000,
    n.burnin = floor(n.iter/2),
    n.thin = max(1, floor(n.chains * (n.iter - n.burnin) / n.sims)),
    n.sims = 1000,  DIC = TRUE, 
    bugs.directory = "c:/Program Files/OpenBUGS/",
    working.directory = NULL, digits = 5, over.relax = FALSE, seed=NULL)
}

\arguments{
  \item{data}{either a named list (names corresponding to variable names
    in the \code{model.file}) of the data for the \pkg{OpenBUGS} model,
    \emph{or} a vector or list of the names of the data objects used by
    the model.  If \code{data} is a one element character vector (such as \code{"data.txt"}), 
    it is assumed that data have already been written to the working directory into that file, 
    e.g. by the function \code{\link{bugs.data}}.}
  \item{inits}{a list with \code{n.chains} elements; each element of the
    list is itself a list of starting values for the \pkg{OpenBUGS} model,
    \emph{or} a function creating (possibly random) initial values.
    Alternatively, if \code{inits} are missing or \code{inits = NULL},
    initial values are generated by \pkg{OpenBUGS}.}
  \item{parameters.to.save}{character vector of the names of the
    parameters to save which should be monitored}
  \item{model.file}{file containing the model written in \pkg{OpenBUGS} code.
    The extension can be either \file{.bug} or \file{.txt}.  If
    \file{.bug}, a copy of the file with extension \file{.txt} will be
    created in the \code{bugs()} call and removed afterwards.  Note that
    similarly named \file{.txt} files will be overwritten.}
  \item{n.chains}{number of Markov chains (default: 3)}
  \item{n.iter}{number of total iterations per chain (including burn in;
    default: 2000)}
  \item{n.burnin}{length of burn in, i.e. number of iterations to
    discard at the beginning.  Default is \code{n.iter/2}, that is,
    discarding the first half of the simulations.}
  \item{n.thin}{thinning rate.  Must be a positive integer.  Set
    \code{n.thin} > 1 to save memory and computation time if
    \code{n.iter} is large.  Default is \code{max(1, floor(n.chains *
    (n.iter-n.burnin) / 1000))} which will only thin if there are at
    least 2000 simulations.}
  \item{n.sims}{The approximate number of simulations to keep after thinning.}
  \item{DIC}{logical; if \code{TRUE} (default), compute deviance, pD, and
    DIC.  This is done in \pkg{BRugs} directly.}
  \item{digits}{number of significant digits used for \pkg{OpenBUGS} input,
    see \code{\link{formatC}}}
  \item{bugs.directory}{directory that contains the \pkg{OpenBUGS} executable
    - currently unused}
  \item{working.directory}{sets working directory during execution of
    this function; \pkg{WinBUGS} in- and output will be stored in this
    directory; if \code{NULL}, a temporary working directory via
    \code{\link{tempdir}} is used.}
  \item{over.relax}{If \code{TRUE}, over-relaxed form of MCMC is used if available from OpenBUGS.}
  \item{seed}{random seed (default is no seed)}
}

\value{A \code{\link{bugs}} object.}

\author{Andrew Gelman, \email{gelman@stat.columbia.edu},
  \url{http:/www.stat.columbia.edu/~gelman/bugsR/}; modifications and
  packaged by Sibylle Sturtz, \email{sturtz@statistik.tu-dortmund.de},
  and Uwe Ligges.
}

\note{By default, BRugs (and hence \code{openbugs()}) is quite verbose. 
    This can be controlled for the whole BRugs package by the option \sQuote{BRugsVerbose} (see \code{\link{options}})
    which is set to \code{TRUE} by default.
}

\seealso{\code{\link{bugs}} and the \pkg{BRugs} package}

\examples{
# An example model file is given in:
model.file <- system.file(package = "R2WinBUGS", "model", "schools.txt")
# Let's take a look:
file.show(model.file)

# Some example data (see ?schools for details):
data(schools)
schools

J <- nrow(schools)
y <- schools$estimate
sigma.y <- schools$sd
data <- list ("J", "y", "sigma.y")
inits <- function(){
    list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100),
         sigma.theta = runif(1, 0, 100))
}
## or alternatively something like:
# inits <- list(
#   list(theta = rnorm(J, 0, 90), mu.theta = rnorm(1, 0, 90),
#        sigma.theta = runif(1, 0, 90)),
#   list(theta = rnorm(J, 0, 100), mu.theta = rnorm(1, 0, 100),
#        sigma.theta = runif(1, 0, 100))
#   list(theta = rnorm(J, 0, 110), mu.theta = rnorm(1, 0, 110),
#        sigma.theta = runif(1, 0, 110)))

parameters <- c("theta", "mu.theta", "sigma.theta")

\dontrun{
## both write access in the working directory and package BRugs required:
schools.sim <- bugs(data, inits, parameters, model.file,
    n.chains = 3, n.iter = 5000,
    program = "openbugs", working.directory = NULL)
print(schools.sim)
plot(schools.sim)
}
}
\keyword{interface}
\keyword{models}
