% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/oc2Sdecision.R
\name{oc2Sdecision}
\alias{oc2Sdecision}
\title{Decision Function for 2 Sample Operating Characteristics}
\usage{
oc2Sdecision(pc = 0.975, qc = 0, lower.tail = TRUE, link = c("identity",
  "logit", "log"))
}
\arguments{
\item{pc}{vector of critical cumulative probabilities of the
difference distribution.}

\item{qc}{vector of respective critical values of the difference
distribution. Must match the length of \code{pc}.}

\item{lower.tail}{logical value selecting if the threshold is a
lower or upper bound.}

\item{link}{enables application of a link function prior to
evaluating the difference distribution. Can take one of the values
\code{identity} (default), \code{logit} or \code{log}.}
}
\value{
The function returns a decision function which takes three
arguments. The first and second argument are expected to be mixture
(posterior) distributions from which the difference distribution is
formed. The third argument determines if the function acts as an
indicator function or if the function returns the distance from the
decision boundary for each condition in log-space, i.e. the
distance is 0 at the decision boundary, negative for a 0 decision
and positive for a 1 decision.
}
\description{
The function sets up a 2 sample one-sided decision function with an
arbitrary number of conditions on the difference distribution.
}
\details{
This function creates a one-sided decision function on the
basis of the difference distribution in a 2 sample situation. To
support double criterion design, see \emph{Neuenschwander et al.,
2010}, an arbitrary number of criterions can be given. The decision
function demands that the probability mass below the critical value
\code{qc} of the difference \eqn{x_1 - x_2} is at least
\code{pc}. Hence, for \code{lower.tail=TRUE} condition \eqn{i} is
equivalent to

\deqn{P(x_1 - x_2 \leq q_{c,i}) > p_{c,i}}

and the decision function is implemented as indicator function on
the basis of the heavy-side step function \eqn{H} which is \eqn{0}
for \eqn{x \leq 0} and \eqn{1} for \eqn{x > 0}. As all conditions
must be met, the final indicator function returns

\deqn{\Pi_i H_i(P(x_1 - x_2 \leq q_{c,i}) - p_{c,i} ),}

which is \eqn{1} if all conditions are met and \eqn{0}
otherwise. For \code{lower.tail=FALSE} differences must be greater
than the given quantile \code{qc}.

Note that whenever a \code{link} other than \code{identity} is
requested, then the underlying densities are first transformed
using the link function and then the probabilties for the
differences are calculated in the transformed space. Hence, for a
binary endpoint the \code{logit} link will lead to decisions based
on the differences in \code{logit}s corresponding to a criterion
based on the log-odds. The \code{log} link will evaluate ratios
instead of absolute differences which could be useful for a binary
endpoint or counting rates. The respective critical quantiles
\code{qc} must be given on the transformed scale.
}
\examples{

# see Gsponer et al., 2010
priorT <- mixnorm(c(1,   0, 0.001), sigma=88, param="mn")
priorP <- mixnorm(c(1, -49, 20   ), sigma=88, param="mn")
# the success criteria is for delta which are larger than some
# threshold value which is why we set lower.tail=FALSE
successCrit  <- oc2Sdecision(c(0.95, 0.5), c(0, 50), FALSE)
# the futility criterion acts in the opposite direction
futilityCrit <- oc2Sdecision(c(0.90)     , c(40),    TRUE)

print(successCrit)
print(futilityCrit)

# consider decision for specific outcomes
postP_interim <- postmix(priorP, n=10, m=-50)
postT_interim <- postmix(priorT, n=20, m=-80)
futilityCrit( postP_interim, postT_interim )
successCrit(  postP_interim, postT_interim )

# Binary endpoint with double criterion decision on log-odds scale
# 95\% certain positive difference and an odds ratio of 2 at least
decL2 <- oc2Sdecision(c(0.95, 0.5), c(0, log(2)), lower.tail=FALSE, link="logit")
# 95\% certain positive difference and an odds ratio of 3 at least
decL3 <- oc2Sdecision(c(0.95, 0.5), c(0, log(3)), lower.tail=FALSE, link="logit")

# data scenario
post1 <- postmix(mixbeta(c(1, 1, 1)), n=40, r=10)
post2 <- postmix(mixbeta(c(1, 1, 1)), n=40, r=18)

# positive outcome and a median odds ratio of at least 2 ...
decL2(post2, post1)
# ... but not more than 3
decL3(post2, post1)

}
\references{
Neuenschwander B, Capkun-Niggli G, Branson M,
Spiegelhalter DJ. Summarizing historical information on controls in
clinical trials. \emph{Clin Trials}. 2010; 7(1):5-18

Gsponer T, Gerber F, Bornkamp B, Ohlssen D, Vandemeulebroecke M,
Schmidli H.A practical guide to Bayesian group sequential
designs. \emph{Pharm. Stat.}. 2014; 13: 71-80
}
\seealso{
Other oc2S: \code{\link{oc2S}}
}
