% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mixess.R
\name{ess}
\alias{ess}
\title{Effective Sample Size for a Conjugate Prior}
\usage{
ess(mix, method = c("moment", "morita"), ...)
}
\arguments{
\item{mix}{Prior (mixture of conjugate distributions).}

\item{method}{Selects the used method. Can be either \code{moment} (default) or \code{morita}.}

\item{...}{Optional arguments applicable to specific methods.}
}
\description{
Calculates the Effective Sample Size (ESS) for a mixture prior. The
ESS indicates how many experimental units the prior is roughly
equivalent to.
}
\details{
The ESS is calculated using the either a moments based
approach or the more sophisticated method by \emph{Morita et
al. (2008)}. The moments based method is the default method and
provides conservative estimates of the ESS.

For the moments method the mean and standard deviation of the
mixture are calculated and then approximated by the conjugate
distribution with the same mean and standard deviation. For
conjugate distributions, the ESS is well defined. See the examples
for a step-wise calculation in the beta mixture case.

The Morita method used here evaluates the mixture prior at the mode
instead of the mean as proposed originally by Morita. The method
may lead to very optimistic ESS values, especially if the mixture
contains many components.
}
\section{Supported Conjugate Prior-Likelihood Pairs}{


\tabular{lccc}{
\strong{Prior/Posterior} \tab \strong{Likelihood} \tab \strong{Predictive} 
 \tab \strong{Summaries} \cr
Beta \tab Binomial \tab Beta-Binomial \tab \code{n}, \code{r} \cr
Normal \tab Normal (\emph{fixed \eqn{\sigma}}) \tab Normal \tab \code{n}, \code{m}, \code{se}  \cr
Gamma \tab Poisson \tab Gamma-Poisson \tab  \code{n}, \code{m} \cr
Gamma \tab Exponential \tab Gamma-Exp (\emph{not supported}) \tab \code{n}, \code{m}
}
}

\examples{
# Conjugate Beta example
a <- 5
b <- 15
prior <- mixbeta(c(1, a, b))

ess(prior)
(a+b) 

# Beta mixture example
bmix <- mixbeta(rob=c(0.2, 1, 1), inf=c(0.8, 10, 2))

ess(bmix)
# moments method is equivalent to
# first calculate moments
bmix_sum <- summary(bmix)
# then calculate a and b of a matching beta
ab_matched <- ms2beta(bmix_sum["mean"], bmix_sum["sd"])
# finally take the sum of a and b which are equivalent
# to number of responders/non-responders respectivley
round(sum(ab_matched))

ess(bmix, method="morita")

# Normal mixture example
nmix <- mixnorm(rob=c(0.5, 0, 2), inf=c(0.5, 3, 4), sigma=10)

ess(nmix)

## the reference scale determines the ESS
sigma(nmix) <- 20
ess(nmix)

# Gamma mixture example
gmix <- mixgamma(rob=c(0.3, 20, 4), inf=c(0.7, 50, 10))

ess(gmix) ## interpreted as appropriate for a Poisson likelihood (default)

likelihood(gmix) <- "exp"
ess(gmix) ## interpreted as appropriate for an exponential likelihood


}
\references{
Morita S, Thall PF, Mueller P.
Determining the effective sample size of a parametric prior.
\emph{Biometrics} 2008;64(2):595-602.
}
