% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/regu-est.R
\name{glm.regu.path}
\alias{glm.regu.path}
\title{Regularied M-estimation for fitting generalized linear models along a regularization path}
\usage{
glm.regu.path(nrho = NULL, rho.seq = NULL, y, x, iw = NULL,
  loss = "cal", n.iter = 100, eps = 1e-06, tune.fac = 0.5,
  tune.cut = TRUE, ann.init = TRUE, nz.lab = NULL)
}
\arguments{
\item{nrho}{The number of tuning parameters in a regularization path.}

\item{rho.seq}{A vector of tuning parameters in a regularization path. If both \code{nrho} and \code{rho.seq} are specified, then \code{rho.seq} overrides \code{nrho}.}

\item{y}{An \eqn{n} x \eqn{1} response vector.}

\item{x}{An \eqn{n} x \eqn{p} matix of covariates, excluding a constant.}

\item{iw}{An \eqn{n} x \eqn{1} weight vector.}

\item{loss}{A loss function, which can be specified as "guas" for continuous responses, or "ml" or "cal" for binary respones.}

\item{n.iter}{The maximum number of iterations allowed as in \code{\link{glm.regu}}.}

\item{eps}{The tolerance used to declare convergence as in \code{\link{glm.regu}}.}

\item{tune.fac}{The multiplier (factor) used to define rho.seq if only \code{nrho} is specified.}

\item{tune.cut}{Logical; if \code{TRUE}, all smaller tuning parameters are skipped once non-convergence is found with a tuning parameter.}

\item{ann.init}{Logical; if \code{TRUE}, the estimates from the previous tuning parameter are used as the inital value when fitting with the current tuning parameter.}

\item{nz.lab}{A \eqn{p} x \eqn{1} logical vector (useful for simulations), indicating which covariates are included when calculating the number of nonzero coefficients.}
}
\value{
\item{rho}{The vector of tuning parameters included in the regularization path.}
\item{non.conv}{A vector indicating the non-convergene status found or imputed if \code{tune.cut=TRUE}, along the regularization path. 
For each tuning parameter, 0 indicates convergence, 1 non-convergence if exceeding \code{n.iter}, 2 non-convergence if exceeding \code{bt.lim}.}
\item{nz.all}{A vector giving the numbers of nonzero coefficients estimated, along the regularization path.}
\item{bet.all}{A matrix giving estimated intercept and coefficients, column by column, along the regularization path.}
\item{fit.all}{A matrix giving fitted values, column by column, along the regularization path.}
}
\description{
This function implements regularized M-estimation for fitting generalized linear models with binary or contiunous responses 
along a regularization path.
}
\details{
If not specified by users, the sequence of tuning parameters (i.e., the regularization path) is defined as a geometric series 
of length \code{nrho}, starting from the value which yields a zero solution, and then decreasing by a factor \code{tune.fac} successively.
}
\examples{
\donttest{
data(simu.data)
n <- dim(simu.data)[1]
p <- dim(simu.data)[2]-2

y <- simu.data[,1]
tr <- simu.data[,2]
x <- simu.data[,2+1:p]
x <- scale(x)

### Example 1: linear regression
out.rgaus.path <- glm.regu.path(rho.seq=c(.01, .02, .05, .1, .2, .5), y=y[tr==1], x=x[tr==1,], 
                                loss="gaus")

# the estimated intercept and coefficients; the first 10 are shown
out.rgaus.path$bet.all[1:10,]

### Example 2: logistic regression using likelihood loss
out.rml.path <- glm.regu.path(rho.seq=c(.002, .005, .01, .02, .05, .1), y=tr, x=x, loss="ml")
out.rml.path$bet.all[1:10,]

### Example 3: logistic regression using calibration loss
out.rcal.path <- glm.regu.path(rho.seq=c(.005, .01, .02, .05, .1, .2), y=tr, x=x, loss="cal")
out.rcal.path$bet.all[1:10,]
}

}
\references{
Tan, Z. (2017) Regularized calibrated estimation of propensity scores with model misspecification and high-dimensional data, arXiv:1710.08074. 

Tan, Z. (2019) Model-assisted inference for treatment effects using regularized calibrated estimation with high-dimensional data, 
\emph{Annals of Statistics}, to appear (preprint arXiv:1801.09817).
}
