#' Optimization of Floating Percentile Model Parameters
#' 
#' Calculate parameter inputs that optimize benchmark performance
#' 
#' @param  data data.frame containing, at a minimum, chemical concentrations as columns and a logical \code{Hit} column classifying toxicity
#' @param  paramList character vector of column names of chemical concentration variables in \code{data}
#' @param  plot logical; whether to generate a plot to visualize the opimization results
#' @param  FN_crit numeric vector over which to optimize false negative thresholds (default = \code{seq(0.1, 0.9, by = 0.05))}
#' @param  alpha numeric vector of type-I error rate values over which to optimize (default = \code{seq(0.05, 0.5, by = 0.05))}
#' @param  simplify logical; whether to export only optimized values or the full results of optimization
#' @param  which numeric or character indicating which type of plot to generate (see Details; default = \code{c(1, 2)})
#' @param  colors values recognizable as colors - text, hexadecimal, numbers, etc. (default = \code{heat.colors(10)}).
#' @param  colsteps numeric value, number of unique colors to include in gradient (default = \code{100})
#' @param  ... additional argument passed to \code{FPM}, \code{chemSig}, and \code{chemSigSelect}
#' @details  
#' \code{optimFPM} was designed to help optimize the predictive capacity of the benchmarks generated by \code{FPM}. The default input parameters to
#' \code{FPM} (i.e., \code{FN_crit = 0.2} and \code{alpha = 0.05}) are arbitrary, and optimization can help to objectively establish more accurate benchmarks.
#' Graphical output from \code{optimFPM} can also help users to understand the relationship(s) between benchmark accuracy/error, \code{FN_crit}, and \code{alpha}.
#' We also recommend that users apply \code{cvFPM} to their data to further inform the selection of \code{FPM} input values.
#' 
#' Default inputs for \code{FN_crit} and \code{alpha} were selected to represent a reasonable range of values to test. Testing over both ranges
#' will result in a two-way optimization, which can be computationally intensive. Alternatively, \code{optimFPM} can be run for one parameter at a time
#' by specifying a single value for \code{FN_crit} or \code{alpha}. Note that inputting single values for both \code{FN_crit} and \code{alpha} will generate unhelpful results. 
#' 
#' Two metrics are used for optimization, one based on the maximum overall reliability (i.e., highest probability of correctly
#' predicting \code{Hit} values) and one based on minimizing the difference between the false negative and false positive rates,
#' which represents a trade-off between under- and overconservatism.
#' 
#' Graphical output will differ depending on whether or not a single value is input for \code{FN_crit} or \code{alpha}. Providing a single value for one
#' of the two arguments will generate a line graph, whereas providing longer vectors (i.e., length > 1) of inputs for both arguments will generate dot matrix plots using \code{colors} to generate
#' a color palette and \code{colsteps} to define the granularity of the color gradient with the palette. The order of \code{colors} will be plotted
#' from more optimal to less optimal; for example, the default of \code{heat.colors(10)} will show optimal colors as red and less optimal colors as yellower.
#' By default, two plots will be generated, however the \code{which} argument can control whether to include either or both plots. Inputs
#' to \code{which} are, by default, \code{c(1, 2)}, but flexible character inputs also can be used, for example \code{which = "OR"} or \code{which = "balanced"}.
#' Black and gray squares indicate the optimal argument values (black for the indicated optimization metric and gray for the other metric).
#' 
#' @return data.frame of optimized \code{FN_crit} and/or \code{alpha} values
#' @seealso FPM, cvFPM
#' @importFrom dplyr bind_rows
#' @importFrom graphics points 
#' @importFrom graphics lines
#' @importFrom graphics abline
#' @importFrom graphics legend
#' @examples
#' paramList = c("Cd", "Cu", "Fe", "Mn", "Ni", "Pb", "Zn")
#' FN_seq <- seq(0.1, 0.3, 0.05)
#' alpha_seq <- seq(0.05, 0.2, 0.05)
#' optimFPM(data = h.tristate, paramList = paramList, alpha = 0.05, FN_crit = FN_seq)
#' optimFPM(data = h.tristate, paramList = paramList, FN_crit = 0.2, alpha = alpha_seq)
#' optimFPM(data = h.tristate, paramList = paramList, alpha = alpha_seq, FN_crit = FN_seq, which = 2)
#' @export
optimFPM <- function(data, 
                      paramList,
                      plot = TRUE,
                      FN_crit = seq(0.1, 0.9, by = 0.05),
                      alpha = seq(0.05, 0.5, by = 0.05), 
                      simplify = TRUE,
                      which = c(1, 2),
                      colors = heat.colors(10),
                      colsteps = 100,
                      ...){
    
    if(length(FN_crit) > 1 & length(alpha) > 1){
        grid <- expand.grid(FN_crit, alpha)
        names(grid) <- c("FN_crit", "alpha")

        x <- apply(grid, 1, function(x) {
            tmp <- FPM(data, paramList, FN_crit = x[1], alpha = x[2], ...)[["FPM"]]
           return(tmp)
        })
        
        z <- as.data.frame(matrix(ncol = length(paramList) + 14,
                                         nrow = 1, data = rep(NA, length(paramList) + 14)))
        names(z) <- c(paramList, "FN_crit", "TP", "FN", "TN", "FP",
                        "pFN", "pFP", "HR", "NR", "PHR", "PNR", "FHR", "FNR", "OR")
        z <- do.call(rbind, lapply(x, function(i) dplyr::bind_rows(z, i)[-1,])) # remove dummy row
        row.names(z) <- NULL
        
        y <- do.call(rbind, lapply(x, function(tmp) {c(FPFN = abs(tmp$pFN - tmp$pFP), OR = tmp$OR)}))
        out <- data.frame(grid, y)
        
        min1 <- out[min(which.min(out$FPFN)),]
        min2 <- out[min(which.min(1 - out$OR)),]
    
        if(plot){
            
            if(any(c(1, "OR", "or", "reliability", "Reliability", "rel", "Rel") %in% which)){
                plot(out[1:2], 
                     col = colorGradient(1 - out[,4], colors = colors, colsteps = colsteps, na.rm = TRUE),
                     pch = 15, xlab = "FN Limit", ylab = "alpha", cex = 3, 
                     main = paste0("Overall Reliability\nRange:", paste(signif(range(out[, "OR"]), 2), collapse = "-")))
                points(x = min2$FN_crit, y = min2$alpha, pch = 0, cex = 4, col = "black")
                points(x = min1$FN_crit, y = min1$alpha, pch = 0, cex = 3.5, col = "gray")
            }
            
            if(any(c(2, "Bal", "bal", "balance", "balanced", "Balance", "Balanced") %in% which)){
                plot(out[1:2], 
                     col = colorGradient(out[,3], colors = colors, colsteps = colsteps, na.rm = TRUE),
                     pch = 15, xlab = "FN Limit", ylab = "alpha", cex = 3, 
                     main = paste0("|FN% - FP%|\nRange:", paste(signif(range(out[, "FPFN"]), 2), collapse = "-")))
                points(x = min1$FN_crit, y = min1$alpha, pch = 0, cex = 4, col = "black")
                points(x = min2$FN_crit, y = min2$alpha, pch = 0, cex = 3.5, col = "gray")
            }
        }
        tmp <- data.frame(t(min1[1:2]), t(min2[1:2]))
        names(tmp) <- c("balance_FN.FP", "max_OR")
        
        if(simplify){
            return(tmp)
        } else {
            return(list(z, tmp))
        }
    } else if(length(FN_crit) > 1 & length(alpha) == 1) {
        if(any(FN_crit >  1) | any(FN_crit <= 0)){
            stop("FN_crit values must be between 0 and 1")
        }
        
        tmp <- sapply(FN_crit, function(x) {
            FPM(data = data, paramList = paramList, FN_crit = x, alpha = alpha, ...)[["FPM"]]
            })
        pFN <- unlist(tmp["pFN",])
        pFP <- unlist(tmp["pFP",])
        OR <- unlist(tmp["OR",])
        pick <- FN_crit[min(which.max(OR))]
        pick2 <- FN_crit[which.min(abs(1 - (pFN/pFP)))]
    
        z <- as.data.frame(matrix(ncol = length(paramList) + 14,
                                         nrow = 1, data = rep(NA, length(paramList) + 14)))
        names(z) <- c(paramList, "FN_crit", "TP", "FN", "TN", "FP",
                        "pFN", "pFP", "HR", "NR", "PHR", "PNR", "FHR", "FNR", "OR")
        
        out <- do.call(dplyr::bind_rows, 
                       list(z,
                     data.frame(FPM(data = data, paramList = paramList, FN_crit = pick2, ...)[["FPM"]]),
                     data.frame(FPM(data = data, paramList = paramList, FN_crit = pick, ...)[["FPM"]])
                     )
        )[-1,] 
        row.names(out) <- c("balance_FN.FP", "max_OR")
        
        if(plot){
            plot(x = FN_crit, y = OR, ylim = c(0,1),
                 xlab = "FN Limit", ylab = "Error rate", type = "l")
            lines(x = FN_crit, y = pFN, col = 2, type = "l")
            lines(x = FN_crit, y = pFP, col = 4, type = "l")
            abline(v = pick, lty = 3, col = "darkorchid", lwd = 2)
            abline(v = pick2, lty = 3, col = "darkorange", lwd = 2)
            legend("top", inset = -0.1, xpd = TRUE, cex = 0.8, ncol = 2, bg = "white",
                   lty = c(1, 1, 1, 3, 3), lwd = c(1, 1, 1, 2, 2),
                   col = c(1, 2, 4, "darkorchid", "darkorange"),
                   legend = c("Overall reliability",
                              "False negative rate",
                              "False positive rate",
                              "Maximum reliability",
                              "Balanced FN:FP"))
        }
        
        if(simplify){
            return(
            c(balance_FN.FP = out["balance_FN.FP", "FN_crit"],
              max_OR = out["max_OR", "FN_crit"])
            )
        } else{
            return(list(optim = out, 
                    optim_FN = c(balance_FN.FP = out["balance_FN.FP", "FN_crit"],
                    max_OR = out["max_OR", "FN_crit"]))
            )
        }
    } else if(length(FN_crit) == 1 & length(alpha) > 1){
        if(any(alpha >  1) | any(alpha <= 0)){
            stop("alpha values must be between 0 and 1")
        }
        
        z <- as.data.frame(matrix(ncol = length(paramList) + 15,
                                         nrow = 1, data = rep(NA, length(paramList) + 15)))
        names(z) <- c(paramList, "FN_crit", "TP", "FN", "TN", "FP",
                        "pFN", "pFP", "HR", "NR", "PHR", "PNR", "FHR", "FNR", "OR", "alpha")
        
        tmp <- dplyr::bind_rows(z, dplyr::bind_rows(sapply(alpha, function(x) {
            FPM(data = data, paramList = paramList, alpha = x, ...)[["FPM"]]
            })))[-1,]
        
        pFN <- unlist(tmp[,"pFN"])
        pFP <- unlist(tmp[,"pFP"])
        OR <- unlist(tmp[,"OR"])
        pick <- alpha[min(which.min(1 - OR))] # lowest of the low Overall Reliability
        pick2 <- alpha[min(which.min(abs(pFN - pFP)))]
    
        out <- do.call(dplyr::bind_rows, 
                       list(z,
                     data.frame(FPM(data = data, paramList = paramList, alpha = pick2, ...)[["FPM"]], alpha = pick2),
                     data.frame(FPM(data = data, paramList = paramList, alpha = pick, ...)[["FPM"]], alpha = pick)
                     )
        )[-1,] ## force data into standard template then remove template row
        row.names(out) <- c("balance_FN.FP", "max_OR")
        
        if(plot){
            plot(x = alpha, y = OR, ylim = c(0,1),
                 xlab = "Alpha level", ylab = "Error rate", type = "l")
            lines(x = alpha, y = pFN, col = 2, type = "l")
            lines(x = alpha, y = pFP, col = 4, type = "l")
            abline(v = pick, lty = 3, col = "darkorchid", lwd = 2)
            abline(v = pick2, lty = 3, col = "darkorange", lwd = 2)
            legend("top", inset = -0.1, xpd = TRUE, cex = 0.8, ncol = 2, bg = "white",
                   lty = c(1, 1, 1, 3, 3), lwd = c(1, 1, 1, 2, 2),
                   col = c(1, 2, 4, "darkorchid", "darkorange"),
                   legend = c("Overall reliability",
                              "False negative rate",
                              "False positive rate",
                              "Maximum reliability",
                              "Balanced FN:FP"))
        }
        
        if(simplify){
            return(
                c(balance_FN.FP = out["balance_FN.FP", "alpha"],
                  max_OR = out["max_OR", "alpha"])
            )
        } else{
            return(
                list(optim = out, 
                    optim_alpha = c(balance_FN.FP = out["balance_FN.FP", "alpha"],
                    max_OR = out["max_OR", "alpha"]))
            )
        }
    }
}## end code
