#' @title Retrieve Parameter Information from RVI file Algorithms
#'
#' @description
#' Reads the processed rvi object and returns a data frame of parameter information for parameters
#' associated with processes in the rvi object.
#'
#' @details
#' Uses the Raven database files in the /extdata folder to (1) associate parameters with particular algorithms,
#' and (2) subset the database parameter list based on the information in the rvi file.
#'
#' These files are stored with the RavenR package and retrieved with this function by default,
#' but a separate link may be provided to a modified file if desired. Note that the database files held in the RavenR
#' package are unofficial copies of those in the official Raven SVN, and any discrepancies should defer to the Raven SVN versions.
#'
#' @param rvi data object generated from the \code{\link{rvn_rvi_read}} routine
#' @param RavenAlgParamsFile (optional) path to RavenAlgParams.dat file
#' @param RavenParamsFile (optional) path to RavenParameters.dat file
#'
#' @return
#' Returns a data frame with parameter information containing the parameter name, parameter class, units, auto (whether
#' the parameter may be autogenerated within Raven), default, min, and max values. Note that the flags 'xxx',
#' -9999, and 9999 are used as missing values within the parameter data frame.
#'
#' @examples
#' # sample workflow of rvn_rvi_read
#' rvi <- system.file("extdata","Nith.rvi", package="RavenR") %>%
#' rvn_rvi_read(.)
#'
#' # get data frame of parameters related to processes in rvi file
#' rvn_rvi_getparams(rvi)
#'
#' @export rvn_rvi_getparams
#' @importFrom utils read.table
rvn_rvi_getparams <- function(rvi,
                              RavenAlgParamsFile=system.file("extdata","RavenAlgParams.dat", package="RavenR"),
                              RavenParamsFile=system.file("extdata","RavenParameters.dat", package="RavenR"))
{

  HPTable <- rvi$HydProcTable

  # update condition and conditional here
  # update reference to RavenProcessConnections.dat with new files

  if (nrow(HPTable)==0){
    print("WARNING (rvn_rvi_getparams): no rows in hydrologicprocess table")
    return (NA)
  }
  if (paste(names(HPTable), collapse=" ") != "ProcessType Algorithm From To Conditional Note"){ #  & ncol(HPTable)!=6
    stop("(rvn_rvi_getparams): improper hydrological process table data frame format")
  }

  # read in AlgParams file
  # note: stops at line 179, numerical methods not read in
  stopifnot(file.exists(RavenAlgParamsFile))
  cnames<-c("Algorithm","AlgorithmType","ParameterName","ParameterClass")
  RavenAlgParamsTable<-read.table(RavenAlgParamsFile,
                            sep="",
                            col.names=cnames,
                            nrows=173,
                            header=FALSE,
                            blank.lines.skip=TRUE,
                            strip.white=TRUE,
                            stringsAsFactors=FALSE,
                            flush=TRUE,
                            comment.char = "#")

  stopifnot(file.exists(RavenParamsFile))
  cnames <- c("param","class_type","units","auto","default","min","max")
  RavenParamsTable<-read.table(RavenParamsFile,
                                  sep="",
                                  col.names=cnames,
                                  header=FALSE,
                                  blank.lines.skip=TRUE,
                                  strip.white=TRUE,
                                  stringsAsFactors=FALSE,
                                  flush=TRUE,
                                  comment.char = "#")

  subset_AlgParams <- RavenAlgParamsTable[RavenAlgParamsTable$Algorithm %in% HPTable$Algorithm,]
  param_df <- RavenParamsTable[RavenParamsTable$param %in% subset_AlgParams$ParameterName,]

  return(param_df)
}
