% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filter_count.R
\name{countPixelsFilter}
\alias{countPixelsFilter}
\title{Filter based on the minimum number of connected pixels in the ROI.}
\usage{
countPixelsFilter(msiData, roiImage, minNumPixels = 9,
  smoothPeakImage = FALSE, smoothSigma = 2, closePeakImage = FALSE,
  closeKernSize = 5, aggressive = 0, verbose = TRUE)
}
\arguments{
\item{msiData}{\link{msi.dataset-class} object. See \link{msiDataset}.}

\item{roiImage}{\link{ms.image-class} object representing the ROI mask. See
\link{msImage}.}

\item{minNumPixels}{integer (default = 9). Smallest number of connected pixels
used to select a peak.}

\item{smoothPeakImage}{logical (default = \code{FALSE}). Whether the peak
images should be smoothed before determining the connected components.}

\item{smoothSigma}{numeric (default = 2). Standard deviation of Gaussian
kernel.}

\item{closePeakImage}{logical (default = \code{FALSE}). Whether morphological
closing should be applied to the binary peak images.}

\item{closeKernSize}{numeric (default = 5). Kernel size for morphological
closing operation. Kernel shape is fixed to \code{diamond}.}

\item{aggressive}{integer (default = 0). Defines the level of aggressiveness
of the filter. See 'Details' section.}

\item{verbose}{logical (default = \code{TRUE}). Additional output text.}
}
\value{
\code{peak.filter} object. See \link{applyPeaksFilter-msi.dataset-method}.
}
\description{
\code{countPixelsFilter} selects peaks which signals are localized in regions
consisting of a minimum number of connected pixels in the ROI.
}
\details{
Count filter tries to determine and remove peaks which signal is
scattered in a region unrelated with the expected ROI. A minimum number of
connected pixels in the ROI is used to trigger the filter. This value should
be carefully set equal to the geometrical size of the smallest expected sub-region.
Each peak image is binarized using Otsu's thresholding and the connected components
are extracted. The filter selects those peaks that show, within the ROI, at least one
connected component of size larger or equal to \code{minNumPixels}. The level
of aggressiveness, associated with increasing values of the parameter
\code{aggressive}, determines whether the size of the connected components
within the ROI should be compared with that of the connected components
localized outside the ROI. If \code{aggressive = 1}, the filter checks whether
the max size of the connected components localized outside the ROI is smaller
or equal to the maximum size of the connected components inside the ROI.
If \code{aggressive = 2}, a stricter filter checks whether the maximum size
of the connected components localized outside the ROI is smaller than
\code{minNumPixels}. Different aggressiveness levels can produce completely
different results, depending on the nature of the analyzed dataset.
}
\examples{
## Load package
library("SPUTNIK")

## Mass spectrometry intensity matrix
X <- matrix(rnorm(200), 20, 40)

## Print original dimensions
print(dim(X))

## m/z vector
mzVector <- seq(600, 900, by = (900 - 600) / 39)

## Read the image size
imSize <- c(5, 4)

## Construct the ms.dataset object
msiX <- msiDataset(X, mzVector, imSize[1], imSize[2])

## Extract the ROI using k-means
ref.roi <- refAndROIimages(msiData = msiX, refMethod = "sum",
                           roiMethod = "otsu", useFullMZRef = TRUE)

## Perform count pixels filtering
count.sel <- countPixelsFilter(msiData = msiX, roiImage = ref.roi$ROI,
                               minNumPixels = 4, aggressive = 1)

## Apply the filter
msiX <- applyPeaksFilter(msiX, count.sel)

## Print new dimensions
print(dim(getIntensityMat(msiX)))
}
\seealso{
applyPeaksFilter
}
\author{
Paolo Inglese \email{p.inglese14@imperial.ac.uk}
}
