% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msi.dataset_methods.R
\name{normIntensity,msi.dataset-method}
\alias{normIntensity,msi.dataset-method}
\alias{normIntensity}
\title{Normalize the peaks intensities.}
\usage{
\S4method{normIntensity}{msi.dataset}(object, method = "median", peaksInd = NULL, offsetZero = 0)
}
\arguments{
\item{object}{\link{msi.dataset-class} object.}

\item{method}{string (default = \code{"median"}). The normalization method to
be used. Valid values are: \code{"median"}, \code{"PQN"}, \code{"TIC"},
\code{TMM}, or \code{"upperQuartile"}.
See 'Details' section.}

\item{peaksInd}{numeric array (default = NULL). Array of peak indices used to
calculate the scaling factors (TIC, median). If NULL, all the peaks are used.}

\item{offsetZero}{numeric (default = 0). This value is added to all the peak
intensities to take into accounts of the zeros.}
}
\value{
object \link{msi.dataset-class} object, with normalized peaks
intensities.

When using TIC scaling, if zeros are present in the matrix, a positive offset
must be added to all the peak intensities through the parameter \code{offsetZero}.
This is necessary for applying the CLR transformation. TIC scaling transforms the
spectra into compositional data; in this case the CLR transformation must be
applied through the varTransform function.
}
\description{
Normalize the peaks intensities.
}
\details{
The valid values for \code{method} are:
\itemize{
  \item \code{"median"}: median of spectrum intensities is scaled to one.
  \item \code{"PQN"}:
  \enumerate{
    \item apply \code{"TIC"} normalization
    \item calculate the median reference spectrum (after removing the zeros)
    \item calculate the quotients of peaks intensities and reference
    \item calculate the median of quotients for each peak (after removing the zeros)
    \item divide all the peak intensities by the median of quotients
  }
  \item \code{"TIC"}: total ion current normalization assign the sum of the
  peaks intensities to one.
  \item \code{"TMM"}: trimmed mean of M-values (TMM with zero pairing).
  Called TMMwzp in edgeR.
  \item \code{"upperQuartile"}: spectra are scaled by their 3rd quartile.
}
}
\examples{
## Load package
library("SPUTNIK")

## Create the msi.dataset-class object
sz <- c(40, 40)
x <- matrix(rnorm(sz[1] * sz[2] * 20) * 1000, sz[1] * sz[2], 20)
x[x < 0] <- 0 # MS data is positive
mz <- sort(sample(100, ncol(x)))
msiX <- msiDataset(x, mz, sz[1], sz[2])

## Normalize and log-transform
msiX <- normIntensity(msiX, "median")
msiX <- varTransform(msiX, "log")

## Create the msi.dataset-class object
sz <- c(40, 40)
x <- matrix(rnorm(sz[1] * sz[2] * 20) * 1000, sz[1] * sz[2], 20)
x[x < 0] <- 0 # MS data is positive
mz <- sort(sample(100, ncol(x)))
msiX <- msiDataset(x, mz, sz[1], sz[2])

## Normalize using PQN
msiX <- normIntensity(msiX, "PQN")
}
\references{
F. Dieterle, A. Ross, G. Schlotterbeck, and Hans Senn. 2006.
Probabilistic quotient normalization as robust method to account for dilution
of complex biological mixtures. Application in 1H NMR metabonomics.
Analytical Chemistry 78(13): 4281-4290.

Robinson MD, Oshlack A (2010). A scaling normalization method for
differential expression analysis of RNA-seq data. Genome Biology 11, R25.
}
\seealso{
\link{msi.dataset-class}
}
\author{
Paolo Inglese \email{p.inglese14@imperial.ac.uk}
}
