% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SCD.R
\name{SCD}
\alias{SCD}
\title{Detecting cellwise outliers using Shapley values.}
\usage{
SCD(
  x,
  mu,
  Sigma,
  Sigma_inv = NULL,
  step_size = 0.1,
  min_deviation = 0,
  max_step = NULL,
  max_iter = 1000,
  q = 0.99,
  method = "cellMCD",
  check = TRUE,
  cells = NULL
)
}
\arguments{
\item{x}{Data vector with \eqn{p} entries or data matrix with \eqn{n \times p} entries containing only numeric entries.}

\item{mu}{Either \code{NULL} (default) or mean vector of \code{x}. If NULL, \code{method} is used for parameter estimation.}

\item{Sigma}{Either \code{NULL} (default) or covariance matrix \eqn{p \times p} of \code{x}. If NULL, \code{method} is used for parameter estimation.}

\item{Sigma_inv}{Either \code{NULL} (default) or Sigma's inverse \eqn{p \times p} matrix.
If \code{NULL}, the inverse of \code{Sigma} is computed using \code{solve(Sigma)}.}

\item{step_size}{Numeric. Step size for the imputation of outlying cells, with \code{step_size} \eqn{\in [0,1]}. Defaults to \eqn{0.1}.}

\item{min_deviation}{Numeric. Detection threshold, with \code{min_deviation} \eqn{\in [0,1]}. Defaults to \eqn{0.2}}

\item{max_step}{Either \code{NULL} (default) or an integer. The maximum number of steps in each iteration.
If \code{NULL}, \code{max_step} \eqn{= p}.}

\item{max_iter}{Integer. The maximum number of iterations.}

\item{q}{Numeric. The quantile of the Chi-squared distribution for detection and imputation of outliers. Defaults to \eqn{0.99}.}

\item{method}{Either "cellMCD" (default) or "MCD". Specifies the method used for parameter estimation if \code{mu} and/or \code{Sigma} are not provided.}

\item{check}{Logical. If \code{TRUE} (default), inputs are checked before running the function
and an error message is returned if one of the inputs is not as expected.}

\item{cells}{Either \code{NULL} (default) or a vector/matrix of the same dimension as \code{x},
indicating the outlying cells. The matrix must contain only zeros and ones, or \code{TRUE}/\code{FALSE}.}
}
\value{
A list of class \code{shapley_algorithm} (\code{\link{new_shapley_algorithm}}) containing the following:
\item{\code{x}}{A \eqn{p}-dimensional vector (or a \eqn{n \times p} matrix) containing the imputed data.}
\item{\code{phi}}{A \eqn{p}-dimensional vector (or a \eqn{n \times p} matrix) containing the Shapley values (outlyingness-scores) of \code{x}; see \code{\link{shapley}}.}
\item{\code{x_original}}{A \eqn{p}-dimensional vector (or a \eqn{n \times p} matrix) containing the original data.}
\item{\code{x_history}}{The path of how the original data vector was modified.}
\item{\code{phi_history}}{The Shapley values corresponding to \code{x_history}.}
\item{\code{S_history}}{The indices of the outlying cells in each iteration.}
}
\description{
The \code{SCD} function indicates outlying cells for
a data vector with \eqn{p} entries or data matrix with \eqn{n \times p} entries containing only numeric entries \code{x}
for a given center \code{mu} and covariance matrix \code{Sigma} using the Shapley value \insertCite{Mayrhofer2022}{ShapleyOutlier}.
}
\examples{
p <- 5
mu <- rep(0,p)
Sigma <- matrix(0.9, p, p); diag(Sigma) = 1
Sigma_inv <- solve(Sigma)
x <- c(0,1,2,2.3,2.5)
SCD_x <- SCD(x = x, mu = mu, Sigma = Sigma)
plot(SCD_x)

library(MASS)
set.seed(1)
n <- 100; p <- 10
mu <- rep(0,p)
Sigma <- matrix(0.9, p, p); diag(Sigma) = 1
X <- mvrnorm(n, mu, Sigma)
X[sample(1:(n*p), 100, FALSE)] <- rep(c(-5,5),50)
SCD_X <- SCD(X, mu, Sigma)
plot(SCD_X, subset = 20)
}
\references{
\insertAllCited{}
}
