\name{rmult.clm}
\alias{rmult.clm}
\title{
Simulating Correlated Ordinal Responses Conditional on a Marginal Cumulative Link Model Specification
}
\description{
Simulates correlated ordinal responses assuming a cumulative link model for the marginal probabilities.
}
\usage{
rmult.clm(clsize, intercepts, betas, xformula, xdata, link, cor.matrix, rlatent = NULL)
}
\arguments{
  \item{clsize}{
integer indicating the common cluster size.
}
  \item{intercepts}{
numerical vector or matrix containing the intercepts of the marginal cumulative link model.
}
  \item{betas}{
numerical vector or matrix containing the value of the marginal regression parameter vector (excluding \code{intercepts}).
}
  \item{xformula}{
formula expression as in other marginal regression models but without including a response variable.  
}
  \item{xdata}{
optional data frame containing the variables provided in \code{xformula}.
}
  \item{link}{
character string indicating the link function in the marginal cumulative link model. Options include \code{"probit"}, \code{"logit"}, \code{"cloglog"} or \code{"cauchit"}. Required when \code{rlatent = NULL}.
}
  \item{cor.matrix}{
matrix indicating the correlation matrix of the multivariate normal distribution when the NORTA method is employed (\code{rlatent = NULL}).
}
 \item{rlatent}{
matrix with \code{clsize} columns containing realizations of the latent random vectors when the NORTA method is not preferred. See details for more info.
}
}
\details{
The assumed marginal cumulative link model is  
\deqn{Pr(Y_{it}\le j |x_{it})=F(\beta_{tj0} +\beta^{'}_{t} x_{it})}
where \eqn{F} is the cumulative distribution function determined by \code{link}. For subject \eqn{i}, \eqn{Y_{it}} is the \eqn{t}-th ordinal response and \eqn{x_{it}} is the associated covariates vector. Finally, \eqn{\beta_{tj0}} is the \eqn{j}-th category-specific intercept at the \eqn{t}-th measurement occasion and \eqn{\beta_{tj}} is the \eqn{j}-th category-specific regression parameter vector at the \eqn{t}-th measurement occasion.

The ordinal response \eqn{Y_{it}} is obtained by extending the approach of \cite{McCullagh (1980)}. See vignette for more info.

When \eqn{\beta_{tj0}=\beta_{j0}} for all \eqn{t}, then \code{intercepts} should be provided as a numerical vector. Otherwise, \code{intercepts} must be a numerical matrix such that row \eqn{t} contains the category-specific intercepts at the \eqn{t}-th measurement occasion.

\code{betas} should be provided as a numeric vector only when \eqn{\beta_{t}=\beta} for all \eqn{t}. Otherwise, \code{betas} must be provided as a numeric matrix with \code{clsize} rows such that the \eqn{t}-th row contains the value of \eqn{\beta_{t}}. In either case, \code{betas} should reflect the order of the terms implied by \code{xformula}.

The appropriate use of \code{xformula} is \code{xformula = ~ covariates}, where \code{covariates} indicate the linear predictor as in other marginal regression models.

The optional argument \code{xdata} should be provided in ``long'' format.

The NORTA method is the default option for simulating the latent random vectors denoted by \eqn{e_{it}} in the threshold approach described in the vignette. To import simulated values for the latent random vectors without utilizing the NORTA method, the user can employ the \code{rlatent} argument. In this case, element (\eqn{i,t}) of \code{rlatent} represents the realization of \eqn{e_{it}}.
}
\value{
Returns a list that has components:
\item{Ysim}{the simulated ordinal responses. Element (\eqn{i},\eqn{t}) represents the realization of \eqn{Y_{it}}.}
\item{simdata}{a data frame that includes the simulated response variables (y), the covariates specified by \code{xformula}, subjects' identities (id) and the corresponding measurement occasions (time).}
\item{rlatent}{the latent random variables denoted by \eqn{e_{it}} in the vignette.}
}
\references{
Cario, M. C. and Nelson, B. L. (1997)  \emph{Modeling and generating random vectors with arbitrary marginal distributions and correlation matrix}. Technical Report, Department of Industrial Engineering and Management Sciences, Northwestern University, Evanston, Illinois.

Li, S. T. and Hammond, J. L. (1975)  Generation of pseudorandom numbers with specified univariate distributions and correlation coefficients. \emph{IEEE Transactions on Systems, Man and Cybernetics} \bold{5}, 557--561.

McCullagh, P. (1980) Regression models for ordinal data. \emph{Journal of the Royal Statistical Society B} \bold{42}, 109--142.

Touloumis, A., Agresti, A. and Kateri, M. (2013) GEE for multinomial responses using a local odds ratios parameterization. \emph{Biometrics} \bold{69}, 633--640.
}
\author{
Anestis Touloumis
}
\seealso{
\code{\link{rmult.bcl}} for simulating correlated nominal responses, \code{\link{rmult.crm}} for simulating correlated ordinal responses and \code{\link{rbin}} for simulating correlated binary responses.
}
\examples{
## See vignette for the sampling scheme.
set.seed(12345)
N <- 500
clsize <- 4
intercepts <- c(-1.5, -0.5, 0.5, 1.5)
betas <- matrix(c(1, 2, 3, 4), 4, 1)
x <- rep(rnorm(N), each = clsize)
cor.matrix <- toeplitz(c(1, 0.85, 0.5, 0.15))
CorOrdRes <- rmult.clm(clsize = clsize, intercepts = intercepts, betas = betas, 
    xformula = ~x, cor.matrix = cor.matrix, link = "probit")

## Same sampling scheme except that the parameter vector is now
## time-stationary.
set.seed(12345)
x <- rep(rnorm(N), each = clsize)
CorOrdRes <- rmult.clm(clsize = clsize, betas = 1, xformula = ~x, cor.matrix = toeplitz(c(1, 
    0.85, 0.5, 0.15)), intercepts = c(-1.5, -0.5, 0.5, 1.5), link = "probit")

## Fit a GEE model (Touloumis et al., 2013) to estimate the regression
## coefficients.
library(multgee)
fitmod <- ordLORgee(y ~ x, id = id, repeated = time, link = "probit", data = CorOrdRes$simdata)
coef(fitmod)
}
