% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary_functions.R
\name{bias}
\alias{bias}
\title{Compute (relative/standardized) bias summary statistic}
\usage{
bias(
  estimate,
  parameter = NULL,
  type = "bias",
  abs = FALSE,
  percent = FALSE,
  unname = FALSE
)
}
\arguments{
\item{estimate}{a \code{numeric} vector or \code{matrix}/\code{data.frame}
of parameter estimates. If a vector,
the length is equal to the number of replications. If a \code{matrix}/\code{data.frame},
the number of rows must equal the number of replications}

\item{parameter}{a \code{numeric} scalar/vector indicating the fixed parameters.
If a single value is supplied and \code{estimate} is a \code{matrix}/\code{data.frame}
then the value will be recycled for each column; otherwise, each element will be associated
with each respective column in the \code{estimate} input.
If \code{NULL} then it will be assumed that the \code{estimate} input is in a deviation
form (therefore \code{mean(estimate))} will be returned)}

\item{type}{type of bias statistic to return. Default (\code{'bias'}) computes the standard bias
(average difference between sample and population), \code{'relative'} computes
the relative bias statistic (i.e., divide the bias by the value
in \code{parameter}; note that multiplying this by 100 gives the "percent bias" measure),
\code{'abs_relative'} computes the relative bias but the absolute values of the parameters
are used in the denominator rather than the (potentially) signed input values,
and \code{'standardized'} computes the standardized bias estimate
(standard bias divided by the standard deviation of the sample estimates)}

\item{abs}{logical; find the absolute bias between the parameters and estimates? This effectively
just applies the \code{\link{abs}} transformation to the returned result. Default is FALSE}

\item{percent}{logical; change returned result to percentage by multiplying by 100?
Default is FALSE}

\item{unname}{logical; apply \code{\link{unname}} to the results to remove any variable
names?}
}
\value{
returns a \code{numeric} vector indicating the overall (relative/standardized)
  bias in the estimates
}
\description{
Computes the (relative) bias of a sample estimate from the parameter value.
Accepts estimate and parameter values, as well as estimate values which are in deviation form.
If relative bias is requested the \code{estimate} and \code{parameter} inputs are both required.
}
\examples{

pop <- 2
samp <- rnorm(100, 2, sd = 0.5)
bias(samp, pop)
bias(samp, pop, type = 'relative')
bias(samp, pop, type = 'standardized')

dev <- samp - pop
bias(dev)

# equivalent here
bias(mean(samp), pop)

# matrix input
mat <- cbind(M1=rnorm(100, 2, sd = 0.5), M2 = rnorm(100, 2, sd = 1))
bias(mat, parameter = 2)
bias(mat, parameter = 2, type = 'relative')
bias(mat, parameter = 2, type = 'standardized')

# different parameter associated with each column
mat <- cbind(M1=rnorm(1000, 2, sd = 0.25), M2 = rnorm(1000, 3, sd = .25))
bias(mat, parameter = c(2,3))

# same, but with data.frame
df <- data.frame(M1=rnorm(100, 2, sd = 0.5), M2 = rnorm(100, 2, sd = 1))
bias(df, parameter = c(2,2))

# parameters of the same size
parameters <- 1:10
estimates <- parameters + rnorm(10)
bias(estimates, parameters)

# relative difference dividing by the magnitude of parameters
bias(estimates, parameters, type = 'abs_relative')

# relative bias as a percentage
bias(estimates, parameters, type = 'abs_relative', percent = TRUE)


}
\references{
Sigal, M. J., & Chalmers, R. P. (2016). Play it again: Teaching statistics with Monte
Carlo simulation. \code{Journal of Statistics Education, 24}(3), 136-156.
\doi{10.1080/10691898.2016.1246953}
}
\seealso{
\code{\link{RMSE}}
}
\author{
Phil Chalmers \email{rphilip.chalmers@gmail.com}
}
