% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVPedSim_Functions.R
\name{trim_ped}
\alias{trim_ped}
\title{Trim pedigree based on proband recall}
\usage{
trim_ped(ped_file, recall_probs = NULL)
}
\arguments{
\item{ped_file}{An object of class \code{ped}. A pedigree generated by \code{sim_ped} or \code{sim_RVped}, or an object created by the function \code{\link{new.ped}}.  See details.}

\item{recall_probs}{Numeric. The proband's recall probabilities for relatives, see details.  If not supplied, the default value of four times kinship coefficient between the proband and the relative is used.}
}
\value{
\code{ped_trim} The trimmed pedigree.
}
\description{
Primarily intended as an internal function, \code{trim_ped} chooses a proband and trims relatives based on the proband's probability of recalling his or her relatives.
}
\details{
By default \code{recall_probs} is four times the kinship coefficient, as defined by Thompson (see references), between the proband and the probands relative, which results in a recall probability of \eqn{2^{-(n-1)}} for a relative of degree \eqn{n}. Alternatively, the user may specify a list of recall probabilities of length \eqn{l > 0}, in which case the first \emph{l-1} items in \code{recall_probs} are the respective proband recall probabilities for relatives of degree \emph{1, 2, ..., l-1}, and the \emph{l}th item in \code{recall_probs} is the proband's recall probability for all relatives of degree \strong{\emph{l} or greater}.  For example if \code{recall_probs = c(1)} all relatives will be recalled by the proband with probability 1.

Occasionally, a trimmed family member must be retained to ensure that the pedigree can be plotted.  When this occurs, family members who share a non-zero kinship coefficient with the proband are censored of all pertinent information, and will always have the following qualities:
\enumerate{
  \item availability status = 0
  \item affected status = NA
  \item birth year = NA
  \item onset year = NA
  \item death year = NA
  \item RR = NA
}

Users who wish to use \code{trim_ped} for pedigrees not generated by \code{sim_ped} or \code{sim_RVped} must use \code{\link{new.ped}} to create an object of class \code{ped}.  The \code{ped} object \emph{must} contain the following variables for each pedigree member:
\tabular{lll}{
\strong{name} \tab \strong{type} \tab \strong{description} \cr
\code{FamID} \tab numeric \tab family identification number \cr
\code{ID} \tab numeric \tab individual identification number \cr
\code{dadID} \tab numeric \tab identification number of father \cr
\code{momID} \tab numeric \tab identification number of mother \cr
\code{sex} \tab numeric \tab gender identification; if male \code{sex = 0}, if female \code{sex = 1} \cr
\code{affected} \tab logical \tab disease-affection status: \cr
\code{proband} \tab logical \tab a proband identifier: \code{proband = TRUE} if the individual is the proband, and \code{FALSE} otherwise.\cr
\tab \tab \code{affected  = TRUE} if affected by disease , and \code{FALSE} otherwise, \cr
\code{birthYr} \tab numeric \tab the individual's birth year.\cr
\code{onsetYr} \tab numeric \tab the individual's disease onset year, when applicable.\cr
\code{deathYr} \tab numeric \tab the individual's death year, when applicable.\cr
\code{RR} \tab numeric \tab the individual's relative risk of disease. \cr
\code{available} \tab logical \tab availibility status; \cr
\tab\tab \code{available = TRUE} if available, and \code{FALSE} otherwise. \cr
}
}
\examples{
#Read in example pedigree to trim
data(EgPeds)
egPeds <- new.ped(EgPeds)

#plot example_ped using kinship2
plot(subset(egPeds, FamID == 1), location = "topright", cex = 0.85)
mtext("Original Pedigree", side = 3, line = 2)


## Trim pedigree examples
# Illustrate the effect of various settings for recall_probs
Recall_Probabilities <- list(c(1),
                             c(1, 0.5),
                             c(1, 0.25, 0.1))


for (k in 1:length(Recall_Probabilities)) {
   set.seed(2)
   #trim pedigree
   TrimPed <- trim_ped(ped_file = subset(egPeds, FamID == 1),
                       recall_probs = Recall_Probabilities[[k]])

   plot(TrimPed, location = "topright", cex = 0.85)
   mtext(paste0("recall_probs = (", sep = "",
                paste(Recall_Probabilities[[k]], collapse = ", "), ')' ),
                side = 3, line = 2 )
}


}
\references{
Nieuwoudt, Christina and Jones, Samantha J and Brooks-Wilson, Angela and Graham, Jinko (2018). \emph{Simulating Pedigrees Ascertained for Multiple Disease-Affected Relatives}. Source Code for Biology and Medicine, 13:2.

Thompson, E. (2000). \emph{Statistical Inference from Genetic Data on Pedigrees.} NSF-CBMS Regional Conference Series in Probability and Statistics, 6, I-169.
}
\seealso{
\code{\link{sim_RVped}}, \code{\link{sim_ped}}, \code{\link{new.ped}}
}
