% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation-spades.R
\name{spades}
\alias{spades}
\alias{spades,simList,ANY,ANY,missing-method}
\alias{spades,ANY,ANY,ANY,logical-method}
\title{Run a spatial discrete event simulation}
\usage{
spades(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  ...
)

\S4method{spades}{simList,ANY,ANY,missing}(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  ...
)

\S4method{spades}{ANY,ANY,ANY,logical}(
  sim,
  debug = getOption("spades.debug"),
  progress = NA,
  cache,
  .plotInitialTime = NULL,
  .saveInitialTime = NULL,
  notOlderThan = NULL,
  ...
)
}
\arguments{
\item{sim}{A \code{simList} simulation object, generally produced by \code{simInit}.}

\item{debug}{Optional logical flag or character vector indicating what to print to
console at each event. See details.
Default is to use the value in \code{getOption("spades.debug")}.}

\item{progress}{Logical (\code{TRUE} or \code{FALSE} show a graphical progress bar),
character (\code{"graphical"}, \code{"text"}) or numeric indicating
the number of update intervals to show in a graphical progress bar.}

\item{cache}{Logical. If \code{TRUE}, then the \code{spades} call will be cached.
This means that if the call is made again with the same simList,
then `spades`` will return the return value from the previous run
of that exact same simList. Default \code{FALSE}. See Details.
See also the vignette on caching for examples.}

\item{.plotInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{.saveInitialTime}{Numeric. Temporarily override the \code{.plotInitialTime}
parameter for all modules. See Details.}

\item{notOlderThan}{Date or time. Passed to \code{reproducible::Cache} to update the cache.
Default is \code{NULL}, meaning don't update the cache.
If \code{Sys.time()} is provided, then it will force a recache,
i.e., remove old value and replace with new value.
Ignored if \code{cache} is \code{FALSE}.}

\item{...}{Any. Can be used to make a unique cache identity, such as "replicate = 1".
This will be included in the \code{Cache} call, so will be unique
and thus \code{spades} will not use a cached copy as long as
anything passed in \code{...} is unique, i.e., not cached previously.}
}
\value{
Invisibly returns the modified \code{simList} object.
}
\description{
Here, we implement a simulation in a more modular fashion so it's easier to add
submodules to the simulation. We use S4 classes and methods, and use `data.table`
instead of `data.frame` to implement the event queue (because it is much faster).
}
\details{
The is the workhorse function in the SpaDES package. It runs simulations by
implementing the rules outlined in the \code{simList}.

This function gives simple access to two sets of module parameters:
\code{.plotInitialTime} and with \code{.plotInitialTime}. The primary use of
these arguments is to temporarily turn off plotting and saving. "Temporary"
means that the \code{simList} is not changed, so it can be used again with
the simList values reinstated. To turn off plotting and saving, use
\code{.plotInitialTime = NA} or \code{.saveInitialTime = NA}. NOTE: if a
module did not use \code{.plotInitialTime} or \code{.saveInitialTime}, then
these arguments will not do anything.

If \code{cache} is TRUE, this allows for a seamless way to "save" results
of a simulation. The  user does not have to intentionally do any saving manually.
Instead, upon a call to \code{spades} in which the simList is identical,
the function will simply return the result that would have come if it had
been rerun. Use this with caution, as it will return exactly the result from
a previous run, even if there is stochasticity internally.
Caching is only based on the input simList.
See also the vignette on caching for examples.
}
\note{
The debug option is primarily intended to facilitate building simulation
models by the user.
Will print additional outputs informing the user of updates to the values of
various \code{simList} slot components.
See \url{https://github.com/PredictiveEcology/SpaDES/wiki/Debugging} for details.
}
\section{\code{debug}}{


\code{debug} can be a logical, character vector or a numeric scalar (currently
1 or 2).
If \code{debug} is specified and is not \code{FALSE}, 2 things could happen:
1) there can be messages sent to console, such as events as they pass by, and
2) if \code{options("spades.browserOnError" = TRUE)} (experimental still) if
there is an error, it will attempt to open a browser
in the event where the error occurred. You can edit, and then press \code{c} to continue
or \code{Q} to quit, plus all other normal interactive browser tools.
\code{c} will trigger a reparse and events will continue as scheduled, starting
with the one just edited. There may be some unexpected consequences if the
\code{simList} objects had already been changed before the error occurred.

If not specified in the function call, the package
option \code{spades.debug} is used. The following
options for debug are available:

\tabular{ll}{
  \code{TRUE} \tab the event immediately following will be printed as it
runs (equivalent to \code{current(sim)}).\cr
  function name (as character string) \tab If a function, then it will be run on the
                                           simList, e.g., "time" will run
                                           \code{time(sim)} at each event.\cr
  moduleName (as character string) \tab All calls to that module will be entered
                                        interactively\cr
  eventName (as character string) \tab All calls that have that event name (in any module)
                                       will be entered interactively\cr
  \code{c(<moduleName>, <eventName>)}  \tab Only the event in that specified module
                                            will be entered into. \cr
  Any other R expression expressed as a character string  \tab
                                Will be evaluated with access to the simList as 'sim'.
                               If this is more than one character string, then all will
                               be printed to the screen in their sequence. \cr
  A numeric scalar, currently 1 or 2 (maybe others) \tab This will print out alternative forms of event
                                          information that users may find useful \cr
                                          information that users may find useful \cr

}
}

\examples{
\dontrun{
mySim <- simInit(
 times = list(start = 0.0, end = 2.0, timeunit = "year"),
 params = list(
   .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
 ),
 modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
 paths = list(modulePath = system.file("sampleModules", package = "SpaDES.core"))
)
spades(mySim)

# set default debug printing for the current session
# setOption(spades.debug = TRUE)

# Different debug options (overrides the package option 'spades.debug')
spades(mySim, debug = TRUE) # Fastest
spades(mySim, debug = "simList")
spades(mySim, debug = "print(table(sim$landscape$Fires[]))")

# Can turn off plotting, and inspect the output simList instead
out <- spades(mySim, .plotInitialTime = NA) # much faster
completed(out) # shows completed events

# use cache -- simInit should generally be rerun each time a spades call is made
#   to guarantee that it is identical. Here, run spades call twice, first
#   time to establish cache, second time to return cached result
for (i in 1:2) {
 mySim <- simInit(
   times = list(start = 0.0, end = 2.0, timeunit = "year"),
   params = list(
     .globals = list(stackName = "landscape", burnStats = "nPixelsBurned")
   ),
   modules = list("randomLandscapes", "fireSpread", "caribouMovement"),
   paths = list(modulePath = system.file("sampleModules", package = "SpaDES.core"))
 )
 print(system.time(out <- spades(mySim, cache = TRUE)))
}
}

}
\references{
Matloff, N. (2011). The Art of R Programming (ch. 7.8.3).
            San Francisco, CA: No Starch Press, Inc..
            Retrieved from \url{https://www.nostarch.com/artofr.htm}
}
\seealso{
\code{\link{SpaDES.core-package}},
\code{\link{simInit}}, and the caching vignette (very important for reproducibility):
\url{https://CRAN.R-project.org/package=SpaDES.core/vignettes/iii-cache.html} which
uses \code{\link[reproducible]{Cache}}.
}
\author{
Alex Chubaty and Eliot McIntire
}
