##' plotGlobalGraph
##'
##'
##' @title Plot a global graph of the drug
##' @description The `plotGlobalGraph()` identifies the drug label entered by the user, and plots an
##' integrated diagram including box plot of the normalized drug-disease reverse association scores, null distribution curves
##' of significant P-value, and heat map of cancer subtype sample distribution.
##' @param data A list of result data generated by function `OCSSD()`.
##' @param drug.label A character string of drug labels to determine which drug to use for visualization.
##' @param overall.main An overall title for the whole graph. If the user
##' does not make any input, the title will display a drug label.
##' @param overall.cex.main The magnification to be used for overall.main (default: 1.5).
##' @param cex.submap.axis The magnification to be used for axis of each submap annotation relative to
##' the current setting of cex.
##' @param cex.submap.lab The magnification to be used for x and y labels of each submap relative to
##' the current setting of cex.
##' @param cex.submap.main The magnification to be used for main titles of each submap relative to the
##' current setting of cex.
##' @param cex.submap.sub The magnification to be used for sub titles of each submap relative to the
##' current setting of cex.
##' @param cex.legend fontsize of labels for legend.
##' @return A plot.
##' @author Xudong Han,
##' Junwei Han,
##' Chonghui Liu
##' @examples
##' ## Get the result data of OCSSD().
##' ## The data is based on the simulated breast cancer subtype data.
##' \donttest{Subtype_drugs<-get("Subtype_drugs")}
##' ## Plot a global graph of the drug pirenperone(1.02e-05M).
##' \donttest{plotGlobalGraph(data=Subtype_drugs,drug.label="pirenperone(1.02e-05M)")}
##' @importFrom graphics boxplot
##' @importFrom graphics par
##' @importFrom BiocGenerics density
##' @importFrom graphics plot
##' @importFrom graphics lines
##' @importFrom graphics title
##' @importFrom graphics layout
##' @importFrom graphics legend
##' @importFrom graphics image
##' @importFrom graphics axis
##' @export

plotGlobalGraph<-function(data,drug.label="",overall.main="",overall.cex.main=1.5,cex.submap.axis=1,cex.submap.lab=1,
                          cex.submap.main=1,cex.submap.sub=1,cex.legend=1){

  phen<-names(table(data[["SampleInformation"]][["sampleSubtype"]]))
  control.label<-data[["Parameter"]][["control.label"]]
  phen<-phen[phen!=control.label]
 if(length(phen)==1){
    stop("There is no drug disease inverse association score matrix in the results of the two sample types")
  }


    sampleid<-match(colnames(data[["DrugMatrix"]]),data[["SampleInformation"]][["sampleId"]])
    sample_v<-data[["SampleInformation"]][["sampleSubtype"]][sampleid]
    drug_index<-which(row.names(data[["DrugMatrix"]])==drug.label)
    colork<-get("Colork")
    sycolor<-colork[1:length(phen)]

    layout(matrix(c(1,3,2,3), 2, 2))
    s_NMDCS<-NULL
    s_p<-NULL
    for(i in 1:length(phen)){
      s_NMDCS<-paste(s_NMDCS,phen[i],"=",data[[i]][drug_index,4])
      s_p<-paste(s_p,phen[i],"=",data[[i]][drug_index,5])
    }
    s_NMDCS<-paste("SDS:",s_NMDCS)
    s_p<-paste("Pvalue:",s_p)
    group<-data.frame(drug=data[["DrugMatrix"]][drug_index,],samp=sample_v)
    boxplot(drug~samp,group,col=sycolor,xlab = "Sample subtype",ylab = "Normalized Drug-disease reverse association score",main="Box-plot",sub=s_NMDCS,
            cex.axis=cex.submap.axis,cex.lab=cex.submap.lab,cex.main=cex.submap.main,cex.sub=cex.submap.sub)

    #
    drug_scores<-data[["DrugMatrix"]][drug_index,]
    phen_scores<-vector("numeric",length(phen))
    for(i in 1:length(phen)){
      phen_scores[i]<-mean(drug_scores[which(sample_v==phen[i])])
    }

    rd_scores<-sapply(1:data[["Parameter"]][["nperm"]], function(r){
       rd_phen_scores<-NULL
       rd_sample_v<-sample(sample_v,replace = FALSE)
       for(i in 1:length(phen)){
        rd_phen_scores<-c(rd_phen_scores,mean(drug_scores[which(rd_sample_v==phen[i])]))
      }
      return(rd_phen_scores)
    })
    rd_scores<-as.vector(rd_scores)

    drug_den<-density(rd_scores)
    plot(drug_den,lwd = 2,ylab = "Frequency",sub = s_p,xlab = "SDS",main="SDS Null Distribution",
         cex.axis=cex.submap.axis,cex.lab=cex.submap.lab,cex.main=cex.submap.main,cex.sub=cex.submap.sub)
    for(i in 1:length(phen)){
      score_wz<-which(drug_den[["x"]]>=phen_scores[i])[1]
      lines(c(drug_den[["x"]][score_wz],drug_den[["x"]][score_wz]),c(0,drug_den[["y"]][score_wz]),lwd = 3,
          col=sycolor[i])
    }
   legend("topright",phen,col=sycolor,lwd=2,bty="n",cex=cex.legend)

   retux<-matrix(NA,nrow = length(phen),ncol = length(data[["DrugMatrix"]][1,]))
   ordindex<-order(data[["DrugMatrix"]][drug_index,],decreasing = TRUE)
   correl<-data[["DrugMatrix"]][drug_index,ordindex]
   ordsample<-sample_v[ordindex]
   revphen<-rev(phen)
   for(p in 1:length(phen)){
     ppindex<-which(ordsample==revphen[p])
     retux[p,ppindex]<-correl[ppindex]
   }
   row.names(retux)<-revphen
   colnames(retux)<-ordsample

   n.rows<-length(retux[,1])
   n.cols<-length(retux[1,])
   rnames1<-row.names(retux)
   cnames1<-colnames(retux)
   mycol <-rev(c("#FF0000", "#FF0D1D", "#FF4040", "#FF5A5A", "#FF7080", "#FF9DB0", "#FFAADA", "#EEE5EE", "#D5D5FF", "#A9A9FF", "#8888FF", "#7070FF", "#4040FF", "#0000FF","#0000FF"))
   image(1:n.cols, 1:n.rows, t(retux), col=mycol, axes=FALSE,main="Heat map of the normalized drug-disease reverse association scores for each subtype sample (red:positive,blue:negative)",xlab = NA,ylab=" ",cex.main=cex.submap.main)
   axis(2, at=1:n.rows , labels=rnames1, adj= 0, tick=FALSE,cex.axis=cex.submap.axis,las = 1, font.axis=1, line=-1)
   size.col.char <- 35/(n.cols)
   axis(1, at=1:n.cols, labels=cnames1, tick=FALSE, las = 3,cex.axis=size.col.char, font.axis=1, line=-1)
   if(overall.main==""){
     title(main=drug.label,outer =TRUE,line=-1,cex.main = overall.cex.main)
   }else{
     title(main=overall.main,outer =TRUE,line=-1,cex.main = overall.cex.main)
   }

}
