% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeTropicalCoordinates}
\alias{computeTropicalCoordinates}
\title{Compute Tropical Coordinates from a Persistence Diagram}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeTropicalCoordinates()} computes the following seven tropical coordinates based on the lifespans (or persistence) \eqn{\lambda_i = d_i - b_i}:

\itemize{
  \item \eqn{F_1 = \max_i \lambda_i}.
  \item \eqn{F_2 = \max_{i<j} (\lambda_i+\lambda_j)}.
  \item \eqn{F_3 = \max_{i<j<k} (\lambda_i+\lambda_j+\lambda_k)}.
  \item \eqn{F_4 = \max_{i<j<k<l} (\lambda_i+\lambda_j+\lambda_k+\lambda_l)}.
  \item \eqn{F_5 = \sum_i \lambda_i}.
  \item \eqn{F_6 = \sum_i \min(r \lambda_i, b_i)}, where \eqn{r} is a positive integer.
  \item \eqn{F_7 = \sum_j \big(\max_i(\min(r \lambda_i, b_i)+\lambda_i)  - (\min(r \lambda_j, b_j)+\lambda_j)\big)}.
}
Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeTropicalCoordinates(D, homDim, r = 1)
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}
\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}
\item{r}{a positive integer used to compute \eqn{F_6} and \eqn{F_7}. Default is 1.}
}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes the seven tropical coordinates based on the filtered data. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.
}

\value{
A (named) numeric vector \eqn{(F_1, F_2, F_3, F_4, F_5, F_6, F_7)}.
}

\author{
Umar Islambekov
}

\references{
1. Kališnik, S., (2019). Tropical coordinates on the space of persistence barcodes. Foundations of Computational Mathematics, 19(1), pp.101-129.

2. Ali, D., Asaad, A., Jimenez, M.J., Nanda, V., Paluzo-Hidalgo, E. and Soriano-Trigueros, M., (2023). A survey of vectorization methods in topological data analysis. IEEE Transactions on Pattern Analysis and Machine Intelligence.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

# Compute tropical coordinates for homological dimension H_0
computeTropicalCoordinates(D, homDim = 0)

# Compute tropical coordinates for homological dimension H_1
computeTropicalCoordinates(D, homDim = 1)
}
