% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/out.data.R
\name{out.data}
\alias{out.data}
\title{Generating TDM output}
\usage{
out.data(
  input,
  vpd.input,
  sr.input,
  prec.input,
  peak.hours = c(10:14),
  low.sr = 150,
  peak.sr = 300,
  vpd.cutoff = 0.5,
  prec.lim = 1,
  method = "env.filt",
  max.quant = 1,
  make.plot = TRUE
)
}
\arguments{
\item{input}{An \code{\link{is.trex}}-compliant time series from \code{\link{tdm_cal.sfd}} outputs
(e.g., \code{X$sfd.mw$sfd})}

\item{vpd.input}{An \code{\link{is.trex}}-compliant object an individual series of VPD in kPa (see vpd).
The extent and temporal resolution should be equal to input.
Use \code{\link{dt.steps}} to correct if needed.}

\item{sr.input}{An \code{\link{is.trex}}-compliant object of an individual series of solar irradiance
(e.g. either PAR or global radiation; see \code{\link{sr}}).
 The extent and temporal resolution should be equal to input. Use \code{\link{dt.steps}} to correct if needed.
 This data is only needed when applying the "env.filt" method.}

\item{prec.input}{An \code{\link{is.trex}}-compliant object of daily precipitation in mm d-1 (see \code{\link{preci}}).
The extent should be equal to input with a daily temporal resolution.
 Use \code{\link{dt.steps}} to correct if needed.
 This data is only needed when applying the "env.filt" method.}

\item{peak.hours}{Numeric vector with hours which should be considered as peak-of-the-day hours
(default = \code{c(10:14)}).
This variable is only needed when the "stat" method is selected.}

\item{low.sr}{Numeric threshold value in the unit of the sr.input time-series (e.g., W m-2)
to exclude cloudy days which impact \eqn{G_{C}}{Gc} (default = 150 W m-2).
 This variable is only needed when the "env.filt" method is selected.}

\item{peak.sr}{Numeric threshold value in the unit of the sr.input time-series (e.g., W m-2)
to exclude hours which are not considered as peak-of-the-day hours (default = 300 W m-2).
This variable is only needed when the "env.filt" method is selected.}

\item{vpd.cutoff}{Numeric threshold value in kPa for peak-of-the-day mean VPD to eliminate unrealistic
and extremely high values of \eqn{G_{C}}{Gc} due to low VPD values or high values of \eqn{G_{C}}{Gc} (default = 0.5 kPa).}

\item{prec.lim}{Numeric threshold value in mm d-1 for daily precipitation to remove rainy days (default = 1 mm d-1).
This variable is only needed when "env.filt" method is selected.}

\item{method}{Character string indicating whether precipitation and solar irradiance data should be used
to determined peak-of-the-day \eqn{G_{C}}{Gc} values and filter the daily \eqn{G_{C}}{Gc} values ("env.filt")
 or not ("stat"; default). When \code{"env.filt"} is selected, \code{input}, \code{vpd.input}, \code{sr.input}, \code{prec.input},
  \code{peak.sr}, \code{low.sr}, \code{vpd.cutoff} and \code{prec.lim} have to be provided.
  When \code{"stat"} is selected only \code{input}, \code{vpd.input} and \code{peak.hours}.}

\item{max.quant}{Numeric, defining the quantile of the \eqn{G_{C}}{Gc} data which should be considered as GC.max (default = 1).}

\item{make.plot}{Logical; if \code{TRUE}, a plot is generated presenting the response of \eqn{G_{C}}{Gc} to VPD.}
}
\value{
A named list of \code{data.frame} objects,
 containing the following items:

\describe{
   \item{raw}{A \code{data.frame} containing the input data and filtered values. Columns include the timestamp [,"timestamp"]
(e.g., "2012-01-01 00:00:00"), year of the data [,"year"], day of year [,"doy"], input solar radiance data [,"sr"],
 daily average radiance data [,"sr"], input vapour pressure deficit data [,"vpd"], isolated peak-of-the-day vapour pressure
 deficit values [,"vpd.filt"], input daily precipitation [,"prec.day"], sap flux density expressed in mmol m-2 s-1 [,"fd"],
 crown conductance expressed in mmol m-2 s-1 kPa-1 [,"gc"], and the filtered crown conductance [,"gc.filt"]}

 \item{peak.mean}{A \code{data.frame} containing the daily mean crown conductance values.
 Columns include the timestamp [,"timestamp"] (e.g., "2012-01-01"), peak-of-the-day vapour pressure deficit [,"vpd.filt"],
  the filtered crown conductance mmol m-2 s-1 kPa-1 [,"gc.filt"], and the normalized crown conductance according
   to the maximum crown conductance [,"gc.norm"].}

  \item{sum.mod}{A model summary object (see \code{\link{summary}()})
   of the model between VPD and \eqn{G_{C}}{Gc}.}


   \item{sum.mod.norm}{A model summary object (see \code{\link{summary}()})
   of the model between VPD and \eqn{G_{C}}{Gc}/\eqn{GC.max}.}



}
}
\description{
Generates relevant outputs from the sap flux density (SFD) values.
This function provides both \eqn{F_{d}}{Fd} (\eqn{SFD} expressed in \eqn{mmol m^{-2} s^{-1}}{mmol m-2 s-1}) and crown conductance values
(\eqn{G_{C}}{Gc}; an analogue to stomatal conductance) in an easily exportable format.
Additionally, the function can perform environmental filtering on \eqn{F_{d}}{Fd} and \eqn{G_{C}}{Gc} and model \eqn{G_{C}}{Gc} sensitivity to vapour pressure deficit (VPD).
The user can choose between in- (\code{method = "env.filt"}) or excluding (\code{method = "stat"}) environmental filtering
on the \eqn{G_{C}}{Gc} and adjust the filter threshold manually.
}
\details{
Various relevant outputs can be derived from the SFD data.
This function provides the option to recalculate SFD to \eqn{F_{d}}{Fd} (expressed in mmol m-2 s-1)
and crown conductance (according to Pappas \emph{et al.} 2018).
\eqn{G_{C}}{Gc} is estimated per unit sapwood area, where \eqn{G_{C} = F_{d} / VPD}{GC = Fd / VPD} (in kPa), assuming that
i) the stem hydraulic capacitance between the height of sensor and the leaves is negligible, and
ii) that the canopy is well coupled to the atmosphere. In order to reduce the effect of stem hydraulic capacitance,
peak-of-the-day \eqn{G_{C}}{Gc} are solely considered for calculating daily average \eqn{G_{C}}{Gc}.
Peak-of-the-day conditions are defined by \code{peak.hours} or \code{peak.sr}. Moreover, to analyse the relationship between \eqn{G_{C}}{Gc}
 and environmental measurements (e.g., VPD), the daily mean peak-of-the-day \eqn{G_{C}}{Gc} values can be restricted to:
 i) non-cloudy days (see \code{low.sr}), to reduce the impact of low irradiance on \eqn{G_{C}}{Gc},
 ii) non-rainy days (see \code{prec.lim}), as wet leaves are not well coupled to the atmosphere, and
 iii) daily mean peak-of-the-day \eqn{G_{C}}{Gc} great then a threshold (see \code{vpd.cutoff}),
 to eliminate unrealistically high \eqn{G_{C}}{Gc} values due to low \eqn{F_{d}}{Fd} or VPD values (when method = \code{"env.filt"}).
 Moreover, the sensitivity of the daily mean peak-of-the-day \eqn{G_{C}}{Gc} to VPD is modelled by fitting the following model:

  \deqn{G_{C} = \alpha + \beta VPD^{-0.5}}{GC = \alpha + \beta VPD^(-0.5)}

  Besides using the raw daily mean peak-of-the-day \eqn{G_{C}}{Gc} values, the function also applies
  a normalization where daily mean peak-of-the-day \eqn{G_{C}}{Gc} is standardized to the maximum conductance (GC.max; see \code{max.quant}).
}
\examples{
\donttest{
#Gc response function
#Gc response function
raw   <- is.trex(example.data(type="doy"), tz="GMT",
                 time.format="\%H:\%M", solar.time=TRUE,
                 long.deg=7.7459, ref.add=FALSE)

input <- dt.steps(input=raw, start="2013-05-01 00:00", end="2013-11-01 00:00",
                   time.int=15, max.gap=60, decimals=10, df=FALSE)

input[which(input<0.2)]<- NA
input <- tdm_dt.max(input, methods=c("dr"), det.pd=TRUE, interpolate=FALSE,
                 max.days=10, df=FALSE)

output.data<- tdm_cal.sfd(input,make.plot=TRUE,df=FALSE,wood="Coniferous", decimals = 6)

input<- output.data$sfd.dr$sfd

output<- out.data(input=input, vpd.input=vpd, sr.input=sr, prec.input=preci,
                  low.sr = 150, peak.sr=300, vpd.cutoff= 0.5, prec.lim=1,
                  method="env.filt", max.quant=0.99, make.plot=TRUE)
head(output)

}

}
