% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVineMLE.R
\name{RVineMLE}
\alias{RVineMLE}
\title{Maximum Likelihood Estimation of an R-Vine Copula Model}
\usage{
RVineMLE(data, RVM, start = RVM$par, start2 = RVM$par2, maxit = 200,
  max.df = 30, max.BB = list(BB1 = c(5, 6), BB6 = c(6, 6), BB7 = c(5, 6),
  BB8 = c(6, 1)), grad = FALSE, hessian = FALSE, se = FALSE, ...)
}
\arguments{
\item{data}{An N x d data matrix (with uniform margins).}

\item{RVM}{An \code{\link{RVineMatrix}} object including the structure and
the pair-copula families and parameters (if known).}

\item{start}{Lower triangular d x d matrix with zero diagonal entries with
starting values for the pair-copula parameters (optional; otherwise they are
calculated via \cr \code{\link{RVineSeqEst}}; default: \code{start =
RVM$par}).}

\item{start2}{Lower triangular d x d matrix with zero diagonal entries with
starting values for the second parameters of pair-copula families with two
parameters (optional; otherwise they are calculated via
\code{\link{RVineSeqEst}}; default: \code{start2 = RVM$par2}).}

\item{maxit}{The maximum number of iteration steps (optional; default:
\code{maxit = 200}).}

\item{max.df}{Numeric; upper bound for the estimation of the degrees of
freedom parameter of the t-copula (default: \code{max.df = 30}; for more
details see \code{\link{BiCopEst}}).}

\item{max.BB}{List; upper bounds for the estimation of the two parameters
(in absolute values) of the BB1, BB6, BB7 and BB8 copulas \cr (default:
\code{max.BB = list(BB1=c(5,6),BB6=c(6,6),BB7=c(5,6),BB8=c(6,1))}).}

\item{grad}{If RVM$family only contains one parameter copula families or the
t-copula the analytical gradient can be used for maximization of the
log-likelihood (see \code{\link{RVineGrad}}; default: \code{grad = FALSE}).}

\item{hessian}{Logical; whether the Hessian matrix of parameter estimates is
estimated (default: \code{hessian = FALSE}). Note that this is not the
Hessian Matrix calculated via \code{\link{RVineHessian}} but via finite
differences.}

\item{se}{Logical; whether standard errors of parameter estimates are
estimated on the basis of the Hessian matrix (see above; default: \code{se =
FALSE}).}

\item{...}{Further arguments for \code{optim} (e.g. \code{factr} controls
the convergence of the "L-BFGS-B" method, or \code{trace}, a non-negative
integer, determines if tracing information on the progress of the
optimization is produced.) \cr For more details see the documentation of
\code{\link{optim}}.}
}
\value{
\item{RVM}{\code{\link{RVineMatrix}} object with the calculated
parameters stored in \code{RVM$par} and \code{RVM$par2}. Additional
information about the fit is added (e.g., log-likelihood, AIC, BIC).}
\item{value}{Optimized log-likelihood value corresponding to the estimated
pair-copula parameters.} \item{convergence}{An integer code indicating
either successful convergence (\code{convergence = 0}) or an error:\cr
\code{1} = the iteration limit \code{maxit} has been reached \cr \code{51} =
a warning from the "L-BFGS-B" method; see component \code{message} for
further details \cr \code{52} = an error from the "L-BFGS-B" method; see
component \code{message} for further details} \item{message}{A character
string giving any additional information returned by \code{\link{optim}}, or
\code{NULL}.} \item{counts}{A two-element integer vector giving the number
of calls to \code{fn} and \code{gr} respectively.  This excludes those calls
needed to compute the Hessian, if requested, and any calls to \code{fn} to
compute a finite-difference approximation to the gradient.}
\item{hessian}{If \code{hessian = TRUE}, the Hessian matrix is returned. Its
calculation is on the basis of finite differences (output of \code{optim}).}
}
\description{
This function calculates the maximum likelihood estimate (MLE) of the
R-vine copula model parameters using sequential estimates as initial values
(if not provided).
}
\note{
\code{RVineMLE} uses the L-BFGS-B method for optimization. \cr If the
analytical gradient is used for maximization, computations may be up to 10
times faster than using finite differences.
}
\examples{

# define 5-dimensional R-vine tree structure matrix
Matrix <- c(5, 2, 3, 1, 4,
            0, 2, 3, 4, 1,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 1)
Matrix <- matrix(Matrix, 5, 5)

# define R-vine pair-copula family matrix
family <- c(0, 1, 3, 4, 4,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 3,
            0, 0, 0, 0, 0)
family <- matrix(family, 5, 5)

# define R-vine pair-copula parameter matrix
par <- c(0, 0.2, 0.9, 1.5, 3.9,
         0, 0, 1.1, 1.6, 0.9,
         0, 0, 0, 1.9, 0.5,
         0, 0, 0, 0, 4.8,
         0, 0, 0, 0, 0)
par <- matrix(par, 5, 5)

# define second R-vine pair-copula parameter matrix
par2 <- matrix(0, 5, 5)

# define RVineMatrix object
RVM <- RVineMatrix(Matrix = Matrix, family = family,
                   par = par, par2 = par2,
                   names = c("V1", "V2", "V3", "V4", "V5"))

# simulate a sample of size 300 from the R-vine copula model
set.seed(123)
simdata <- RVineSim(300, RVM)

# compute the MLE
mle <- RVineMLE(simdata, RVM, grad = TRUE, trace = 0)

# compare parameters
round(mle$RVM$par - RVM$par, 2)

}
\references{
Dissmann, J. F., E. C. Brechmann, C. Czado, and D. Kurowicka
(2013). Selecting and estimating regular vine copulae and application to
financial returns. Computational Statistics & Data Analysis, 59 (1), 52-69.

Stoeber, J. and U. Schepsmeier (2013). Estimating standard errors in regular
vine copula models. Computational Statistics, 1-29
\url{http://link.springer.com/article/10.1007/s00180-013-0423-8#}.
}
\seealso{
\code{\link{RVineSeqEst}},
\code{\link{RVineStructureSelect}},
\code{\link{RVineMatrix}},
\code{\link{RVineGrad}},
\code{\link{RVineHessian}}
}
\author{
Ulf Schepsmeier, Jeffrey Dissmann
}
