% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{WH_2d}
\alias{WH_2d}
\title{2D Whittaker-Henderson Smoothing}
\usage{
WH_2d(
  d,
  ec,
  lambda,
  criterion,
  method,
  max_dim = 200,
  p,
  q = c(2, 2),
  framework,
  y,
  wt,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{d}{Matrix of observed events, whose rows and columns must be named.}

\item{ec}{Matrix of central exposure. The central exposure corresponds to the
sum of the exposure duration over the insured population. An individual
experiencing an event of interest during the year will no longer be exposed
afterward and the exposure should be computed accordingly.}

\item{lambda}{Smoothing parameter vector of size \code{2}. If missing, an
optimization procedure will be used to find the optimal smoothing
parameter. If supplied, no optimal smoothing parameter search will take
place unless the \code{method} argument is also supplied, in which case \code{lambda}
will be used as the starting parameter for the optimization procedure.}

\item{criterion}{Criterion to be used for the selection of the optimal
smoothing parameter. Default is \code{"REML"} which stands for restricted
maximum likelihood. Other options include \code{"AIC"}, \code{"BIC"} and \code{"GCV"}.}

\item{method}{Method to be used to find the optimal smoothing parameter.
Default to \code{"fixed_lambda"} if \code{lambda} is supplied, meaning no
optimization is performed. Otherwise, default to \code{"perf"} which means the
faster performance iteration method is used. The alternative \code{"outer"}
method is safer but slower. Both those methods rely on the \code{optim} function
from package \code{stats}.}

\item{max_dim}{Number of parameters to be kept in the optimization problem.
Default is \code{200}. Values higher than \code{1000} may result in very high
computation times and memory usage.}

\item{p}{Optional vector of size \code{2}. Maximum number of eigenvectors to keep
on each dimension after performing the eigen decomposition of the
penalization matrix. If missing, will be automatically computed so that the
number of elements of the (square) matrices involved in the optimization
problem remains lower that the \code{max_dim} argument}

\item{q}{Order of penalization vector of size \code{2}. Polynoms of degrees
\verb{(q[[1]] - 1,q[[2]] - 1)} are considered smooth and are therefore
unpenalized. Should be left to the default of \code{c(2,2)} for most practical
applications.}

\item{framework}{Default framework is \code{"ml"} which stands for maximum
likelihood unless the \code{y} argument is also provided, in which case an
\code{"reg"} or (approximate) regression framework is used.}

\item{y}{Optional matrix of observations whose rows and columns should be
named. Used only in the regression framework and if the \code{d} and \code{ec}
arguments are missing (otherwise \code{y} is automatically computed from \code{d} and
\code{ec}). May be useful when using Whittaker-Henderson smoothing outside of
the survival analysis framework.}

\item{wt}{Optional matrix of weights. As for the observation vector \code{y}, Used
only in the regression framework and if the \code{d} and \code{ec} arguments are
missing (otherwise \code{wt} is set equal to \code{d}). May be useful when using
Whittaker-Henderson smoothing outside of the survival analysis framework.}

\item{quiet}{Should messages and warnings be silenced ? Default to \code{FALSE},
may be set to \code{TRUE} is the function is called repeatedly.}

\item{...}{Additional parameters passed to the smoothing function called.}
}
\value{
An object of class \code{WH_2d} i.e. a list containing :
\itemize{
\item \code{d} The inputed matrix of observed events (if supplied as input)
\item \code{ec} The inputed matrix of central exposure (if supplied as input)
\item \code{y} The observation matrix, either supplied or computed as y = log(d) - log(ec)
\item \code{wt} The inputed matrix of weights, either supplied or set to \code{d}
\item \code{y_hat} The matrix of values fitted using Whittaker-Henderson smoothing
\item \code{std_y_hat} The matrix of standard deviations associated with the fit
\item \code{res} The matrix of deviance residuals associated with the fit
\item \code{edf_obs} The matrix of effective degrees of freedom associated with each observation
\item \code{edf_par} The matrix of effective degrees of freedom associated with each eigenvector
\item \code{diagnosis} A data.frame with one line containing the sum of effective degrees of freedom
for the model, the deviance of the fit as well as the AIC, BIC, GCV and
REML criteria
\item \code{Psi} The variance-covariance matrix associated with the fit, which is required
for the extrapolation step.
\item \code{lambda} The vector of smoothing parameters used.
\item \code{p} The number of eigenvectors kept on each dimension if the rank reduction method
is used.
\item \code{q} The supplied vector of orders for the penalization matrices.
}
}
\description{
#' Main package function to apply Whittaker-Henderson smoothing in a
bidimensional survival analysis framework. It takes as input a matrix of
observed events and a matrix of associated central exposure, both depending
on two covariates, and build a smooth version of the log-hazard rate.
Smoothing parameters may be supplied or automatically chosen according to a
specific criterion such as \code{"REML"} (the default), \code{"AIC"}, \code{"BIC"} or
\code{"GCV"}. Whittaker-Henderson may be applied in a full maximum likelihood
framework or an approximate gaussian framework. As Whittaker-Henderson
smoothing relies on full-rank smoothers, computation time and memory usage in
the bidimensional case may prove overwhelming and the function integrates a
rank-reduction procedure to avoid such issues.
}
\examples{
keep_age <- which(rowSums(portfolio_LTC$ec) > 5e2)
keep_duration <- which(colSums(portfolio_LTC$ec) > 1e3)

d  <- portfolio_LTC$d[keep_age, keep_duration]
ec <- portfolio_LTC$ec[keep_age, keep_duration]

y <- log(d / ec) # observation vector
y[d == 0] <- - 20
wt <- d

# Maximum likelihood
WH_2d(d, ec, lambda = c(1e2, 1e2))
WH_2d(d, ec) # performance iteration default method
WH_2d(d, ec, method = "outer") # slower but safer outer iteration method
WH_2d(d, ec, criterion = "GCV")
# alternative optimization criteria for smoothing parameter selection

# Regression
WH_2d(y = y, wt = wt, lambda = c(1e2, 1e2)) # regression framework is triggered when y is supplied
WH_2d(d, ec, framework = "reg", lambda = c(1e2, 1e2))
# setting framework = "reg" forces computation of y from d and ec

# Rank reduction
keep_age <- which(rowSums(portfolio_LTC$ec) > 1e2)
keep_duration <- which(colSums(portfolio_LTC$ec) > 1e2)

d  <- portfolio_LTC$d[keep_age, keep_duration]
ec <- portfolio_LTC$ec[keep_age, keep_duration]

prod(dim(d)) # problem dimension is 627 !
WH_2d(d, ec)
# rank-reduction is used to find an approximate solution using 200 parameters

}
