% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\encoding{UTF-8}
\name{wave}
\alias{wave}
\title{Weakly associated vectors sampling}
\usage{
wave(X, pik, bound = 1, tore = FALSE, shift = FALSE,
  toreBound = -1, comment = FALSE)
}
\arguments{
\item{X}{matrix representing the spatial coordinates.}

\item{pik}{vector of the inclusion probabilities. The length should be equal to N.}

\item{bound}{a scalar representing the bound to reach. See Details. Default is 1.}

\item{tore}{an optional logical value, if we are considering the distance on a tore. See Details. Default is \code{TRUE}.}

\item{shift}{an optional logical value, if you would use a shift perturbation. See Details. Default is \code{FALSE}.}

\item{toreBound}{a numeric value that specify the size of the grid. Default is -1.}

\item{comment}{an optional logical value, indicating some informations during the execution. Default is \code{FALSE}.}
}
\value{
A vector of size \eqn{N} with elements equal 0 or 1. The value 1 indicates that the unit is selected while the value 0 is for non-chosen unit.
}
\description{
Select a spread sample from inclusion probabilities using the weakly associated vectors sampling method.
}
\details{
The main idea is derived from the cube method (Devill and Tillé, 2004). At each step, the inclusion probabilities vector \code{pik}
is randomly modified. This modification is carried out in a direction that best preserves the spreading of the sample.

A stratification matrix \eqn{\bf A} is computed from the spatial weights matrix calculated from the function \code{\link{wpik}}.
Depending if \eqn{\bf A} is full rank or not, the vector giving the direction is not selected in the same way.

If matrix \eqn{\bf A} is not full rank, a vector that is contained in the right null space is selected:
\deqn{ Null(\bf A) = \{ \bf x \in R^N | \bf A\bf x = \bf 0  \}. }

If matrix \eqn{\bf A} is full rank, we find \eqn{\bf v}, \eqn{\bf u} the singular vectors associated to the 
smallest singular value \eqn{\sigma } of \eqn{\bf A} such that

\deqn{ \bf A\bf v = \sigma \bf u,~~~~ \bf A^\top \bf u = \sigma \bf v.}

Vector \eqn{ \bf v } is then centered to ensure fixed sample size. At each step, inclusion probabilities is modified and at least on component is set to 0 or 1. Matrix \eqn{\bf A } is updated 
from the new inclusion probabilities. The whole procedure it repeated until it remains only one component that is not equal to 0 or 1.

For more informations on the options \code{tore} and \code{toreBound}, see \code{\link{distUnitk}}. If \code{tore} is set up \code{TRUE} and \code{toreBound} not specified the \code{toreBound} is equal to 
\deqn{N^{1/p}}
where \eqn{p} is equal to the number of column of the matrix \code{X}.

For more informations on the option \code{shift}, see \code{\link{wpik}}.
}
\examples{

#------------
# Example 2D
#------------

N <- 50
n <- 15
X <- as.matrix(cbind(runif(N),runif(N)))
pik <- sampling::inclusionprobabilities(runif(N),n)
s <- wave(X,pik)

#------------
# Example 2D grid 
#------------

N <- 36 # 6 x 6 grid
n <- 12 # number of unit selected
x <- seq(1,sqrt(N),1)
X <- as.matrix(cbind(rep(x,times = sqrt(N)),rep(x,each = sqrt(N))))
pik <- rep(n/N,N)
s <- wave(X,pik, tore = TRUE,shift = FALSE)

#------------
# Example 1D
#------------

N <- 100
n <- 10
X <- as.matrix(seq(1,N,1))
pik <- rep(n/N,N)
s <- wave(X,pik,tore = TRUE,shift =FALSE,comment = TRUE)


}
\references{
Deville, J. C. and Tillé, Y. (2004). Efficient balanced sampling: the cube method. Biometrika, 91(4), 893-912
}
\seealso{
\code{\link{wpik}}, \code{\link{distUnitk}}.
}
\author{
Raphaël Jauslin \email{raphael.jauslin@unine.ch}
}
