#' Counterfactual evaluation
#'
#' @param formula An optional formula without a dependent variable that is of
#' class "formula" and that follows standard R conventions for formulas, e.g.
#' `~ x1 + x2`. Allows you to transform or otherwise re-specify combinations of
#' the variables in both `data` and `cfact`. To use this parameter, both data and
#' `cfact` must be coercable to data frames; the variables of both `data` and
#' `cfact` must be labeled; and all variables appearing in formula must also
#' appear in both `data` and `cfact`. Otherwise, errors are returned. The
#' intercept is automatically dropped. Default is `NULL`.
#' @param data May take one of the following forms:
#' 1. A R model output object, such as the output from calls to `lm`, `glm`, and
#' `zelig`. If it is not a `zelig` object, such an output object must be a `list`.
#' It must additionally have either a `formula` or `terms` component and either
#'  a `data` or `model` component; if it does not, an error is returned. Of the
#'  latter, `whatif` first looks for `data`, which should contain either the
#'  original data set supplied as part of the model call (as in `glm`), which is
#'  assumed to reside in the global environment. If data does not exist,
#'  `whatif` then looks for `model`, which should contain the model frame (as in
#'  `lm`). The intercept is automatically dropped from the extracted observed
#'  covariate data set if the original model included one.
#' 2. A n-by-k non-character (logical or numeric) matrix or data frame of
#' observed covariate data with n data points or units and k covariates. All
#' desired variable transformations and interaction terms should be included in
#' this set of k covariates unless formula is alternatively used to produce
#' them. However, an intercept should not be. Such a matrix may be obtained by
#' passing model output (e.g., output from a call to `lm`) to `model.matrix` and
#' excluding the intercept from the resulting matrix if one was fit. Note that
#' `whatif` will attempt to coerce data frames to their internal numeric values.
#' Hence, data frames should only contain logical, numeric, and factor columns;
#' character columns will lead to an error being returned.
#' 3. A string. Either the complete path (including file name) of the file
#' containing the data or the path relative to your working directory. This file
#' should be a white space delimited text file. If it contains a header, you
#' must include a column of row names as discussed in the help file for the R
#' function `read.table`. The data in the file should be as otherwise described
#' in (2).
#'
#' Missing data is allowed and will be dealt with via the argument missing. It
#' should be flagged using R's standard representation for missing data, `NA`.
#' @param cfact A R object or a string. If a R object, a m-by-k non-character
#' matrix or data frame of counterfactuals with m counterfactuals and the same
#' k covariates (in the same order) as in data. However, if formula is used to
#' select a subset of the k covariates, then `cfact` may contain either only
#' these j <= K covariates or the complete set of k covariates. An intercept
#' should not be included as one of the covariates. It will be automatically
#' dropped from the counterfactuals generated by Zelig if the original model
#' contained one. Data frames will again be coerced to their internal numeric
#' values if possible. If a string, either the complete path (including file
#' name) of the file containing the counterfactuals or the path relative to your
#' working directory. This file should be a white space delimited text file. See
#' the discussion under data for instructions on dealing with a header. All
#' counterfactuals should be fully observed: if you supply counterfactuals with
#' missing data, they will be list-wise deleted and a warning message will be
#' printed to the console.
#' @param range An optional numeric vector of length k, where k is the number of
#' covariates. Each element represents the range of the corresponding covariate
#' for use in calculating Gower distances. Use this argument when covariate data
#' do not represent the population of interest, such as selection by
#' stratification or experimental manipulation. By default, the range of each
#' covariate is calculated from the data (the difference of its maximum and
#' minimum values in the sample), which is appropriate when a simple random
#' sampling design was used. To supply your own range for the kth covariate, set
#' the kth element of the vector equal to the desired range and all other
#' elements equal to `N`A. Default is `NULL`.
#' @param freq An optional numeric vector of any positive length, the elements
#' of which comprise a set of distances. Used in calculating cumulative
#' frequency distributions for the distances of the data points from each
#' counterfactual. For each such distance and counterfactual, the cumulative
#' frequency is the fraction of observed covariate data points with distance to
#' the counterfactual less than or equal to the supplied distance value. The
#' default varies with the distance measure used. When the Gower distance
#' measure is employed, frequencies are calculated for the sequence of Gower
#' distances from 0 to 1 in increments of 0.05. When the Euclidian distance
#' measure is employed, frequencies are calculated for the sequence of Euclidian
#' distances from the minimum to the maximum observed distances in twenty equal
#' increments, all rounded to two decimal places. Default is `NULL`.
#' @param nearby An optional scalar indicating which observed data points are
#' considered to be nearby (i.e., withing 'nearby' geometric variances of) the
#' counterfactuals. Used to calculate the summary statistic returned by the
#' function: the fraction of the observed data nearby each counterfactual. By
#' default, the geometric variance of the covariate data is used. For example,
#' setting nearby to 2 will identify the proportion of data points within two
#' geometric variances of a counterfactual. Default is `NULL`.
#' @param distance An optional string indicating which of two distance measures
#' to employ. The choices are either ``"gower"``, Gower's non-parametric
#' distance measure ($G^2$), which is suitable for both qualitative and
#' quantitative data; or `"euclidian"`, squared Euclidian distance, which is
#' only suitable for quantitative data. The default is `"gower"`.
#' @param miss An optional string indicating the strategy for dealing with
#' missing data in the observed covariate data set. `whatif` supports two
#' possible missing data strategies: `"list"`, list-wise deletion of missing
#' cases; and `"case"`, ignoring missing data case-by-case. Note that if
#' `"case"` is selected, cases with missing values are deleted listwise for the
#' convex hull test and for computing Euclidian distances, but pairwise deletion
#' is used in computing the Gower distances to maximally use available
#' information. The user is strongly encouraged to treat missing data using
#' specialized tools such as Amelia prior to feeding the data to `whatif`.
#' Default is `"list"`.
#' @param choice An optional string indicating which analyses to undertake. The
#' options are either `"hull"`, only perform the convex hull membership test;
#' `"distance"`, do not perform the convex hull test but do everything else,
#' such as calculating the distance between each counterfactual and data point;
#' or `"both"`, undertake both the convex hull test and the distance
#' calculations (i.e., do everything). Default is `"both"`.
#' @param return.inputs A Boolean; should the processed observed covariate and
#' counterfactual data matrices on which all whatif computations are performed
#' be returned? Processing refers to internal `whatif` operations such as the
#' subsetting of covariates via formula, the deletion of cases with missing
#' values, and the coercion of data frames to numeric matrices. Primarily
#' intended for diagnostic purposes. If `TRUE`, these matrices are returned as
#' a list. Default is `FALSE`.
#' @param return.distance A Boolean; should the matrix of distances between each
#' counterfactual and data point be returned? If `TRUE`, this matrix is returned
#' as part of the output; if `FALSE`, it is not. Default is FALSE due to the
#' large size that this matrix may attain.
#' @param ... Further arguments passed to and from other methods.
#'
#' @description Implements the methods described in King and Zeng (2006a, 2006b)
#' for evaluating counterfactuals.
#'
#' @details This function is the primary tool for evaluating your counterfactuals.
#' Specifically, it:
#'
#' 1. Determines whether or not your counterfactuals are in the convex hull of
#' the observed covariate data.
#'
#' 2. Computes the distance of your counterfactuals from each of the n observed
#' covariate data points. The default distance function used is Gower's
#' non-parametric measure.
#'
#' 3. Computes a summary statistic for each counterfactual based on the
#' distances in (2): the fraction of observed covariate data points with
#' distances to your counterfactual less than a value you supply. By default,
#' this value is taken to be the geometric variability of the observed data.
#'
#' 4. Computes the cumulative frequency distribution of each counterfactual for
#' the distances in (2) using values that you supply. By default, Gower
#' distances from 0 to 1 in increments of 0.05 are used.
#'
#' @return   An object of class "whatif", a list consisting of the following
#' six or seven elements:
#'   \item{call}{The original call to \code{whatif}.}
#'   \item{inputs}{A list with two elements, \code{data} and \code{cfact}.  Only
#'     present if \code{return.inputs} was set equal to \code{TRUE} in the call
#'     to \code{whatif}.  The first element is the processed observed
#'     covariate data matrix on which all \code{whatif} computations were
#'     performed.  The second element is the processed counterfactual data
#'     matrix.}
#'   \item{in.hull}{A logical vector of length \eqn{m}, where \eqn{m} is the number
#'     of counterfactuals.  Each element of the vector is \code{TRUE} if the corresponding
#'     counterfactual is in the convex hull and \code{FALSE} otherwise.}
#' \item{dist}{A \eqn{m}-by-\eqn{n} numeric matrix, where \eqn{m} is
#'     the number of counterfactuals and \eqn{n} is the number of data points
#'     (units).  Only present if \code{return.distance} was set equal to \code{TRUE}
#'   in the call to \code{whatif}.  The \eqn{[i, j]}th entry of the matrix contains the
#'     distance between the \eqn{i}th counterfactual and the \eqn{j}th data point.}
#'   \item{geom.var}{A scalar.  The geometric variability of the observed covariate
#'     data.}
#'   \item{sum.stat}{A numeric vector of length \eqn{m}, where \eqn{m} is the
#'     number of counterfactuals.   The \eqn{m}th element contains the summary
#'     statistic for the corresponding counterfactual.  This summary statistic is
#'     the fraction of data points with distances to the counterfactual
#'     less than the argument \code{nearby}, which by default is the geometric
#'     variability of the covariates.}
#'   \item{cum.freq}{A numeric matrix.  By default, the matrix has
#'     dimension \eqn{m}-by-21, where \eqn{m} is the number of
#'     counterfactuals; however, if you supplied your own frequencies via
#'     the argument \code{freq}, the matrix has dimension \eqn{m}-by-\eqn{f},
#'     where \eqn{f} is the length of \code{freq}.  Each row of the
#'     matrix contains the cumulative frequency distribution for the
#'     corresponding counterfactual calculated using either the distance
#'     measure-specific default set of distance values or the set that you supplied (see
#'     the discussion under the argument \code{freq}).  Hence, the \eqn{[i, j]}th
#'     entry of the matrix is the fraction of data points with
#'     distances to the \eqn{i}th counterfactual less than or equal to the
#'     value represented by the \eqn{j}th column.  The column names contain these
#'     values.}
#'
#' @examples
#' ##  Create example data sets and counterfactuals
#' my.cfact <- matrix(rnorm(3 * 5), ncol = 5)
#' my.data <- matrix(rnorm(100 * 5), ncol = 5)
#'
#' ##  Evaluate counterfactuals
#' my.result <- whatif(data = my.data, cfact = my.cfact)
#'
#' ##  Evaluate counterfactuals and supply own gower distances for
#' ##  cumulative frequency distributions
#' my.result <- whatif(cfact = my.cfact, data = my.data,
#'                     freq = c(0, .25, .5, 1, 1.25, 1.5))
#'
#' @source King, Gary and Langche Zeng.  2006.  "The Dangers of Extreme
#' Counterfactuals."  \emph{Political Analysis} 14 (2). Available from
#' <http://gking.harvard.edu>.
#'
#' King, Gary and Langche Zeng.  2007.  "When Can History Be Our Guide?
#' The Pitfalls of Counterfactual Inference."  `International Studies Quarterly`
#' 51 (March).  Available from <http://gking.harvard.edu>.
#'
#' @seealso   \code{\link{plot.whatif}},
#' \code{\link{summary.whatif}},
#' \code{\link{print.whatif}},
#' \code{\link{print.summary.whatif}}
#'
#' @md
#' @export



whatif <- function(formula = NULL, data, cfact, range = NULL, freq = NULL,
                   nearby = 1,  distance = "gower", miss = "list",
                   choice= "both", return.inputs = FALSE,
                   return.distance = FALSE, ...)  {

    #DATA PROCESSING AND RELATED USER INPUT ERROR CHECKING
    #Initial processing of cfact
  print("Preprocessing data ...")

    if (grepl('Zelig*', class(data)) & missing(cfact))
        cfact <- zelig_setx_to_df(data)
    if (grepl('Zelig*', class(data)) & !missing(cfact)) {
        formula <- formula(delete.response(terms(data$formula)))
        data <- data$zelig.out$z.out[[1]]$model
    }

  if(!((is.character(cfact) && is.vector(cfact) && length(cfact) == 1) ||
       is.data.frame(cfact) || (is.matrix(cfact) && !is.character(cfact)))) {
    stop("'cfact' must be either a string, a R data frame, or a R non-character matrix")
  }
  if (is.character(cfact))  {
    cfact <- read.table(cfact)
  }
  if (dim(cfact)[1] == 0)  {
    stop("no counterfactuals supplied: 'cfact' contains zero rows")
  }
  if (!any(complete.cases(cfact)))  {
    stop("there are no cases in 'cfact' without missing values")
  }
  if ("(Intercept)" %in% dimnames(cfact)[[2]])  {
    cfact <- cfact[, -(which(dimnames(cfact)[[2]] == "(Intercept)"))]
  }
    #Initial processing of data
  if (is.list(data) && !(is.data.frame(data)))  {
    if (!((("formula" %in% names(data)) || ("terms" %in% names(data))) &&
          (("data" %in% names(data)) || ("model" %in% names(data)))))  {
      stop("the list supplied to 'data' is not a valid output object")
    }
    tt <- terms(data)
    attr(tt, "intercept") <- rep(0, length(attr(tt, "intercept")))
    if ("data" %in% names(data))  {
      if (is.data.frame(data$data))  {
        data <- model.matrix(tt, model.frame(tt, data = data$data, na.action = NULL))
      }  else  {
        data <- model.matrix(tt, model.frame(tt, data = eval(data$data, envir = .GlobalEnv), na.action = NULL))
      }
    }  else  {
      data <- model.matrix(tt, data = data$model)
    }
    if (!(is.matrix(data)))  {
      stop("observed covariate data could not be extracted from output object")
    }
    rm(tt)
  } else {
    if(!((is.character(data) && is.vector(data) && length(data) == 1) ||
         is.data.frame(data) || (is.matrix(data) && !is.character(data)))) {
      stop("'data' must be either a string, a R data frame, a R non-character matrix, or an output object")
    }
    if (is.character(data))  {
      data <- read.table(data)
    }
  }
  if (dim(data)[1] == 0)  {
    stop("no observed covariate data supplied: 'data' contains zero rows")
  }
  if (!any(complete.cases(data)))  {
    stop("there are no cases in 'data' without missing values")
  }
  #Secondary processing of data and cfact: use formula
  if (!(is.null(formula)))  {
    if (identical(class(formula), "formula"))  {
      if (!(is.data.frame(as.data.frame(data))))  {
        stop("'data' must be coercable to a data frame in order to use 'formula'")
      }
      if (!(is.data.frame(as.data.frame(cfact))))  {
        stop("'cfact' must be coercable to a data frame in order to use 'formula'")
      }
      formula <- update.formula(formula, ~ . -1)
      ttvar <- all.vars(formula)
      for (i in 1:length(ttvar))  {
        if (!(ttvar[i] %in% dimnames(data)[[2]])){
          stop("variable(s) in 'formula' either unlabeled or not present in 'data'")
        }
        if (!(ttvar[i] %in% dimnames(cfact)[[2]])){
          stop("variable(s) in 'formula' either unlabeled or not present in 'cfact'")
        }
      }
      rm(ttvar)
      data <- model.matrix(formula, data = model.frame(formula, as.data.frame(data),
                                      na.action = NULL))
      cfact <- model.matrix(formula, data = model.frame(formula, as.data.frame(cfact),
                                       na.action = NULL))
    } else {
      stop("'formula' must be of class 'formula'")
    }
  }
  if (!(identical(complete.cases(cfact), rep(TRUE, dim(cfact)[1])))) {
    cfact <- na.omit(cfact)
    print("Note:  counterfactuals with missing values eliminated from cfact")
  }
    #Tertiary processing of data and cfact:  convert to numeric matrices
  if (is.data.frame(data))  {
    if (is.character(as.matrix(data)))  {
      stop("observed covariate data not coercable to numeric matrix due to character column(s)")
    }
    data <- suppressWarnings(data.matrix(data))
  }  else  {
    data <- data.matrix(as.data.frame(data))
  }
  if (is.data.frame(cfact))  {
    if (is.character(as.matrix(cfact)))  {
      stop("counterfactual data not coercable to numeric matrix due to character column(s)")
    }
    cfact <- suppressWarnings(data.matrix(cfact))
  }  else  {
    cfact <- data.matrix(as.data.frame(cfact))
  }
   #Final checks on data and cfact
  if (!(is.matrix(data) && is.numeric(data)))  {
    stop("observed covariate data not coercable to numeric matrix")
  }
  if (!(is.matrix(cfact) && is.numeric(cfact)))  {
    stop("counterfactual data not coercable to numeric matrix")
  }
  na.fail(cfact)

    #NON DATA-PROCESSING RELATED USER INPUT ERROR CHECKING
    #Check if cfact, data have the same number of dimensions, k
  if (!identical(ncol(cfact), ncol(data)))  {
    stop("number of columns of 'cfact' and 'data' are not equal")
  }
    #Check format of range if user supplies an argument
  if (!(is.null(range)))  {
    if (!(is.vector(range) && is.numeric(range)))  {
      stop("'range' must be a numeric vector")
    }
    if (!identical(length(range), ncol(data)))  {
      stop("length of 'range' does not equal number of columns of 'data'")
    }
  }
   #Check format of freq if user supplies an argument
  if (!(is.null(freq)))  {
    if (!(is.vector(freq) && is.numeric(freq)))  {
      stop("'freq' must be a numeric vector")
    }
    na.fail(freq)
  }
    #Check if nearby argument is numeric, a scalar, and >= 0, if supplied
  if (!(is.null(nearby))) {
    if (!(is.numeric(nearby) && is.vector(nearby) && length(nearby) == 1 &&
          nearby >= 0))  {
      stop("'nearby' must be numeric, greater than or equal to 0, and a scalar")
    }
  }
    #Check if miss argument is valid
  if (!(identical(miss, "list") || identical(miss, "case")))  {
    stop("'miss' must be either ''case'' or ''list''")
  }

    #Check if distance argument is valid
  if (!(identical(distance, "gower") || identical(distance, "euclidian")))  {
    stop("'distance' must be either ''gower'' or ''euclidian''")
  }

    #Check if choice argument is valid
  if (!(identical(choice, "both") || identical(choice, "hull") || identical(choice, "distance")))  {
    stop("'choice' must be either ''both'', ''hull'', or ''distance''")
  }

   #Check if return.distance argument is valid
  if (!(is.logical(return.inputs)))  {
    stop("'return.inputs' must be logical, i.e. either TRUE or FALSE")
  }

   #Check if return.distance argument is valid
  if (!(is.logical(return.distance)))  {
    stop("'return.distance' must be logical, i.e. either TRUE or FALSE")
  }

    #KEY LOCAL VARIABLES
  n = nrow(data)  #Number of data points in observed data set (initially including missing)

    #LOCAL FUNCTIONS
  convex.hull.test <- function(x, z)  {

  #Create objects required by lp function, adding a row of 1s to
  #transposed matrix s and a 1 to counterfactual vector z[m,].  Note that "A" here
  #corresponds to "A'" in King and Zeng 2006, Appendix A, and "B" and
  #"C" to their "B" and "C", respectively.

    n <- nrow(x)
    k <- ncol(x)
    m <- nrow(z)

    A <- rbind(t(x), rep(1, n))
    C <- c(rep(0, n))
    D <- c(rep("=", k + 1))

    hull = rep(0,m)

    for (i in 1:m)  {
      B <- c(z[i,],1)
      lp.result <- lp(objective.in=C, const.mat=A, const.dir=D, const.rhs=B)
      if (lp.result$status==0)
        hull[i]<-1
    }
    hull <- as.logical(hull)
    return(hull)
  }

  calc.gd <- function(dat, cf, range) {
      #If range =  0 for a variable k, set the normalized difference
      #equal to 0 if, for a given observed data point p, its
      #kth element minus the kth element of the counterfactual is 0.
      #Otherwise set equal to NA, thus ignoring the contribution of the kth
      #variable to the calculation of Gower's distance.
      #Note that an element of the range vector should only be 0 in degenerate
      #cases.
    n<-nrow(dat)
    m<-nrow(cf)
    dat=t(dat)
    dist=matrix(0,m,n,dimnames=list(1:m,1:n))
    for (i in 1:m) {
      temp<-abs(dat-cf[i,])/range
      if (any(range==0)) {
        temp[is.nan(temp)]<-0
        temp[temp==Inf]<-NA
      }
      dist[i,]<-colMeans(temp,na.rm=T)
    }
    return(t(dist))
   }

  calc.ed <- function(dat, cf)  {
    n<-nrow(dat)
    m<-nrow(cf)
    dat<-t(dat)
    dist=matrix(0,m,n,dimnames=list(1:m,1:n))
    for (i in 1:m) {
      temp<-(dat-cf[i,])^2
      dist[i,]<-(colSums(temp))
    }
      return(t(dist))
    }

  geom.var <- function(dat,rang) {
    n <- nrow(dat)
    dat<-t(dat)
    ff<-function(x){
      temp<-abs(dat-x)/rang
      if (any(rang==0)) {
        temp[is.nan(temp)]<-0
        temp[temp==Inf]<-NA
      }
      tmp<-sum(colMeans(temp,na.rm=TRUE) )
      return(tmp)
    }
    sum.gd.x<-sum(apply(dat,2,ff),na.rm=TRUE)
    gv.x <- (0.5 * sum.gd.x)/(n^2)
    return (gv.x)
  }

  calc.cumfreq <- function(freq, dist)  {
    m<-length(freq)
    n<-ncol(dist)
    res<-matrix(0,n,m)
    for(i in 1:m)
      res[,i]<-(colSums(dist <= freq[i]))/nrow(dist)
    return(res)
  }

    #MISSING DATA
if (identical(miss, "list"))  {
  data <- na.omit(data)
  n <- nrow(data)
}

    #CONVEX HULL TEST
if ((choice=="both")|(choice=="hull"))  {
  print("Performing convex hull test ...")
  test.result <- convex.hull.test(x = na.omit(data), z = cfact)
}

    #CALCULATE DISTANCE

if ((choice=="both")|(choice=="distance"))  {
  print("Calculating distances ....")
  if (identical(distance, "gower"))  {
    samp.range <- apply(data, 2, max, na.rm = TRUE) - apply(data, 2, min, na.rm = TRUE)
    if(!is.null(range)) {
      w<-which(!is.na(range))
      samp.range[w]<-range[w]
    }
    if (identical(TRUE, any(samp.range == 0)))  {
      print("Note:  range of at least one variable equals zero")
    }
    dist <- calc.gd(dat = data, cf = cfact, range=samp.range)
  }  else {
    dist <- calc.ed(dat = na.omit(data), cf = cfact)
  }

     #GEOMETRIC VARIANCE
  print("Calculating the geometric variance...")
  if (identical(distance, "gower"))  {
    gv.x <- geom.var(dat = data,rang = samp.range)
  }  else {
    gv.x<-.5*mean(calc.ed(dat = na.omit(data), cf = na.omit(data)))
  }

    #SUMMARY STATISTIC
  if (identical(miss, "case") && identical(distance, "euclidian"))  {
    summary <- colSums(dist <= nearby*gv.x) * (1/nrow(na.omit(data)))
  }  else  {
    summary <- colSums(dist <= nearby*gv.x) * (1/n)
  }

    #CUMULATIVE FREQUENCIES
  print("Calculating cumulative frequencies ...")
   if (is.null(freq))  {
     if (identical(distance, "gower"))  {
       freqdist <- seq(0, 1, by = 0.05)
     }  else {
       min.ed <- min(dist)
       max.ed <- max(dist)
       freqdist <- round(seq(min.ed, max.ed, by = (max.ed - min.ed)/20), 2)
     }
  } else {
    freqdist <- freq
  }
  cumfreq <- calc.cumfreq(freq = freqdist, dist = dist)
  dimnames(cumfreq) <- list(seq(1, nrow(cfact), by = 1), freqdist)
}

print("Finishing up ...")

  #RETURN

if (return.inputs)  {
  if (choice=="both")  {
    if (return.distance)  {
      out <- list(call = match.call(), inputs = list(data = data, cfact = cfact), in.hull = test.result, dist = t(dist), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }  else  {
      out <- list(call = match.call(), inputs = list(data = data, cfact = cfact), in.hull = test.result, geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }
  }

  if (choice=="distance")  {
    if (return.distance)  {
      out <- list(call = match.call(), inputs = list(data = data, cfact = cfact), dist = t(dist), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }  else {
      out <- list(call = match.call(), inputs = list(data = data, cfact = cfact), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }
  }

  if (choice=="hull") {
      out <- list(call = match.call(), inputs = list(data = data, cfact = cfact), in.hull = test.result)
  }

}  else  {
  if (choice=="both")  {
    if (return.distance)  {
      out <- list(call = match.call(), in.hull = test.result, dist = t(dist), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }  else {
      out <- list(call = match.call(), in.hull = test.result, geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }
  }

  if (choice=="distance")  {
    if (return.distance)  {
      out <- list(call = match.call(), dist = t(dist), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }  else {
      out <- list(call = match.call(), geom.var = gv.x, sum.stat = summary, cum.freq = cumfreq)
    }
  }

  if (choice=="hull")  {
      out <- list(call = match.call(), in.hull = test.result)
  }

}

  class(out) <- "whatif"
  return(invisible(out))
}
