% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregation-trees.R
\name{build_aggtree}
\alias{build_aggtree}
\alias{inference_aggtree}
\title{Aggregation Trees}
\usage{
build_aggtree(
  y,
  D,
  X,
  honest_frac = 0.5,
  method = "aipw",
  scores = NULL,
  cates = NULL,
  is_honest = NULL,
  ...
)

inference_aggtree(object, n_groups, boot_ci = FALSE, boot_R = 2000)
}
\arguments{
\item{y}{Outcome vector.}

\item{D}{Treatment vector.}

\item{X}{Covariate matrix (no intercept).}

\item{honest_frac}{Fraction of observations to be allocated to honest sample.}

\item{method}{Either \code{"raw"} or \code{"aipw"}, controls how node predictions are computed.}

\item{scores}{Optional, vector of scores to be used in computing node predictions. Useful to save computational time if scores have already been estimated. Ignored if \code{method == "raw"}.}

\item{cates}{Optional, estimated CATEs. If not provided by the user, CATEs are estimated internally via a \code{\link[grf]{causal_forest}}.}

\item{is_honest}{Logical vector denoting which observations belong to the honest sample. Required only if the \code{cates} argument is used.}

\item{...}{Further arguments from \code{\link[rpart]{rpart.control}}.}

\item{object}{An \code{aggTrees} object.}

\item{n_groups}{Number of desired groups.}

\item{boot_ci}{Logical, whether to compute bootstrap confidence intervals.}

\item{boot_R}{Number of bootstrap replications. Ignored if \code{boot_ci == FALSE}.}
}
\value{
\code{\link{build_aggtree}} returns an \code{aggTrees} object.\cr

\code{\link{inference_aggtree}} returns an \code{aggTrees.inference} object, which in turn contains the \code{aggTrees} object used
in the call.
}
\description{
Nonparametric data-driven approach to discovering heterogeneous subgroups in a selection-on-observables framework.
The approach constructs a sequence of groupings, one for each level of granularity. Groupings are nested and
feature an optimality property. For each grouping, we obtain point estimation and standard errors for the group average
treatment effects (GATEs). Additionally, we assess whether systematic heterogeneity is found by testing the hypotheses
that the differences in the GATEs across all pairs of groups are zero. Finally, we investigate the driving mechanisms of
effect heterogeneity by computing the average characteristics of units in each group.
}
\details{
Aggregation trees are a three-step procedure. First, the conditional average treatment effects (CATEs) are estimated using any
estimator. Second, a tree is grown to approximate the CATEs. Third, the tree is pruned to derive a nested sequence of optimal
groupings, one for each granularity level. For each level of granularity, we can obtain point estimation and inference about
the GATEs.\cr

To implement this methodology, the user can rely on two core functions that handle the various steps.\cr
\subsection{Constructing the Sequence of Groupings}{

\code{\link{build_aggtree}} constructs the sequence of groupings (i.e., the tree) and estimate the GATEs in each node. The
GATEs can be estimated in several ways. This is controlled by the \code{method} argument. If \code{method == "raw"}, we
compute the difference in mean outcomes between treated and control observations in each node. This is an unbiased estimator
in randomized experiment. If \code{method == "aipw"}, we construct doubly-robust scores and average them in each node. This
is unbiased also in observational studies. Honest regression forests and 5-fold cross fitting are used to estimate the
propensity score and the conditional mean function of the outcome (unless the user specifies the argument \code{scores}).\cr

The user can provide a vector of the estimated CATEs via the \code{cates} argument. If so, the user needs to specify a logical
vector to denote which observations belong to the honest sample. If honesty is not desired, \code{is_honest} must be a
vector of \code{FALSE}s. If no vector of CATEs is provided, these are estimated internally via a
\code{\link[grf]{causal_forest}}.\cr
}

\subsection{GATEs Estimation and Inference}{

\code{\link{inference_aggtree}} takes as input an \code{aggTrees} object constructed by \code{\link{build_aggtree}}. Then, for
the desired granularity level, chosen via the \code{n_groups} argument, it provides point estimation and standard errors for
the GATEs. Additionally, it performs some hypothesis testing to assess whether we find systematic heterogeneity and computes
the average characteristics of the units in each group to investigate the driving mechanisms.
\subsection{Point estimates and standard errors for the GATEs}{

GATEs and their standard errors are obtained by fitting an appropriate linear model. If \code{method == "raw"}, we estimate
via OLS the following:

\deqn{Y_i = \sum_{l = 1}^{|T|} L_{i, l} \gamma_l + \sum_{l = 1}^{|T|} L_{i, l} D_i \beta_l + \epsilon_i}

with \code{L_{i, l}} a dummy variable equal to one if the i-th unit falls in the l-th group, and |T| the
number of groups. If the treatment is randomly assigned, one can show that the betas identify the GATE of
each group. However, this is not true in observational studies due to selection into treatment. In this case, the user is
expected to use \code{method == "aipw"} when calling \code{\link{build_aggtree}}. In this case,
\code{\link{inference_aggtree}} uses the scores in the following regression:

\deqn{score_i = \sum_{l = 1}^{|T|} L_{i, l} \beta_l + \epsilon_i}

This way, betas again identify the GATEs.\cr

Regardless of \code{method}, standard errors are estimated via the Eicker-Huber-White estimator.\cr

If \code{boot_ci == TRUE}, the routine also computes asymmetric bias-corrected and accelerated 95\% confidence intervals using 2000 bootstrap
samples. Particularly useful when the honest sample is small-ish.
}

\subsection{Hypothesis testing}{

\code{\link{inference_aggtree}} uses the standard errors obtained by fitting the linear models above to test the hypotheses
that the GATEs are different across all pairs of leaves. Here, we adjust p-values to account for multiple hypotheses testing
using Holm's procedure.
}

\subsection{Average Characteristics}{

\code{\link{inference_aggtree}} regresses each covariate on a set of dummies denoting group membership. This way, we get the
average characteristics of units in each leaf, together with a standard error. Leaves are ordered in increasing order of their
predictions (from most negative to most positive). Standard errors are estimated via the Eicker-Huber-White estimator.
}

}

\subsection{Caution on Inference}{

Regardless of the chosen \code{method}, both functions estimate the GATEs, the linear models, and the average characteristics
of units in each group using only observations in the honest sample. If the honest sample is empty (this happens because the
user either sets \code{honest_frac = 0} or passes a vector of \code{FALSE}s as \code{is_honest} when calling
\code{\link{build_aggtree}}), the same data used to construct the tree are used to estimate the above quantities. This is
fine for prediction but invalidates inference.
}
}
\examples{
\donttest{## Generate data.
set.seed(1986)

n <- 1000
k <- 3

X <- matrix(rnorm(n * k), ncol = k)
colnames(X) <- paste0("x", seq_len(k))
D <- rbinom(n, size = 1, prob = 0.5)
mu0 <- 0.5 * X[, 1]
mu1 <- 0.5 * X[, 1] + X[, 2]
y <- mu0 + D * (mu1 - mu0) + rnorm(n)

## Construct sequence of groupings. CATEs estimated internally.
groupings <- build_aggtree(y, D, X, method = "aipw")

## Alternatively, we can estimate the CATEs and pass them.
splits <- sample_split(length(y), training_frac = 0.5)
training_idx <- splits$training_idx
honest_idx <- splits$honest_idx

y_tr <- y[training_idx]
D_tr <- D[training_idx]
X_tr <- X[training_idx, ]

y_hon <- y[honest_idx]
D_hon <- D[honest_idx]
X_hon <- X[honest_idx, ]

library(grf)
forest <- causal_forest(X_tr, y_tr, D_tr) # Use training sample.
cates <- predict(forest, X)$predictions

groupings <- build_aggtree(y, D, X, method = "aipw", cates = cates,
                           is_honest = 1:length(y) \%in\% honest_idx)

## We have compatibility with generic S3-methods.
summary(groupings)
print(groupings)
plot(groupings) # Try also setting 'sequence = TRUE'.

## To predict, do the following.
tree <- subtree(groupings$tree, cv = TRUE) # Select by cross-validation.
head(predict(tree, data.frame(X)))

## Inference with 4 groups.
results <- inference_aggtree(groupings, n_groups = 4)

summary(results$model) # Coefficient of leafk is GATE in k-th leaf.

results$gates_diff_pairs$gates_diff # GATEs differences.
results$gates_diff_pairs$holm_pvalues # leaves 1-2 not statistically different.

## LATEX.
print(results, table = "diff")
print(results, table = "avg_char")}

}
\references{
\itemize{
\item R Di Francesco (2022). Aggregation Trees. CEIS Research Paper, 546. \doi{10.2139/ssrn.4304256}.
}
}
\seealso{
\code{\link{plot.aggTrees}} \code{\link{print.aggTrees.inference}}
}
\author{
Riccardo Di Francesco
}
